/*
 * (c) 2018-2021 Wolfgang Scherr, OE8WOZ, http://www.oe8woz.at,
 * all rights reserved - use at your own risk for ham radio purposes.
 *
 * This source code is released for the "Funkamateur" magazine, to be
 * found in the download area. See also https://www.funkamateur.de/
 *
 * If you find this project useful, he would be happy about a "thanks"
 * via email:   mail <at> oe8woz.at    (replace " <at> " by "@")
 *
 * LICENSE AND DISCLAIMER AT THE END OF THIS FILE.
 *
 * GenSeqV2.c - Use with Atmel Studio V7, may also work with WinAVR.
 * Setup a project with ATTINY13A target and use this file as "main.c".
 *
 * Created: 26.09.2018 21:02:10
 * Last updated: 02.02.2021 21:02:10
 * Author : oe8woz
 * Version: V2 - for GenSEQ SMD v1.3  (released in FA 03/21)
 *
 *  ATTINY13A to PCB pin out
 * --------------------------
 *
 *  =====================================================================
 *  uC# ... uC dir ... definition                  ... PCB # ... name
 *  =====================================================================
 *                     12V                         ... Pin 1
 *                     GND                         ... Pin 7
 *  =====================================================================
 *	PB0 ... OUTPUT ... to driver for REL2          ... Pin 6 ... P_REL2
 *	PB1 ... OUTPUT ... to driver for REL1          ... Pin 5 ... P_REL1
 *	PB2 ... OUTPUT ... to driver for PA (inverted) ... Pin 2 ... P_PA_N
 *  PB3 ... I/O    ... to SEL (with pull-up)       ... Pin 8 ... P_SEL
 *  PB4 ... I/O    ... to CTRL_IN (opt. pull-up)   ... Pin 4 ... P_CTRL
 *                     to driver of CTRL_OUT       ... Pin 3
 *  =====================================================================
 *  PB5 ... reset  ... to enable of drivers
 *  VCC ...            5V
 *  GND ...            GND
 *  =====================================================================
 */ 
#define F_CPU 1200000UL		// 9.6MHz/8 = 1.2MHz  (internal RC OSC)

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <avr/eeprom.h>
#include <avr/wdt.h>
#include <avr/fuse.h>

#include <util/delay.h>

// basic HW definitions
#define P_REL2 PB0
#define P_REL1 PB1
#define P_PA_N PB2
#define P_SEL  PB3
#define P_CTRL PB4

// helper macros
#define CI_STATE (PINB >> P_CTRL) & 1
#define SEL_STATE (PINB >> P_SEL) & 1

#define SET_REL1 PORTB = (PORTB & (~_BV(P_REL2))) | _BV(P_PA_N) | _BV(P_REL1)
#define SET_REL2 PORTB = (PORTB & (~_BV(P_REL1))) | _BV(P_PA_N) | _BV(P_REL2)
#define SET_PA   PORTB = (PORTB & (~_BV(P_PA_N)) & (~_BV(P_REL1)) & (~_BV(P_REL2)))
#define CLR      PORTB = (PORTB & (~_BV(P_REL1)) & (~_BV(P_REL2))) | _BV(P_PA_N)

#define SET_REL1i PORTB = (PORTB & (~_BV(P_REL1))) | _BV(P_PA_N) | _BV(P_REL2)
#define SET_REL2i PORTB = (PORTB & (~_BV(P_REL2))) | _BV(P_PA_N) | _BV(P_REL1)
#define SET_PAi   PORTB = (PORTB & (~_BV(P_PA_N))) | _BV(P_REL1) | _BV(P_REL2)
#define CLRi      PORTB = PORTB | _BV(P_PA_N) | _BV(P_REL1) | _BV(P_REL2)

// no difference pre/post PA activation, so SET_REL1f = SET_REL2f (no own macro)
#define SET_REL2f PORTB = (PORTB & (~_BV(P_REL2))) | _BV(P_PA_N) | _BV(P_REL1)
#define SET_PAf   PORTB = (PORTB & (~_BV(P_PA_N)) & (~_BV(P_REL2))) | _BV(P_REL1)
#define CLRf      PORTB = (PORTB & (~_BV(P_REL1))) | _BV(P_PA_N) | _BV(P_REL2)

/* 
 *  FUSE bytes - explanations
 *  =====================================================================
 *
 *                 S DBBR
 *                 P WDDS
 *                 E E10D
 * High byte:   1111 1011   0xFB	(3V BOD)
 *
 *              SEWD SSCC
 *              PEDV UUKK
 *              ISO8 TTSS
 *              EVNE 1010
 * Low byte:    0110 0110   0x66   (9,6MHz/8, 14ck+4ms, EEP preserve)
 *
 *  =====================================================================
 */
FUSES = {
	.low = (FUSE_SPIEN & FUSE_CKDIV8 & FUSE_SUT1 & FUSE_CKSEL0),	// serial programming, 
	                                                                // 9.6MHZ/8, 20CK+4ms startup
	.high = (FUSE_BODLEVEL1),										// 3V brownout
};

// EEPROM variables
uint8_t  EEMEM eep_invert = 0;		// ADDRESS 2 (3x button switch in conf.)	
									// 1...inverted / 0...non-inverted
uint8_t  EEMEM eep_failsafe = 0;	// ADDRESS 1 (2x button switch in conf.)
									// 1...failsafe / 0...latching
uint8_t  EEMEM eep_time = 0;		// ADDRESS 0 (1x button switch in conf.)
									// 1...160ms / 0...20ms   (may be adapted, see "default_time")

// sequencer variables (state machine)
typedef enum { rx, rel2, tx, cool, rel1 } seqstate_t;
seqstate_t seqstate = rx;
uint8_t invert = 0;
uint8_t failsafe = 0;
uint16_t base_time = 0;	

// constants
const uint16_t default_time = 585;	// =~20ms of course "only" with the AVR OSC accuracy
									// this means: 29LSB maps to about 1 millisecond

// prototype - EEPROM change
void set_config(void);

// watchdog disable (after soft reset)
void wdt_init(void) __attribute__((naked)) __attribute__((section(".init3")));
void wdt_init(void)
{
	MCUSR = 0;
	wdt_disable();
	return;
}

/*  SETUP AND MAIN LOOP
 *  =====================================================================
 *
 *  Here we initialize everything and check if we need to enter config
 *  mode (see below, separate routine). Otherwise we go in an endless
 *  loop and wait for a pin change interrupt to handle output switching.
 *
 *  =====================================================================
 */
int main(void)
{
	/* setup I/O */
	PORTB = 0b00000000;   // set all pins to LOW
	PORTB |= _BV(P_PA_N); // set PA pin HIGH (disable)
	PORTB |= _BV(P_SEL);  // set SEL pin high (pull up with weak external pull-down or driven - see below)
	//PORTB |= _BV(P_CTRL); // set CTRL pin via pull-up (not good with series resistor in place!)
	invert = eeprom_read_byte(&eep_invert);
	failsafe = eeprom_read_byte(&eep_failsafe);
	if (failsafe) {
		PORTB |= _BV(P_REL2);  // set REL2 pin HIGH
	} else if (invert) {
		PORTB |= _BV(P_REL2);  // set REL2 pin HIGH
		PORTB |= _BV(P_REL1);  // set REL1 pin HIGH
	}
	DDRB = 0b00000000;    // set all pins as INPUT
	// DDRB |= _BV(P_SEL);   // set SEL pin as OUTPUT - this is needed, if a "strong" pull-down (<100kOhm) is used!
							// if you set SEL as output, reprogramming EEP bits is no longer possible !!!
	DDRB |= _BV(P_REL2);  // set REL2 pin as OUTPUT
	DDRB |= _BV(P_REL1);  // set REL1 pin as OUTPUT
	DDRB |= _BV(P_PA_N);  // set PA pin as OUTPUT

	/* setup delay time via EEPROM */
	base_time = default_time << (eep_time<<3);		//  Either "fast" or 8x slower (may be adapted)

	/* initially set latching switch */
	if (!failsafe) {
		/* disable PA and set RF switch to RX */
		if (invert)	SET_REL1i; else SET_REL1;
		for(uint16_t i = 0; i<base_time; ++i) _delay_us(1);
		if (invert)	CLRi; else CLR;
		seqstate=rx;
	}

	/* check we are in config mode 
       (SEL and CI set to GND on start up) */
	if (!(SEL_STATE) & !(CI_STATE)) {
		set_config();
	}

	/* this interrupt will react on CTRL (PTT) changes */
	PCMSK |= (1<<P_CTRL); // pin change interrupt mask
	GIMSK |= _BV(PCIE);   // enable PCINTx
	sei(); // enable global interrupts

	/* done, now we wait on changes */
	set_sleep_mode(SLEEP_MODE_PWR_DOWN);

	/* main loop, just enter sleep if we wake up by an interrupt */
    while (1) 
    {
		sleep_mode();   // go to sleep and wait for interrupt on CTRL (PTT)...
    }
}

/*  PIN CHANGE INTERRPUT:
 *  =====================================================================
 *
 *  Here we actually run the state machine for non-overlapping switching
 *
 *  =====================================================================
 */
ISR(PCINT0_vect)
{	uint8_t pinstat = CI_STATE; // 0...TX, 1...RX
	uint16_t timer=0;	 // us

	// state machine handling PA, REL1 and REL2 outputs
	// we loop here until we reach a stable TX or RX state
	while(1) {
		switch (seqstate) {
			case rx:
				if (failsafe)
					CLRf;
				else if (invert) CLRi; else CLR;
				if (pinstat) {
					return;
				} else { // (!pinstat)
					timer=base_time;
					seqstate=rel2;
				}
				break;
			case rel2:
				if (failsafe)
					SET_REL2f;
				else if (invert) SET_REL2i; else SET_REL2;
				if (pinstat) {
					timer=base_time;
					seqstate=rel1;
				} else { // (!pinstat)
					if (timer) {
						--timer;
					} else { // (!timer)
						seqstate=tx;
					}
				}
				break;
			case tx:
				if (failsafe)
					SET_PAf;
				else if (invert) SET_PAi; else SET_PA;
				if (pinstat) {
					timer=base_time;
					seqstate=cool;
				} else { // (!pinstat)
					return;
				}
				break;
			case cool:
				if (failsafe)
					SET_REL2f;
				else if (invert) CLRi; else CLR;
				if (pinstat) {
					if (timer) {
						--timer;
					} else { // (!timer)
						timer=base_time;
						seqstate=rel1;
					}
				} else { // (!pinstat)
					seqstate=tx;
				}
				break;
			case rel1:
				if (failsafe)
					CLRf;
				else if (invert) SET_REL1i; else SET_REL1;
				if (pinstat) {
					if (timer) {
						--timer;
					} else { // (!timer)
						seqstate=rx;
					}
				} else { // (!pinstat)
					timer=base_time;
					seqstate=rel2;
				}
				break;
		}
		pinstat = CI_STATE;
	}
}

/*
 *  Set new configuration
 *  =====================================================================
 *
 *  Enter config mode:
 *  --------------------------
 *
 *  1) Connect a LED with 4,7k resistor between REL1 and GND
 *     Connect button switch between SEL and GND and CI and GND
 *
 *  2) Press SEL and CI button and power board, REL2 will light up
 *
 *  3) Release buttons, REL2 goes off - go to 4 for config or go to 6
 *
 *  4) Press SEL button
 *	   Check LED on REL1: all 3 bits are shown, one after the other
 *                        long..."1"   short..."0"
 *
 *  5) Keep SEL button pressed, press and release CI button 1x, 2x or 3x 
 *     (depending which bit to toggle)
 *     then release SEL button (REL2 will blink shortly, EEP value is updated)
 *
 *  6) Go back to step 4 for further programming or 
 *     press CI button to reset board.
 *
 *  =====================================================================
 */

// we show the config on SEL
void set_config(void) {
	uint8_t *cnt = 0;

	// show REL2, wait for release
	SET_REL2;
	while (!(SEL_STATE) || !(CI_STATE)) {}
	CLR;

	// programming loop
	while (1) {
		if ((SEL_STATE) && !(CI_STATE)) {
			// PTT low with SEL high -> reset 
			wdt_enable(WDTO_15MS); while(1) {};
		}
		if (!(SEL_STATE) && (CI_STATE)) {
			// SEL pressed -->
			// blink actual EEPROM settings
			for(uint8_t* i=0;i<(uint8_t *)3;++i) {
				uint8_t val = eeprom_read_byte(i);
				SET_REL1;
				if (val) {
					_delay_ms(500);
				} else {
					_delay_ms(100);
				}
				CLR;
				if (val) {
					_delay_ms(200);
				} else {
					_delay_ms(600);
				}
			}
			// wait to allow toogle a bit
			while (!(SEL_STATE) && (CI_STATE)) {}
			while (1) {
				if (!(SEL_STATE) && !(CI_STATE)) {
					// PTT low with SEL low --> count
					++cnt;
					// wait to allow release of SEL or CI
					while (!(SEL_STATE) && !(CI_STATE)) {
						_delay_ms(5);
					}
				}
				// SEL released, store bit (if any count)
				if (SEL_STATE) {
					if (cnt--) {
						SET_REL2;
						eeprom_write_byte(cnt,!eeprom_read_byte(cnt));
						_delay_ms(300);
						CLR;
					}
					cnt=0;
					break;
				}
			}
		}
	}
}

/* LICENSE AND DISCLAIMER:
 * =====================================================================
 *
 * (c) 2018-2021 Wolfgang Scherr, OE8WOZ, http://www.oe8woz.at,
 * all rights reserved - use at your own risk for ham radio purposes.
 *
 * Redistribution and use in source and compiled forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * Redistributions in compiled form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * Neither the name of the author nor the names of other contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 *
 * The code and any derivatives must not be used for commercial purposes
 * without specific prior written permission by the author.
 *
 * THIS CODE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS CODE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * =====================================================================
*/