//********************************************************************************************    
/*FA-Leserservice:	Uhrzeitanzeige mit Drehspulinstrumenten und Arduino nano (FA 09/22)
	Die Firmware steuert drei Drehspulmessinstrumente zur Anzeige der Uhrzeit. Diese wird
  von einem RTC-Modul mit DS3231 zur Verfügung gestellt. Die Steuerung erfolgt mithilfe
  von PWM-Signalen an den Arduino-Pins D9, D10 und D11. 
  
Versionsnummer: 1.0

*/
//******************************************************************************************
/*Anschlussbelegung des Arduino Nano:  
  D7    Jumper zur Einstellung des Vollausschlags
  D8    Jumper zum Stellen der Uhr
  D9    Messwerk Stunde
  D10   Messwerk Minute
  D11   Messwerk Sekunde
  A0    Uhr stellen, Stunde/Vollauschlag
  A1    Uhr stellen, Minute/Messwerkauswahl
  VIN   Betriebsspannungseingang 7...10V
  GND   Masse
 Alle nicht aufgeführten Anschlüsse sind unbeschaltet.
*/
//******************************************************************************************
//Bibliothek einbinden und RTC definieren:
#include <RTClib.h>
RTC_DS3231 rtc; //RTC vom Typ DS3231 am I2C-Anschluss

//Definition der globalen Variablen:
int stunden;
int minuten;
int sekunden;

//Zuordnung der Arduino-Pins:
#define JPVA 7    //Jumper Vollauschlag
#define JPZT 8    //Jumper Uhr stellen
#define MSTD 9    //Stundenanzeige
#define MMIN 10   //Minutenanzeige
#define MSEK 11   //Sekundenanzeige
#define STDE A0   //Analogpin zum Einlesen des Stundenwerts beim Stellen der Uhr
#define MINE A1   //Analogpin zum Einlesen des Minutenwerts beim Stellen der Uhr

//********************************************************************************************
//Setup-Einstellungen:
void setup() 
{
//Ein- und Ausgänge definieren:
pinMode (JPVA, INPUT);
pinMode (JPZT, INPUT);
pinMode (MSTD, OUTPUT);
pinMode (MMIN, OUTPUT);
pinMode (MSEK, OUTPUT);

//Referenzspannung des ADU festelegen:
analogReference (DEFAULT);  //Referenzspannung des ADU ist die Betriebsspannung

//Öffnen der Schnittstelle zum RTC-Modul und kurze Pause
rtc.begin();
delay (200);
}
//********************************************************************************************
//Hauptprogrammschleife:
void loop() 
{
/*Funktionsaufrufe zur Prüfung, ob die Jumper gesteckt sind. Diese Funktionen starten bei offenen 
Steckbrücken das Stellen der Uhr bzw. des Vollauschlags der Instrumente.
*/

vollauschlag ();
uhr_stellen();
  
//RTC auslesen und Zeitwerte als PWM ausgeben:
  DateTime now = rtc.now();

  stunden = now.hour();             //Stunde
  minuten = now.minute();           //Minute
  sekunden = now.second();          //Sekunde
  analogWrite(MSTD,(stunden *11));  //Stunde als PWM-Wert ausgeben
  analogWrite(MMIN,(minuten *4));   //Minute als PWM-Wert ausgeben
  analogWrite(MSEK,(sekunden *4));  //Sekunde als PWM-Wert ausgeben

  delay(500);                      //Pause bis zum nächsten Durchlauf
}
//******************************************************************************************
//Funktionen zum Justieren der Instrumentenanzeige und zum Stellen der Uhr:
void vollauschlag (void) 
{
/*
Jumper J1 gesteckt? Wenn NICHT, werden die jeweiligen PWM-Signale für h=23, min=59, sek=59 
an die Ausgänge gegeben. Damit ist die Einstellung des Endausschlags mittels R1 bis R3 möglich. 
Wenn der J1 wieder gesteckt ist, wird diese Programmschleife verlassen.
*/
while (digitalRead(JPVA)==HIGH) 
  {
  analogWrite(MSTD,253);  //23 Schritte in Stufen zu 11 ergeben 253
  analogWrite(MMIN,236);  //59 Schritte in Stufen zu 4 ergeben 236
  analogWrite(MSEK,236); 
  }
}

void uhr_stellen(void)
{
/*
Jumper J2 gesteckt? Wenn NICHT, wird die Uhr in den Einstellmodus versetzt. Einstellwiderstand R9 
ist für die Stunden und R11 für die Minuten zuständig, Sekunden = 0. 
Nach dem Stecken von J2 wird diese Schleife verlassen und die Variable zst gesetzt. 
Beim Verlassen der Funktion erfolgt die Übertragung der Zeitdaten in den Uhrenschaltkreis. 
Das Datum ist in dieser Anwendung ohne Bedeutung, Füllzahlen: 1.1.2022.
 */  
int std_set;
int min_set;
int zst = 0;
 
while (digitalRead(JPZT)==HIGH) 
  {
   std_set = analogRead(STDE)/40; //Auslesen des Analogwerts vom Pin A0 (STDE) und Teilung durch 40
   min_set = analogRead(MINE)/15; //Auslesen des Analogwerts vom Pin A1 (MINE) und Teilung durch 15
   stunden = constrain(std_set,0,23); //Begrenzung des Wertebereichs auf 0 bis 23
   minuten = constrain(min_set,0,59); //Begrenztung des Wertebereichs auf 0 bis 59
   sekunden = 0;
   analogWrite(MSTD,(stunden *11)); //Ausgabe der Werte als PWM-Signal
   analogWrite(MMIN,(minuten *4));
   analogWrite(MSEK,(sekunden *4));
   zst = 1;
  }
if (zst==1) //Wenn J2 offen war, werden nun die Einstellwerte in das RTC-Modul übertragen
{
  rtc.adjust(DateTime(2022, 1, 1, stunden, minuten, 0));
}
}
//**********************************************************************************************
//Ende des Programms
