//Stromzähler-Programm.
//Zeigt verbrauchten Strom mittels Frequenzzähler an. hier 10.000 Impulse pro kWh.
//Geschrieben von Henry Arndt, DL2TM 3.7.2021

/*Befehle LCD:
  lcd.noDisplay();//AUSschalten
  lcd.display(); EINschalten
  WEB: https://www.arduino.cc/en/Reference/LiquidCrystal

  SD-Card: https://www.arduino.cc/en/Reference/SD

  Beispiel Datenlogger: https://draeger-it.blog/arduino-lektion-27-datenloggen-mit-dem-logging-shield/
  Beispiel Setzen/Anzeige Zeit: https://create.arduino.cc/projecthub/tittiamo68/clock-set-date-time-0d46a4

  DateTime: https://adafruit.github.io/RTClib/html/class_date_time.html#aad62a6709a1e65e6860b1efaf8c347e9

  Für SD-Karte unbedingt die SD-Bibliothek ersetzen. orginal funktiniert wegen der Pin-Belegung zu einem MEGA-Bord nicht.
  Gute Beschreibung wie man es macht findet man hier:
  https://www.az-delivery.de/blogs/azdelivery-blog-fur-arduino-und-raspberry-pi/daten-logger-shield-am-mega-r3

  Berechnung Impulslänge in Leistung:
  http://www.sonnenkiste.de/s0-impulszaehler-berechnung/


  Da Gleitkommaberechnung Schwierigkeiten gemacht hat, wurde die Leistungsberechnung auf Festkomma umgestellt, mit 2 Stellen nach dem Komma.

  Messzeit: Zwischen 5:00 Uhr und 22:00 Uhr. Außerhalb dürfte wohl keine ungepufferte Solaranlage Strom liefern
  Messinterval: 1 Minute über RTC realisiert

  Für die Einstellung der Uhrzeit in dem RTC-Chip ist diese Zeile auszuführen:
  (Kommentar entfernen, Bord programmieren, danach Kommentar wieder aktivieren und Bord erneut programmieren):
  rtc.adjust(DateTime(2021, 7, 12, 19, 13, 0)); //RTC mit Datum setzen (Y,M/D,h,m,s);
*/


// includes the library
//LCD:
#include <LiquidCrystal.h>
//RTC:
#include "RTClib.h"
#include <Wire.h>

// include the SD library für SD-CARD:
#include <SPI.h>
#include <SD.h>


//Variablen
//Für LCD-Anzeige:
const int rs = 8, en = 9, d4 = 4, d5 = 5, d6 = 6, d7 = 7;//Zuordnung der einzelnen Pins
LiquidCrystal lcd(rs, en, d4, d5, d6, d7);//Instanz LCD

//RTC:
RTC_DS1307 rtc;

DateTime now;
DateTime lastminute;//merkt sich die Minute der letzten Messung. Messabstand ist eine Minute. 100 erzwingt sofortigen Start


// set up variables using the SD utility library functions:
Sd2Card card;
File myFile;
SdVolume volume;
//SdFile root;
char Filename[] = "strom.txt";

//Für die Berechung der Leistung:
const long countperkWh = 10000; ///10.000 Impulse pro kWh. Steht so auf dem Zähler.
unsigned long sek;// = (3600*1000) / countperkWh; //Zeit für einen Impuls. 3600 Sekunden hat eine Stunde, * 1000: umwandlung Sekunde in Millisekunden
unsigned long power;
const long messinterval = 10000l; //Zeitraum zwischen den Messungen in ms. Hier 10s.
const long maxwait = 1200000l; //Maximale Zeit in ms die gewartet wird, bis die zweite Flanke eintritt. Hier 120s=2 Minuten

const byte interruptPin = 2;//Festlegung PIN-Nummer, wo der Fototransistor angeschlossen ist.
volatile bool bFlanke = false; //Wird TRUE durch den INTERRUPT. Als Zeichen für den Flankenwechsel. volatile ist ein muss...
unsigned long startzeit, differenz, ziel; //Pufer im Zusammenhang mit der Funktion millis
//int i = 0; //allgemeine Variable
bool bWechselTagNacht = true; //wenn ein Übergang von Tag/Nacht erfolgt, dann True. Dient dazu, dass die LCD-Anzeige nicht ständig überschrieben wird

char buffer[100];//für alles mögliche

//--------->Anfangsroutine:
void setup() {
  Serial.begin(9600); //Serielle Kommunikation mit 9600 Baud beginnen
  //Serial.println("Start");

  //noInterrupts();//INTERRUPTs verbieten

  pinMode(interruptPin, INPUT_PULLUP);//Interrupt-PIN initialisieren

  lcd.begin(16, 2);//INIT LCD
  lcd.setCursor(0, 0); //COL,ROW 0-Basiert
  lcd.print("Stromzaehler");
  delay(1000);
  lcd.clear(); //COL,ROW 0-Basiert

  //RTC-Init:
  Serial.println("RTC-Check");
  lcd.print("RTC: ");

  //rtc.begin();

  if (! rtc.begin())
  {
    lcd.setCursor(5, 0);
    lcd.print("Fehler");
    Serial.println("RTC-Fehler");
    while (1);
  }
    
  lcd.write("OK!");
  
  //--->Einmalig ausführen; wenn Akku vorhanden ist, wird die if-Anweisung nicht mehr ausgeführt...
  //rtc.adjust(DateTime(2021, 7, 12, 19, 13, 0)); //RTC mit Datum setzen (Y,M/D,h,m,s);

  if (! rtc.isrunning())
  {
    rtc.adjust(DateTime(2021, 7, 12, 19, 13, 0)); //RTC mit Datum setzen (Y,M/D,h,m,s)
    //rtc.adjust(DateTime(2021, 7, 4, 15, 58, 0)); //RTC mit Datum setzen (6.7.2021 15:58:00)
    //DateTime (uint16_t year, uint8_t month, uint8_t day, uint8_t hour=0, uint8_t min=0, uint8_t sec=0)
  }


  //Check SD-Card:
  //Serial.println("SD-Karte");
  lcd.setCursor(0, 1); //COL,ROW 0-Basiert
  lcd.write("SD: ");

  // On the Ethernet Shield, CS is pin 4. It's set as an output by default.
  // Note that even if it's not used as the CS pin, the hardware SS pin
  // (10 on most Arduino boards, 53 on the Mega) must be left as an output
  // or the SD library functions will not work.
  pinMode(SS, OUTPUT);

  if (!SD.begin( 10, 11, 12, 13))
  {
    lcd.clear();
    lcd.print("SD-Fehler!");
    Serial.println("initialization failed. Things to check:");
    Serial.println("* is a card is inserted?");
    Serial.println("* Is your wiring correct?");
    Serial.println("* did you change the chipSelect pin to match your shield or module?");
    while (1);
  }
  //SD-Karte geht
  lcd.write("OK!");

  //Falls Datei schon da ist, löschen:
  if (SD.exists(Filename)) {
    SD.remove(Filename);
  }
  myFile = SD.open(Filename, FILE_WRITE);
  if (!myFile) {
    //DateiFehler
    lcd.clear();
    lcd.setCursor(0, 0); //COL,ROW 0-Basiert
    lcd.print("Fehler SD-Card");
    while (1);
  }

  delay(2000);//Für den Benutzer, damit er die letzte Anzeige noch sieht

  //aktuelle zeit anzeigen:
  now = rtc.now();//aktuele Zeit vom RTC holen
  
  //Zeit anzeigen:

  //Datum:
  lcd.setCursor(0, 0); //COL,ROW 0-Basiert
  snprintf(buffer, sizeof(buffer), "%02u.%02u.%02u\0", now.day(), now.month(), now.year());
  lcd.print(buffer);

  //Uhrzeit:
  lcd.setCursor(0, 1); //COL,ROW 0-Basiert
  snprintf(buffer, sizeof(buffer), "%02u:%02u:%02u\0", now.hour(), now.minute(), now.second());
  lcd.print(buffer);
  delay(2000);

  lcd.clear();
  lastminute = now;


  //berechnungswerte initialisieren:
  sek = (3600l * 1000l) / countperkWh; //Zeit für einen Impuls. 3600 Sekunden hat eine Stunde, * 1000: umwandlung Sekunde in Millisekunden


  //INTERRUPT:
  attachInterrupt(digitalPinToInterrupt(interruptPin), Flanke, RISING); //Zuweisen Interrupt-Routine zu PIN und wann INTERRUPT ausgelöst wird (hier Steigende Flanke)

  //Letzter Befehl: Interrupts erlauben:
  interrupts();

}



//--------->Programmschleife:
void loop() {
  long x;

  //Ist es Nacht? Wenn ja dies anzeigen und keine Messung durchführen:
Anfang:
  now = rtc.now();//aktuele Zeit vom RTC holen
  if ((now.hour() > 20) || (now.hour() < 6))
  {
    if (bWechselTagNacht)
    {
      bWechselTagNacht = false;
      lcd.clear();
      lcd.print("Nachtruhe");
    }
  }
  else
  {
    bWechselTagNacht = true;


    ziel = millis() + messinterval; //Standard: alle 30 Sekunden

    //Warten auf beginn neuer Messzyklus:
    while (now.minute() == lastminute.minute())
    {
      //Während auf den nächsten Meßzyklus gewartet wird, Uhrzeit anzeigen.
      now = rtc.now();//aktuele Zeit vom RTC holen
      snprintf(buffer, sizeof(buffer), "%02u:%02u\0", now.hour(), now.minute()); //Uhrzeit in String wandeln
      //lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print(buffer);
      lcd.setCursor(8, 0);

      snprintf(buffer, sizeof(buffer), "Warte %02u\0", ((ziel - millis()) / 1000l)); //Aus Systemzeit Sekunden filtern und absteigenden Counter erzeugen/anzeigen
      lcd.print(buffer);
      //eine Sekunde warten:
      delay(1000);
    }

    //Neuer Messzyklus startet:
    lastminute = now;
    lcd.clear();
    lcd.setCursor(0, 1);
    lcd.print("Warte auf Flanke");

    bFlanke = false;
    while (!bFlanke);

    //Flanke gewechselt!
    startzeit = millis();

    bFlanke = false;
    //warte auf nächsten Flankenwechsel:
    while (!bFlanke)
    {
      if ((startzeit + maxwait) < millis())
      {
        //Maximale wartezeit überschritten! Abbruch!
        lcd.clear();
        lcd.setCursor(0, 1);
        lcd.print("Zeitüberlauf!");
        //Sekunden warten:
        delay(2000);
        goto Anfang;
      }
    }

    //Flanke gewechselt!
    differenz = millis() - startzeit;

    //Abstand zwischen den Impulsen ist nun da. Dies nun in Leistung umrechnen.
    //Erklärt auf dieser Webseite: http://www.sonnenkiste.de/s0-impulszaehler-berechnung/
    //Erster Wert:
    x = (differenz * 100l) / sek; //Ermittelte Dauer dividiert durch Zeitdauer für einen Impuls. wegen Ganzzahldivision mit 2 Stellen nach Komma: *100

    if (x < 1) x = 1; //division durch 0 vermeiden

    power = (1000l * 100l) / x; //1000*100= 1kW=1000W. Wegen Ganzzahldivision mit 2 Stellen nach Komma: *100 Ergebnis ist in Watt

    if (power > 10000) power = 10000; //Maximale Leistung auf 10kW begrenzen
    snprintf(buffer, sizeof(buffer), "%uW", power);

    //Uhrzeit anzeigen:
    now = rtc.now();//aktuele Zeit vom RTC holen
    snprintf(buffer, sizeof(buffer), "%02u:%02i\0", now.hour(), now.minute()); //Uhrzeit in String wandeln

    lcd.clear();
    lcd.print(buffer);

    //Leistung anzeigen:
    lcd.setCursor(0, 1);
    snprintf(buffer, sizeof(buffer), "%uW", power);
    lcd.print(buffer);//Leistung in Watt anzeigen

    //now = rtc.now();//aktuele Zeit vom RTC holen
    //Alle Daten auf SD-Karte speichern:
    snprintf(buffer, sizeof(buffer), "%02u.%02u.%02u;%02u:%02u:%02u;%lu;\0",  now.day(), now.month(), now.year(), now.hour(), now.minute(), now.second(), power); //Datum, Uhrzeit
    writeContent(buffer);
  
    bFlanke = false;
  }
}


void writeContent(char wert[]) {
  myFile = SD.open(Filename, FILE_WRITE); //Öffnet bzw. erzeugt die Datei im Modus schreibend
  if (myFile) { //Wenn die Datei existiert dann...
    myFile.println(wert);
    myFile.close(); //Schließen der Datei (Dieses ist wichtig da sonst beim beenden des Sketches dies Daten verloren gehen können.)
  } else {
    //Dieser Block wird ausgeführt wenn die Datei nicht erzeugt werden konnte.
    lcd.clear();
    Serial.println("Fehler procedure writeContent: Datei lässt sich nicht beschreiben");
    lcd.print("Dateifehler");
    while (1);
  }
}

//Interrupt-Service-Routine
void Flanke()
{
  bFlanke = true;
}

/*Berechnung Leistung:
   Zeit in ms*100 (100: Kommaverschiebung um 2 Stellen nach Links)
   Ergebnis teilen durch 360 (360: Siehe definition zu "sek": 3600 Sekunden hat eine Stunde, das dividiert durch Zählerauflösung (10000 Imp=1kWh)*1000 als Umrechnung Sekunde nach Millisekunde
   1000W * 100. Ergebnis teilen durch Ergebnis davor. (100: Kommaverschiebung um 2 Stellen nach Links)
*/
