//Solar-Messprogramm zur Aufnahme der abgegebenen leistung von 2 kleinen Solarmodulen (hier 5V * 250mA)
//Gedacht für die Variante mit Operationsverstärker. Damit Nutzung des internen 10-Bit AD-Wandlers im Board.
//Zusatz 3.8.2021: Nutzung des 16-Bit AD-Wandlers auf Basis ADS1115 eingebaut.
//Mittels bedingter Compilierung muss ausgewählt werden, welche Variante benutzt wird
//Geschrieben von Henry Arndt, DL2TM 25.7.2021

/*Befehle LCD:
  lcd.noDisplay();//AUSschalten
  lcd.display(); EINschalten
  WEB: https://www.arduino.cc/en/Reference/LiquidCrystal

  SD-Card: https://www.arduino.cc/en/Reference/SD

  Beispiel Datenlogger: https://draeger-it.blog/arduino-lektion-27-datenloggen-mit-dem-logging-shield/
  Beispiel Setzen/Anzeige Zeit: https://create.arduino.cc/projecthub/tittiamo68/clock-set-date-time-0d46a4

  DateTime: https://adafruit.github.io/RTClib/html/class_date_time.html#aad62a6709a1e65e6860b1efaf8c347e9

  Für SD-Karte unbedingt die SD-Bibliothek ersetzen. orginal funktiniert wegen der Pin-Belegung zu einem MEGA-Bord nicht.
  Gute Beschreibung wie man es macht findet man hier:
  https://www.az-delivery.de/blogs/azdelivery-blog-fur-arduino-und-raspberry-pi/daten-logger-shield-am-mega-r3

  Messzeit: Zwischen 5:00 Uhr und 22:00 Uhr. Außerhalb dürfte wohl keine ungepufferte Solaranlage strom liefern

  Für die Einstellung der Uhrzeit in dem RTC-Chip ist diese Zeile auszuführen:
  (Kommentar entfernen, Bord programmieren, danach Kommentar wieder aktivieren und Bord erneut programmieren):
  rtc.adjust(DateTime(2021, 7, 12, 19, 13, 0)); //RTC mit Datum setzen (Y,M/D,h,m,s);



*/

// includes the library
//LCD:
#include <LiquidCrystal.h>
//RTC:
#include "RTClib.h"
#include <Wire.h>


//include the SD library für SD-CARD:
#include <SPI.h>
#include <SD.h>


//Variablen
//Für LCD-Anzeige:
const int rs = 8, en = 9, d4 = 4, d5 = 5, d6 = 6, d7 = 7;//Zuordnung der einzelnen Pins
LiquidCrystal lcd(rs, en, d4, d5, d6, d7);//Instanz LCD

//RTC:
RTC_DS1307 rtc;

DateTime now;


// set up variables using the SD utility library functions:
Sd2Card card;
File myFile;
SdVolume volume;
//SdFile root;
char Filename[] = "solar.txt";

//Für die Berechung der Leistung:
const float Uref = (2.448f / 1024.0f); //U-Referenzspannung von 2,448V am AREF-Pin gemessen geteilt durch 2^10Bit= Spannung pro bit
const float kfU = 2.84f; //Korrekturfaktor für die gemessene Spannung, nur bei internen AD-Wandler nötig!
const float kfI = 0.128205f; //Korrekturfaktor für den gemessenen Strom. Der OV multipliziert mit Faktor 7,8, der wird hier zurückgerechnet. Nur bei internen AD-Wandler nötig!
float U1, U2, I1, I2, P1, P2;

DateTime lastminute;//merkt sich die Minute der letzten Messung. Messabstand ist eine Minute. 100 erzwingt sofortigen Start

unsigned long startzeit, differenz, ziel; //Pufer im Zusammenhang mit der Funktion millis
bool bWechselTagNacht = true; //wenn ein Übergang von Tag/Nacht erfolgt, dann True. Dient dazu, dass die LCD-Anzeige nicht ständig überschrieben wird
const int RelaisPin = 53;     // Pinnummer für das Relais

char buffer[100];//für alles mögliche
int i1, i2; //für alles mögliche

/*Analogpins:
  Analogeingänge:
  AN8: Spannung SZ1 (SZ=Solarzelle)
  AN9: Strom SZ1
  AN10: Spannung SZ2
  AN11: Strom SZ2
  Einlesen mit z.B.: USZ1 = analogRead(USZ1);
*/
//Zuordnung der Pins:
int Upin1 = A8;
int Ipin1 = A9;
int Upin2 = A10;
int Ipin2 = A11;


//Werte vom AD-Wandler (egal welcher). Achtung: Sind Teile von der Referenzspannung, keine mV!
int16_t USZ1;
int16_t ISZ1;
int16_t USZ2;
int16_t ISZ2;




//--------->Anfangsroutine:
void setup() {
  Serial.begin(9600); //Serielle Kommunikation mit 9600 Baud beginnen

  pinMode(RelaisPin, OUTPUT);//Status Relais setzen
  digitalWrite(RelaisPin, LOW);//Relais ausschalten, also Spannungsmessung

  analogReference(INTERNAL2V56);//Referenz des internen AD-Wandlers auf 2,56V setzen. Damit darf keine Spannung am Analogeingang größer als 2,56V sein.
  
  lcd.begin(16, 2);//INIT LCD
  lcd.setCursor(0, 0); //COL,ROW 0-Basiert
  lcd.print("Solarmeter");
  delay(1000);
  lcd.clear(); //COL,ROW 0-Basiert

  //RTC-Init:
  Serial.println("RTC-Check");
  lcd.print("RTC: ");

  if (! rtc.begin())
  {
    lcd.setCursor(5, 0);
    lcd.print("Fehler");
    Serial.println("RTC-Fehler");
    while (1);
  }
  lcd.write("OK!");
  //delay(1000);


  //--->Einmalig ausführen; wenn Akku vorhanden ist, wird die if-Anweisung nicht mehr ausgeführt...
  //rtc.adjust(DateTime(2021, 7, 12, 19, 13, 0)); //RTC mit Datum setzen (Y,M/D,h,m,s);

  if (! rtc.isrunning())
  {
    rtc.adjust(DateTime(2021, 7, 12, 19, 13, 0)); //RTC mit Datum setzen (Y,M/D,h,m,s)
    //rtc.adjust(DateTime(2021, 7, 4, 15, 58, 0)); //RTC mit Datum setzen (6.7.2021 15:58:00)
    //DateTime (uint16_t year, uint8_t month, uint8_t day, uint8_t hour=0, uint8_t min=0, uint8_t sec=0)
  }


  //Check SD-Card:
  //Serial.println("SD-Karte");
  lcd.setCursor(0, 1); //COL,ROW 0-Basiert
  lcd.write("SD: ");

  // On the Ethernet Shield, CS is pin 4. It's set as an output by default.
  // Note that even if it's not used as the CS pin, the hardware SS pin
  // (10 on most Arduino boards, 53 on the Mega) must be left as an output
  // or the SD library functions will not work.
  pinMode(SS, OUTPUT);
  
  if (!SD.begin( 10, 11, 12, 13))
  {
    lcd.clear();
    lcd.print("SD-Fehler!");
    Serial.println("initialization failed. Things to check:");
    Serial.println("* is a card is inserted?");
    Serial.println("* Is your wiring correct?");
    while (1);
  }
  //SD-Karte geht
  lcd.write("OK!");

  //Falls Datei schon da ist, löschen:
  //if (SD.exists(Filename)) {
  //  SD.remove(Filename);
  //}
  myFile = SD.open(Filename, FILE_WRITE);
  if (!myFile) {
    //DateiFehler
    lcd.clear();
    lcd.setCursor(0, 0); //COL,ROW 0-Basiert
    lcd.print("Fehler SD-Card");
    while (1);
  }

  delay(2000);//Für den Benutzer, damit er die letzte Anzeige noch sieht

  //aktuelle zeit anzeigen:
  now = rtc.now();//aktuele Zeit vom RTC holen

  //Zeit anzeigen:
  //Datum:
  lcd.setCursor(0, 0); //COL,ROW 0-Basiert
  snprintf(buffer, sizeof(buffer), "%02u.%02u.%02u\0", now.day(), now.month(), now.year());
  lcd.print(buffer);

  //Uhrzeit:
  lcd.setCursor(0, 1); //COL,ROW 0-Basiert
  snprintf(buffer, sizeof(buffer), "%02u:%02u:%02u\0", now.hour(), now.minute(), now.second());
  lcd.print(buffer);
  delay(2000);

  lcd.clear();
  lastminute = now;
}



//--------->Programmschleife:
void loop() {
  long x;

  //Ist es Nacht? Wenn ja dies anzeigen und keine Messung durchführen:
Anfang:
  now = rtc.now();//aktuele Zeit vom RTC holen
  if ((now.hour() > 20) || (now.hour() < 6))
  {
    if (bWechselTagNacht)
    {
      bWechselTagNacht = false;
      lcd.clear();
      lcd.print("Nachtruhe");
      lastminute = now;
    }
  }
  else
  {
    bWechselTagNacht = true;

    //Warten auf beginn neuer Messzyklus:
    if (now.minute() == lastminute.minute())
    {
      //kein neuer Messzyklus
      //Uhrzeit anzeigen.
      snprintf(buffer, sizeof(buffer), "%02u:%02u\0", now.hour(), now.minute()); //Uhrzeit in String wandeln
      //lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print(buffer);
      lcd.setCursor(8, 0);

      snprintf(buffer, sizeof(buffer), "Warte %02u\0", ((59 - now.second()))); //Aus Systemzeit Sekunden filtern und absteigenden Counter erzeugen/anzeigen
      lcd.print(buffer);
    }
    else
    {
      //Neuer Messzyklus startet:
      lastminute = now;
      lcd.clear();
      lcd.setCursor(0, 0);

      //Spannung messen von Modul1:
      //interner AD-Wandler:
      USZ1 = analogRead(Upin1);
      U1 = float(USZ1);
      U1 = U1 * Uref * kfU;


      //Spannung messen von Modul2:
      USZ2 = analogRead(Upin2);
      U2 = float(USZ2);
      U2 = U2 * Uref * kfU;

      //Strom messen:
      //Relais einschalten:
      digitalWrite(RelaisPin, HIGH);//Relais einschalten, also Strommessung
      delay(20);//Prellzeit

      ISZ1 = analogRead(Ipin1);//Strom Modul1 einlesen
      ISZ2 = analogRead(Ipin2);//Strom Modul2 einlesen

      digitalWrite(RelaisPin, LOW);//Relais ausschalten
      delay(20);

      I1 = float(ISZ1) * Uref * kfI;
      I2 = float(ISZ2) * Uref * kfI;

      //Leistung berechnen
      //Leistung Modul1:
      P1 = U1 * I1;
      P1 = P1 * 1000.0f; //Ergebnis in mW
      P1 = P1 * 0.75f; //Füllfaktor 75%

      //Leistung Modul2:
      P2 = U2 * I2;
      P2 = P2 * 1000.0f; //Ergebnis in mW
      P2 = P2 * 0.75f; //Füllfaktor 75%


      //Wert anzeigen:
      lcd.print("Modul1  Modul2");
      lcd.setCursor(0, 1);
      //dtostrf(P, 3, 0, buffer); //0 Stellen nach Komma
      snprintf(buffer, sizeof(buffer), "%1umW\0", int(P1));
      lcd.print(buffer);
      lcd.setCursor(8, 1);
      snprintf(buffer, sizeof(buffer), "%1umW \0", int(P2));
      lcd.print(buffer);

      now = rtc.now();
      snprintf(buffer, sizeof(buffer), "%02u.%02u.%02u;%02u:%02u:%02u;%i;%i\0",  now.day(), now.month(), now.year(), now.hour(), now.minute(), now.second(), int(P1), int(P2)); //Datum, Uhrzeit, Leistung
      writeContent(buffer);

      delay(5000);
      lcd.clear();
    }
  }
}


void writeContent(char wert[]) {
  myFile = SD.open(Filename, FILE_WRITE); //Öffnet bzw. erzeugt die Datei im Modus schreibend
  if (myFile) { //Wenn die Datei existiert dann...
    myFile.println(wert);
    myFile.close(); //Schließen der Datei (Dieses ist wichtig da sonst beim beenden des Sketches dies Daten verloren gehen können.)
  } else {
    //Dieser Block wird ausgeführt wenn die Datei nicht erzeugt werden konnte.
    lcd.clear();
    Serial.println("Fehler procedure writeContent: Datei lässt sich nicht beschreiben");
    lcd.print("Dateifehler");
    while (1);
  }
}
