use work.all;
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use work.Definitions.all;

entity memory_manager is
    port(
        res_n : in std_logic;
        clk : in std_logic;
        clk_sdram : in std_logic;
		  N_samples : in std_logic_vector(15 downto 0);
        
        -- pins to ram device
        sdram_data_Dq : inout std_logic_vector(15 downto 0);
        o_sdram_addr : out std_logic_vector(12 downto 0);
        o_sdram_clk : out std_logic;
        o_sdram_cke : out std_logic;
        o_sdram_cs_l : out std_logic;
        o_sdram_ras_l : out std_logic;
        o_sdram_cas_l : out std_logic;
        o_sdram_we_l : out std_logic;
        o_sdram_ba : out std_logic_vector(1 downto 0);
        o_sdram_dqml : out std_logic;
        o_sdram_dqmh : out std_logic;
		  
		  -- pins for all application interfaces
		  mem_ready_o : out std_logic;
        
        -- pins for write interface
        write_req_1 : in std_logic;
        write_req_1_done : out std_logic;
        write_idx_1 : in std_logic_vector(15 downto 0);
        write_req_2 : in std_logic;
        write_req_2_done : out std_logic;
        write_idx_2 : in std_logic_vector(15 downto 0);
        data_i : in reg_array_t;
        
        -- pins for read interface
        read_req : in std_logic;
        read_req_done : out std_logic;
        read_idx : idx_t;
        data_o : out reg_array_t;
        
        -- pins for test write interface
        write_addr_test : in std_logic_vector(15 downto 0);
        write_data_test : in std_logic_vector(15 downto 0);
        write_req_test : in std_logic
    );
end memory_manager;

architecture behav of memory_manager is
	 type state_t is (idle, write_1, write_1_delay, write_1_wait,
	                        write_2, write_2_delay, write_2_wait,
									read_s, read_s_delay, read_s_wait, write_test, write_test_delay, write_test_wait);
    signal current_state : state_t;
    
    signal write_req_1_reg : std_logic;
    signal write_req_1_d : std_logic;
    signal write_idx_1_reg : idx_t;
    signal write_req_2_reg : std_logic;
    signal write_req_2_d : std_logic;
    signal write_idx_2_reg : idx_t;
    signal data_reg : reg_array_t;
    
    signal read_req_reg : std_logic;
    signal read_req_d : std_logic;
    signal read_idx_reg : idx_t;
    
    signal write_req_test_reg : std_logic;
    signal write_req_test_d : std_logic;
    
    signal c_idx : integer range 0 to 15;    -- hint
	 signal N_samples_samp : std_logic_vector(15 downto 0);
	 signal addr_calc : unsigned(15 downto 0);
    
    signal i_sdram_data : std_logic_vector(15 downto 0);
    signal o_sdram_data : std_logic_vector(15 downto 0);
    signal o_sdram_buf_we : std_logic;
    
    signal i_ram_data : std_logic_vector(15 downto 0);
    signal o_ram_data : std_logic_vector(15 downto 0);
    signal i_ram_addr : std_logic_vector(23 downto 0);
    signal i_ram_len : std_logic_vector(9 downto 0);
    signal i_ram_read_req : std_logic;
    signal o_ram_read_valid : std_logic;
    signal i_ram_write_req : std_logic;
    signal o_ram_write_valid : std_logic;
    signal o_ready : std_logic;
    
    constant SDRAM_ROW_WIDTH_BIT  : integer := 13;
    constant SDRAM_COL_WIDTH_BIT  : integer := 9;
    constant SDRAM_BANK_WIDTH_BIT : integer := 2;
    constant SDRAM_DATA_WIDTH_BIT : integer := 16;
    constant SDRAM_CAS            : integer := 3;
begin
    read_req_done <= read_req_d;
    write_req_1_done <= write_req_1_d;
    write_req_2_done <= write_req_2_d;
	 mem_ready_o <= o_ready;
    
    reg_p: process(res_n, clk) is
    begin
        if res_n = '0' then
        elsif clk'event and clk = '1' then
            if write_req_1 = '1' then
                write_req_1_reg <= '1';
                data_reg(0 to 3) <= data_i(0 to 3);
                write_idx_1_reg <= write_idx_1;
            end if;
				if write_req_1_d = '1' and write_req_1 = '0' then
                write_req_1_reg <= '0';
            end if;
				
            if write_req_2 = '1' then
                write_req_2_reg <= '1';
                data_reg(4 to 7) <= data_i(4 to 7);
                write_idx_2_reg <= write_idx_2;
            end if;
            if write_req_2_d = '1' and write_req_2 = '0' then
                write_req_2_reg <= '0';
            end if;
            
            if read_req = '1' then
                read_req_reg <= '1';
                read_idx_reg <= read_idx;
            end if;
            if read_req_d = '1' and read_req = '0' then
                read_req_reg <= '0';
            end if;
            
            if write_req_test = '1' then
                write_req_test_reg <= '1';
            end if;
            if write_req_test_d = '1' and write_req_test = '0' then
                write_req_test_reg <= '0';
            end if;
        end if;
    end process;
    
    work_p: process(res_n, clk) is
    begin
        if res_n = '0' then
        elsif clk'event and clk ='1' then
            case current_state is
                when idle =>
                    c_idx <= 0;
						  if write_req_1_reg = '1' and write_req_1_d = '0' then
								current_state <= write_1;
                    end if;
						  if write_req_2_reg = '1' and write_req_2_d = '0' then
								current_state <= write_2;
                    end if;
                    if read_req_reg = '1' and read_req_d = '0' then
								current_state <= read_s;
                    end if;
                    if write_req_test_reg = '1' and write_req_test_d = '0' then
                        current_state <= write_test;
                    end if;
                    read_req_d <= '0';
                    write_req_1_d <= '0';
                    write_req_2_d <= '0';
                    i_ram_write_req <= '0';
                    i_ram_read_req <= '0';
                    write_req_test_d <= '0';
						  N_samples_samp <= N_samples;
                when write_1 =>
                    i_ram_data <= data_reg(c_idx);
						  i_ram_addr <= std_logic_vector(resize(shift_left(unsigned(N_samples_samp), c_idx) + unsigned(write_idx_1_reg), i_ram_addr'length));
                    i_ram_len <= std_logic_vector(to_unsigned(1, i_ram_len'length));
                    i_ram_read_req <= '0';
                    i_ram_write_req <= '1';
                    if o_ready = '1' then
                        current_state <= write_1_delay;
                    end if;
                when write_1_delay =>
                    i_ram_write_req <= '0';
                    if o_ram_write_valid = '1' then
                        current_state <= write_1_wait;
                    end if;
                when write_1_wait =>
                    i_ram_write_req <= '0';
                    if o_ready = '1' then
                        if c_idx < 3 then
                            c_idx <= c_idx + 1;
									 current_state <= write_1;
                        else
                            current_state <= idle;
                            write_req_1_d <= '1';
                        end if;
                    end if;
                when write_2 =>
                    i_ram_data <= data_reg(c_idx + 4);
						  i_ram_addr <= std_logic_vector(resize(shift_left(unsigned(N_samples_samp), c_idx + 4) + unsigned(write_idx_2_reg), i_ram_addr'length));
                    i_ram_len <= std_logic_vector(to_unsigned(1, i_ram_len'length));
                    i_ram_read_req <= '0';
                    i_ram_write_req <= '1';
                    if o_ready = '1' then
                        current_state <= write_2_delay;
                    end if;
                when write_2_delay =>
                    i_ram_write_req <= '1';
                    if o_ram_write_valid = '1' then
                        current_state <= write_2_wait;
                    end if;
                when write_2_wait =>
                    i_ram_write_req <= '0';
                    if o_ready = '1' then
                        if c_idx < 3 then
                            c_idx <= c_idx + 1;
									 current_state <= write_2;
                        else
                            current_state <= idle;
                            write_req_2_d <= '1';
                        end if;
                    end if;
                when read_s =>
						  i_ram_addr <= std_logic_vector(resize(shift_left(unsigned(N_samples_samp), c_idx) + unsigned(read_idx_reg), i_ram_addr'length));
                    i_ram_len <= std_logic_vector(to_unsigned(1, i_ram_len'length));
                    i_ram_read_req <= '1';
                    i_ram_write_req <= '0';
                    if o_ready = '1' then
                        current_state <= read_s_delay;
                    end if;
                when read_s_delay =>
                    if o_ram_read_valid = '1' then
                        i_ram_read_req <= '0';
                        current_state <= read_s_wait;
                    else
                        i_ram_read_req <= '1';
                    end if;
                when read_s_wait =>
                    i_ram_read_req <= '0';
                    if o_ready = '1' then
                        data_o(c_idx) <= o_ram_data;
                        if c_idx < 7 then
                            c_idx <= c_idx + 1;
									 current_state <= read_s;
                        else
                            current_state <= idle;
                            read_req_d <= '1';
                        end if;
                    end if;
                when write_test =>
                    i_ram_data <= write_data_test;
                    i_ram_addr <= X"00" & write_addr_test;
                    i_ram_len <= std_logic_vector(to_unsigned(1, i_ram_len'length));
                    i_ram_read_req <= '0';
                    i_ram_write_req <= '1';
                    if o_ready = '1' then
                        current_state <= write_test_delay;
                    end if;
                when write_test_delay =>
                    i_ram_write_req <= '1';
                    if o_ram_write_valid = '1' then
								i_ram_write_req <= '0';
                        current_state <= write_test_wait;
                    end if;
                when write_test_wait =>
                    i_ram_write_req <= '0';
                    if o_ready = '1' then
                        current_state <= idle;
                        write_req_test_d <= '1';
                    end if;
            end case;
        end if;
    end process;
    
    sdram_controller_e: entity SDRAM_Controller
    port map(
        res_n => res_n,
        clk => clk,
        clk_sdram => clk_sdram,
        
        -- application pins
        i_ram_data => i_ram_data,
        o_ram_data => o_ram_data,
        i_ram_addr => i_ram_addr,
        i_ram_len => i_ram_len,
        i_ram_read_req => i_ram_read_req,
        o_ram_read_valid => o_ram_read_valid,
        i_ram_write_req => i_ram_write_req,
        o_ram_write_valid => o_ram_write_valid,
        o_ready => o_ready,
        
        -- pins to ram device
        i_sdram_data => i_sdram_data,
        o_sdram_data => o_sdram_data,
        o_sdram_addr => o_sdram_addr,
        o_sdram_clk => o_sdram_clk,
        o_sdram_cke => o_sdram_cke,
        o_sdram_cs_l => o_sdram_cs_l,
        o_sdram_ras_l => o_sdram_ras_l,
        o_sdram_cas_l => o_sdram_cas_l,
        o_sdram_we_l => o_sdram_we_l,
        o_sdram_buf_we => o_sdram_buf_we,
        o_sdram_ba => o_sdram_ba,
        o_sdram_dqml => o_sdram_dqml,
        o_sdram_dqmh => o_sdram_dqmh
    );
    
	 data_switcher_p1: process(o_sdram_buf_we, o_sdram_data, sdram_data_Dq) is
    begin
       if o_sdram_buf_we = '1' then
           sdram_data_Dq <= o_sdram_data;
       else
           sdram_data_Dq <= (others => 'Z');
       end if;
    end process;
    i_sdram_data <= sdram_data_Dq;
end architecture;
