use work.all;
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity main_logic is
	port(
		res_n : in std_logic;
		clk : in std_logic;
		RX : in std_logic_vector(7 downto 0);
		RX_d : in std_logic;
		TX : out std_logic_vector(7 downto 0);
		TX_d : out std_logic;
		TX_active : in std_logic;
		
		software_reset : out std_logic;
		led_reg : out std_logic_vector(15 downto 0);
		start_o_1 : in std_logic;
		start_o_2 : in std_logic;
		start_i_1 : out std_logic;
		start_i_2 : out std_logic;
		voltage_current_1 : out std_logic;
		voltage_current_2 : out std_logic;
		calibration_1 : out std_logic;
		calibration_2 : out std_logic;
		N_samples : out std_logic_vector(15 downto 0);
		
		set_point_1_reg : out std_logic_vector(15 downto 0);
		Kp_1_reg : out std_logic_vector(15 downto 0);
		Ki_1_reg : out std_logic_vector(15 downto 0);
		Kd_1_reg : out std_logic_vector(15 downto 0);
		Kf_1_reg : out std_logic_vector(15 downto 0);
		Mp_1_reg : out std_logic_vector(15 downto 0);
		Mi_1_reg : out std_logic_vector(15 downto 0);
		Md_1_reg : out std_logic_vector(15 downto 0);
		Mf_1_reg : out std_logic_vector(15 downto 0);
		bias_high_1_reg : out std_logic_vector(15 downto 0);
		bias_low_1_reg : out std_logic_vector(15 downto 0);
		limit_1 : out std_logic_vector(15 downto 0);
		
		set_point_2_reg : out std_logic_vector(15 downto 0);
		Kp_2_reg : out std_logic_vector(15 downto 0);
		Ki_2_reg : out std_logic_vector(15 downto 0);
		Kd_2_reg : out std_logic_vector(15 downto 0);
		Kf_2_reg : out std_logic_vector(15 downto 0);
		Mp_2_reg : out std_logic_vector(15 downto 0);
		Mi_2_reg : out std_logic_vector(15 downto 0);
		Md_2_reg : out std_logic_vector(15 downto 0);
		Mf_2_reg : out std_logic_vector(15 downto 0);
		bias_high_2_reg : out std_logic_vector(15 downto 0);
		bias_low_2_reg : out std_logic_vector(15 downto 0);
		limit_2 : out std_logic_vector(15 downto 0);
		
		read_addr_reg : out std_logic_vector(15 downto 0);
		
		ADC_1_V_data_reg : in std_logic_vector(15 downto 0);
		ADC_1_C_data_reg : in std_logic_vector(15 downto 0);
		DAC_1_high_data_reg : in std_logic_vector(15 downto 0);
		DAC_1_low_data_reg : in std_logic_vector(15 downto 0);
		
		ADC_2_V_data_reg : in std_logic_vector(15 downto 0);
		ADC_2_C_data_reg : in std_logic_vector(15 downto 0);
		DAC_2_high_data_reg : in std_logic_vector(15 downto 0);
		DAC_2_low_data_reg : in std_logic_vector(15 downto 0);
		
		read_req : out std_logic;
		read_req_done : in std_logic;
		
		write_addr_test : out std_logic_vector(15 downto 0);
		write_data_test : out std_logic_vector(15 downto 0);
		write_req_test : out std_logic
	);
end main_logic;

architecture behav of main_logic is
	constant N_reg : integer := 2**6;
	type UART_state_t is (reset_state, register_state, rx_data_state_1, rx_data_state_2, rx_data_state_3, rx_data_state_4, rx_data_state_5,
	                      tx_data_state_1, tx_data_wait_state_1, tx_data_wait_state_1e, tx_data_wait_state_1ee, tx_data_wait_state_1eee, tx_data_state_2, tx_data_wait_state_2, tx_data_wait_state_2e, tx_data_wait_state_2ee, tx_data_wait_state_2eee,
								 tx_bulk_fetch, tx_bulk_fetch_wait, tx_bulk_fetch_wait_2, tx_bulk_send_byte_1, tx_bulk_send_byte_1_wait, tx_bulk_send_byte_1_wait_e, tx_bulk_send_byte_1_wait_ee, tx_bulk_send_byte_1_wait_eee,
								 tx_bulk_send_byte_2, tx_bulk_send_byte_2_wait, tx_bulk_send_byte_2_wait_e, tx_bulk_send_byte_2_wait_ee, tx_bulk_send_byte_2_wait_eee, tx_bulk_done);
	
	signal UART_state : UART_state_t := reset_state;
	signal UART_next_state : UART_state_t := reset_state;
	signal reg_idx : integer range 0 to N_reg := 0;
	signal read_idx : integer range 0 to 8;
	signal read_addr : integer range 0 to 16384;
	signal N_samples_int : integer range 0 to 16384;
	
	signal tmp_reg : std_logic_vector(15 downto 0);
	signal config_reg : std_logic_vector(15 downto 0);
begin
	start_i_1 <= '0' when start_o_1 = '0' else config_reg(0);
	start_i_2 <= '0' when start_o_2 = '0' else config_reg(1);
	voltage_current_1 <= config_reg(2);
	voltage_current_2 <= config_reg(3);
	calibration_1 <= config_reg(4);
	calibration_2 <= config_reg(5);
	software_reset <= config_reg(15);
   
	next_state_p: process(clk, res_n) is
	begin
	   if res_n = '0' then
			UART_state <= reset_state;
		elsif clk'event and clk = '1' then
			UART_state <= UART_next_state;
		end if;
	end process;
	
	g: process(RX_d, RX, TX_active, UART_state, tmp_reg, read_req_done, read_idx, read_addr, N_Samples_int) is
	begin
		UART_next_state <= UART_state;
		case UART_state is
			when reset_state =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					else
						UART_next_state <= register_state;
					end if;
				end if;
			when register_state =>
				if RX(6) = '0' then
					UART_next_state <= rx_data_state_1;
				else
					UART_next_state <= tx_data_state_1;
				end if;
			when rx_data_state_1 =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					else
						UART_next_state <= rx_data_state_2;
					end if;
				end if;
			when rx_data_state_2 =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					else
						UART_next_state <= rx_data_state_3;
					end if;
				end if;
			when rx_data_state_3 =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					else
						UART_next_state <= rx_data_state_4;
					end if;
				end if;
			when rx_data_state_4 =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					else
						UART_next_state <= rx_data_state_5;
					end if;
				end if;
			when rx_data_state_5 =>
				if tmp_reg = X"FFFF" then
					UART_next_state <= tx_bulk_fetch;
				else
					UART_next_state <= reset_state;
				end if;
			when tx_data_state_1 =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					end if;
				end if;
				UART_next_state <= tx_data_wait_state_1;
			when tx_data_wait_state_1 =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					end if;
				end if;
				if TX_active = '0' then
					UART_next_state <= tx_data_wait_state_1e;
				end if;
         when tx_data_wait_state_1e =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					end if;
				end if;
				if TX_active = '0' then
					UART_next_state <= tx_data_wait_state_1ee;
				end if;
         when tx_data_wait_state_1ee =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					end if;
				end if;
				if TX_active = '0' then
					UART_next_state <= tx_data_wait_state_1eee;
				end if;
			when tx_data_wait_state_1eee =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					end if;
				end if;
				if TX_active = '0' then
					UART_next_state <= tx_data_state_2;
				end if;
			when tx_data_state_2 =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					end if;
				end if;
				if TX_active = '0' then
					UART_next_state <= tx_data_wait_state_2;
				end if;
			when tx_data_wait_state_2 =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					end if;
				end if;
				if TX_active = '0' then
					UART_next_state <= tx_data_wait_state_2e;
				end if;
         when tx_data_wait_state_2e =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					end if;
				end if;
				if TX_active = '0' then
					UART_next_state <= tx_data_wait_state_2ee;
				end if;
			when tx_data_wait_state_2ee =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					end if;
				end if;
				if TX_active = '0' then
					UART_next_state <= tx_data_wait_state_2eee;
				end if;
         when tx_data_wait_state_2eee =>
				if RX_d = '1' then
					if RX(7) = '1' then
						UART_next_state <= reset_state;
					end if;
				end if;
				if TX_active = '0' then
					UART_next_state <= reset_state;
				end if;
			
			when tx_bulk_fetch =>
				UART_next_state <= tx_bulk_fetch_wait;
			
			when tx_bulk_fetch_wait =>
				UART_next_state <= tx_bulk_fetch_wait_2;
			
			when tx_bulk_fetch_wait_2 =>
				if read_req_done = '1' then
					UART_next_state <= tx_bulk_send_byte_1;
				else
					UART_next_state <= tx_bulk_fetch_wait_2;
				end if;
			
			when tx_bulk_send_byte_1 =>
				UART_next_state <= tx_bulk_send_byte_1_wait;
			
			when tx_bulk_send_byte_1_wait =>
				if TX_active = '0' then
					UART_next_state <= tx_bulk_send_byte_1_wait_e;
				else
					UART_next_state <= tx_bulk_send_byte_1_wait;
				end if;
			
			when tx_bulk_send_byte_1_wait_e =>
				if TX_active = '0' then
					UART_next_state <= tx_bulk_send_byte_1_wait_ee;
				else
					UART_next_state <= tx_bulk_send_byte_1_wait_e;
				end if;
			
			when tx_bulk_send_byte_1_wait_ee =>
				if TX_active = '0' then
					UART_next_state <= tx_bulk_send_byte_1_wait_eee;
				else
					UART_next_state <= tx_bulk_send_byte_1_wait_ee;
				end if;
			
			when tx_bulk_send_byte_1_wait_eee =>
				if TX_active = '0' then
					UART_next_state <= tx_bulk_send_byte_2;
				else
					UART_next_state <= tx_bulk_send_byte_1_wait_eee;
				end if;
			
			when tx_bulk_send_byte_2 =>
				UART_next_state <= tx_bulk_send_byte_2_wait;
			
			when tx_bulk_send_byte_2_wait =>
				if TX_active = '0' then
					UART_next_state <= tx_bulk_send_byte_2_wait_e;
				else
					UART_next_state <= tx_bulk_send_byte_2_wait;
				end if;
			
			when tx_bulk_send_byte_2_wait_e =>
				if TX_active = '0' then
					UART_next_state <= tx_bulk_send_byte_2_wait_ee;
				else
					UART_next_state <= tx_bulk_send_byte_2_wait_e;
				end if;
			
			when tx_bulk_send_byte_2_wait_ee =>
				if TX_active = '0' then
					UART_next_state <= tx_bulk_send_byte_2_wait_eee;
				else
					UART_next_state <= tx_bulk_send_byte_2_wait_ee;
				end if;
			
			when tx_bulk_send_byte_2_wait_eee =>
				if TX_active = '0' then
					UART_next_state <= tx_bulk_done;
				else
					UART_next_state <= tx_bulk_send_byte_2_wait_eee;
				end if;
			
			when tx_bulk_done =>
				if read_idx < 8 then
					UART_next_state <= tx_bulk_send_byte_1;
				else
					if read_addr < (N_samples_int - 1) then    -- off by one error, because read_addr is incremented just in tx_bulk_done
						UART_next_state <= tx_bulk_fetch;
					else
						UART_next_state <= reset_state;
					end if;
				end if;
			
		end case;
	end process;

	work_p: process(res_n, clk) is
	begin
		if res_n = '0' then
			led_reg <= (others => '0');
		elsif clk'event and clk = '1' then
			case UART_state is
				when reset_state =>
					reg_idx <= 0;
					TX_d <= '0';
					config_reg(0) <= '0';
					config_reg(1) <= '0';
					read_req <= '0';
					write_req_test <= '0';
				when register_state =>
					reg_idx <= to_integer(unsigned(RX(5 downto 0)));
					TX_d <= '0';
				when rx_data_state_1 =>
					tmp_reg(15 downto 12) <= RX(3 downto 0);
					TX_d <= '0';
				when rx_data_state_2 =>
					tmp_reg(11 downto 8) <= RX(3 downto 0);
					TX_d <= '0';
				when rx_data_state_3 =>
					tmp_reg(7 downto 4) <= RX(3 downto 0);
					TX_d <= '0';
				when rx_data_state_4 =>
					tmp_reg(3 downto 0) <= RX(3 downto 0);
					TX_d <= '0';
				when rx_data_state_5 =>
					case reg_idx is
						when 0  => led_reg <= tmp_reg;
						when 1  => config_reg <= tmp_reg;
						when 2  =>
							N_samples <= tmp_reg;
							N_samples_int <= to_integer(unsigned(tmp_reg));
						
						when 3  => set_point_1_reg <= tmp_reg;
						when 4  => Kp_1_reg <= tmp_reg;
						when 5  => Ki_1_reg <= tmp_reg;
						when 6  => Kd_1_reg <= tmp_reg;
						when 7  => Kf_1_reg <= tmp_reg;
						when 8  => Mp_1_reg <= tmp_reg;
						when 9  => Mi_1_reg <= tmp_reg;
						when 10 => Md_1_reg <= tmp_reg;
						when 11 => Mf_1_reg <= tmp_reg;
						when 12 => bias_high_1_reg <= tmp_reg;
						when 13 => bias_low_1_reg <= tmp_reg;
						when 14 => limit_1 <= tmp_reg;
						
						when 15 => set_point_2_reg <= tmp_reg;
						when 16 => Kp_2_reg <= tmp_reg;
						when 17 => Ki_2_reg <= tmp_reg;
						when 18 => Kd_2_reg <= tmp_reg;
						when 19 => Kf_2_reg <= tmp_reg;
						when 20 => Mp_2_reg <= tmp_reg;
						when 21 => Mi_2_reg <= tmp_reg;
						when 22 => Md_2_reg <= tmp_reg;
						when 23 => Mf_2_reg <= tmp_reg;
						when 24 => bias_high_2_reg <= tmp_reg;
						when 25 => bias_low_2_reg <= tmp_reg;
						when 26 => limit_2 <= tmp_reg;
						
						when 27 =>
							if tmp_reg = X"FFFF" then
								read_idx <= 0;
								read_addr <= 0;
							else
								read_addr_reg <= tmp_reg;
								read_req <= '1';
							end if;
						
						when 36 =>
							write_addr_test <= tmp_reg;
							write_req_test <= '1';
						when 37 =>
							write_data_test <= tmp_reg;
						
						when others =>
					end case;
					TX_d <= '0';
				when tx_data_state_1 =>
					case reg_idx is
						when 1 => tmp_reg <= config_reg;
						
						when 28 => tmp_reg <= ADC_1_V_data_reg;
						when 29 => tmp_reg <= ADC_1_C_data_reg;
						when 30 => tmp_reg <= DAC_1_high_data_reg;
						when 31 => tmp_reg <= DAC_1_low_data_reg;
						
						when 32 => tmp_reg <= ADC_2_V_data_reg;
						when 33 => tmp_reg <= ADC_2_C_data_reg;
						when 34 => tmp_reg <= DAC_2_high_data_reg;
						when 35 => tmp_reg <= DAC_2_low_data_reg;
						
						when others => tmp_reg <= (others => '0');
					end case;
					TX <= tmp_reg(15 downto 8);
					TX_d <= '0';
				when tx_data_wait_state_1 =>
					TX <= tmp_reg(15 downto 8);
					TX_d <= '1';
                when tx_data_wait_state_1e =>
					TX <= tmp_reg(15 downto 8);
					TX_d <= '0';
            when tx_data_wait_state_1ee =>
					TX <= tmp_reg(15 downto 8);
					TX_d <= '0';
				when tx_data_wait_state_1eee =>
					TX <= tmp_reg(15 downto 8);
					TX_d <= '0';
				when tx_data_state_2 =>
					TX <= tmp_reg(7 downto 0);
					TX_d <= '0';
				when tx_data_wait_state_2 =>
					TX <= tmp_reg(7 downto 0);
				   TX_d <= '1';
            when tx_data_wait_state_2e =>
					TX <= tmp_reg(7 downto 0);
					TX_d <= '0';
            when tx_data_wait_state_2ee =>
					TX <= tmp_reg(7 downto 0);
					TX_d <= '0';
				when tx_data_wait_state_2eee =>
					TX <= tmp_reg(7 downto 0);
					TX_d <= '0';
				
				when tx_bulk_fetch =>
					read_addr_reg <= std_logic_vector(to_unsigned(read_addr, read_addr_reg'length));
					read_req <= '1';
					TX_d <= '0';
				
				when tx_bulk_fetch_wait =>
					read_req <= '0';
					TX_d <= '0';
					tmp_reg <= X"0000";    -- bug fix: avoid several bulk send runs
				
				when tx_bulk_fetch_wait_2 =>
					read_req <= '0';
					TX_d <= '0';
				
				when tx_bulk_send_byte_1 =>
					case read_idx is
						when 0 => TX <= ADC_1_V_data_reg(15 downto 8);
						when 1 => TX <= ADC_1_C_data_reg(15 downto 8);
						when 2 => TX <= DAC_1_high_data_reg(15 downto 8);
						when 3 => TX <= DAC_1_low_data_reg(15 downto 8);
						when 4 => TX <= ADC_2_V_data_reg(15 downto 8);
						when 5 => TX <= ADC_2_C_data_reg(15 downto 8);
						when 6 => TX <= DAC_2_high_data_reg(15 downto 8);
						when 7 => TX <= DAC_2_low_data_reg(15 downto 8);
						when others =>
					end case;
					TX_d <= '1';
				
				when tx_bulk_send_byte_1_wait =>
					TX_d <= '0';
				
				when tx_bulk_send_byte_1_wait_e =>
					TX_d <= '0';
				
				when tx_bulk_send_byte_1_wait_ee =>
					TX_d <= '0';
				
				when tx_bulk_send_byte_1_wait_eee =>
					TX_d <= '0';
				
				when tx_bulk_send_byte_2 =>
					case read_idx is
						when 0 => TX <= ADC_1_V_data_reg(7 downto 0);
						when 1 => TX <= ADC_1_C_data_reg(7 downto 0);
						when 2 => TX <= DAC_1_high_data_reg(7 downto 0);
						when 3 => TX <= DAC_1_low_data_reg(7 downto 0);
						when 4 => TX <= ADC_2_V_data_reg(7 downto 0);
						when 5 => TX <= ADC_2_C_data_reg(7 downto 0);
						when 6 => TX <= DAC_2_high_data_reg(7 downto 0);
						when 7 => TX <= DAC_2_low_data_reg(7 downto 0);
						when others =>
					end case;
					TX_d <= '1';
					
				when tx_bulk_send_byte_2_wait =>
					TX_d <= '0';
					read_idx <= read_idx + 1;
				
				when tx_bulk_send_byte_2_wait_e =>
					TX_d <= '0';
				
				when tx_bulk_send_byte_2_wait_ee =>
					TX_d <= '0';
				
				when tx_bulk_send_byte_2_wait_eee =>
					TX_d <= '0';
				
				when tx_bulk_done =>
					TX_d <= '0';
					if read_idx = 8 then
						read_idx <= 0;
						read_addr <= read_addr + 1;
					end if;
				
			end case;
		end if;
	end process;
end architecture;
