use work.all;
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use work.Definitions.all;

entity controller is
	port(
		res_n : in std_logic;
		software_reset : in std_logic;
		clk_h : in std_logic;
		N_samples : in std_logic_vector(15 downto 0);
		
		DAC_high_nSYNC : out std_logic;
		DAC_high_SCLK : out std_logic;
		DAC_high_DIN : out std_logic;
		
		DAC_low_nSYNC : out std_logic;
		DAC_low_SCLK : out std_logic;
		DAC_low_DIN : out std_logic;
		
		ADC_V_nRST : out std_logic;
		ADC_V_SDI : out std_logic;
		ADC_V_nCS : out std_logic;
		ADC_V_SCLK : out std_logic;
		ADC_V_SDO_0 : in std_logic;
		ADC_V_RVS : in std_logic;
		
		ADC_C_nRST : out std_logic;
		ADC_C_SDI : out std_logic;
		ADC_C_nCS : out std_logic;
		ADC_C_SCLK : out std_logic;
		ADC_C_SDO_0 : in std_logic;
		ADC_C_RVS : in std_logic;
		
		voltage_current : in std_logic;
		start_i : in std_logic;
		start_o : out std_logic;
		calibration_i : in std_logic;
		
		set_point_reg : in std_logic_vector(15 downto 0);
		Kp_reg : in std_logic_vector(15 downto 0);
		Ki_reg : in std_logic_vector(15 downto 0);
		Kd_reg : in std_logic_vector(15 downto 0);
		Kf_reg : in std_logic_vector(15 downto 0);
		Mp_reg : in std_logic_vector(15 downto 0);
		Mi_reg : in std_logic_vector(15 downto 0);
		Md_reg : in std_logic_vector(15 downto 0);
		Mf_reg : in std_logic_vector(15 downto 0);
		bias_high_reg : in std_logic_vector(15 downto 0);
		bias_low_reg : in std_logic_vector(15 downto 0);
		limit : in std_logic_vector(15 downto 0);
		
		mem_idx : out std_logic_vector(15 downto 0);
		strobe : out std_logic;
		mem_ready : in std_logic;
		ADC_V_data_o : out std_logic_vector(15 downto 0);
		ADC_C_data_o : out std_logic_vector(15 downto 0);
		DAC_high_data_o : out std_logic_vector(15 downto 0);
		DAC_low_data_o : out std_logic_vector(15 downto 0)
	);
end controller;

architecture behav of controller is
	signal DAC_high_data : std_logic_vector(15 downto 0);
	signal DAC_high_send_d : std_logic;
	signal DAC_low_data : std_logic_vector(15 downto 0);
	signal DAC_low_send_d : std_logic;
	
	signal ADC_V_data : std_logic_vector(15 downto 0);
	signal ADC_V_data_samp : std_logic_vector(15 downto 0);
	signal ADC_V_data_d : std_logic;
	signal ADC_C_data : std_logic_vector(15 downto 0);
	signal ADC_C_data_samp : std_logic_vector(15 downto 0);
	signal ADC_C_data_d : std_logic;
	
	signal cycles : integer range 0 to 2**16 := to_integer(unsigned(N_samples));
	type controller_state_t is (reset_state, reset_wait_state, pre_ready_state, ready_state,
	                            collect_data_state, diff_state, accu_state, mult_state, div_state, calc_state_1, calc_state_2, post_adjust_state, set_DAC_state);
	signal current_state : controller_state_t;
	signal next_state : controller_state_t;
begin
	set_back_p: process(cycles) is
	begin
		if cycles = 0 then
			start_o <= '1';
		else
			start_o <= '0';
		end if;
	end process;
	
	g_p: process(res_n, clk_h) is
	begin
		if res_n = '0' then
			current_state <= reset_state;
		elsif clk_h'event and clk_h = '1' then
			current_state <= next_state;
		end if;
	end process;
	
	calc_next_state_p: process(current_state, mem_ready, ADC_V_data_d, ADC_C_data_d, voltage_current, start_i, cycles, N_samples) is
	begin
		case current_state is
			when reset_state =>
				next_state <= reset_wait_state;
			when reset_wait_state =>
				if start_i = '1' then
					next_state <= pre_ready_state;
				else
					next_state <= reset_wait_state;
				end if;
			when pre_ready_state =>
				next_state <= pre_ready_state;
				if voltage_current = '1' then
					if ADC_V_data_d = '1' then
						next_state <= ready_state;
					end if;
				else
					if ADC_C_data_d = '1' then
						next_state <= ready_state;
					end if;
				end if;
			when ready_state =>
				next_state <= ready_state;
				if (ADC_V_data_d = '1') and (ADC_C_data_d = '1') then    -- since all ADCs run synchronously, this ugly 'and-condition' will work...
					next_state <= collect_data_state;
				end if;
			when collect_data_state =>
				next_state <= diff_state;
			when diff_state =>
				next_state <= accu_state;
			when accu_state =>
				next_state <= mult_state;
			when mult_state =>
				next_state <= div_state;
			when div_state =>
				next_state <= calc_state_1;
			when calc_state_1 =>
				next_state <= calc_state_2;
			when calc_state_2 =>
				next_state <= post_adjust_state;
			when post_adjust_state =>
				next_state <= set_DAC_state;
			when set_DAC_state =>
				if cycles = (to_integer(unsigned(N_samples)) - 1) then    -- need to subtract 1 since cycles is incremented just inside set_DAC_state.
					next_state <= reset_state;
				else
					next_state <= ready_state;
				end if;
		end case;
	end process;
	
	work_p: process(res_n, clk_h) is
		type control_type_t is (control_voltage, control_current);
		variable cntrl_type : control_type_t;
		variable first_cycle : std_logic;
		variable ist : integer range 0 to 2**16-1;
		variable soll : integer range 0 to 2**16-1;
		variable e : integer;
		variable eold : integer;
		variable eold_m_e : integer;
		variable esum : integer;
		variable stell : integer;
		variable stell_h : integer range 0 to 2**16-1;
		variable stell_l : integer range 0 to 2**16-1;
		variable Kp : integer range 0 to 2**16-1;
		variable Mp : integer range 0 to 2**16-1;
		variable Ki : integer range 0 to 2**16-1;
		variable Mi : integer range 0 to 2**16-1;
		variable Kd : integer range 0 to 2**16-1;
		variable Md : integer range 0 to 2**16-1;
		variable Kf : integer range 0 to 2**16-1;
		variable Mf : integer range 0 to 2**16-1;
		variable bias_high : integer range 0 to 2**16-1;
		variable bias_low : integer range 0 to 2**16-1;
		variable p_portion : signed(31 downto 0);
		variable i_portion : signed(31 downto 0);
		variable d_portion : signed(31 downto 0);
		variable f_portion : signed(31 downto 0);
		variable p_portion_sh : signed(31 downto 0);
		variable i_portion_sh : signed(31 downto 0);
		variable d_portion_sh : signed(31 downto 0);
		variable f_portion_sh : signed(31 downto 0);
	begin
		if res_n = '0' then
		elsif clk_h'event and clk_h = '1' then
			case current_state is
				when reset_state =>
					DAC_high_data <= std_logic_vector(to_unsigned(0, DAC_high_data'length));
					DAC_low_data <= std_logic_vector(to_unsigned(0, DAC_low_data'length));
					DAC_high_send_d <= '1';
					DAC_low_send_d <= '1';
					strobe <= '0';
				when reset_wait_state =>
				   esum := 0;
					eold := 0;
					first_cycle := '1';
					
					if voltage_current = '1' then
						cntrl_type := control_voltage;
					else
						cntrl_type := control_current;
					end if;
					
					cycles <= 0;
					DAC_high_send_d <= '0';
					DAC_low_send_d <= '0';
					strobe <= '0';
				when pre_ready_state =>
					cycles <= 0;
					DAC_high_send_d <= '0';
					DAC_low_send_d <= '0';
					strobe <= '0';
				when ready_state =>
					soll := to_integer(unsigned(set_point_reg));
					Kp := to_integer(unsigned(Kp_reg));
					Mp := to_integer(unsigned(Mp_reg));
					Ki := to_integer(unsigned(Ki_reg));
					Mi := to_integer(unsigned(Mi_reg));
					Kd := to_integer(unsigned(Kd_reg));
					Md := to_integer(unsigned(Md_reg));
					Kf := to_integer(unsigned(Kf_reg));
					Mf := to_integer(unsigned(Mf_reg));
					bias_high := to_integer(unsigned(bias_high_reg));
					bias_low := to_integer(unsigned(bias_low_reg));
					
					DAC_high_send_d <= '0';
					DAC_low_send_d <= '0';
					strobe <= '0';
				when collect_data_state =>
					if cntrl_type = control_voltage then
						ist := to_integer(unsigned(ADC_V_data));
					else
						ist := to_integer(unsigned(ADC_C_data)) - 32768;
					end if;
					strobe <= '0';
					
					ADC_V_data_samp <= ADC_V_data;
					ADC_C_data_samp <= ADC_C_data;
				when diff_state =>
					e := soll - ist;
					strobe <= '0';
				when accu_state =>
					if first_cycle = '1' then
						first_cycle := '0';
						eold := e;
						eold_m_e := 0;
					else
						eold_m_e := e - eold;
					end if;
					esum := esum + e;
					strobe <= '0';
				when mult_state =>
					p_portion := to_signed(Kp * e, p_portion'length);
					i_portion := to_signed(Ki * esum, i_portion'length);
					d_portion := to_signed(Kd * eold_m_e, d_portion'length);
					f_portion := to_signed(Kf * soll, f_portion'length);
					strobe <= '0';
				when div_state =>
					p_portion_sh := shift_right(p_portion, Mp);
					i_portion_sh := shift_right(i_portion, Mi);
					d_portion_sh := shift_right(d_portion, Md);
					f_portion_sh := shift_right(f_portion, Mf);
					strobe <= '0';
				when calc_state_1 =>
					stell := to_integer(p_portion_sh) + to_integer(i_portion_sh);
					strobe <= '0';
				when calc_state_2 =>
					stell := stell + to_integer(d_portion_sh) + to_integer(f_portion_sh);
					strobe <= '0';
				when post_adjust_state =>
					-- limit and anti wind up
					if stell > 65535 then
						esum := esum - e;
						stell := 65535;
					end if;
					if stell < (-65535) then
						esum := esum - e;
						stell := (-65535);
					end if;
					
					-- cross over between high side and low side DAC/MOSFET
					if stell >= 0 then
						stell_h := stell + bias_high;
						stell_l := bias_low;
					else
						stell_h := bias_high;
						stell_l := (-stell) + bias_low;
					end if;
					
					DAC_high_send_d <= '0';
					DAC_low_send_d <= '0';
					strobe <= '0';
				when set_DAC_state =>
					eold := e;
					
					if calibration_i = '0' then
						cycles <= cycles + 1;
					end if;
					
					DAC_high_data <= std_logic_vector(to_unsigned(stell_h, DAC_high_data'length));
					DAC_high_data_o <= std_logic_vector(to_unsigned(stell_h, DAC_high_data_o'length));
					DAC_high_send_d <= '1';
					
					DAC_low_data <= std_logic_vector(to_unsigned(stell_l, DAC_low_data'length));
					DAC_low_data_o <= std_logic_vector(to_unsigned(stell_l, DAC_low_data_o'length));
					DAC_low_send_d <= '1';
					
					ADC_V_data_o <= ADC_V_data_samp;
					ADC_C_data_o <= ADC_C_data_samp;
					mem_idx <= std_logic_vector(to_unsigned(cycles, mem_idx'length));
					strobe <= '1';
			end case;
		end if;
	end process;
	
	dac_h: entity DAC
	port map(
		res_n => res_n,
		clk => clk_h,
		DAC_nSYNC => DAC_high_nSYNC,
		DAC_SCLK => DAC_high_SCLK,
		DAC_DIN => DAC_high_DIN,
		DAC_data => DAC_high_data,
		DAC_send_d => DAC_high_send_d
	);
	
	dac_l: entity DAC
	port map(
		res_n => res_n,
		clk => clk_h,
		DAC_nSYNC => DAC_low_nSYNC,
		DAC_SCLK => DAC_low_SCLK,
		DAC_DIN => DAC_low_DIN,
		DAC_data => DAC_low_data,
		DAC_send_d => DAC_low_send_d
	);

	adc_v_h: entity ADC
	generic map(
		range_sel => uni_1_25_V_ref
	)
	port map(
		res_n => res_n,
		clk => clk_h,
		software_reset => software_reset,
		ADC_nRST => ADC_V_nRST,
		ADC_SDI => ADC_V_SDI,
		ADC_nCS => ADC_V_nCS,
		ADC_SCLK => ADC_V_SCLK,
		ADC_SDO_0 => ADC_V_SDO_0,
		ADC_RVS => ADC_V_RVS,
		
		ADC_data => ADC_V_data,
		ADC_data_d => ADC_V_data_d
	);
	
	adc_c_h: entity ADC
	generic map(
		range_sel => bipol_1_25_V_ref
	)
	port map(
		res_n => res_n,
		clk => clk_h,
		software_reset => software_reset,
		ADC_nRST => ADC_C_nRST,
		ADC_SDI => ADC_C_SDI,
		ADC_nCS => ADC_C_nCS,
		ADC_SCLK => ADC_C_SCLK,
		ADC_SDO_0 => ADC_C_SDO_0,
		ADC_RVS => ADC_C_RVS,
		
		ADC_data => ADC_C_data,
		ADC_data_d => ADC_C_data_d
	);
end architecture;
