use work.all;
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity SDRAM_Controller is
    generic(
        -- ram device bus widths
        SDRAM_ROW_WIDTH_BIT  : integer := 13;
        SDRAM_COL_WIDTH_BIT  : integer := 9;
        SDRAM_BANK_WIDTH_BIT : integer := 2;
        SDRAM_DATA_WIDTH_BIT : integer := 16;
        
        -- timing parameters
        SDRAM_CAS            : integer := 3;
        tPOWERUP             : integer := 14285;
        tREF                 : integer := 64;    -- 64ms
        tREF_COUNT           : integer := 4;  -- 8192
        tCK                  : integer := 7
    );
    port(
        res_n             : in    std_logic;
        clk               : in    std_logic;
        clk_sdram         : in    std_logic;
        
        -- application pins
        i_ram_data        : in    std_logic_vector(SDRAM_DATA_WIDTH_BIT-1 downto 0);
        o_ram_data        : out   std_logic_vector(SDRAM_DATA_WIDTH_BIT-1 downto 0);
        i_ram_addr        : in    std_logic_vector(SDRAM_ROW_WIDTH_BIT + SDRAM_COL_WIDTH_BIT + SDRAM_BANK_WIDTH_BIT - 1 downto 0);
        i_ram_len         : in    std_logic_vector(SDRAM_COL_WIDTH_BIT downto 0);
        i_ram_read_req    : in    std_logic;
        o_ram_read_valid  : out   std_logic;
        i_ram_write_req   : in    std_logic;
        o_ram_write_valid : out   std_logic;
        o_ready           : out   std_logic;
        
        -- pins to ram device
        i_sdram_data      : in    std_logic_vector(SDRAM_DATA_WIDTH_BIT-1 downto 0);
        o_sdram_data      : out   std_logic_vector(SDRAM_DATA_WIDTH_BIT-1 downto 0);
        o_sdram_addr      : out   std_logic_vector(SDRAM_ROW_WIDTH_BIT-1 downto 0);
        o_sdram_clk       : out   std_logic;
        o_sdram_cke       : out   std_logic;
        o_sdram_cs_l      : out   std_logic;
        o_sdram_ras_l     : out   std_logic;
        o_sdram_cas_l     : out   std_logic;
        o_sdram_we_l      : out   std_logic;
        o_sdram_buf_we    : out   std_logic;
        o_sdram_ba        : out   std_logic_vector(SDRAM_BANK_WIDTH_BIT-1 downto 0);
        o_sdram_dqml      : out   std_logic;
        o_sdram_dqmh      : out   std_logic
    );
end SDRAM_Controller;

architecture behav of SDRAM_Controller is
    -- timing constants
    constant tRP  : integer := SDRAM_CAS;
    constant tRC  : integer := 6 + SDRAM_CAS;
    constant tMRD : integer := 2;
    constant tRCD : integer := SDRAM_CAS;
    
    -- command definition
    subtype cmd_t is std_logic_vector(3 downto 0);
                           -- CS RAS CAS WE
    constant CMD_DESL  : cmd_t := "1111";    -- device deselect
    constant CMD_NOP   : cmd_t := "0111";    -- no operation
    constant CMD_BST   : cmd_t := "0110";    -- burst stop
    constant CMD_READ  : cmd_t := "0101";    -- read
    constant CMD_WRITE : cmd_t := "0100";    -- write
    constant CMD_ACT   : cmd_t := "0011";    -- bank activate
    constant CMD_PRE   : cmd_t := "0010";    -- precharge select bank
    constant CMD_REF   : cmd_t := "0001";    -- CBR auto-refresh
    constant CMD_MRS   : cmd_t := "0000";    -- mode register set
    signal sdram_cmd   : cmd_t;
    
    constant ADDR_WIDTH : integer := SDRAM_ROW_WIDTH_BIT + SDRAM_COL_WIDTH_BIT + SDRAM_BANK_WIDTH_BIT - 1;
    
    type state_t is (init_start, init_powerup, init_pre, init_nop1, init_nop2,
                     init_ref1, init_wait1, init_ref2, init_wait2, init_wait3,
                     init_LMR, init_LMR1, init_LMR2,
                     idle,
                     ref, ref_nop1, ref_nop2, ref_aref, ref_wait,
                     write, write_act, write_nop1, write_nop2, write_D0, write_N, write_stop0, write_stop1, write_stop_pre,
                     read, read_act, read_nop1, read_nop2, read_start, read_wait1, read_wait2, read_wait3, read_data, read_end, read_push_back);
    signal current_state : state_t;
    
    signal wait_cnt_start : std_logic;
    signal wait_cnt_end : std_logic;
    signal wait_cnt_load : integer;
    
    signal ref_en : std_logic;
    signal ref_req : std_logic;
    signal ref_ok : std_logic;
    signal ram_write_valid : std_logic;
    signal ram_read_valid : std_logic;
    signal sdram_buf_we : std_logic;
    
    signal ready : std_logic;
    signal sdram_cke : std_logic;
    signal sdram_dqml : std_logic;
    signal sdram_dqmh : std_logic;
    signal sdram_addr : std_logic_vector(o_sdram_addr'range);
    signal sdram_ba : std_logic_vector(o_sdram_ba'range);
    signal ram_col : std_logic_vector(SDRAM_COL_WIDTH_BIT-1 downto 0);
    signal ram_row : std_logic_vector(SDRAM_ROW_WIDTH_BIT-1 downto 0);
    signal ram_ba : std_logic_vector(SDRAM_BANK_WIDTH_BIT-1 downto 0);
    
    signal i_sdram_data_smp : std_logic_vector(SDRAM_DATA_WIDTH_BIT-1 downto 0);
begin
    o_sdram_clk       <= not clk_sdram;
    o_sdram_cs_l      <= sdram_cmd(3);
    o_sdram_ras_l     <= sdram_cmd(2);
    o_sdram_cas_l     <= sdram_cmd(1);
    o_sdram_we_l      <= sdram_cmd(0);
    o_sdram_buf_we    <= sdram_buf_we;
    o_sdram_cke       <= sdram_cke;
    o_sdram_dqml      <= sdram_dqml;
    o_sdram_dqmh      <= sdram_dqmh;
    o_sdram_addr      <= sdram_addr;
    o_sdram_ba        <= sdram_ba;
    
    o_ram_write_valid <= ram_write_valid;
    o_ram_read_valid  <= ram_read_valid;
    o_ready           <= ready and not ref_req;
    
    -- sample i_sdram_data at rising edge of clk_sdram
    smp_p: process(res_n, clk_sdram) is
    begin
        if res_n = '0' then
        elsif clk_sdram'event and clk_sdram = '1' then
				if current_state = read_end then -- hint!
                i_sdram_data_smp <= i_sdram_data;
            end if;
        end if;
    end process;
    
    -- wait counter
    wait_cnt_p: process(res_n, clk) is
        type w_st_t is (state_0, state_1);
        variable w_st : w_st_t;
        variable wait_cnt : integer;
    begin
        if res_n = '0' then
            w_st := state_0;
        elsif clk'event and clk = '1' then
            case w_st is
                when state_0 =>
                    if wait_cnt_start = '1' then
                        wait_cnt_end <= '0';
                        wait_cnt := wait_cnt_load;
                        w_st := state_1;
                    end if;
                when state_1 =>
                    if wait_cnt /= 0 then
                        wait_cnt := wait_cnt -1;
                        wait_cnt_end <= '0';
                    else
                        wait_cnt_end <= '1';
                        w_st := state_0;
                    end if;
            end case;
        end if;
    end process;
    
    -- refresh counter
    ref_cnt_p: process(res_n, clk) is
        constant REFRESH_INTERVAL : integer := (((tREF * 1000000) / tREF_COUNT) / tCK);
        variable ref_cnt : integer := 0;
    begin
        if res_n = '0' then
            ref_cnt := 0;
            ref_req <= '0';
        elsif clk'event and clk = '1' then
            if ref_en = '1' then
                if ref_cnt = REFRESH_INTERVAL then
                    ref_cnt := 0;
                    ref_req <= '1';
                else
                    ref_cnt := ref_cnt + 1;
                    if ref_ok = '1' then
                        ref_req <= '0';
                    end if;
                end if;
            end if;
        end if;
    end process;

    work_p: process(res_n, clk) is
    begin
        if res_n = '0' then
        elsif clk'event and clk = '1' then
            case current_state is
                when init_start =>
                    current_state <= init_powerup;
                    ready <= '0';
                    sdram_cke <= '1';
                    sdram_dqml <= '0';
                    sdram_dqmh <= '0';
                    sdram_addr <= (others => '0');
                    sdram_ba <= (others => '0');
                    sdram_cmd <= CMD_NOP;
                    wait_cnt_load <= tPOWERUP;
                    wait_cnt_start <= '1';
						  ref_en <= '0';
                when init_powerup =>
                    if wait_cnt_end = '1' then
                        current_state <= init_pre;
                    else
                        current_state <= init_powerup;
                    end if;
                    wait_cnt_start <= '0';
                    sdram_cmd <= CMD_NOP;
                when init_pre =>
                    current_state <= init_nop1;
                    sdram_cmd <= CMD_PRE;
                    sdram_addr(10) <= '1';
                when init_nop1 =>
                    if tRP = 3 then
                        current_state <= init_nop2;
                    else
                        current_state <= init_ref1;
                    end if;
                    sdram_cmd <= CMD_NOP;
                when init_nop2 =>
                    current_state <= init_ref1;
                    sdram_cmd <= CMD_NOP;
                when init_ref1 =>
                    current_state <= init_wait1;
                    sdram_cmd <= CMD_REF;
                    --wait_cnt_load <= tRC - 4;
                    wait_cnt_load <= tRC - 2;
                    wait_cnt_start <= '1';
                when init_wait1 =>
                    current_state <= init_ref2;
                    wait_cnt_start <= '0';
                    sdram_cmd <= CMD_NOP;
                when init_ref2 =>
                    if wait_cnt_end = '1' then
                        current_state <= init_wait2;
                    else
                        current_state <= init_ref2;
                    end if;
                    if wait_cnt_end = '1' then
                        sdram_cmd <= CMD_REF;
                    end if;
                    --wait_cnt_load <= tRC - 4;
                    wait_cnt_load <= tRC - 2;
                    wait_cnt_start <= '1';
                when init_wait2 =>
                    current_state <= init_wait3;
                    sdram_cmd <= CMD_NOP;
                    wait_cnt_start <= '0';
                when init_wait3 =>
                    if wait_cnt_end = '1' then
                        current_state <= init_LMR;
                    else
                        current_state <= init_wait3;
                    end if;
                when init_LMR =>
                    current_state <= init_LMR1;
                    sdram_cmd <= CMD_MRS;
                    --sdram_addr <= (others => '0');
                    --sdram_addr(6 downto 4) <= std_logic_vector(to_unsigned(SDRAM_CAS, 3));  -- latency mode
                    --sdram_addr(3 downto 0) <= "0111";  -- burst length = full page
                    
                    sdram_addr <= "0001000110000";    -- single address access
                    
                    sdram_ba <= (others => '0');
                when init_LMR1 =>
                    current_state <= init_LMR2;
                    sdram_cmd <= CMD_NOP;
                when init_LMR2 =>
                    current_state <= idle;
                    sdram_cmd <= CMD_NOP;
                    ready <= '1';
                    ref_en <= '1';  -- enable refresh
                when idle =>
                    if ref_req = '1' then
                        current_state <= ref;
                        --ready <= '0';
                    elsif i_ram_read_req = '1' then
                        current_state <= read;
                        --ready <= '0';
                    elsif i_ram_write_req = '1' then
                        current_state <= write;
                        --ready <= '0';
                    else
                        current_state <= idle;
                        sdram_cmd <= CMD_NOP;
                        --ready <= '1';
                    end if;
                    ready <= '1';
                    sdram_cmd <= CMD_NOP;
                when ref =>
                    current_state <= ref_nop1;
                    ready <= '0';
                    ref_ok <= '1';
                    sdram_cmd <= CMD_PRE;
                    sdram_addr(10) <= '1';  -- all banks
                when ref_nop1 =>
                    current_state <= ref_nop2;
                    sdram_cmd <= CMD_NOP;
                when ref_nop2 =>
                    current_state <= ref_aref;
                    sdram_cmd <= CMD_NOP;
                    wait_cnt_load <= tRC - 2;
                    wait_cnt_start <= '1';
                when ref_aref =>
                    current_state <= ref_wait;
                    ref_ok <= '0';
                    sdram_cmd <= CMD_REF;
                    wait_cnt_start <= '0';
                when ref_wait =>
                    if wait_cnt_end = '1' then
                        current_state <= idle;
                    else
                        current_state <= ref_wait;
                    end if;
                    sdram_cmd <= CMD_NOP;
                when write =>
                    current_state <= write_act;
                    ready <= '0';
                    ram_col <= i_ram_addr(SDRAM_COL_WIDTH_BIT-1 downto 0);
                    ram_row <= i_ram_addr(SDRAM_ROW_WIDTH_BIT + SDRAM_COL_WIDTH_BIT-1 downto SDRAM_COL_WIDTH_BIT);
                    ram_ba  <= i_ram_addr(SDRAM_ROW_WIDTH_BIT + SDRAM_COL_WIDTH_BIT + SDRAM_BANK_WIDTH_BIT-1 downto SDRAM_ROW_WIDTH_BIT + SDRAM_COL_WIDTH_BIT);
                when write_act =>
                    current_state <= write_nop1;
                    sdram_cmd <= CMD_ACT;
                    sdram_addr <= ram_row;
                    sdram_ba <= ram_ba;
                    if tRCD = 2 then
                        ram_write_valid <= '1';
                    end if;
                when write_nop1 =>
                    if tRCD = 3 then
                        current_state <= write_nop2;
                    else
                        current_state <= write_D0;
                    end if;
                    sdram_cmd <= CMD_NOP;
                    if tRCD = 3 then
                        ram_write_valid <= '1';
                    end if;
                when write_nop2 =>
                    current_state <= write_D0;
                    sdram_cmd <= CMD_NOP;
                    sdram_buf_we <= '0';
                when write_D0 =>
                    current_state <= write_N;
                    sdram_cmd <= CMD_WRITE;
                    o_sdram_data <= i_ram_data;
                    sdram_buf_we <= '1';
                    sdram_addr(SDRAM_COL_WIDTH_BIT-1 downto 0) <= ram_col;
                    sdram_ba <= ram_ba;
                when write_N =>
                    current_state <= write_stop0;
                    sdram_cmd <= CMD_NOP;
                    ram_write_valid <= '0';
                when write_stop0 =>
                    current_state <= write_stop1;
                    sdram_cmd <= CMD_NOP;
                    sdram_buf_we <= '0';
                when write_stop1 =>
                    current_state <= write_stop_pre;
                    sdram_cmd <= CMD_NOP;
                when write_stop_pre =>
                    current_state <= idle;
                    sdram_cmd <= CMD_PRE;
                    sdram_addr(10) <= '1';  -- precharge all banks
                when read =>
                    current_state <= read_act;
                    ram_read_valid <= '0';
                    ready <= '0';
                    ram_col <= i_ram_addr(SDRAM_COL_WIDTH_BIT-1 downto 0);
                    ram_row <= i_ram_addr(SDRAM_ROW_WIDTH_BIT + SDRAM_COL_WIDTH_BIT-1 downto SDRAM_COL_WIDTH_BIT);
                    ram_ba <= i_ram_addr(SDRAM_ROW_WIDTH_BIT + SDRAM_COL_WIDTH_BIT + SDRAM_BANK_WIDTH_BIT-1 downto SDRAM_ROW_WIDTH_BIT + SDRAM_COL_WIDTH_BIT);
                when read_act =>
                    current_state <= read_nop1;
                    sdram_cmd <= CMD_ACT;
                    sdram_addr <= ram_row;
                    sdram_ba <= ram_ba;
                when read_nop1 =>
                    if tRCD = 3 then
                        current_state <= read_nop2;
                    else
                        current_state <= read_start;
                    end if;
                    sdram_cmd <= CMD_NOP;
                when read_nop2 =>
                    current_state <= read_start;
                    sdram_cmd <= CMD_NOP;
                    sdram_buf_we <= '0';
                when read_start =>
                    current_state <= read_wait1;
                    sdram_cmd <= CMD_READ;
                    sdram_buf_we <= '0';
                    sdram_addr(SDRAM_COL_WIDTH_BIT-1 downto 0) <= ram_col;
                    sdram_addr(10) <= '0';    -- no auto-precharge (do it manually in state read_end)
                    sdram_ba <= ram_ba;
                when read_wait1 =>
                    if SDRAM_CAS = 3 then
                        current_state <= read_wait2;
                    else
                        current_state <= read_wait3;
                    end if;
                    sdram_cmd <= CMD_NOP;
                when read_wait2 =>
                    current_state <= read_wait3;
                    sdram_cmd <= CMD_NOP;
                when read_wait3 =>
                    current_state <= read_data;
                    sdram_cmd <= CMD_NOP;
                    ram_read_valid <= '1';
                when read_data =>
                    current_state <= read_end;
                    ram_read_valid <= '0';
                    sdram_cmd <= CMD_NOP;
                when read_end =>
                    current_state <= read_push_back;
                    ram_read_valid <= '0';
                    sdram_cmd <= CMD_PRE;
                    sdram_addr(10) <= '1';  -- precharge all banks
					 when read_push_back =>
						  current_state <= idle;
                    ram_read_valid <= '0';
                    sdram_cmd <= CMD_NOP;
                    sdram_addr(10) <= '0';
						  o_ram_data <= i_sdram_data_smp;
            end case;
        end if;
    end process;
end architecture;
