use work.all;
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use work.Definitions.all;

entity ADC is
	generic(
		range_sel : range_sel_t
	);
	port(
		res_n : in std_logic;
		clk : in std_logic;
		software_reset : in std_logic;
		
		ADC_nRST : out std_logic;
		ADC_SDI : out std_logic;
		ADC_nCS : out std_logic;
		ADC_SCLK : out std_logic;
		ADC_SDO_0 : in std_logic;
		ADC_RVS : in std_logic;
		
		ADC_data : out std_logic_vector(15 downto 0);
		ADC_data_d : out std_logic
	);
end ADC;

architecture behav of ADC is
	type state_ll_t is (ll_reset, ll_apply_data, ll_wait_1, ll_wait_1e, ll_clk_re, ll_wait_2, ll_wait_2e, ll_clk_fe);
	signal ll_current_state : state_ll_t := ll_reset;
	signal ll_next_state : state_ll_t := ll_reset;
	signal ll_start_com : std_logic := '0';
	signal ll_com_d : std_logic;
	signal ll_tx_data : std_logic_vector(31 downto 0);
	signal ll_rx_data : std_logic_vector(31 downto 0);
	signal ll_data_idx : integer range 0 to 32;
	signal ll_data_next_idx : integer range 0 to 32;
	
	type state_hl_t is (hl_reset, ADC_reset, ADC_reset_boot, hl_set_config_reg_1, hl_set_config_reg_2, hl_read_adc, hl_read_adc_d);
	signal hl_current_state : state_hl_t := hl_reset;
	signal hl_next_state : state_hl_t := hl_reset;
	
	signal ADC_reset_cnt : integer range 0 to 32767;
	signal ADC_reset_cnt_trigger : std_logic;
	signal ADC_reset_cnt_boot : integer range 0 to 8388607;
	signal ADC_reset_cnt_boot_trigger : std_logic;
begin
	ADC_reset_cnt_p: process(res_n, clk) is
	begin
		if res_n = '0' then
			ADC_reset_cnt <= 0;
		elsif clk'event and clk = '1' then
			if ADC_reset_cnt_trigger = '1' then
				ADC_reset_cnt <= ADC_reset_cnt + 1;
			else
				ADC_reset_cnt <= 0;
			end if;
		end if;
	end process;
	
	ADC_reset_cnt_boot_p: process(res_n, clk) is
	begin
		if res_n = '0' then
			ADC_reset_cnt_boot <= 0;
		elsif clk'event and clk = '1' then
			if ADC_reset_cnt_boot_trigger = '1' then
				ADC_reset_cnt_boot <= ADC_reset_cnt_boot + 1;
			else
				ADC_reset_cnt_boot <= 0;
			end if;
		end if;
	end process;

	hl_g_p: process(res_n, clk) is
	begin
		if res_n = '0' then
			hl_current_state <= hl_reset;
		elsif clk'event and clk = '1' then
			hl_current_state <= hl_next_state;
		end if;
	end process;
	
	hl_calc_next_state: process(hl_current_state, ll_com_d, ll_data_idx, software_reset, ADC_reset_cnt, ADC_reset_cnt_boot) is
	begin
		case hl_current_state is
			when hl_reset =>
				if ll_com_d = '1' then
					if software_reset = '0' then
						hl_next_state <= hl_reset;
					else
						hl_next_state <= ADC_reset;
					end if;
				else
					hl_next_state <= hl_reset;
				end if;
			when ADC_reset =>
				--if ADC_reset_cnt = 16384 then
				if ADC_reset_cnt = 26384 then
					hl_next_state <= ADC_reset_boot;
				else
					hl_next_state <= ADC_reset;
				end if;
			when ADC_reset_boot =>
				--if ADC_reset_cnt_boot = 4194303 then
				if ADC_reset_cnt_boot = 8194303 then
					hl_next_state <= hl_set_config_reg_1;
				else
					hl_next_state <= ADC_reset_boot;
				end if;
			when hl_set_config_reg_1 =>
				if ll_data_idx = 32 then
					hl_next_state <= hl_set_config_reg_2;
				else
					hl_next_state <= hl_set_config_reg_1;
				end if;
			when hl_set_config_reg_2 =>
				if ll_data_idx = 32 then
					hl_next_state <= hl_read_adc;
				else
					hl_next_state <= hl_set_config_reg_2;
				end if;
			when hl_read_adc =>
				if ll_data_idx = 32 then
					hl_next_state <= hl_read_adc_d;
				else
					hl_next_state <= hl_read_adc;
				end if;
			when hl_read_adc_d =>
				if software_reset = '1' then
					hl_next_state <= hl_read_adc;
				else
					hl_next_state <= hl_reset;
				end if;
		end case;
	end process;
	
	hl_work_p: process(hl_current_state, ll_rx_data) is
	begin
		case hl_current_state is
			when hl_reset =>
				ADC_reset_cnt_trigger <= '0';
				ADC_reset_cnt_boot_trigger <= '0';
				ADC_nRST <= '0';
				ll_tx_data <= (others => '0');
				ll_start_com <= '0';
				ADC_data <= X"0000";
				ADC_data_d <= '0';
			when ADC_reset =>
				ADC_reset_cnt_trigger <= '1';
				ADC_reset_cnt_boot_trigger <= '0';
				ADC_nRST <= '0';
				ll_tx_data <= (others => '0');
				ll_start_com <= '0';
				ADC_data <= X"0000";
				ADC_data_d <= '0';
			when ADC_reset_boot =>
				ADC_reset_cnt_trigger <= '0';
				ADC_reset_cnt_boot_trigger <= '1';
				ADC_nRST <= '1';
				ll_tx_data <= (others => '0');
				ll_start_com <= '0';
				ADC_data <= X"0000";
				ADC_data_d <= '0';
			when hl_set_config_reg_1 =>
				ADC_reset_cnt_trigger <= '0';
				ADC_reset_cnt_boot_trigger <= '0';
				ADC_nRST <= '1';
				ll_tx_data <= X"D0160000";    -- RANGE_SEL_REG[31:16] = 0x0000 (see page 43 and page 54 of ADS8681 data sheet)
				ll_start_com <= '1';
				ADC_data <= X"0000";
				ADC_data_d <= '0';
			when hl_set_config_reg_2 =>
				ADC_reset_cnt_trigger <= '0';
				ADC_reset_cnt_boot_trigger <= '0';
				ADC_nRST <= '1';
				case range_sel is
					when uni_1_25_V_ref =>
						ll_tx_data <= X"D014000B";    -- RANGE_SEL_REG[15:0] = 0x000B (see page 43 and page 54 of ADS8681 data sheet)
					when bipol_1_25_V_ref =>
						ll_tx_data <= X"D0140003";    -- RANGE_SEL_REG[15:0] = 0x0003 (see page 43 and page 54 of ADS8681 data sheet)
				end case;
				ll_start_com <= '1';
				ADC_data <= X"0000";
				ADC_data_d <= '0';
			when hl_read_adc =>
				ADC_reset_cnt_trigger <= '0';
				ADC_reset_cnt_boot_trigger <= '0';
				ADC_nRST <= '1';
				ll_tx_data <= X"00000000";
				ll_start_com <= '1';
				--ADC_data <= X"0000";
				ADC_data <= ll_rx_data(31 downto 16);
				ADC_data_d <= '0';
			when hl_read_adc_d =>
				ADC_reset_cnt_trigger <= '0';
				ADC_reset_cnt_boot_trigger <= '0';
				ADC_nRST <= '1';
				ll_tx_data <= X"00000000";
				ll_start_com <= '0';
				ADC_data <= ll_rx_data(31 downto 16);
				ADC_data_d <= '1';
		end case;
	end process;

-- low level stuff below here --
	ll_g_p: process(res_n, clk) is
	begin
		if res_n = '0' then
			ll_current_state <= ll_reset;
			ll_data_idx <= 0;
		elsif clk'event and clk = '1' then
			ll_current_state <= ll_next_state;
			ll_data_idx <= ll_data_next_idx;
		end if;
	end process;
	
	ll_calc_next_state: process(ll_current_state, ll_start_com, ll_data_idx) is
	begin
		case ll_current_state is
			when ll_reset =>
				if ll_start_com = '1' then
					ll_next_state <= ll_apply_data;
				else
					ll_next_state <= ll_reset;
				end if;
				ll_data_next_idx <= 0;
			when ll_apply_data =>
				ll_next_state <= ll_wait_1;
				ll_data_next_idx <= ll_data_idx;
			when ll_wait_1 =>
				ll_next_state <= ll_wait_1e;
				ll_data_next_idx <= ll_data_idx;
			when ll_wait_1e =>
				ll_next_state <= ll_clk_re;
				ll_data_next_idx <= ll_data_idx;
			when ll_clk_re =>
				ll_next_state <= ll_wait_2;
				ll_data_next_idx <= ll_data_idx;
			when ll_wait_2 =>
				ll_next_state <= ll_wait_2e;
				ll_data_next_idx <= ll_data_idx;
			when ll_wait_2e =>
				ll_next_state <= ll_clk_fe;
				ll_data_next_idx <= ll_data_idx + 1;
			when ll_clk_fe =>
				if ll_data_idx < 32 then
					ll_next_state <= ll_apply_data;
					ll_data_next_idx <= ll_data_idx;
				else
					ll_next_state <= ll_reset;
					ll_data_next_idx <= 0;
				end if;
		end case;
	end process;
	
	ll_work_p: process(ll_current_state, ll_tx_data, ll_data_idx, ll_start_com) is
	begin
		case ll_current_state is
			when ll_reset =>
				ADC_nCS <= '1';
				ADC_SCLK <= '0';
				ADC_SDI <= '0';
            ll_com_d <= '1';
			when ll_apply_data =>
				ADC_nCS <= '0';
				ADC_SCLK <= '0';
				ADC_SDI <= ll_tx_data(31 - ll_data_idx);
				ll_com_d <= '0';
			when ll_wait_1 =>
				ADC_nCS <= '0';
				ADC_SCLK <= '0';
				ADC_SDI <= ll_tx_data(31 -ll_data_idx);
				ll_com_d <= '0';
			when ll_wait_1e =>
				ADC_nCS <= '0';
				ADC_SCLK <= '1';
				ADC_SDI <= ll_tx_data(31 - ll_data_idx);
				ll_com_d <= '0';
			when ll_clk_re =>
				ADC_nCS <= '0';
				ADC_SCLK <= '1';
				ADC_SDI <= ll_tx_data(31 - ll_data_idx);
				ll_com_d <= '0';
			when ll_wait_2 =>
				ADC_nCS <= '0';
				ADC_SCLK <= '1';
				ADC_SDI <= ll_tx_data(31 - ll_data_idx);
				ll_com_d <= '0';
			when ll_wait_2e =>
				ADC_nCS <= '0';
				ADC_SCLK <= '0';
				ADC_SDI <= ll_tx_data(31 - ll_data_idx);
				ll_com_d <= '0';
			when ll_clk_fe =>
				ADC_nCS <= '0';
				ADC_SCLK <= '0';
				if ll_data_idx < 32 then
                ADC_SDI <= ll_tx_data(31 - ll_data_idx);
            else
                ADC_SDI <= '0';
            end if;
            ll_com_d <= '0';
		end case;
	end process;
	
	samp_p: process(res_n, clk) is
	begin
		if res_n = '0' then
			ll_rx_data <= (others => '0');
		elsif clk'event and clk ='1' then
			if ll_current_state = ll_clk_re then
				ll_rx_data(31 - ll_data_idx) <= ADC_SDO_0;
			end if;
		end if;
	end process;
end architecture;
