/*
    PROGRAM IC705_get_F  reads the current frequency via Bluetooth connection
    the 0.96" OLED Display shows VFO-frequency and Ham-band in meter (if wanted)
    program version  2.5 from 04/08/2022  edited by Dr.sc.Peter Heblik , DD6USB
    with an ESP32 and libraries from Wilfried(Willi)Dilling,DK8RW "CIVbusLib"
    if you use the 0.96" OLED the wiring is:
                ESP32    0.96"OLED
                3.3V   -   VCC
                GND    -   GND
                21     -   SDA
                22     -   SCL
    wiring between ESP32<->AD558<->StepUp_converter you find in a seperate file

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#define VERSION_STRING "Program: ESP32 BT to IC705"

#include <ICradio.h>
#include <CIVmaster.h>
#include <Wire.h>

//***********************************************************************************************
//    ***************>>  BEGIN of user definitions   <<*********************
//***********************************************************************************************
//#define debug                // scratch out if not used
#define oled                   // if exist SSD1306 OLED - otherwise scratch it out
#define V3                     // if DAC using 2.56V range
//#define V10                  // if DAC using 9.96V range
//***********************************************************************************************
// here define theESP32 output-pins for the DAC AD558
#define bit0_pin  26    // LSB
#define bit1_pin  25
#define bit2_pin  32
#define bit3_pin  33
#define bit4_pin  27
#define bit5_pin  14
#define bit6_pin  12
#define bit7_pin  13    // MSB
//************************************************************************************************
// band in m        160    80    60    40    30    20    17    15    12    10     6
//************************************************************************************************
float band_val[] = {0.23, 0.46, 0.69, 0.92, 1.15, 1.38, 1.61, 1.84, 2.07,  2.3,  2.53} ; // in Volts !
//************************************************************************************************
//   *****************>>   END of user definitions  <<***********************
//************************************************************************************************

const uint8_t bit_pins[] = {bit0_pin, bit1_pin, bit2_pin, bit3_pin, bit4_pin, bit5_pin, bit6_pin, bit7_pin} ; // the bit_pin-array

#define screen_width 128
#define screen_heigth 64

#ifdef oled
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
Adafruit_SSD1306  ol(screen_width, screen_heigth, &Wire) ; //instance for screen
#endif

CIV           civ ;
ICradio IC705(TypeIC705, CIV_ADDR_705) ;

#define LED   2                         // built in LED
#define BASELOOP_TICK 10                // determined in main libraries 

unsigned long old_VFO , VFO ;           // holds the VFO frequencies
unsigned long time_current_baseloop ;   // temporary time of the baseloop entry for calculations
unsigned long time_last_baseloop ;      // remember last value
uint8_t DAC_val  ;                      // DAC_value
String band_S = "" ;                    // band-string
bool paired ;                           // state of pairing
#ifdef V3
float max_Volt = 2.55 ;                 // max. output DAC in low Volt range
#endif
#ifdef V10                              // max. output DAC in high Volt range
float max_Volt = 9.98 ;
#endif
float bit_step = 0.0 , bval ;

//*************************************************************
void setup()
{
#ifdef debug                        // initialize the serial interface (for debug messages)
  Serial.begin(19200);
  delay(100);                       // give time to Serial start
  Serial.println (VERSION_STRING);
#endif
  pinMode(LED, OUTPUT) ;
  digitalWrite(LED, LOW) ;          // set pairing LED off
#ifdef oled
  init_oled() ;                     // init OLED (if used)
#endif
  civ.setupp(true);                 // initialize the civ object/module (true means "use BT")
  IC705.setupp(millis());           // registering of the CIV-address in civ is done
  time_current_baseloop = millis();
  time_last_baseloop = time_current_baseloop ;
#ifdef debug
  Serial.print("CIV_Addr of IC 705 = ") ;
  Serial.println(IC705.getCIVaddr(), HEX) ;
#endif
  for (uint8_t i = 0 ; i < 8 ; i++)
  {
    pinMode(bit_pins[i], OUTPUT) ;
    digitalWrite(bit_pins[i], LOW) ;     // init the DAC-output (set to 0V)
  }
  bit_step = max_Volt / 256.0 ;          // calculate single bit_step for the concerning range
#ifdef debug
  Serial.print("max_Volt = ") ;
  Serial.print(max_Volt) ;
  Serial.print (" , bit_step = ") ;
  Serial.println(bit_step, 5) ;
  Serial.println("init ends") ;
#endif
}

//*************************************************************

void loop(void)
{
  time_current_baseloop = millis() ;
  if ((time_current_baseloop - time_last_baseloop) > BASELOOP_TICK)
  {
    if (time_current_baseloop > t_RadioCheck)
    {
      IC705.loopp(time_current_baseloop) ; // loop after t_RadioCheck ms (completion of the bootup-phase of the controller)
    }
    paired = IC705.getAvailability() ;     // is radio on or off? -> update database
    if (paired)
    {
      digitalWrite(LED, HIGH);             // blue LED on = paired !
      determine_bands() ;                  // calculate ham-band and switch PA
      set_DAC() ;                          // set DAC-bit-pattern
    }
    else
    {
      digitalWrite(LED, LOW) ;             // if back from pairing
      DAC_val = 0 ;
      set_DAC() ;
    }
    time_last_baseloop = time_current_baseloop;
  }
}

//*************************************************************

void determine_bands(void)
{
  VFO = IC705.getFrequency() / 1e3 ;   // get freq in kHz
  if (old_VFO != VFO)
  {
    old_VFO = VFO ;
#ifdef debug
    Serial.print("VFO in f[kHz]: " + String(VFO)) ;
#endif
    switch (VFO)
    {
      case   1810 ...   2000 : band_S = "160 m band" ; bval = band_val[0] ; break ;
      case   3500 ...   3800 : band_S = " 80 m band" ; bval = band_val[1] ; break ;
      case   5351 ...   5366 : band_S = " 60 m band" ; bval = band_val[2] ; break ;
      case   7000 ...   7200 : band_S = " 40 m band" ; bval = band_val[3] ; break ;
      case  10100 ...  10150 : band_S = " 30 m band" ; bval = band_val[4] ; break ;
      case  14000 ...  14350 : band_S = " 20 m band" ; bval = band_val[5] ; break ;
      case  18068 ...  18168 : band_S = " 17 m band" ; bval = band_val[6] ; break ;
      case  21000 ...  21450 : band_S = " 15 m band" ; bval = band_val[7] ; break ;
      case  24890 ...  24990 : band_S = " 12 m band" ; bval = band_val[8] ; break ;
      case  28000 ...  29700 : band_S = " 10 m band" ; bval = band_val[9] ; break ;
      case  50300 ...  51990 : band_S = "  6 m band" ; bval = band_val[10]; break ;
      case 144000 ... 146000 : band_S = "  2 m band" ; bval =     0       ; break ;
      case 430000 ... 440000 : band_S = "70 cm band" ; bval =     0       ; break ;
      case  88000 ... 106000 : band_S = " FM Radio " ; bval =     0       ; break ;
      case 117975 ... 137000 : band_S = " AIR band " ; bval =     0       ; break ;
      case 137001 ... 140000 : band_S = "WeatherSat" ; bval =     0       ; break ;
      default :                band_S = " no bands " ; bval =     0       ; break ;
    }
    DAC_val = bval / bit_step ;                          // calculate DAC value
#ifdef debug
    bval != 0 ? Serial.println(" , band = " + band_S) : Serial.println(band_S) ;
    Serial.print("bval = ") ;
    Serial.print(bval) ;
    Serial.print(" , ") ;
    Serial.print(DAC_val, DEC) ;
    Serial.print(" , ") ;
    Serial.println(DAC_val,BIN) ;
#endif
#ifdef oled
    show_oled() ;                     // show frequ & band on OLED
#endif
  }
}

//**********************************************************

void set_DAC(void)
{
  for (uint8_t Bitpos = 0 ; Bitpos < 8 ; Bitpos++)
  {
    digitalWrite(bit_pins[Bitpos], bitRead(DAC_val, Bitpos)) ;
#ifdef debug1
    Serial.print(bitRead(DAC_val, Bitpos)) ;
    Serial.print(" , pin = ") ;
    Serial.print(Bitpos) ;
    Serial.print(" , bit = ") ;
    Serial.println(bitRead(DAC_val, Bitpos)) ;
    Serial.println("======================================") ;
#endif
  }
}

//**********************************************************

#ifdef oled
void show_oled(void)
{
  char buf[16] ;              // ol print_buffer
  ol.fillRect(0, 29, 78, 55, BLACK) ;     // clear old frequency
  ol.fillRect(0, 50 , 127 , 30, BLACK) ;  // clear old messages
  ol.setCursor(2, 30) ;
  sprintf(buf, "%6d", VFO) ;              // write new frequency
  ol.print(buf) ;
  ol.setCursor(0, 50) ;
  ol.print(band_S) ;                      // write band message 
  yield() ;                               // ESP32 interna
  ol.display() ;                          // show oled-screen
}

//*************************************************************

void init_oled(void)
{
  ol.begin(SSD1306_SWITCHCAPVCC, 0x3C) ;
  ol.fillScreen(BLACK) ;
  ol.setRotation(0) ;
  ol.setTextSize(1) ;
  ol.setTextColor(WHITE) ;
  ol.setCursor(4, 7) ;
  ol.print("IC705 - BT interface") ;
  ol.drawRect(0, 0, ol.width(), 20, WHITE) ;
  ol.setTextSize(2) ;
  ol.setCursor(80, 30);
  ol.print("kHz") ;
  ol.display() ;
}
#endif

//**************************************************************
// EOM
//**************************************************************
