
;**************************************************************************
;*                  Display SA23-12RT / BU2090                            *
;*                     Simple I2C Clock Demo                              *
;**************************************************************************
;**************************************************************************
;*          v 2.04 - 13.05.2016                                           *
;**************************************************************************
;*          Hardw.: PIC16F1826 used                                       *
;*          OSC.......: Internal osc. used     POWER.....:   5V extern.   *
;**************************************************************************


 INCLUDE "P16F1826.INC"



_U001			     EQU     B'00100110000100'		;see Page 48 of Datasheet
_U002			     EQU     B'01100011111111'		;see Page 49 of Datasheet


	__CONFIG	_CONFIG1, _U001
	__CONFIG	_CONFIG2, _U002


;--------------------------------------------------------------------------
;RA0 = DATA
;RA1 = CLK1
;RA2 = CLK2
;RA3 = CLK3
;--------------------------------------------------------------------------


;==========================================================================
;       Variable Definition
;==========================================================================
TIMER1		EQU	H'20'		;delay routine
TIMER2		EQU	H'21'		; "	"	"

v_t1		EQU	H'22'
v_t2		EQU	H'23'

v_numx		EQU	H'24'
v_cnt_1		EQU	H'25'
v_cnt_2		EQU	H'26'
v_cnt_3		EQU	H'27'
v_cnt_4		EQU	H'28'

v_i2c_add	EQU	H'29'
;----------------------------
v_setup_mode	EQU	H'2A'
v_rtc_type	EQU	H'2B'
;----------------------------
v_bit		EQU	H'2F'

v_pos1		EQU	H'30'
v_pos2		EQU	H'31'
v_pos3		EQU	H'32'
v_pos4		EQU	H'33'

v_disp1		EQU	H'34'
v_disp2		EQU	H'35'
v_disp3		EQU	H'36'
v_disp4		EQU	H'37'

v_tmp		EQU	H'38'
v_no_rtc	EQU	H'39'

v_rtc_d1	EQU	H'40'
v_rtc_d2	EQU	H'41'
v_rtc_m1	EQU	H'42'
v_rtc_m2	EQU	H'43'
v_rtc_y1	EQU	H'44'
v_rtc_y2	EQU	H'45'
v_rtc_y3	EQU	H'46'
v_rtc_y4	EQU	H'47'
v_rtc_h1	EQU	H'48'
v_rtc_h2	EQU	H'49'
v_rtc_n1	EQU	H'4A'
v_rtc_n2	EQU	H'4B'
v_rtc_s1	EQU	H'4C'
v_rtc_s2	EQU	H'4D'

v_rtc_sec	EQU	H'4E'
v_rtc_min	EQU	H'4F'
v_rtc_hrs	EQU	H'50'
v_rtc_wday	EQU	H'51'		;day in the week
v_rtc_day	EQU	H'52'
v_rtc_mnt	EQU	H'53'
v_rtc_year	EQU	H'54'

;--------------------------------------------------------------------------
;==========================================================================

		ORG	0		;Reset vector address
		GOTO	init_main	;goto RESET routine when boot.

;----------------------------------------------------------------------------
;----------------------------------------------------------------------------
;
;	       **********************************
;              **  RESET :  main boot routine  **
;              **********************************
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
; initialization
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

init_main	nop

		clrf	BSR
;
		clrf	INTCON		;disable interrupts

		banksel	OPTION_REG	;BANK 1
		movlw	B'00111111'
		movwf	OPTION_REG	;see datasheet Page 176

		banksel	ANSELA
		clrf	ANSELA
		clrf	ANSELB

		banksel	TRISA
		movlw	B'00000000'
		movwf	TRISA

		movlw	B'11111111'
		movwf	TRISB

;no OSCCON change - default 500kHz used

		banksel	WPUB
		movlw	B'00001101'
		movwf	WPUB

		clrf	BSR			;select BANK 0

		clrf	PORTA
		clrf	PORTB

		call	i2c_init
		clrf	v_rtc_type
		clrf	v_setup_mode
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

		clrf	INTCON

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;                      Initialising continues...
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
main_prg	clrf	BSR

		movlw	H'0A'
		movwf	v_disp1
		movwf	v_disp3
		movlw	H'0B'
		movwf	v_disp2
		movwf	v_disp4
		call	translate3
		call	send_all
		call	dr4

		movlw	H'0B'
		movwf	v_disp1
		movwf	v_disp3
		movlw	H'0A'
		movwf	v_disp2
		movwf	v_disp4
		call	translate3
		call	send_all
		call	dr4
;------------------------------------------------------------------------------

		goto	demo_clock

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
clr_pos1	movlw	H'0B'
		call	wa_numx
		movwf	v_pos1
		return
;------------------------------------------------------------------------------
clr_all		movlw	H'0B'
		movwf	v_disp1
		movwf	v_disp2
		movwf	v_disp3
		movwf	v_disp4
		call	translate3
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
translate2	btfsc	v_no_rtc,D'000'
		call	no_rtc_mm

translate3	movf	v_disp1,0
		movwf	v_numx
		call	wa_numx
		movwf	v_pos1

		movf	v_disp2,0
		movwf	v_numx
		call	wa_numx
		movwf	v_pos2

		movf	v_disp3,0
		movwf	v_numx
		call	wa_numx
		movwf	v_pos3

		movf	v_disp4,0
		movwf	v_numx
		call	wa_numx
		movwf	v_pos4

;-------------------------------------------
;no first zero....
;-------------------------------------------
		movf	v_disp1,0
		xorlw	H'00'
		btfsc	STATUS,Z
		call	clr_pos1
		movf	v_disp1,0
		xorlw	'0'
		btfsc	STATUS,Z
		call	clr_pos1

		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
no_rtc_mm	movlw	H'0A'
		movwf	v_disp1
		movwf	v_disp2
		movwf	v_disp3
		movwf	v_disp4
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
send_all	call	send_pos2
		call	send_pos1
		call	send_pos3
		call	send_pos4
		call	strobe_1
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
wa_numx		andlw	B'01111111'
		brw
		retlw	B'11000000'	;00h -> 0
		retlw	B'11111001'	;01h -> 1
		retlw	B'10100100'	;02h -> 2
		retlw	B'10110000'	;03h -> 3
		retlw	B'10011001'	;04h -> 4
		retlw	B'10010010'	;05h -> 5
		retlw	B'10000010'	;06h -> 6
		retlw	B'11011000'	;07h -> 7
		retlw	B'10000000'	;09h -> 8
		retlw	B'10010000'	;09h -> 9
		retlw	B'10111111'	;0Ah -> -
		retlw	B'11111111'	;0Bh -> space
		retlw	B'11000110'	;0Ch -> C
		retlw	B'11000111'	;0Dh -> L
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------



;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;Sending Data to the BU2090 chips
;every chip contains 12 bits
;always the MSb needs to be send as first bit and LSb as last bit
;in case more than 12 bits are send to the BU2090 before strobe
;the last 12 bits are used (the are just cyclic overwritten)
;
;segment on the SA23-12 is on, if the corresponding bit = 0
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
send_pos1	nop

;bit 0
		movf	v_pos1,0
		movwf	v_bit
		btfsc	v_setup_mode,D'000'
		bcf	v_bit,D'007'		;decimal point in setup mode
		call	send_bit_1

;bit 1
		lslf	v_bit,1
		call	send_bit_1

;bit 2
		lslf	v_bit,1
		call	send_bit_1

;bit 3
		lslf	v_bit,1
		call	send_bit_1

;bit 4
		lslf	v_bit,1
		call	send_bit_1

;bit 5
		lslf	v_bit,1
		call	send_bit_1

;bit 6
		lslf	v_bit,1
		call	send_bit_1

;bit 7
		lslf	v_bit,1
		call	send_bit_1

		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
send_pos2	nop

;bit 0
		movf	v_pos2,0
		movwf	v_bit
		btfsc	v_setup_mode,D'000'
		bcf	v_bit,D'007'		;decimal point in setup mode
		call	send_bit_2

;bit 1
		lslf	v_bit,1
		call	send_bit_2

;bit 2
		lslf	v_bit,1
		call	send_bit_2

;bit 3
		lslf	v_bit,1
		call	send_bit_2

;bit 4
		lslf	v_bit,1
		call	send_bit_1

;bit 5
		lslf	v_bit,1
		call	send_bit_1

;bit 6
		lslf	v_bit,1
		call	send_bit_1

;bit 7
		lslf	v_bit,1
		call	send_bit_1

		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
send_pos3	nop

;bit 0
		movf	v_pos3,0
		movwf	v_bit
		btfsc	v_setup_mode,D'000'
		bcf	v_bit,D'007'		;decimal point in setup mode

		call	send_bit_2

;bit 1
		lslf	v_bit,1
		call	send_bit_2

;bit 2
		lslf	v_bit,1
		call	send_bit_2

;bit 3
		lslf	v_bit,1
		call	send_bit_2

;bit 4
		lslf	v_bit,1
		call	send_bit_2

;bit 5
		lslf	v_bit,1
		call	send_bit_2

;bit 6
		lslf	v_bit,1
		call	send_bit_2

;bit 7
		lslf	v_bit,1
		call	send_bit_2

		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
send_pos4	nop

;dummy bit 1
		movf	v_pos4,0
		movwf	v_bit
		call	send_bit_3
;dummy bit 2
		call	send_bit_3
;dummy bit 3
		call	send_bit_3
;dummy bit 4
		call	send_bit_3

;bit 0
		movf	v_pos4,0
		movwf	v_bit
		btfsc	v_setup_mode,D'000'
		bcf	v_bit,D'007'		;decimal point in setup mode

		call	send_bit_3

;bit 1
		lslf	v_bit,1
		call	send_bit_3

;bit 2
		lslf	v_bit,1
		call	send_bit_3

;bit 3
		lslf	v_bit,1
		call	send_bit_3

;bit 4
		lslf	v_bit,1
		call	send_bit_3

;bit 5
		lslf	v_bit,1
		call	send_bit_3

;bit 6
		lslf	v_bit,1
		call	send_bit_3

;bit 7
		lslf	v_bit,1
		call	send_bit_3

		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;send bit 7 of v_bit register to the BU2090
;at return stays the CLK1 line on High
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
send_bit_1	bcf	PORTA,D'000'		;clear DATA
		nop
		bcf	PORTA,D'001'		;clear CLK1
;set DATA line
		btfsc	v_bit,D'007'
		bsf	PORTA,D'000'

		bsf	PORTA,D'001'		;CLK1
		nop

		bcf	PORTA,D'000'		;clear DATA line
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
send_bit_2	bcf	PORTA,D'000'		;clear DATA
		nop
		bcf	PORTA,D'002'		;clear CLK2
;set DATA line
		btfsc	v_bit,D'007'
		bsf	PORTA,D'000'

		bsf	PORTA,D'002'		;CLK2
		nop

		bcf	PORTA,D'000'		;clear DATA line
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
send_bit_3	bcf	PORTA,D'000'		;clear DATA
		nop
		bcf	PORTA,D'003'		;clear CLK3
;set DATA line
		btfsc	v_bit,D'007'
		bsf	PORTA,D'000'

		bsf	PORTA,D'003'		;CLK3
		nop

		bcf	PORTA,D'000'		;clear DATA line
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
strobe_1	bsf	PORTA,D'000'		;set DATA line to High
		nop
		bcf	PORTA,D'001'		;clear CLK1 line --> strobe
		bcf	PORTA,D'002'		;clear CLK2 line --> strobe
		bcf	PORTA,D'003'		;clear CLK3 line --> strobe
		nop
		bcf	PORTA,D'000'		;clear DATA line
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
; delay routines
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
d2		MOVLW   D'010'
		MOVWF   TIMER2
DEL_LOOP1	MOVLW   D'255'
		MOVWF   TIMER1

DEL_LOOP2	DECFSZ  TIMER1,F
		GOTO    DEL_LOOP2
		DECFSZ  TIMER2,F
		GOTO    DEL_LOOP1
		return
;------------------------------------------------
d1		MOVLW   D'007'
		MOVWF   TIMER2
DEL_LOOP1b	MOVLW   D'255'
		MOVWF   TIMER1

DEL_LOOP2b	DECFSZ  TIMER1,F
		GOTO    DEL_LOOP2b
		DECFSZ  TIMER2,F
		GOTO    DEL_LOOP1b
		return

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

;------------------------------------
dr2		call	d2
		call	d2
		call	d2
		return
;------------------------------------
dr4		call	dr2
		call	dr2
		call	dr2
		return


;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;I2C
;set up I2C BUS for communication as a master
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
i2c_init	movlw	H'04'
		movwf	BSR		;select BANK 4

		movlw	B'00101000'		;see Datasheet Page 277
;			bit 3-0 SSPM<3:0>: Synchronous Serial Port Mode Select bits
;			1000 = I2C Master mode, clock = FOSC / (4 * (SSPADD+1))
		movwf	SSP1CON1

;set-up I2C speed
		movlw	B'00111111'
		movwf	SSP1ADD
		clrf	SSP1STAT

		clrf	BSR		;select BANK 0

		return
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;
; I2C communication routines
;
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

;-----------------------------------------------------------------------------
i2c_start	bsf	BSR,D'002'		;select register BANK 4
		bsf	SSP1CON2, SEN
		bcf	BSR,D'002'		;select register BANK 0
		call	d55
		return
;-----------------------------------------------------------------------------
i2c_stop	bsf	BSR,D'002'
		bsf	SSP1CON2, PEN
		bcf	BSR,D'002'
		call	d55
		return
;-----------------------------------------------------------------------------
i2c_ack		bsf	BSR,D'002'
		bcf	SSP1CON2, ACKDT		;ACK
		bsf	SSP1CON2, ACKEN
		bcf	BSR,D'002'
		call	d55
		return
;-----------------------------------------------------------------------------
i2c_not_ack	bsf	BSR,D'002'
		bsf	SSP1CON2, ACKDT		;NOT ACK
		bsf	SSP1CON2, ACKEN
		bcf	BSR,D'002'
		call	d55
		return
;-----------------------------------------------------------------------------
;output: received byte is in W register
;
i2c_receive	bsf	BSR,D'002'
		bsf	SSP1CON2, RCEN
		bcf	BSR,D'002'
		call	d55
		bsf	BSR,D'002'
		movf	SSP1BUF,0
		bcf	BSR,D'002'
		return
;-----------------------------------------------------------------------------
;input: transmitted byte have to be in W register
;
i2c_send	bsf	BSR,D'002'
		movwf	SSP1BUF
		bcf	BSR,D'002'
		call	d55			;wait a while
		return
;-----------------------------------------------------------------------------
i2c_restart	bsf	BSR,D'002'
		bsf	SSP1CON2, RSEN
		bcf	BSR,D'002'
		call	d55
		return
;-----------------------------------------------------------------------------
;-------------------------------------------------------------------------------
d55		movlw	D'255'
		movwf	TIMER2
d55_loop	decfsz	TIMER2,F
		goto	d55_loop
		return
;----------------------------------------------------------------------------
;----------------------------------------------------------------------------

;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
; Real Time clock I2C communication routines for ISL12022
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;read current date/time
;
rtc_i2c_read_is	clrf	v_no_rtc

;Step 1 - Start condition
		call	i2c_start

;Step 2: addressing the RTC device
		movf	v_i2c_add,0		;get the RTC device address
		andlw	B'11111110'		;set bit 0 to "0" -> write
		call	i2c_send

;Step 3: send memory location
		movlw	B'00000000'		;memory address (= 0)
		call	i2c_send

;Step 4: START condition
		call	i2c_restart

;Step 5: addressing the RTC device
		movf	v_i2c_add,0		;get the RTC device address
		iorlw	B'00000001'		;set bit 0 to "1" -> read
		call	i2c_send

;-------------------------------------
;Step 6: read data
;6.1 - sec
		call	i2c_receive		;received byte - sec
		movwf	v_rtc_sec
		call	i2c_ack

;6.2 - min
		call	i2c_receive		;received byte - min
		movwf	v_rtc_min
		call	i2c_ack

;6.3 - hours
		call	i2c_receive		;received byte
		movwf	v_rtc_hrs
		call	i2c_ack
		bcf	v_rtc_hrs,D'007'	;clear the MIL bit of HR register

;6.5 - day
		call	i2c_receive		;received byte
		movwf	v_rtc_day
		call	i2c_ack

;6.6 - month
		call	i2c_receive		;received byte
		movwf	v_rtc_mnt
		call	i2c_ack

;6.7 - year
		call	i2c_receive		;received byte
		movwf	v_rtc_year
		call	i2c_ack


;6.4 - day in a week
		call	i2c_receive		;received byte
		movwf	v_rtc_wday
		call	i2c_not_ack
		incf	v_rtc_wday,1		;for ISL12022 is DW 0-6; for translation we need 1-7


;Step 7: send STOP condition (stop the read)
		call	i2c_stop

		movf	v_rtc_hrs,0
		xorlw	H'7F'
		btfsc	STATUS,Z
		bsf	v_no_rtc,D'000'		;no RTC device

		return

;--------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
;IS12022 write routine
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
rtc_i2c_wr_isl	nop

;1. set the WRTC bit of INT register to 1

		call	i2c_start
		movlw	B'11011110'		;I2C address of ISL12022 - RTC part
		call	i2c_send
		movlw	H'08'
		call	i2c_send
		movlw	B'01000001'
		call	i2c_send
		call	i2c_stop
;----------------------------------------------------
;2. set the date and time

		call	i2c_start
		movlw	B'11011110'		;I2C address of ISL12022 - RTC part
		call	i2c_send
		movlw	H'00'
		call	i2c_send

		movf	v_rtc_sec,0
		call	i2c_send
;
		movf	v_rtc_min,0
		call	i2c_send
;
		movf	v_rtc_hrs,0
		addlw	D'128'			;set MIL bit --> 24 hrs format to be used
		call	i2c_send
;
		movf	v_rtc_day,0
		call	i2c_send
;
		movf	v_rtc_mnt,0
		call	i2c_send
;
		movf	v_rtc_year,0
		call	i2c_send
;
		movf	v_rtc_wday,0
		call	i2c_send

		call	i2c_stop

;not used here -----------------------------
;		call	isl_DST_set
; ------------------------------------------
		return
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
;Maintain DST registers for ISL12022
;
;European convention:
;	start of daylight saving time: last Sunday in March at 1.00 AM
;       end   of daylight saving time: last Sunday in October at 1.00 AM
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
isl_DST_set	call	i2c_start
		movlw	B'11011110'		;I2C address of ISL12022 - RTC part
		call	i2c_send
		movlw	H'20'			;start of DST Forward Registers
		call	i2c_send

;Forward
		movlw	H'03'			;Month = March
		addlw	D'128'			;set bit DSTE (Enable DST)
		call	i2c_send

		movlw	B'01111110'
					;bit  6     - DwFdE = 1   --> Week in Month / Day in Week option
					;bits 5-4-3 = Week  = 111 --> Week = 7 --> last week in the Month
					;bits 2-1-0 = Day   = 110 --> Day =  6 --> Sunday
		call	i2c_send

		movlw	D'000'			;date - not used
		call	i2c_send

		movlw	H'02'			;hour
		call	i2c_send
;-----------------------------------------------------------
;Backward
		movlw	H'10'		;Month = October
		call	i2c_send

		movlw	B'01111110'
					;bit  6     - DwFdE = 1   --> Week in Month / Day in Week option
					;bits 5-4-3 = Week  = 111 --> Week = 7 --> last week in the Month
					;bits 2-1-0 = Day   = 110 --> Day =  6 --> Sunday
		call	i2c_send

		movlw	D'000'			;date - not used
		call	i2c_send

		movlw	H'02'			;hour
		call	i2c_send

		return
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------

;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
; Real Time clock I2C communication routines - MT41T00CAP
;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;read current date/time
rtc_i2c_read	nop

;Step 1 - Start condition
		call	i2c_start

;Step 2: addressing the RTC device
		movf	v_i2c_add,0		;get the RTC device address
		andlw	B'11111110'		;set bit 0 to "0" -> write
		call	i2c_send

;Step 3: send memory location
		movlw	B'00000000'		;memory address (= 0)
		call	i2c_send

;Step 4: send STOP condition (stop the write)
		call	i2c_stop

;Step 5: START condition
		call	i2c_start

;Step 6: addressing the RTC device
		movf	v_i2c_add,0		;get the RTC device address
		iorlw	B'00000001'		;set bit 0 to "1" -> read
		call	i2c_send

;-------------------------------------
;Step 7: read data
;7.1 - sec
		call	i2c_receive		;received byte - sec
		movwf	v_rtc_sec
		call	i2c_ack

;7.2 - min
		call	i2c_receive		;received byte - min
		movwf	v_rtc_min
		call	i2c_ack

;7.3 - hours
		call	i2c_receive		;received byte
		movwf	v_rtc_hrs
		call	i2c_ack

;7.4 - day in a week
		call	i2c_receive		;received byte
		movwf	v_rtc_wday
		call	i2c_ack

;7.5 - day
		call	i2c_receive		;received byte
		movwf	v_rtc_day
		call	i2c_ack

;7.6 - month
		call	i2c_receive		;received byte
		movwf	v_rtc_mnt
		call	i2c_ack

;7.7 - year
		call	i2c_receive		;received byte
		movwf	v_rtc_year
		call	i2c_not_ack

;Step 8: send STOP condition (stop the read)
		call	i2c_stop

		movf	v_rtc_hrs,0
		xorlw	H'FF'
		btfsc	STATUS,Z
		bsf	v_no_rtc,D'000'		;no RTC device

		return

;-----------------------------------------------------------------------------
;-----------------------------------------------------------------------------
rtc_i2c_write	nop

;Step 1: Start condition
		call	i2c_start
;
;Step 2: addressing the RTC device

		movf	v_i2c_add,0
		andlw	B'11111110'		;set bit 0 to "0" -> write
		call	i2c_send

;Step 3: send memory location
		movlw	B'00000000'		;memory address (= 0)
		call	i2c_send
;-------------------------------------
;Step 4: write data
;4.1 - sec
		movf	v_rtc_sec,0
		call	i2c_send

;4.2 - min
		movf	v_rtc_min,0
		call	i2c_send

;4.3 - hours
		movf	v_rtc_hrs,0
		call	i2c_send

;4.4 - day in a week
		movf	v_rtc_wday,0
		call	i2c_send

;4.5 - day
		movf	v_rtc_day,0
		call	i2c_send

;4.6 - month
		movf	v_rtc_mnt,0
		call	i2c_send

;4.7 - year
		movf	v_rtc_year,0
		call	i2c_send

;Step 5: send STOP condition (stop the read)
		call	i2c_stop

		return
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
rtc_translate	nop

		swapf	v_rtc_hrs,0
		andlw	B'00000011'
		movwf	v_disp1

		movf	v_rtc_hrs,0
		andlw	B'00001111'
		movwf	v_disp2

;minutes
		swapf	v_rtc_min,0
		andlw	B'00000111'
		movwf	v_disp3

		movf	v_rtc_min,0
		andlw	B'00001111'
		movwf	v_disp4

		return
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;--------------------------------------------------------------------
;--------------------------------------------------------------------
rtc_back_trans	nop
;hours -----------------------------------
		movf	v_disp1,0
		movwf	v_tmp
		swapf	v_tmp,1

		movf	v_disp2,0
		addwf	v_tmp,0
		movwf	v_rtc_hrs
;minutes --------------------------------
		movf	v_disp3,0
		movwf	v_tmp
		swapf	v_tmp,1

		movf	v_disp4,0
		addwf	v_tmp,0
		movwf	v_rtc_min
;seconds -------------------------------
		clrf	v_rtc_sec

		return
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------


;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
demo_clock	nop
;first - find out, if we have an RTC chip connected; and if so which one: MT41T00CAP or ISL12022

		movlw	B'00000010'
		movwf	v_rtc_type		;RTC = MT41T00CAP
		clrf	v_no_rtc
		movlw	B'11010000'		;RTC = 1101000x - ST Clock
		movwf	v_i2c_add
		call	rtc_i2c_read
		btfss	v_no_rtc,D'000'
		goto	main_loop_c1

		movlw	B'00000001'
		movwf	v_rtc_type		;RTC = ISL12022
		clrf	v_no_rtc
		movlw	B'11011110'		;RTC = 1101111x - ILS12022 Clock
		movwf	v_i2c_add
		call	rtc_i2c_read_is
		btfss	v_no_rtc,D'000'
		goto	main_loop_c2
;no RTC device found
		clrf	v_rtc_type
		goto	main_loop
;-----------------------------------------------------------------------------------------
main_loop_c1	movlw	H'0C'			;C
		movwf	v_disp1
		movlw	H'0D'			;L
		movwf	v_disp2
		movlw	H'0B'			;space
		movwf	v_disp3
		movlw	H'01'			;1
		movwf	v_disp4

		call	translate2
		call	send_all
		call	dr4
		call	dr4
		goto	main_loop
;-----------------------------------------------------------------------------------------
main_loop_c2	movlw	H'0C'			;C
		movwf	v_disp1
		movlw	H'0D'
		movwf	v_disp2
		movlw	H'0B'			;space
		movwf	v_disp3
		movlw	H'02'			;2
		movwf	v_disp4

		call	translate2
		call	send_all
		call	dr4
		call	dr4
		goto	main_loop
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
main_loop	btfsc	v_rtc_type,D'001'
		call	rtc_i2c_read
		btfsc	v_rtc_type,D'000'
		call	rtc_i2c_read_is

		call	rtc_translate
		call	translate2

		bcf	v_pos2,D'007'		;decimal point

		btfsc	v_no_rtc,D'000'
		call	clr_all

		btfss	PORTB,D'000'
		call	set_clock
		clrf	v_setup_mode

		call	send_all		;show data
		call	dr2
		call	dr2
		call	dr2
;----------------------------------------------------------------------
		btfsc	v_rtc_type,D'001'
		call	rtc_i2c_read
		btfsc	v_rtc_type,D'000'
		call	rtc_i2c_read_is

		call	rtc_translate
		call	translate2

		call	send_all		;show data
		call	dr2
		call	dr2
		call	dr2

		goto	main_loop
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
set_clock	movf	v_rtc_type,0
		btfsc	STATUS,Z
		return				;no RTC device connected

		movf	v_pos1,0
		xorlw	H'0B'
		btfsc	STATUS,Z
		clrf	v_pos1

		bsf	v_setup_mode,D'000'

		btfss	PORTB,D'002'
		call	inc_hrs
		btfss	PORTB,D'003'
		call	inc_min

		call	send_all		;show data
		call	dr2			;take small break...

		btfss	PORTB,D'000'
		goto	set_clock
;---------------------------------------------------
;now we are leaving clock set-up mode
		call	rtc_back_trans


		btfsc	v_rtc_type,D'001'
		call	rtc_i2c_write
		btfsc	v_rtc_type,D'000'
		call	rtc_i2c_wr_isl
;---------------------------------------------------
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
inc_hrs		incf	v_pos2,1
		movf	v_pos2,0
		xorlw	D'004'
		btfss	STATUS,Z
		goto	ih_t2
;
		movf	v_pos1,0
		xorlw	D'002'
		btfsc	STATUS,Z
		goto	clr_hrs
;
ih_t2		movf	v_pos2,0
		xorlw	D'010'
		btfss	STATUS,Z
		return
;
		clrf	v_pos2
		incf	v_pos1,1
		return
;
clr_hrs		clrf	v_pos1
		clrf	v_pos2
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
inc_min		incf	v_pos4,1
		movf	v_pos4,0
		xorlw	D'010'
		btfss	STATUS,Z
		return
;
		clrf	v_pos4
		incf	v_pos3,1
		movf	v_pos3,0
		xorlw	D'006'
		btfss	STATUS,Z
		return
;
		clrf	v_pos3
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
		END