/*****************************************************************************

	Controller328_Mini
		
	Controllerboard mit einem ATMega328 auf einer gedrucken Schaltung im viertel Europakartenformat.
	Es ist eine LCD, ein Drehencoder, vier Tasten, ein 12 Bit DAC und ein Lithium Akku auf dem Print.

	Funktions: Grundlage fr einige Projekte.

	key1:		im Display erscheint "key1"	
	key2:		~
	key3:		~
	key4:		~
	encoder:	im Display erschein ein Wert von 1000, der bei Drehen auf- und abgezhlt wird.
	encoder key:im Display erscheint "key5"  
	LED:		Lebenslicht blinkt mit 5 Hz

	Nach dem Einschalten wird die aktuelle Spannung des Akkus im Display ausgegeben. Wenn < 3V,
	dann sollte nicht lnger eingeschaltet sein.

	Nach Tastendruck wird nach ca. 2 Sekunden das Menu wieder eingeblendet.

	Am DAC wird eine Rampe bis 3,3 V erzeugt.

	License:
    Copyright (C) 2016 Clemens Verstappen DL3ETW

	This program is free software; you can redistribute it and/or modify it under the terms  
 	of the GNU General Public License as published by the Free Software Foundation; either 
	version 3 of the License, or (at your option) any later version.	 
	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
  	without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
	See the GNU General Public License for more details.You should have received a copy of 
	the GNU General Public License along with this program; if not, see <http://www.gnu.org/licenses/>. 

	Code for uart from Peter Fleury.
	http://homepage.hispeed.ch/peterfleury/avr-software.html

	Code for lcd from Peter Dannegger.

	history:
	V1.00 - first version October 2016

*****************************************************************************/
#include 	"lcd_drv.h"						//  LCD EADOG config
#include 	"LCD_DRV.C"
#include 	"i2cmaster.h"
#include 	<inttypes.h>                	// general lib
#include 	<avr/io.h>
#include 	<avr/interrupt.h>
#include 	<stdlib.h>						
#include 	<util/delay.h>	
#include 	"uart.h"

			
#define 	UART_BAUD_RATE      9600      
#define 	F_CPU 				8000000UL 	// xtal frequency

#define		LEBENSLICHT 		PD2			// port pin for LED

#define 	KEY1				0b00000001	// mask
#define 	KEY2				0b00000010	// mask
#define 	KEY3				0b00001000	// mask
#define 	KEY4				0b00010000	// mask
#define 	KEY5				0b00100000	// mask

#define 	MCP4226				0b11000000	// A2 device address of DAC Converter, see datasheet

#define		TIMER_100MS			0x030E		// 	time value for 100ms interrups of realtime_update 

volatile 	uint8_t 			ad_channel, dac_l, dac_h, ret,i,display_timer = 20;	// start value for display menue 
volatile 	uint16_t 			dac_value, encoder = 1000;		// start value for encoder
volatile 	float 				adc_result;
volatile 	char 				ascii[16]; 			// string for lcd	

unsigned 	char 				ZEICHEN=65;

volatile struct BitVariablen				//	status of keys 
{									
	int 	key1			:1;				// key1
	int 	key2			:1;
	int 	key3			:1;
	int 	key4			:1;
	int 	key5			:1;				// encoder key
	int 	encoder_ratation:1;				// encoder flag, if turned
	int		realtime_update :1;				// flag for main routine
	int		display_timer	:1;

}status = {1};

SIGNAL(PCINT0_vect)							// interrupt routine for pin change
{
	uint8_t tmp;
	tmp = ~PINB;							// read portb
	tmp &= 0b00111011;						// blank unused
	if ((tmp) & (KEY1)) 					// set flag
		status.key1 = 1;

	if ((tmp) & (KEY2)) 					// set flag 
		status.key2 = 1;

	if ((tmp) & (KEY3)) 					// set flag 
		status.key3 = 1;

	if ((tmp) & (KEY4)) 					// set flag
		status.key4 = 1;

	if ((tmp) & (KEY5)) 					// set flag
		status.key5 = 1;

} 

SIGNAL(TIMER1_COMPA_vect) 					// MAIN INTERVAL, start for main loop
{											// Signal(Timer_Compa_vect) Interrupt each 100 ms
	status.realtime_update = 1;				// indicate new calculation in main program
} 

SIGNAL(INT1_vect) 							// external interrupt, rising edge of encoder B 
{					
	if (PIND & (1 << PD5)) 					// what is level of encoder A? In what direction was the encoder turned?
	{
		encoder--;							// 16 bit access o.k., no other interrups possible
		status.encoder_ratation = 1; 		// flag for main 
	}
	else
	{
		encoder++;
		status.encoder_ratation = 1;
	}
}

void lcd_message(uint16_t disp_value) 		// this routine calculates the telegram for the dra818 
{		
	ultoa (disp_value, (char*) ascii, 10); 
	lcd_pos( 0, 0 );
  	lcd_text( "Encoder         " );
	lcd_pos( 0, 9);
	lcd_text((char*) ascii);
}			

void set_dac_value(uint16_t dac_value)		// load via I2C DAC 
{
uint8_t dac_l, dac_h;
	
	dac_l = dac_value;
	dac_h = dac_value >> 8;

	i2c_start_wait(MCP4226+I2C_WRITE);   	// tuts 
	i2c_write(dac_h);						// high byte 
	i2c_write(dac_l);						// low byte 
	i2c_stop();
}


// *************************************************************************

int main (void) 
{
	// set pin ports to output

	DDRD 	|= (1<<LEBENSLICHT);	

	// turn pull up resistor in controller on

	PORTB 	|= ((1<<PB0) | (1 << PB1) | (1 << PB3) | (1 << PB4) | (1 << PB5));	// switch on pull ups

	// timer1 configuration

	OCR1A	= TIMER_100MS;										// timer 1, compare match
	TCCR1B	|= ((1 << WGM12) | (1 << CS12) | (1 << CS10)); 		// clear on compare match, divide by 1024
	TIMSK1	|= (1 << OCIE1A);									// timer1 compare match 1 interrupt enable

	// enable pin change interrupt

	PCMSK0 	|= (1 << PCINT0) |(1 << PCINT1) |(1 << PCINT3) |(1 << PCINT4) |(1 << PCINT5);	// enable two switches 
	PCICR	|= (1 << PCIE0);						

	// enable external interrup for encoder_a

	EICRA	|= (1<<ISC11)|(1<<ISC10);					// rising edge for external interrupt1, encoder B
	EIMSK	|= (1<<INT1);								// enable external interrupt1	

	i2c_init();                                			// init I2C interface

	lcd_init();	
	uart_init( UART_BAUD_SELECT(UART_BAUD_RATE,F_CPU) );

	lcd_clear();										// welcome message on LCD
	lcd_pos( 0, 0 );
  	lcd_text( "Controll328_MINI" );
	lcd_pos( 1, 0 );
  	lcd_text( "BY   DL3ETW     " );
	_delay_ms(2000);

// ------------------------------------------------		// initialize ad converter and display battery voltage				

	ADMUX	|= (1 << REFS1) | (1 << REFS0);				// internal 1.1V referenz, 
	ADCSRA	|= (1 << ADEN) | (1 << ADSC) | (1 << ADATE);// enable AD, ADC Start Conversion,
	PORTC 	&= ~(1 << PC0);								// switch to input

	_delay_ms(1);										 
 	adc_result = ADC;									// read second value

	adc_result = adc_result * 4.73 / 1024;				// with 100 k by 330 k use 4,73 ( here calibrated with DVM)
														// 	* 1,1 V * 1/ ( 100 k / (100 k + 330 k))  V / 1024 bit
														// use 1 % resistors. 
	dtostrf (adc_result, 2,2, ascii);					// zwo digits left of decimal, two right
	lcd_pos( 1,0);
  	lcd_text( "U-batt:         " );
	lcd_pos( 1,8);
	lcd_text((char*) ascii);
	lcd_pos( 1,13);
  	lcd_text("V");
	_delay_ms(1000);									// wait to show display value


// ------------------------------------------------ 

	sei();												// enable interrups
	status.display_timer = 1;
												
// ------------------------------------------------ 
  	while(1) 											// main loop
	{
		if (status.realtime_update)						// each 100ms work here
		{
			PORTD ^= (1<<LEBENSLICHT);					// toggle Lebenslicht

			if (status.encoder_ratation == 1)			// encoder UP, check flag
			{
				lcd_message(encoder);
				status.encoder_ratation = 0;
			}
		
// -------------------------------------------------
			if (status.key1)							// TASTER1, check flag 
			{
				lcd_pos( 1, 0 );
 				lcd_text("KEY1            ");			// commend 2. LCD line
// here code
				status.display_timer = 1;
				status.key1 = 0;
			}
// -------------------------------------------------
			if (status.key2)							// TASTER2
			{
				lcd_pos( 1, 0 );
 				lcd_text("KEY2            ");			// commend 2. LCD line
// here code				
				status.display_timer = 1;
				status.key2 = 0;
			}
// -------------------------------------------------
			if (status.key3)							// TASTER3
			{
				lcd_pos( 1, 0 );
 				lcd_text("KEY3            ");			// commend 2. LCD line
// here code				
				status.display_timer = 1;
				status.key3 = 0;
			}
// -------------------------------------------------			
			if (status.key4)							// TASTER4
			{
				lcd_pos( 1, 0 );
 				lcd_text("KEY4            ");			// commend 2. LCD line
// here code				
				status.display_timer = 1;
				status.key4 = 0;
			}

// -------------------------------------------------			
			if (status.key5)							// TASTER4
			{
				lcd_pos( 1, 0 );
 				lcd_text("KEY5            ");			// commend 2. LCD line
// here code				
				status.display_timer = 1;
				status.key5 = 0;
			}

// -------------------------------------------------
			if (status.display_timer)					// overwrite the  2. LCD line
			{											// with menu for four switches
				display_timer--;							
				if (display_timer == 0)
				{
					lcd_pos( 1, 0 );
 					lcd_text("MN1 MN2 MN3 MN4 ");
					display_timer = 20;
					status.display_timer = 0;
				}
			}

			if (dac_value <= 4000)						// ic2 access, pull ups have to be connected !
			dac_value = dac_value +100;					// if not, delete 	
			else
			dac_value = 0;

			set_dac_value(dac_value);					// load via I2C DAC 

		status.realtime_update = 0;

 	 	} // interrup close, each 100 ms
	} // main while
} // main close


	// Belegung der Ports						Function			Signal name EAGLE Schaltplan Controler328_Mini
	// Port B Bit 0 PB0 (PCINT0/CKL0/ICP1)		key1				PB0_KEY1		
	//        Bit 1 PB1 (OC1A/PCINT1)			key2				PB1_KEY2
	//        Bit 2 PB2	(SS/OC1B/PCINT3)		DI	LCD				PB2_SI
  	//        Bit 3 PB3	(MOSI/OC2A/PCINT3))		MOSI / key3			PB3_SPI_MOSI_KEY3
  	//        Bit 4 PB4	(MISO/PCINT4))			MISO / key4			PB4_SPI_MISO_KEY4
  	//        Bit 5 PB5 (SCK/PCINT5)			SCK  / encoder_key	PB5_SPI_SCK_KEY5
  	//        Bit 6 PB6	(PCINT6/XTAL1/TOSC1)	8MHz X-TAL			PB6
	//		  Bit 7 PB7	(PCINT7/XTAL2/TOSC2)	8MHz X-TAL			PB7

 	// Port C Bit 0 PC0 (ADC0/PCINT8)			Voltage Akku		PC0_ADC0
  	//        Bit 1 PC1	(ADC1/PCINT9)			-					PC1_ADC1
 	//        Bit 2 PC2	(ADC2/PCINT10)			-					PC2_ADC2	
  	//        Bit 3 PC3	(ADC3/PCINT11)			-					PC3_ADC3
  	//        Bit 4 PC4	(ADC4/SDA/PCINT12)		SDA					PC4_SDA
  	//        Bit 5 PC5	(ADC5/SCL/PCINT13)		SCL					PC5_SCL
  	//        Bit 6 PC6	(PCINT/RESET)			Reset
  			          										
	// Port D Bit 0 PD0	(PCINT16/RXD)			-					PD0_RX			
	//        Bit 1 PD1	(PCINT17/TXD)			-					PD1_TX
  	//        Bit 2 PD2	(PCINT18/INT0)			Lebenslicht			PD2_SI
  	//        Bit 3 PD3	(PCINT19/OC2B/INT1)		Encoder A			PD3_ENCODERA
  	//        Bit 4 PD4	(PCINT20/XCK/T0)		RS LCD				PD4_RS
  	//        Bit 5 PD5	(PCINT21/OC0B/T1)		Encoder B			PD5_ENCODERB
  	//        Bit 6 PD6	(PCINT22/OC0A/AIN0)		CSB	LCD				PD6_CSB
  	//        Bit 7 PD7	(PCINT23/AIN1)			SCK	LCD				PD7_SCK


