.cseg
; ***********************************************************************************************
; Hauptptogramm fr ATtiny45 mit internen 8 MHz-Oszillator			
; ***********************************************************************************************
;
; Das Programm darf fr eigene Zwecke modifiziert werden. Die ursprngliche Quellangabe und dieser
; Text muss dabei erhalten bleiben. Eine kommerzielle Nutzung ist nicht gestattet. (c) 2017 DC7GB
;
;
; ACHTUNG: Beim Programmieren diese CONFIG-Bits ndern:
;
;	RSTDISBL: einschalten 
;	CKDIV8	: ausschalten
;	CLKSEL	: 0010		(interner RC-Oszillator 8MHz)
;	SUT	: 10		(slow rising Vcc, interner Reset 65ms)
;	BOD	: 111		(brown-out detector AUS)
;
;
; Zur Fehlersuche und Ausgabe des Strommesswertes an PB0 die nachfolgende Definition frei geben. 
;
;	.equ	_DEBUG	=	1
;
;
;	07.01.2017
;	----------
;     *	Urversion fr 5V Stabilisator V3.0 mit LED-Ausgabe und Eingang  TASTE. Whrend des PowerOn
;	wird  TASTE  abgefragt. Bleibt sie bis zur ersten LED-Anzeige gedrckt, so ist die
;	berstromabschaltung deaktiviert. Beim nchsten PowerOn wird die Stromgrenzwertauswertung
;	ohne Tastendruck wieder eingeschaltet. Die Grenzwerte sind durch die Konstanten  UMAX  (in
;	Millivolt) und  IMAX  (in mA) fest im Code vorgegeben.
;
;	Steigt die Ausgangsspannung ber den in  UMAX  eingetragenen Spannungswert in Millivolt (oder
;	sofern aktiviert der Strom be den Wert in  IMAX), so schaltet der Regler ab und zeigt ein
;	doppeltes Blinkmuster an. Der Regler wird durch Druck auf  TASTE  erneut gestartet.
;
;
; -------
; Port B:
; -------

.equ	DIRPB		= 0b00001001	; 0=Input / 1= Output
.equ	PUPB		= 0b00000010	; 1=PullUp-Widerstand bzw. Ausgang auf 1
;
.equ	ERRORLED	= PB0		; OUT: 0= LED ein / 1= LED aus
.equ	TASTE		= PB1		; INP: 0= Test-Taste gedrckt
.equ	STROM_M		= PB2		; ANA: (ADC1) -Eingang fr Strommessung
.equ	REGLEREIN	= PB3		; OUT: 0= Regler aus / 1= Regler ein
.equ	UAUSGANG	= PB4		; ANA: (ADC2) berwachung Ausgangsspannung
.equ	STROM_P		= PB5		; ANA: (ADC0) +Eingang fr Strommessung


;---------------------------------------
; Zeiten und Zhler

.equ	TAKT		= 8000000			; Taktfrequenz in Hz
.equ	SYSTAKT		= 1000				; Taktfrequenz in Hz
.equ	TICS1		= (TAKT/32)/SYSTAKT-1		; 250 = 1ms mit Vorteiler 32


; -------------------------------------
; Bit-Codes

.equ	BIT0		= 0
.equ	BIT1		= 1
.equ	BIT2		= 2
.equ	BIT3		= 3
.equ	BIT4		= 4
.equ	BIT5		= 5
.equ	BIT6		= 6
.equ	BIT7		= 7


;---------------------------------------
; Konstanten fr 10ms-Timer:

.equ	DELAY100MS	= 10		; 100ms 


;---------------------------------------
; Konstanten fr 100ms-Timer:

.equ	DELAY2S		= 20		; 2s


;---------------------------------------
; Grenzwerte fr Strom/Spannung:

.equ	IMAX		= 3000		; Maximalstrom in 1mA-Schritten
.equ	UMAX		= 5500		; Maximalspannung in 1mV-Schritten
;
.equ	UREF		= 1100		; Referenzspannung des ADC in 1mV-Schritten
.equ	MESSLEITWERT	= 10		; 10 S = 0,1 Ohm
.equ	TEILER		= 11		; Teilerfaktor am AD-Eingang
.equ	VIADC		= 20		; Verstrkung des I-Wandlers
.equ	VUADC		= 1		; Verstrkung des U-Wandlers
.equ	ADCMAX		= 255		; maximaler 8Bit-Wert der Wandler
;
.equ	IGRENZ		= (ADCMAX*VIADC*IMAX)/(MESSLEITWERT*TEILER*UREF); (=126 @ 3A) 8Bit Stromgrenzwert
.equ	UGRENZ		= (ADCMAX*VUADC*UMAX)/(TEILER*UREF)		; (=115 @ 5,5V) 8Bit Spannungsgrenzwert


;---------------------------------------
; Flags in GPIO-Registern:

.equ	flags		= GPIOR0	; freies internes Universalregister fr Bits
;
.equ	TIMER		= 0		; 0= Timer noch nicht abgelaufen / 1=10ms vorbei
.equ	LASTBLINK	= 1		; 0= (blinker) zuletzt auf 0 / 1= (blinker) zuletzt auf 1
.equ	STROMMESSUNG	= 2		; 0= keine Strommessung / 1=Strommessung eingeschaltet


;---------------------------------------
; Variablen in CPU-Registern

.def	akku0		= r0		; allgemeines
.def	akku1		= r1		; ...16Bit-Rechenregister
;
.def	blinker		= r14		; durchlaufender 100ms-Blinkzhler
.def	const		= r15		; allgemeiner Konstanten-Speicher
; -----------------------------------------------------------------------------------------------
; ACHTUNG: Nur ab r16 folgene Befehle: adiw, cbr, cpi, sbr, sbci, sbiw, subi, ldi, andi, ori
; -----------------------------------------------------------------------------------------------
.def	temp		= r16		; temporres Hilfsregister
.def	buffer		= r17		; allgemeines Hilfsregister
.def	counter		= r18		; allgemeiner Zhler fr Schleifen (Bitzhler)
.def	mainstate	= r19		; Zustand des Hauptprogramms
.def	blinkmuster	= r20		; 8Bit Blinkmuster fr 1,6s (200ms pro Bit)
.def	musterbit	= r21		; enthlt ein alle 200ms verschobenes 1-Bit fr die Blinkabfrage
.def	irq1ms		= r22		; Zhler bis 10ms

; -----------------------------------------------------------------------------------------------
; ACHTUNG:  ab r26 Pointer-Register XL, XH!
; -----------------------------------------------------------------------------------------------



; ***********************************************************************************************

RESET:		rjmp	START			; Startpunkt
EXT_INT0:	rjmp	EXT_INT0		; IRQ0-Handler
PC_INT0:	rjmp	PC_INT0			; PCINT0-Handler
		rjmp	TIM1_COMPA		; Timer1 CompareA Handler
TIM1_OVF:	rjmp	TIM1_OVF		; Timer1 Overflow Handler
TIM0_OVF:	rjmp	TIM0_OVF		; Timer0 Overflow Handler
EE_RDY:		rjmp	EE_RDY			; EEPROM Ready Handler
ANA_COMP:	rjmp	ANA_COMP		; Analog Comparator Handler
		rjmp	ADC_CONV		; ADC Conversion Handler
TIM1_COMPB:	rjmp	TIM1_COMPB		; Timer1 CompareB Handler
TIM0_COMPA:	rjmp	TIM0_COMPA		; Timer0 CompareA Handler
TIM0_COMPB:	rjmp	TIM0_COMPB		; Timer0 CompareB Handler
WDT:		rjmp	WDT			; Watchdog Handler
USI_START:	rjmp	USI_START		; USI Start Handler
USI_OVF:	rjmp	USI_OVF			; USI Overflow Handler


.include	"macros.asm"
.include	"subroutinen.asm"
.include	"interrupt.asm"


; ***********************************************************************************************
; Initialisierung:
; ***********************************************************************************************

START:		ldi	temp,low(RAMEND)	;; lower 8 Bit
		out	SPL,temp		;; ...des SP laden
		ldi	temp,high(RAMEND)	;; higher 8 Bit
		out	SPH,temp		;; ...des SP laden
;
		ldiw	Z,memstart
INIT1:		st	Z+,temp			;; RAM-Zellen ab <memstart> lschen
		cpiw	Z,(RAMEND+1)		;; RAM-Ende erreicht?
		brne	init1			;; nein...

; ----------------------------------------------
; Ports initialisieren:

		ldi	temp,DIRPB		;; PortB einstellen
		out	DDRB,temp
		ldi	temp,PUPB		;; Pull-up an B-Eingngen aktivieren
		out	PORTB,temp

; ---------------------------------------------- 
; Timer1 fr 1ms Systemtakt programieren:
	
		ldi	temp,TICS1		;; Timerkonstante
		out	OCR1A,temp		;; ...fr Interrupt OCIE1A
		out	OCR1C,temp		;; ...fr CTC-Funktion
		ldi	temp,0b10000110		;; CTC1 aktiviert / TAKT:32
		out	TCCR1,temp
		ldi	temp,(1<<OCIE1A)	;; Timer/Counter1 Compare-A Match Interrupt
		out	TIMSK,temp		;; ...frei geben

; ----------------------------------------------
; A/D-Wandler:

		ldi	temp,0b10001011		;; Ref. 1,1V / +ADC0 -ADC1 v=20
		out	ADMUX,temp
		ldi	temp,0b10001110		;; ADC ein / freerun / ADIE / 125kHz Takt = max. 150us/Sample
		out	ADCSRA,temp
		ldi	temp,0b00000000		;; Unipolar-Mode / free-run
		out	ADCSRB,temp
		ldi	temp,0b00110100		;; ADC0,1 und 2-Eingang vom Digital-Pin
		out	DIDR0,temp		;; ...trennen	
	
; ----------------------------------------------
; Initialiserung von Variablen:

		clr	temp			;; alle Flags in den
		out	flags,temp		;; ...Ruhezustand setzen
		ldi	musterbit,1		;; Abfragebit voreinstellen
		ldi	irq1ms,10		;; 10ms-Zhler auf Anfang setzen

; ----------------------------------------------
; Watchdog:

		ldi	temp,0b00011010		;; WDIE=0 / WDCE+WDE / 64ms
		out	WDTCR,temp		;; Watchdog fr RESET aktivieren

; ***********************************************************************************************
; Hauptprogammschleife:
; ***********************************************************************************************

		clr	mainstate		;; beginne mit Initialisierung
WAIT:		sei				; Interrupts frei geben
		wdr				; 1s-Watchdog beruhigen
		sbis	flags,TIMER		; 10ms abgelaufen?
		rjmp	wait			; nein...
		cbi	flags,TIMER		; 10ms-Marke neu initialisieren

; ---------------------------------------------
; alle 10ms:

		mov	temp,mainstate		; nchsten Zustand laden
		ldiw	Z,MAINJUMP		; Sprungtabelle laden,
		rcall	select			; ...zugehrige Sprungadresse nach Z holen
		icall				; ...und Zustand aufrufen
.ifndef	_DEBUG
		rcall	led_manager		; aktuelles Blinkmuster alle 200ms ndern und ausgeben
.endif
		rjmp	wait



; ***********************************************************************************************
; Zustandsautomat des Hauptprogramms:		(alle 10 ms)
; ***********************************************************************************************


MAINJUMP:	.dw	Z0			; Startpunkt nach PowerOn
		.dw	Z1			; Warte auf Blinkende
		.dw	Z2			; berprfe Messwerte
		.dw	Z3			; Error: Warte auf Tasteneingabe



; ---------------------------------------------
; Initialisierung nach PowerOn:
; ---------------------------------------------

Z0:		sbi	flags,STROMMESSUNG	; Strommessung aktivieren
		sbis	PINB,TASTE		; Taste gedrckt?
		cbi	flags,STROMMESSUNG	; ja, dann Strommessung deaktivieren
;
		ldi	blinkmuster,0b01010101	; Dauerblinkmuster fr LED laden
		sbis	flags,STROMMESSUNG	; Strommessung eingeschaltet?
		ldi	blinkmuster,$ff		; nein, Dauerlicht beim Einschalten
		ldi	temp,DELAY2S		; Wartezeit zur Systemberuhigung
		sts	tim100wait,temp		; ...nach PowerOn starten
		ldi	mainstate,1		; ...und auf Timerende warten
		ret



; ---------------------------------------------
; Spannung ein und testen, warte auf Blinkende:
; ---------------------------------------------

Z1:		sbi	PORTB,REGLEREIN		; Regler einschalten
		ldi	buffer,0b01011111	; U-ERROR: 2x Blinken vorladen
		lds	temp,adbuffer+1		; U-Messwert
		cpi	temp,UGRENZ		; ...berschritten?
		brcc	z2alarm			; ja, Regler sofort ausschalten...
		lds	temp,tim100wait		; Blinkzeit
		tst	temp			; ...abgelaufen?
		brne	z1end			; nein...
		clr	blinkmuster		; LED ausschalten (blinkt nicht mehr)
		ldi	mainstate,2		; weiter mit Messvorgang
Z1END:		ret



; ---------------------------------------------
; Vergleiche Messwerte auf Grenzberschreitung:
; ---------------------------------------------

Z2:		clr	blinkmuster		; Vorgabe: LED ausschalten (blinkt nicht mehr)
		sbis	PINB,TASTE		; Taste gedrckt?
		ldi	blinkmuster,0b11111111	; ja, Dauerlicht, so lange Taste gedrckt ist
		sbi	PORTB,REGLEREIN		; Regler dauernd einschalten
;
		sbis	flags,STROMMESSUNG	; Strommessung aktiviert?
		rjmp	z2u			; nein...
Z2I:		ldi	buffer,0b01111111	; I-ERROR: 1x Blinken vorladen
		lds	temp,adbuffer		; I-Messwert
		cpi	temp,IGRENZ		; ...berschritten?
		brcc	z2alarm			; ja...
;
Z2U:		ldi	buffer,0b01011111	; U-ERROR: 2x Blinken vorladen
		lds	temp,adbuffer+1		; U-Messwert
		cpi	temp,UGRENZ		; ...berschritten?
		brcs	z2end			; nein...
;
; Grenzwerte berschritten:
;
Z2ALARM:	cbi	PORTB,REGLEREIN		; Regler sofort ausschalten
		mov	blinkmuster,buffer	; Alarm-Blinkmuster bernehmen
		ldi	mainstate,3		; Alarm anzeigen
Z2END:		ret



; ---------------------------------------------
; ERROR: Taste abfragen, Regler neu starten:
; ---------------------------------------------

Z3:		clr	temp			; Wartezeit
		sts	tim100wait,temp		; ...lschen
		sbis	PINB,TASTE		; Taste gedrckt?
		ldi	mainstate,1		; ja, Regler ohne Blinken neu starten
		ret				; (auf Taste warten / Regler neu starten)



.dseg
; ***********************************************************************************************
; Daten im RAM
; ***********************************************************************************************

memstart:

; --------------------------------------
; alle Variablen des A/D-Wandlers:

adstate:	.db	1		; Zustand des AD-Automaten
adbuffer:	.db	1		; 8Bit I-Sample
		.db	1		; 8Bit U-Sample

; --------------------------------------
; alle Timer-Variablen:

tim100:		.db	1		; zyklischer 100ms-Timer
tim100wait:	.db	1		; allgemeiner Wartetimer 100ms...25s
