
;**************************************************************************
;*                    LED Strip driver with IR control                    *
;**************************************************************************
;**************************************************************************
;*          v 3.20 - 04.02.2016                                           *
;*                                                                        *
;**************************************************************************
;*          Hardw.: PIC16F1829 used                                       *
;*          OSC.......: Internal osc. used     POWER.....:   5V extern.   *
;**************************************************************************

;Supported IRs:
;	Philips TV
;	SONY


 INCLUDE "P16F1829.INC"



_U001			     EQU     B'00100110000100'		;see Page 48 of Datasheet
_U002			     EQU     B'01100011111111'		;see Page 49 of Datasheet


	__CONFIG	_CONFIG1, _U001
	__CONFIG	_CONFIG2, _U002



;==========================================================================
;       Variable Definition
;==========================================================================
TIMER1		EQU	H'20'		;delay routine
TIMER2		EQU	H'21'		; "	"	"

v_ttt		EQU	H'22'
v_cnt1		EQU	H'23'
v_cnt2		EQU	H'24'
kbd_buffer	EQU	H'25'
v_tst1		EQU	H'26'
v_tst2		EQU	H'27'
v_tst3		EQU	H'28'
disp_pos	EQU	H'29'

v_ir_bit	EQU	H'2A'
v_itimer1	EQU	H'2B'
v_itimer2	EQU	H'2C'
v_ir_tmp	EQU	H'2D'
v_ir_new	EQU	H'2E'
v_ir_byte	EQU	H'2F'

v_ir_byte1	EQU	H'30'
v_ir_byte2	EQU	H'31'
v_ir_byte3	EQU	H'32'
v_ir_byte4	EQU	H'33'
v_ir_byte5	EQU	H'34'
v_ir_byte6	EQU	H'35'
v_ir_byte7	EQU	H'36'
v_ir_byte8	EQU	H'37'

v_ir_byte1_s	EQU	H'38'
v_ir_byte2_s	EQU	H'39'
v_ir_byte3_s	EQU	H'3A'
v_ir_byte4_s	EQU	H'3B'
v_ir_byte5_s	EQU	H'3C'
v_ir_byte6_s	EQU	H'3D'
v_ir_byte7_s	EQU	H'3E'
v_ir_byte8_s	EQU	H'3F'

v_ph1d_m	EQU	H'40'
v_ph1d_l	EQU	H'41'
v_ph2d_m	EQU	H'42'
v_ph2d_l	EQU	H'43'

;--------------------------------------------------------------------------
v_ir_code	EQU	H'44'		;selected IR-code (pulse width)

	;01 = Philips RC6
	;02 = Sharp
;--------------------------------------------------------------------------
;IR additional variables

v_stage		EQU	H'45'

v_ir_comm	EQU	H'46'
v_comm_x	EQU	H'47'
v_loop_cnt2	EQU	H'48'
;--------------------------------------------------------------------------
;PWM Variables

v_poti_1	EQU	H'49'		;Blue   / AN0 = RA0
v_poti_2	EQU	H'4A'		;Red    / AN1 = RA1
v_poti_3	EQU	H'4B'		;Green  / AN3 = RA4
v_mode		EQU	H'4C'		;Mode Selection - Poti on AN6 = RC2
;--------------------------------------------------------------------------
v_FSR1L_save	EQU	H'4D'
v_FSR1H_save	EQU	H'4E'

v_loop_cnt	EQU	H'4F'
;--------------------------------------------------------------------------
v_valid_code	EQU	H'50'
v_code_len	EQU	H'51'
v_first_byte	EQU	H'52'
;--------------------------------------------------------------------------
;==========================================================================

		ORG	0		;Reset vector address
		GOTO	init_main	;goto RESET routine when boot.

		ORG	4
		goto	ir_main		;interrupt service routine

;----------------------------------------------------------------------------
;----------------------------------------------------------------------------
;
;	       **********************************
;              **  RESET :  main boot routine  **
;              **********************************
;
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
; initialization
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

init_main	nop

		clrf	BSR
;
		clrf	INTCON		;disable interrupts

		banksel	OPTION_REG	;BANK 1
		movlw	B'00111111'
		movwf	OPTION_REG	;see datasheet Page 176

		movlw	H'03'
		movwf	BSR		;select BANK 3
		clrf	ANSELA
		clrf	ANSELB
		clrf	ANSELC

		banksel	TRISA
		movlw	B'11111111'
		movwf	TRISA

		movlw	B'00000111'
		movwf	TRISC

		movlw	B'01111111'
		movwf	TRISB


		movf	OSCCON,0
		andlw	B'11111111'
		iorlw	B'01111000'
		movwf	OSCCON		;set 16MHz internal clock (bit 6:3 = 1111) - Datasheet Page 68

		movlw	H'04'
		movwf	BSR		;select BANK 4
		movlw	B'00100100'
		movwf	WPUA		;enable weak pull-ups

		movlw	B'01111111'
		movwf	WPUB

		clrf	WPUC
		clrf	BSR		;select BANK 0

		clrf	PORTA
		clrf	PORTB
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;                      Set-up PWM
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

		nop

;---------------------------------------------------------------------------
;PWM set-up
;set-up PWM1 (output = RC5) for Blue channel
;set-up PWM2 (output = RC3) for Red channel

;1. disable output
		banksel	TRISC
		bsf	TRISC,D'005'		;Blue  - CCP1
		bsf	TRISC,D'003'		;Red   - CCP2
		bsf	TRISC,D'006'		;Green - CCP4

;2 set-up PRx register
		banksel	PR2
		movlw	H'FF'
		movwf	PR2

		banksel	T2CON
		movlw	B'0000111'
		movwf	T2CON		;prescaler = 1 --> see Page 190 of the Datasheet
					;PWM Frequency = 15.625 kHz @ 16MHz internal clock


;3 set-up CCP1CON
		banksel	CCP1CON
		movlw	B'00001100'
		movwf	CCP1CON
		movwf	CCP2CON

		banksel CCP4CON
		movwf	CCP4CON

;4 set-up duty cycle
		bcf	CCP1CON,D'005'		;LSb of duty cycle
		bcf	CCP1CON,D'004'		;LSb2 of duty cycle
		bcf	CCP2CON,D'005'		;LSb of duty cycle
		bcf	CCP2CON,D'004'		;LSb2 of duty cycle
		movlw	B'10000000'		;Duty cycle = 50%
		movwf	CCPR1L			;Duty cycle = 50%
		movwf	CCPR2L			;Duty cycle = 50%

		banksel	CCPR4L
		movwf	CCPR4L
		bcf	CCP4CON,D'005'		;LSb of duty cycle
		bcf	CCP4CON,D'004'		;LSb2 of duty cycle

;5 select timer as an input for PWM
		banksel	CCPTMRS
		movlw	B'00000000'
		movwf	CCPTMRS			;for all PWMs the TIMER2 is selected (see Page 225 of the Datasheet)

;6. clear PIR1 register
		banksel	PIR1
		clrf	PIR1

;7. enable output
		banksel	TRISC
		bcf	TRISC,D'005'	;enable output
		bcf	TRISC,D'003'	;enable output
		bcf	TRISC,D'006'	;enable output
		clrf	BSR		;select bank 0

;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;set-up interrupt in INT-pin (RA2)
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

		clrf	INTCON

;bit INTEDG (bit 6) of OPTION_REG needs to be cleared - is defined above

		bsf	INTCON,INTE			;Enables the INT external interrupt
		bsf	INTCON,GIE			;Enables all active interrupts
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------


;------------------------------------------------------------------------------
;switch all LEDs off
		movlw	H'00'
		movwf	v_poti_1		;Blue
		movlw	H'00'
		movwf	v_poti_2		;Red
		movlw	H'00'
		movwf	v_poti_3		;Green
		call	set_duty_c
;------------------------------------------------------------------------------
		goto	main_prg


;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;                      Interrupt service routine
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

ir_main		nop
		btfss	INTCON,INTF		;unkown event -- return
		goto	leave_ir

		bcf	INTCON,D'007'		;disable interrupts
		movf	PORTA,0
		bcf	INTCON,INTF

		btfsc	v_ir_new,D'000'
		goto	leave_ir		;last value not processed yet

;-----------------------------------------------------------------------------
;Interrupt business logic starts here....
;-----------------------------------------------------------------------------

		bsf	PORTA,D'000'		;switch the LED on - we got signal...
		btfsc	PORTA,D'002'
		goto	ir_cancel		;we expect level 0 now...

		call	ir_get_byte		;receive 1st byte
		movf	v_ir_byte,0
		movwf	v_ir_byte1_s

		call	ir_get_byte		;receive 2nd byte
		movf	v_ir_byte,0
		movwf	v_ir_byte2_s

		call	ir_get_byte		;receive 3rd byte
		movf	v_ir_byte,0
		movwf	v_ir_byte3_s

		call	ir_get_byte		;receive 4th byte
		movf	v_ir_byte,0
		movwf	v_ir_byte4_s

		call	ir_get_byte		;receive 5th byte
		movf	v_ir_byte,0
		movwf	v_ir_byte5_s

		call	ir_get_byte		;receive 6th byte
		movf	v_ir_byte,0
		movwf	v_ir_byte6_s

		call	ir_get_byte		;receive 7th byte
		movf	v_ir_byte,0
		movwf	v_ir_byte7_s

		call	ir_get_byte		;receive 8th byte
		movf	v_ir_byte,0
		movwf	v_ir_byte8_s

		movf	v_ir_byte1_s,0
		movwf	v_ir_byte1
		movf	v_ir_byte2_s,0
		movwf	v_ir_byte2
		movf	v_ir_byte3_s,0
		movwf	v_ir_byte3
		movf	v_ir_byte4_s,0
		movwf	v_ir_byte4
		movf	v_ir_byte5_s,0
		movwf	v_ir_byte5
		movf	v_ir_byte6_s,0
		movwf	v_ir_byte6
		movf	v_ir_byte7_s,0
		movwf	v_ir_byte7
		movf	v_ir_byte8_s,0
		movwf	v_ir_byte8

		clrf	v_ir_new		;store the code
		bsf	v_ir_new,D'000'		;set flag - new character received

;----------------------------------------
ir_cancel	bcf	PORTA,D'000'		;switch the LED off
;-----------------------------------------------------------------------------
;Interrupt business logic ends here....
;-----------------------------------------------------------------------------
leave_ir	nop
		bcf	PORTA,D'000'		;switch the LED off
		bsf	INTCON,INTE
		bsf	INTCON,D'007'
		RETFIE				;return from interrupt with restore of STAUS, WREG and BSR
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
ir_get_byte	clrf	v_ir_byte
		call	ir_get_bit
		btfsc	v_ir_bit,D'000'
		bsf	v_ir_byte,D'007'

		call	ir_get_bit
		btfsc	v_ir_bit,D'000'
		bsf	v_ir_byte,D'006'

		call	ir_get_bit
		btfsc	v_ir_bit,D'000'
		bsf	v_ir_byte,D'005'

		call	ir_get_bit
		btfsc	v_ir_bit,D'000'
		bsf	v_ir_byte,D'004'

		call	ir_get_bit
		btfsc	v_ir_bit,D'000'
		bsf	v_ir_byte,D'003'

		call	ir_get_bit
		btfsc	v_ir_bit,D'000'
		bsf	v_ir_byte,D'002'

		call	ir_get_bit
		btfsc	v_ir_bit,D'000'
		bsf	v_ir_byte,D'001'

		call	ir_get_bit
		btfsc	v_ir_bit,D'000'
		bsf	v_ir_byte,D'000'
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
ir_get_bit	clrf	v_ir_bit
		call	wait_Ph1
		btfsc	PORTA,D'002'
		bsf	v_ir_bit,D'000'
		call	wait_Ph2
		return
;------------------------------------------------------------------------------
ir_get_bit2	clrf	v_ir_bit
		call	wait300us
		btfsc	PORTA,D'002'
		bsf	v_ir_bit,D'000'
		call	wait460us
		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
; interrupt delay routines
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
d350		MOVLW   D'255'
		MOVWF   v_itimer2
iDEL_LOOP1	MOVLW   D'255'
		MOVWF   v_itimer1

iDEL_LOOP2	DECFSZ  v_itimer1,F
		GOTO    iDEL_LOOP2
		DECFSZ  v_itimer2,F
		GOTO    iDEL_LOOP1
		return
;----------------------------------------------------------------------------
;----------------------------------------------------------------------------
;Phase 1 wait routine
;----------------------------------------------------------------------------
;----------------------------------------------------------------------------
wait_Ph1	movf	v_ph1d_m,0
		movwf	v_itimer2
iw2_l2		movf	v_ph1d_l,0
		movwf	v_itimer1

iw2_l1		decfsz	v_itimer1,1		;0,25us
		goto	iw2_l1			;0,50us

		decfsz	v_itimer2,1
		goto	iw2_l2
		return

;----------------------------------------------------------------------------
;----------------------------------------------------------------------------
;Phase 2 wait routine
;----------------------------------------------------------------------------
;----------------------------------------------------------------------------
wait_Ph2	movf	v_ph2d_m,0
		movwf	v_itimer2
iw28_l2		movf	v_ph2d_l,0
		movwf	v_itimer1

iw28_l1		decfsz	v_itimer1,1		;0,25us
		goto	iw28_l1			;0,50us (16MHz clock)

		decfsz	v_itimer2,1
		goto	iw28_l2
		return

;----------------------------------------------------------------------------
;----------------------------------------------------------------------------
;----------------------------------------------------------------------------
wait300us	movlw	D'002'
		movwf	v_itimer2
iw3_l2		movlw	D'210'
		movwf	v_itimer1

iw3_l1		decfsz	v_itimer1,1		;0,5us / 2
		goto	iw3_l1			;1,0us / 2

		decfsz	v_itimer2,1
		goto	iw3_l2
		return

;----------------------------------------------------------------------------
wait460us	movlw	D'004'
		movwf	v_itimer2
iw4_l2		movlw	D'155'
		movwf	v_itimer1

iw4_l1		decfsz	v_itimer1,1		;0,5us / 2
		goto	iw4_l1			;1,0us / 2

		decfsz	v_itimer2,1
		goto	iw4_l2
		return

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;                      Waiting times definitions
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

;----------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;----------------------------------------------------
;delay values for Philips code RC6
;PIC Clock = 16MHz
;in total: 444us
;
;Phase 1 in total ~ 1 * 245 * 0,75us = 183,75 us
;Phase 2 in total ~ 2 * 169 * 0,75us = 253,5 us
;----------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;----------------------------------------------------

wt1_philips1	movlw	D'001'
		movwf	v_ph1d_m
		movlw	D'245'
		movwf	v_ph1d_l

		movlw	D'002'
		movwf	v_ph2d_m
		movlw	D'169'
		movwf	v_ph2d_l

		return

;----------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;----------------------------------------------------
;delay values for Sharp
;PIC Clock = 16MHz
;in total: 300us
;
;Phase 1 in total ~ 1 * 133 * 0,75us = 99,75 us
;Phase 2 in total ~ 2 * 133 * 0,75us = 199,5 us
;----------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;----------------------------------------------------

wt2_sharp1	movlw	D'001'
		movwf	v_ph1d_m
		movlw	D'133'
		movwf	v_ph1d_l

		movlw	D'002'
		movwf	v_ph2d_m
		movlw	D'133'
		movwf	v_ph2d_l

		return

;----------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;----------------------------------------------------
;delay values for Sony
;PIC Clock = 16MHz
;in total: 600us
;
;Phase 1 in total ~ 2 * 200 * 0,75us = 300,0 us
;Phase 2 in total ~ 2 * 200 * 0,75us = 300,0 us
;----------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;----------------------------------------------------

wt3_sony1	movlw	D'002'
		movwf	v_ph1d_m
		movlw	D'200'
		movwf	v_ph1d_l


		movlw	D'002'
		movwf	v_ph2d_m
		movlw	D'200'
		movwf	v_ph2d_l

		return

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
select_times	movf	v_ir_code,0			;Philips TV
		xorlw	D'000'
		btfsc	STATUS,Z
		goto	wt1_philips1			;set-up times for Philips RC6 code

		movf	v_ir_code,0
		xorlw	D'001'
		btfsc	STATUS,Z
		goto	wt3_sony1			;sony (TV)

		movf	v_ir_code,0
		xorlw	D'002'
		btfsc	STATUS,Z
		goto	wt3_sony1			;sony (DVD)

		movf	v_ir_code,0
		xorlw	D'003'
		btfsc	STATUS,Z
		goto	wt3_sony1			;sony (Video)

		movf	v_ir_code,0			;Philips DVD
		xorlw	D'004'
		btfsc	STATUS,Z
		goto	wt1_philips1			;set-up times for Philips RC6 code

;---------------------------------------------------------------------------------
		clrf	v_ir_code			;default - Philips RC6
		goto	wt1_philips1
		return

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;                      Main Loop
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

main_prg	clrf	BSR

		movlw	H'FF'
		movwf	v_ir_byte1
		movwf	v_ir_new

		movf	PORTB,0
		movwf	v_ir_code
		bcf	v_ir_code,D'007'
		lsrf	v_ir_code,1
		lsrf	v_ir_code,1
		lsrf	v_ir_code,1
		lsrf	v_ir_code,1

		call	select_times

		clrf	v_cnt1
		clrf	v_cnt2

		clrf	v_mode
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
main_prg2	nop
		clrf	v_ir_new

		movf	v_mode,0
		xorlw	D'000'
		btfsc	STATUS,Z
		goto	mp_mode0

		movf	v_mode,0
		xorlw	D'001'
		btfsc	STATUS,Z
		goto	mp_mode1

		movf	v_mode,0
		xorlw	D'002'
		btfsc	STATUS,Z
		goto	mp_mode2

		movf	v_mode,0
		xorlw	D'003'
		btfsc	STATUS,Z
		goto	mp_mode3

		movf	v_mode,0
		xorlw	D'004'
		btfsc	STATUS,Z
		goto	mp_mode4

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Mode 0
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
mp_mode0	call	set_duty_c
		bcf	PORTB,D'007'
		btfss	v_ir_new,D'000'
		goto	mp_mode0
;now we received something
		call	execute_comm
		goto	main_prg2
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Mode 1
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
mp_mode1	call	set_duty_c
		bcf	PORTB,D'007'

		movlw	H'FF'
		movwf	v_poti_1
		movlw	H'00'
		movwf	v_poti_2
		movlw	H'00'
		movwf	v_poti_3
		call	set_duty_c
		call	d2

		movlw	H'00'
		movwf	v_poti_1
		movlw	H'FF'
		movwf	v_poti_2
		movlw	H'00'
		movwf	v_poti_3
		call	set_duty_c
		call	d2

		movlw	H'00'
		movwf	v_poti_1
		movlw	H'00'
		movwf	v_poti_2
		movlw	H'FF'
		movwf	v_poti_3
		call	set_duty_c
		call	d2

		btfss	v_ir_new,D'000'
		goto	mp_mode1
;now we received something
		call	execute_comm

		movf	v_mode,0
		xorlw	D'001'
		btfsc	STATUS,Z
		goto	mp_mode1

		goto	main_prg2
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Mode 2
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
mp_mode2	call	set_duty_c
		bcf	PORTB,D'007'

		movlw	H'FF'
		movwf	v_poti_1
		movlw	H'00'
		movwf	v_poti_2
		movlw	H'80'
		movwf	v_poti_3
		call	set_duty_c
		call	dr2

		movlw	H'80'
		movwf	v_poti_1
		movlw	H'FF'
		movwf	v_poti_2
		movlw	H'00'
		movwf	v_poti_3
		call	set_duty_c
		call	dr2

		movlw	H'00'
		movwf	v_poti_1
		movlw	H'80'
		movwf	v_poti_2
		movlw	H'FF'
		movwf	v_poti_3
		call	set_duty_c
		call	dr2

		btfss	v_ir_new,D'000'
		goto	mp_mode2
;now we received something
		call	execute_comm

		movf	v_mode,0
		xorlw	D'002'
		btfsc	STATUS,Z
		goto	mp_mode2

		goto	main_prg2
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Mode 3
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
mp_mode3	movlw	HIGH e_mode3
		movwf	FSR1H
		movlw	LOW e_mode3
		movwf	FSR1L

		bcf	PORTB,D'007'

		movlw	D'069'			;sequence length
		movwf	v_loop_cnt

e_m3_loop	moviw	FSR1++
		movwf	v_poti_1
		moviw	FSR1++
		movwf	v_poti_2
		moviw	FSR1++
		movwf	v_poti_3
		call	set_duty_c
		call	d2

		btfsc	v_ir_new,D'000'
		goto	mp_m3_ir_w		;now we received something

e_m3_lc		decfsz	v_loop_cnt,1
		goto	e_m3_loop

		goto	mp_mode3

;--------------------------------------------
;now we received something
mp_m3_ir_w	call	execute_comm

		movf	v_mode,0
		xorlw	D'003'
		btfsc	STATUS,Z
		goto	e_m3_lc

		goto	main_prg2
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Mode 4
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
mp_mode4	movlw	HIGH e_mode4
		movwf	FSR1H
		movlw	LOW e_mode4
		movwf	FSR1L

		bcf	PORTB,D'007'

		movlw	D'044'			;sequence length
		movwf	v_loop_cnt

e_m4_loop	moviw	FSR1++
		movwf	v_poti_1
		moviw	FSR1++
		movwf	v_poti_2
		moviw	FSR1++
		movwf	v_poti_3
		call	set_duty_c
		call	d2

		btfsc	v_ir_new,D'000'
		goto	mp_m4_ir_w		;now we received something

e_m4_lc		decfsz	v_loop_cnt,1
		goto	e_m4_loop

		goto	mp_mode4

;--------------------------------------------
;now we received something
mp_m4_ir_w	call	execute_comm

		movf	v_mode,0
		xorlw	D'004'
		btfsc	STATUS,Z
		goto	e_m4_lc

		goto	main_prg2
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------


;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;Prepare for SONY TV Code
;------------------------------------------------------------------------------
chk_vc_01	bsf	v_valid_code,D'000'

		movf	v_ir_byte5,0
		xorlw	H'FF'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code

		movf	v_ir_byte6,0
		xorlw	H'FF'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code

;------------------------------------------------------------------------
		movlw	HIGH css_code_k1
		movwf	FSR1H
		movlw	LOW css_code_k1
		movwf	FSR1L

;set pointer FSR0 to the received codes
		movlw	LOW v_ir_byte1
		movwf	v_first_byte

		movlw	D'003'
		movwf	v_code_len

		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Prepare for SONY DVD Code
;------------------------------------------------------------------------------
chk_vc_02	bsf	v_valid_code,D'000'

		movf	v_ir_byte8,0
		xorlw	H'FF'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code

;------------------------------------------------------------------------
		movlw	HIGH cssd_code_k1
		movwf	FSR1H
		movlw	LOW cssd_code_k1
		movwf	FSR1L

;set pointer FSR0 to the received codes
		movlw	LOW v_ir_byte1
		movwf	v_first_byte

		movlw	D'003'
		movwf	v_code_len

		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Prepare for SONY Video Code
;------------------------------------------------------------------------------
chk_vc_03	bsf	v_valid_code,D'000'

		movf	v_ir_byte8,0
		xorlw	H'FF'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code

;------------------------------------------------------------------------
		movlw	HIGH cssv_code_k1
		movwf	FSR1H
		movlw	LOW cssv_code_k1
		movwf	FSR1L

;set pointer FSR0 to the received codes
		movlw	LOW v_ir_byte1
		movwf	v_first_byte

		movlw	D'003'
		movwf	v_code_len

		return
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Prepare for Philips TV Code
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
chk_vc_00	bsf	v_valid_code,D'000'

		movf	v_ir_byte1,0
		xorlw	H'03'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code

		movf	v_ir_byte2,0
		xorlw	H'6A'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code

;ignore byte 3 (can be 3A or CA)

		movf	v_ir_byte4,0
		xorlw	H'AA'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code

		movf	v_ir_byte5,0
		xorlw	H'AA'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code

		movf	v_ir_byte8,0
		xorlw	H'FF'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code
;------------------------------------------------------------------------
		movlw	HIGH csp_code_k1
		movwf	FSR1H
		movlw	LOW csp_code_k1
		movwf	FSR1L

;set pointer FSR0 to the received codes
		movlw	LOW v_ir_byte6
		movwf	v_first_byte

		movlw	D'002'
		movwf	v_code_len

		return

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Prepare for Philips DVD Code
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
chk_vc_04	bsf	v_valid_code,D'000'

		movf	v_ir_byte1,0
		xorlw	H'03'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code

		movf	v_ir_byte2,0
		xorlw	H'6A'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code

;ignore byte 3 (can be 3A or CA)

		movf	v_ir_byte8,0
		xorlw	H'FF'
		btfss	STATUS,Z
		bcf	v_valid_code,D'000'		;not a valid code
;------------------------------------------------------------------------
		movlw	HIGH cspd_code_k1
		movwf	FSR1H
		movlw	LOW cspd_code_k1
		movwf	FSR1L

;set pointer FSR0 to the received codes
		movlw	LOW v_ir_byte5
		movwf	v_first_byte

		movlw	D'003'
		movwf	v_code_len

		return
;------------------------------------------------------------------------------


;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;analyze the received IR-code and execute if there is something to be executed
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
execute_comm	nop
		movf	FSR1H,0
		movwf	v_FSR1H_save
		movf	FSR1L,0
		movwf	v_FSR1L_save

		clrf	v_valid_code

		movf	v_ir_code,0
		xorlw	D'000'
		btfsc	STATUS,Z
		call	chk_vc_00		;Philips TV code

		movf	v_ir_code,0
		xorlw	D'001'
		btfsc	STATUS,Z
		call	chk_vc_01		;SONY TV code

		movf	v_ir_code,0
		xorlw	D'002'
		btfsc	STATUS,Z
		call	chk_vc_02		;SONY DVD code

		movf	v_ir_code,0
		xorlw	D'003'
		btfsc	STATUS,Z
		call	chk_vc_03		;SONY Video code

		movf	v_ir_code,0
		xorlw	D'004'
		btfsc	STATUS,Z
		call	chk_vc_04		;Philips DVD code

		btfss	v_valid_code,D'000'
		goto	irc_s_no_vc		;not a valid code

;-----------------------------------------------------------------------------
		movlw	D'001'
		movwf	v_ir_comm		;command = 1 (switch off)
;-----------------------------------------
irc_s_l2	nop
;set pointer FSR0 to the received codes
		movlw	H'00'
		movwf	FSR0H
		movf	v_first_byte,0
		movwf	FSR0L

		movwf	H'FF'
		movwf	v_comm_x

		movf	v_code_len,0		;data stream length
		movwf	v_loop_cnt2

;find if any command definition fits to received code -----------------
irc_s_l1	clrf	BSR
		moviw	FSR1++
		xorwf	INDF0,0
		btfss	STATUS,Z
		clrf	v_comm_x		;no match
		incf	FSR0L,1
		decfsz	v_loop_cnt2,1
		goto	irc_s_l1		

		movf	v_comm_x,0
		btfss	STATUS,Z
		goto	irc_s_cf		;we found a command code fitting with definition

		incf	v_ir_comm		;try next one
		movf	v_ir_comm,0
		xorlw	D'010'			;this one doesn't exist anymore
		btfsc	STATUS,Z
		goto	irc_s_no_vc		;no valid code found

		goto	irc_s_l2		;try next command code
;------------------------------------------------------------------------------------------
;------------------------------------------------------------------------------------------
irc_s_cf	bsf	PORTB,D'007'

		movf	v_ir_comm,0
		xorlw	D'001'
		btfsc	STATUS,Z
		goto	sw_b_inc

		movf	v_ir_comm,0
		xorlw	D'002'
		btfsc	STATUS,Z
		goto	sw_b_dec

		movf	v_ir_comm,0
		xorlw	D'003'
		btfsc	STATUS,Z
		goto	sw_g_inc

		movf	v_ir_comm,0
		xorlw	D'004'
		btfsc	STATUS,Z
		goto	sw_g_dec

		movf	v_ir_comm,0
		xorlw	D'005'
		btfsc	STATUS,Z
		goto	sw_r_inc

		movf	v_ir_comm,0
		xorlw	D'006'
		btfsc	STATUS,Z
		goto	sw_r_dec


		movf	v_ir_comm,0
		xorlw	D'007'
		btfsc	STATUS,Z
		goto	sw_w_on_off


		movf	v_ir_comm,0
		xorlw	D'008'
		btfsc	STATUS,Z
		goto	sw_m_i


		movf	v_ir_comm,0
		xorlw	D'009'
		btfsc	STATUS,Z
		goto	sw_m_d
;-------------------------------------------------------------------------------------------
irc_s_no_vc	movf	v_FSR1H_save,0
		movwf	FSR1H
		movf	v_FSR1L_save,0
		movwf	FSR1L

		clrf	v_ir_new
		return
;---------------------------------------------------------------
irc_s_fin	movf	v_FSR1H_save,0
		movwf	FSR1H
		movf	v_FSR1L_save,0
		movwf	FSR1L

		call	set_duty_c
		call	d2			;wait a while
		clrf	v_ir_new
		return
;-------------------------------------------------------------------------------------------
;-------------------------------------------------------------------------------------------
;----------------------------------------------
sw_b_inc	movlw	H'10'
		addwf	v_poti_1,1
		btfss	STATUS,C
		goto	irc_s_fin
		movlw	H'FF'
		movwf	v_poti_1
		goto	irc_s_fin
;----------------------------------------------
sw_b_dec	movlw	H'10'
		subwf	v_poti_1,1
		btfsc	STATUS,C
		goto	irc_s_fin
		movlw	H'00'
		movwf	v_poti_1
		goto	irc_s_fin
;----------------------------------------------
sw_g_inc	movlw	H'10'
		addwf	v_poti_3,1
		btfss	STATUS,C
		goto	irc_s_fin
		movlw	H'FF'
		movwf	v_poti_3
		goto	irc_s_fin
;----------------------------------------------
sw_g_dec	movlw	H'10'
		subwf	v_poti_3,1
		btfsc	STATUS,C
		goto	irc_s_fin
		movlw	H'00'
		movwf	v_poti_3
		goto	irc_s_fin
;----------------------------------------------
sw_r_inc	movlw	H'10'
		addwf	v_poti_2,1
		btfss	STATUS,C
		goto	irc_s_fin
		movlw	H'FF'
		movwf	v_poti_2
		goto	irc_s_fin
;----------------------------------------------
sw_r_dec	movlw	H'10'
		subwf	v_poti_2,1
		btfsc	STATUS,C
		goto	irc_s_fin
		movlw	H'00'
		movwf	v_poti_2
		goto	irc_s_fin
;----------------------------------------------
sw_w_on_off	clrf	v_mode			;mode = 0

		movf	v_poti_1,0
		btfss	STATUS,Z
		goto	sw_w_off
		movf	v_poti_2,0
		btfss	STATUS,Z
		goto	sw_w_off
		movf	v_poti_3,0
		btfss	STATUS,Z
		goto	sw_w_off
;all LEDs are off --> all LEDs to highest bright
		movlw	H'FF'
		movwf	v_poti_1
		movwf	v_poti_2
		movwf	v_poti_3
		call	set_duty_c
		call	dr2			;wait a while
		goto	irc_s_fin
;--------------------------------------
;switch all LEDs off
sw_w_off	movlw	H'00'
		movwf	v_poti_1
		movwf	v_poti_2
		movwf	v_poti_3
		call	set_duty_c
		call	dr2			;wait a while
		goto	irc_s_fin
;----------------------------------------------
sw_m_i		incf	v_mode,1
		call	dr2
		movf	v_mode,0
		xorlw	D'005'			;non existing mode
		btfss	STATUS,Z
		goto	irc_s_fin
		clrf	v_mode
		goto	irc_s_fin
;----------------------------------------------
;----------------------------------------------
sw_m_d		decf	v_mode,1
		call	dr2
		movf	v_mode,0
		xorlw	D'255'
		btfss	STATUS,Z
		goto	irc_s_fin
		movlw	D'004'			;last mode
		movwf	v_mode
		goto	irc_s_fin
;----------------------------------------------

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
; delay routines
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
d2		MOVLW   D'255'
		MOVWF   TIMER2
DEL_LOOP1	MOVLW   D'255'
		MOVWF   TIMER1

DEL_LOOP2	DECFSZ  TIMER1,F
		GOTO    DEL_LOOP2
		DECFSZ  TIMER2,F
		GOTO    DEL_LOOP1
		return
;------------------------------------------------------------------------------
d2c		MOVLW   D'090'
		MOVWF   TIMER2
cDEL_LOOP1	MOVLW   D'255'
		MOVWF   TIMER1

cDEL_LOOP2	DECFSZ  TIMER1,F
		GOTO    cDEL_LOOP2
		DECFSZ  TIMER2,F
		GOTO    cDEL_LOOP1
		return
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

;------------------------------------
dr2		call	d2
		call	d2
		call	d2
		return
;------------------------------------
dr4		call	dr2
		call	dr2
		call	dr2
		return


;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------

set_duty_c	clrf	BSR
;set the duty cycle Blue
		movf	v_poti_1,0
		banksel	CCPR1L
		movwf	CCPR1L
		clrf	BSR

;set the duty cycle Red
		movf	v_poti_2,0
		banksel	CCPR2L
		movwf	CCPR2L
		clrf	BSR

;set the duty cycle Green
		movf	v_poti_3,0
		banksel	CCPR4L
		movwf	CCPR4L
		clrf	BSR

		return
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------------------------------------------------
;------------------------------------------------------------------------------------------------------------------------
;------------------------------------------------------------------------------------------------------------------------
;------------------------------------------------------------------------------------------------------------------------
;IR-Codes / Commands
;------------------------------------------------------------------------------------------------------------------------
;Philips TV
;------------------------------------------------------------------------------------------------------------------------
csp_code_k1	da	H'AA', H'9F'					;key 1 / code 1
csp_code_k7	da	H'A9', H'5F'					;key 7 / code 2
csp_code_k2	da	H'AA', H'6F'					;key 2 / code 3
csp_code_k8	da	H'A6', H'AF'					;key 8 / code 4
csp_code_k3	da	H'AA', H'5F'
csp_code_k9	da	H'A6', H'9F'
csp_code_k_on	da	H'A5', H'AF'					;on / off
csp_code_pp	da	H'6A', H'AF'					;P+
csp_code_pm	da	H'6A', H'9F'					;P-
;------------------------------------------------------------------------------------------------------------------------

;------------------------------------------------------------------------------------------------------------------------
;Philips DVD
;------------------------------------------------------------------------------------------------------------------------
cspd_code_k1	da	H'AA', H'AA', H'9F'				;key 1 / code 1
cspd_code_k7	da	H'AA', H'A9', H'5F'				;key 7 / code 2
cspd_code_k2	da	H'AA', H'AA', H'6F'				;key 2 / code 3
cspd_code_k8	da	H'AA', H'A6', H'AF'				;key 8 / code 4
cspd_code_k3	da	H'AA', H'AA', H'5F'
cspd_code_k9	da	H'AA', H'A6', H'9F'
cspd_code_k_on	da	H'A5', H'A9', H'5F'				;on / off
cspd_code_pp	da	H'A9', H'96', H'AF'				;P+
cspd_code_pm	da	H'A9', H'96', H'9F'				;P-
;------------------------------------------------------------------------------------------------------------------------
;------------------------------------------------------------------------------------------------------------------------
;------------------------------------------------------------------------------------------------------------------------
;SONY TV
;------------------------------------------------------------------------------------------------------------------------
css_code_k1	da	H'0A', H'AA', H'A5'				;key 1 / code 1
css_code_k7	da	H'0A', H'4A', H'A9'				;key 7 / code 2
css_code_k2	da	H'09', H'55', H'52'				;key 2 / code 3
css_code_k8	da	H'09', H'25', H'54'				;key 8 / code 4
css_code_k3	da	H'0A', H'55', H'52'
css_code_k9	da	H'0A', H'A5', H'52'
css_code_k_on	da	H'09', H'4A', H'54'				;on / off
css_code_pp	da	H'0A', H'A9', H'52'				;P+
css_code_pm	da	H'09', H'54', H'A9'				;P-
;------------------------------------------------------------------------------------------------------------------------

;------------------------------------------------------------------------------------------------------------------------
;SONY DVD
;------------------------------------------------------------------------------------------------------------------------
cssd_code_k1	da	H'0A', H'AA', H'A9'				;key 1 / code 1
cssd_code_k7	da	H'0A', H'4A', H'AA'				;key 7 / code 2
cssd_code_k2	da	H'09', H'55', H'54'				;key 2 / code 3
cssd_code_k8	da	H'09', H'25', H'55'				;key 8 / code 4
cssd_code_k3	da	H'0A', H'55', H'54'	
cssd_code_k9	da	H'0A', H'A5', H'54'	
cssd_code_k_on	da	H'09', H'4A', H'55'				;on / off
cssd_code_pp	da	H'09', H'52', H'49'				;P+
cssd_code_pm	da	H'0A', H'52', H'49'				;P-
;------------------------------------------------------------------------------------------------------------------------

;------------------------------------------------------------------------------------------------------------------------
;SONY Video
;------------------------------------------------------------------------------------------------------------------------
cssv_code_k1	da	H'0A', H'AA', H'A4'				;key 1 / code 1
cssv_code_k7	da	H'0A', H'4A', H'A9'				;key 7 / code 2
cssv_code_k2	da	H'09', H'55', H'52'				;key 2 / code 3
cssv_code_k8	da	H'09', H'25', H'54'				;key 8 / code 4
cssv_code_k3	da	H'0A', H'55', H'52'	
cssv_code_k9	da	H'0A', H'A5', H'52'	
cssv_code_k_on	da	H'09', H'4A', H'54'				;on / off
cssv_code_pp	da	H'0A', H'A9', H'52'				;P+
cssv_code_pm	da	H'09', H'54', H'A9'				;P-
;------------------------------------------------------------------------------------------------------------------------
;-----------------------------------------------------------------------------------------
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Sequence definition for Mode 3
;------------------------------------------------------------------------------
e_mode3		da	H'00', H'00', H'00'
		da	H'01', H'00', H'00'
		da	H'03', H'00', H'00'
		da	H'05', H'00', H'00'
		da	H'08', H'00', H'00'
		da	H'0C', H'00', H'00'
		da	H'10', H'00', H'00'
		da	H'18', H'00', H'00'
		da	H'20', H'00', H'00'
		da	H'40', H'00', H'00'
		da	H'80', H'00', H'00'
		da	H'FF', H'00', H'00'
		da	H'80', H'00', H'00'
		da	H'40', H'00', H'00'
		da	H'20', H'00', H'00'
		da	H'18', H'00', H'00'
		da	H'10', H'00', H'00'
		da	H'0C', H'00', H'00'
		da	H'08', H'00', H'00'
		da	H'05', H'00', H'00'
		da	H'03', H'00', H'00'
		da	H'01', H'00', H'00'
		da	H'00', H'00', H'00'

		da	H'00', H'00', H'00'
		da	H'00', H'01', H'00'
		da	H'00', H'03', H'00'
		da	H'00', H'05', H'00'
		da	H'00', H'08', H'00'
		da	H'00', H'0C', H'00'
		da	H'00', H'10', H'00'
		da	H'00', H'18', H'00'
		da	H'00', H'20', H'00'
		da	H'00', H'40', H'00'
		da	H'00', H'80', H'00'
		da	H'00', H'FF', H'00'
		da	H'00', H'80', H'00'
		da	H'00', H'40', H'00'
		da	H'00', H'20', H'00'
		da	H'00', H'18', H'00'
		da	H'00', H'10', H'00'
		da	H'00', H'0C', H'00'
		da	H'00', H'08', H'00'
		da	H'00', H'05', H'00'
		da	H'00', H'03', H'00'
		da	H'00', H'01', H'00'
		da	H'00', H'00', H'00'

		da	H'00', H'00', H'00'
		da	H'00', H'00', H'01'
		da	H'00', H'00', H'03'
		da	H'00', H'00', H'05'
		da	H'00', H'00', H'08'
		da	H'00', H'00', H'0C'
		da	H'00', H'00', H'10'
		da	H'00', H'00', H'18'
		da	H'00', H'00', H'20'
		da	H'00', H'00', H'40'
		da	H'00', H'00', H'80'
		da	H'00', H'00', H'FF'
		da	H'00', H'00', H'80'
		da	H'00', H'00', H'40'
		da	H'00', H'00', H'20'
		da	H'00', H'00', H'18'
		da	H'00', H'00', H'10'
		da	H'00', H'00', H'0C'
		da	H'00', H'00', H'08'
		da	H'00', H'00', H'05'
		da	H'00', H'00', H'03'
		da	H'00', H'00', H'01'
		da	H'00', H'00', H'00'
;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
;Sequence definition for Mode 3
;------------------------------------------------------------------------------
e_mode4		da	H'00', H'80', H'00'
		da	H'01', H'40', H'00'
		da	H'03', H'20', H'00'
		da	H'05', H'18', H'00'
		da	H'08', H'10', H'00'
		da	H'0C', H'0C', H'00'
		da	H'10', H'08', H'00'
		da	H'18', H'05', H'00'
		da	H'20', H'03', H'00'
		da	H'40', H'01', H'00'
		da	H'80', H'00', H'00'

		da	H'FF', H'00', H'00'
		da	H'80', H'00', H'01'
		da	H'40', H'00', H'03'
		da	H'20', H'00', H'05'
		da	H'18', H'00', H'08'
		da	H'10', H'00', H'0C'
		da	H'0C', H'00', H'10'
		da	H'08', H'00', H'20'
		da	H'05', H'00', H'40'
		da	H'03', H'00', H'80'
		da	H'01', H'00', H'FF'
		da	H'00', H'00', H'FF'

		da	H'00', H'01', H'80'
		da	H'00', H'03', H'40'
		da	H'00', H'05', H'20'
		da	H'00', H'08', H'10'
		da	H'00', H'0C', H'0C'
		da	H'00', H'10', H'08'
		da	H'00', H'18', H'05'
		da	H'00', H'20', H'03'
		da	H'00', H'40', H'02'
		da	H'00', H'80', H'01'
		da	H'00', H'FF', H'00'

		da	H'01', H'80', H'00'
		da	H'03', H'40', H'00'
		da	H'05', H'20', H'00'
		da	H'08', H'18', H'00'
		da	H'0C', H'10', H'00'
		da	H'10', H'0C', H'00'
		da	H'20', H'08', H'00'
		da	H'40', H'05', H'00'
		da	H'80', H'03', H'00'
		da	H'FF', H'01', H'00'
;------------------------------------------------------------------------------

;------------------------------------------------------------------------------
;xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx


		END