/* **********************************************************
*															*
*	1-Kanal Analog Messpanal 4 ... 20mA 					*
*	Version 1.1												*
*															*
*	Funktionsumfang:										*
*	Messmodul, welches eine 4 .20mA Stromschnittstelle 		*
*	eines Transmitters mittels linearer Funktion zu einem 	*
*	Anzeigert fr die physikalische Messgre des			*
*	Transmitters macht.										*
*															*
*	Grenzwertberwachung und Ansteuerung eines Relais.(V1.1)*
*															*
*	Anzeige auf einem zweizeiligen LCD						*
*															*
*	Prozessor: ATMEGA 32									*
************************************************************/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>

//Defines
#define TIMER0_10MS	144		//Ladewert Timer 0 fr 10ms
#define ENTPRELLZEIT  4		//Tasten Entprellzeit = TIMER0_10MS * ENTPRELLZEIT
#define ANZ_VERS_NR   3		//Anzeigedauer Versionsnummer in Sekunden
#define MESSTAKT	  1		//Messtakt 1 Sekunde
#define TIME_PASS	  0		//Rckmeldung, dass Zeit abgelaufen ist.
#define ABGLEICHPUNKT_20MA	1	//20mA Abgleichpunkt
#define ABGLEICHPUNKT_4MA	2	// 4mA Abgleichpunkt
#define EEPROM_ADR_OFFSET	 10	//EEPROM Adresse 10dez. fr Offsetwert
#define EEPROM_ADR_STEIGUNG  20	//EEPROM Adresse 20dez. fr Steigung
#define EEPROM_ADR_GRENZWERT 30	//EEPROM Adresse 30dez fr Grenzwert

/********************************************************
*	!!! ToDo !!!										*
*   Individuell fr die Messgre anzupassende Werte	*
********************************************************/
#define NACHKOMMASTELLEN 2	//Anzahl Nachkommastellen

//Function declaration
void Init_Timer0(void);
void Init_ADC(void);
unsigned int ADC_12Bit(unsigned char MuxKanal);
void Convert_Int_Bcd(signed long int Zahl);
void Convert_BCD_ASCII(void);
void Init_LCD(void);
void LCD_Zeile1(unsigned char *pLCD);
void LCD_Zeile2(unsigned char *pLCD);
void WR_COMAND(unsigned char i);
void WR_DATA(unsigned char i);
void Wait(unsigned int i);
void Pause_Timer (unsigned char Zeit);
void Wait_Tasten_Open(void);
signed long int Edit_Abgleich(unsigned char Abgleichpunkt);
signed long int Edit_Abgleichwert(unsigned char * i);
void Berechnung_Abgleich(signed long int Abgl20MA, signed int long Abgl4MA);
void Pruefung_Grenzwert(signed long int Messwert, signed long int Grenzwert);
signed long int Edit_Grenzwert(signed long int Grenzwert);
unsigned char EEPROM_Read_CHAR (unsigned int uiAdresse);
void EEPROM_Write_CHAR (unsigned int uiAdresse, unsigned char ucDaten);
float EEPROM_Read_FLOAT (unsigned int uiAdresse);
void EEPROM_Write_FLOAT (unsigned int uiAdresse, float fDaten);


//Globales
unsigned char Sekunde;					//Kennzeichnungsflag fr Sekunde
unsigned char SecTakt;					//Sekundenzhler;
unsigned char PausenTime;				//Zhlvariable fr Pausenzeiten
unsigned char PausenTakt;				//Arbeitsvariable fr PausenTime (sec. Takt)
unsigned char LCD_String1[] = "4-20mA Messmodul";
unsigned char LCD_String2[] = "DO6GM  Vers. 1.1";
unsigned char BCD_Numb[7];	//BCD Ergebnisspeicher
signed long int AbgleichPunkt1 =  50000;	//Defaultwert fr Abgleichpunkt 20mA
signed long int AbgleichPunkt2 = -10000;	//Defaultwert fr Abgleichpunkt  4mA 
signed long int Grenzwert = 100;			//Defaultwert fr ersten Grenzwert
unsigned char Prellcounter;
unsigned char Tasten_Neu;
unsigned char Tasten_Alt;
unsigned char Tastenzustand;
unsigned char Taste_Enter;
unsigned char Taste_Rechts;
unsigned char Taste_Up;

/********************************************************
*	!!! ToDo !!!										*
*   Individuell fr die Messgre anzupassende Werte	*
********************************************************/
unsigned char LCD_String3[] = "  Wasserpegel   ";	//16 Zeichen fr die Messgre
unsigned char Einheit[] = " cm ";					// 4 Zeichen fr die Einheit
		
int main(void)
{
	unsigned char Textzeile1[17];
	unsigned char i, j, k;
	unsigned int ADC_Rohwert;
	signed long int Messwert;
	float Messwert_Berechnet, Offset, Steigung;
	
	Init_LCD();		//Initialisierung
	Init_ADC();
	Init_Timer0();
	DDRB = 0x08;				//Tastaturport (Port B) auf Input
								//Port B3 (Grenzwertrealis) au Output schalten
	PORTB = 0xF7;				//Port B (Input) Pull Up aktiv Grenzwertrealis aus
	LCD_Zeile1(LCD_String1);
	LCD_Zeile2(LCD_String2);
	wdt_enable(WDTO_2S);
	sei();			//Interrupts freigeben
	Prellcounter = 0;
	Tasten_Neu = PINB;			//Tastatur initialisieren
	Tasten_Alt = Tasten_Neu;
	Tastenzustand = 0b00000111;	//keine Taste gedrckt
	Taste_Enter = 0;
	Taste_Rechts = 0;
	Taste_Up = 0;
	Pause_Timer(ANZ_VERS_NR);	//Anzeigedauer Versions Nr.
	LCD_Zeile1(LCD_String3);

	float EEPROM_Read_FLOAT (unsigned int uiAdresse);
	void EEPROM_Write_FLOAT (unsigned int uiAdresse, float fDaten);

	//Offset und Steigung aus EEPROM laden
	Offset = EEPROM_Read_FLOAT(EEPROM_ADR_OFFSET);
	Steigung = EEPROM_Read_FLOAT(EEPROM_ADR_STEIGUNG);
	Grenzwert = (signed long int)EEPROM_Read_FLOAT(EEPROM_ADR_GRENZWERT);

	while (1)
	{
		ADC_Rohwert = ADC_12Bit(0x00);		//ADC-Kanal 0 messen
		Messwert_Berechnet = Steigung * (float)ADC_Rohwert + Offset;

		Messwert = (signed long int)Messwert_Berechnet;
		Convert_Int_Bcd(Messwert);
		Convert_BCD_ASCII();

		Textzeile1[0] = ' ';
		Textzeile1[1] = ' ';
		Textzeile1[2] = ' ';
		Textzeile1[3] = ' ';

		i = NACHKOMMASTELLEN;
		if (i == 0)
		{
			Textzeile1[4] = ' ';	//kein Komma
			i = 5;
			k = 0;
		}
		else
		{
			i = 4;
			k = i + (7 - NACHKOMMASTELLEN);	//Positionszeiger fr Komma
		}

		j = 0;
		while (i < 12)
		{
			if (k == i)
			{
				Textzeile1[i] = ',';		//Komma einfgen
				i++;
			}
			else 
			{
				Textzeile1[i] = BCD_Numb[j];
				i++;
				j++;
			}
		}
		Textzeile1[12] = Einheit[0];
		Textzeile1[13] = Einheit[1];
		Textzeile1[14] = Einheit[2];
		Textzeile1[15] = Einheit[3];

		LCD_Zeile2(Textzeile1);

		Pruefung_Grenzwert(Messwert, Grenzwert);

		Pause_Timer(MESSTAKT);	//Wartezeit = Messtakt

		if (Tastenzustand == 0b00000100)
		{		
			AbgleichPunkt1 = Edit_Abgleich(ABGLEICHPUNKT_20MA);
			AbgleichPunkt2 = Edit_Abgleich(ABGLEICHPUNKT_4MA);
			Berechnung_Abgleich(AbgleichPunkt1, AbgleichPunkt2);
			Offset = EEPROM_Read_FLOAT(EEPROM_ADR_OFFSET);
			Steigung = EEPROM_Read_FLOAT(EEPROM_ADR_STEIGUNG);
			LCD_Zeile1(LCD_String3);
		}
		if (Tastenzustand == 0b00000001)
		{
			Grenzwert = (signed long int)EEPROM_Read_FLOAT(EEPROM_ADR_GRENZWERT);
			Grenzwert = Edit_Grenzwert(Grenzwert);
			EEPROM_Write_FLOAT (EEPROM_ADR_GRENZWERT, (float)Grenzwert);

			LCD_Zeile1(LCD_String3);
		}

		wdt_reset();
	}
}

/****************************************************************
*	Integer auf BCD  Convertierung								*
*																*
*	Der bergebene signed Integerwert wird in eine              *
*	6-stellige BCD Zahl gewandelt.								*
*	Das Ergebnis steht in BCD_Numb[] 							*
****************************************************************/
void Convert_Int_Bcd(signed long int Zahl)           //6-stellig
{
	signed long int i;

	i = Zahl;
	i = i & 0x80000000;
	if (i == 0)
	{
		//Zahl ist positiv
		BCD_Numb[0] = ' ';	//Vorzeichen (fr + blank)
	}
	else
	{
		//Zahl ist negativ
		BCD_Numb[0] = '-';	//Vorzeichen
		Zahl = ~Zahl;		//Zweier Komplement bilden
		Zahl = Zahl + 1;
	}
	BCD_Numb[6] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[5] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[4] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[3] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[2] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[1] = Zahl % 10;
}      

/****************************************************************
*	BCD auf ASCII Convertierung									*
*																*
*	Die in BCD_Numb[] stehende 6-stellige BCD Zahl wird auf 	*
*	ASCII gewandelt und wieder in BCD_Numb[]gespeichert.		*
*   Das auf BCD_Numb[0] stehende Vorzeichen ist schon auf ASCII.*
****************************************************************/
void Convert_BCD_ASCII(void)
{
	unsigned char i;

	for (i = 1; i < 7; i++)
	{
		BCD_Numb[i] = BCD_Numb[i] + 0x30;
	}
}


/****************************************************************
*	AD-Wandler Funktionen										*
****************************************************************/

/*	ADC als 12 Bit ADC fr singel conversion, singel endet Betrieb
    mit interner Referenz initialisieren.
	ADC-Takt = Oszillatortakt / 64								*/
void Init_ADC(void)
{
	ADCSRA = 0b10000110;	//Enable ADC, Trigger Off, no Interrupt, Osz/64
	SFIOR = 0x00;
	ADMUX = 0b11000000;		//Vref = 2,56V, linksbndig, Mux = 0
}

/*	ADC als 12 Bit Wandler.
	4 Wandlungen auf dem durch Mux Kanal bergebenen Eingangskanal.
	Addition der vier Wandlungsergebnisse zu einer 12 Bit Zahl. */
unsigned int ADC_12Bit(unsigned char MuxKanal)
{
	char cHelp, i;
	unsigned int ADC_Ergebnis = 0x0000;

	MuxKanal = MuxKanal & 0b00000111;	//Mux singel endet Kanalnummer isolieren
	cHelp = ADMUX;
	cHelp = cHelp & 0b11111000;			//Kanalnummer mit ADMUX vereinen
	cHelp = cHelp | MuxKanal;
	ADMUX = cHelp;						//Kanalnummer setzen
	for (i = 0; i < 4; i++)
	{
		cHelp = ADCSRA;
		cHelp = cHelp | 0b01000000;			//AD-Wandler starten
		ADCSRA = cHelp;
		cHelp = 0b0100000;
		while (cHelp != 0)					//Warten auf ADC ready warten
		{
			cHelp = ADCSRA;
			cHelp = cHelp & 0b01000000;
		}

		ADC_Ergebnis = ADC_Ergebnis + ADCL;
		ADC_Ergebnis = ADC_Ergebnis + (ADCH << 8);
	}
	
	return(ADC_Ergebnis);	
}


/****************************************************************
*	LCD Funktionen												*
****************************************************************/

/*	Init LCD
	Programmteil zur Initialisierung des LCD Port und des LCD */
void Init_LCD (void)
{
	DDRC = 0xFF;	//LCD Port auf Ausgang schalten
	Wait(6000);		//Wartezeit fr stabile Betriebsspannung  (min. 15ms)
	WR_COMAND(0b00000011);
	Wait(1000);		//min. 4,1ms
	WR_COMAND(0b00000011);
	Wait(200);		//min. 100s
	WR_COMAND(0b00000011);
	Wait(1000);
	WR_COMAND(0b00000010);	//Function SET
	Wait(1000);
	WR_COMAND(0b00000010);	//set interface to 4-Bit
	Wait(1000);
	WR_COMAND(0b00001000);	//2 - lines, 5x7 Font
	Wait(1000);
	WR_COMAND(0b00000000);	//Display OFF (1. Nibble)
	Wait(1000);
	WR_COMAND(0b00001000);	//Display OFF (2. Nibble)
	Wait(1000);
	WR_COMAND(0b00000000);	//Display Clear (1. Nibble)
	Wait(1000);
	WR_COMAND(0b00000001);	//Display Clear (2. Nibble)
	Wait(1000);
	WR_COMAND(0b00000000);	//Entry Mode set (1. Nibble)
	Wait(1000);
	WR_COMAND(0b00000110);	//Entry Mode set (2. Nibble)
	Wait(1000);
	WR_COMAND(0b00000000);	//Switch Display ON (1. Nibble)
	Wait(1000);
	WR_COMAND(0b00001110);	//Switch Display ON (2. Niblle)
	Wait(1000);
}


/*	LCD Ausgabe auf Zeile 1	*/
void LCD_Zeile1(unsigned char *pLCD)
{
	unsigned char i;
	unsigned char j;

	WR_COMAND(0b00001000);	//Set Adresse 0x80 fr 1. Zeile (low Nibble)
	WR_COMAND(0b00000000);	//Set Adresse 0x80 fr 1. Zeile (high Nibble)
	Wait(500);
	for (i = 0; i < 16; i++)
	{
		j = *pLCD;
		WR_DATA(j);
		pLCD++;
	}
}


/*	LCD Ausgabe auf Zeile 2	*/
void LCD_Zeile2(unsigned char *pLCD)
{
	unsigned char i;
	unsigned char j;

	WR_COMAND(0b00001100);	//Set Adresse 0xC0 fr 1. Zeile (low Nibble)
	WR_COMAND(0b00000000);	//Set Adresse 0xC0 fr 1. Zeile (high Nibble)
	Wait(500);
	for (i = 0; i < 16; i++)
	{
		j = *pLCD;
		WR_DATA(j);
		pLCD++;
	}
}


/* 	Write Commando an LCD
	Daten auf PortC.0 - PortC.3, RS auf PortC.4, E auf PortC.5 */
void WR_COMAND(unsigned char i)
{
	i = i & 0x0F;		//Daten isolieren (RS und E auf 0 schalten)
	PORTC = i;
	Wait(10);
	i = i | 0b00100000;	//E Signal setzen
	PORTC = i;
	Wait(10);
	i = i & 0x0F;		//E Signal zurcknehmen
	PORTC = i;	
	Wait(10);
}


/*	Write Data an LCD
	Daten auf PortC.0 - PortC.3, RS auf PortC.4, E auf PortC.5 */
void WR_DATA(unsigned char i)
{
	unsigned char j;

	j = i;
	j = j & 0xF0;		//oberes Nibble zuerst senden
	j = j>>4;
	j = j | 0b00010000;	//RS fr Daten auf 1 setzen
	PORTC = j;
	Wait(10);
	j = j | 0b00100000;	//E Signal setzen
	PORTC = j;
	Wait(10);
	j = j & 0x1F;		//E Signal zurcknehmen
	PORTC = j;
	Wait(10);
	j = i;				//unters Nibble senden
	j = j & 0x0F;
	j = j | 0b00010000;	//RS fr Daten auf 1 setzen
	PORTC = j;
	Wait(10);
	j = j | 0b00100000;	//E Signal setzen
	PORTC = j;
	Wait(10);
	j = j & 0x1F;		//E Signal zurcknehmen
	PORTC = j;
	Wait(10);
}

/*	Abgleichfunktion */

signed long int Edit_Abgleich(unsigned char Abgleichpunkt)
{
	unsigned char i, j, k;
	unsigned char LCD_Abgleich[17];
	signed long int ReturnValue;

	if(Abgleichpunkt == ABGLEICHPUNKT_20MA)
	{
		LCD_Abgleich[0] = 'A';
		LCD_Abgleich[1] = 'b';
		LCD_Abgleich[2] = 'g';
		LCD_Abgleich[3] = 'l';
		LCD_Abgleich[4] = 'e';
		LCD_Abgleich[5] = 'i';
		LCD_Abgleich[6] = 'c';
		LCD_Abgleich[7] = 'h';
		LCD_Abgleich[8] = ' ';
		LCD_Abgleich[9] = '2';
		LCD_Abgleich[10] = '0';
		LCD_Abgleich[11] = 'm';
		LCD_Abgleich[12] = 'A';
		LCD_Abgleich[13] = ' ';
		LCD_Abgleich[14] = '-';
		LCD_Abgleich[15] = '>';
	}
	if(Abgleichpunkt == ABGLEICHPUNKT_4MA)
	{
		LCD_Abgleich[0] = 'A';
		LCD_Abgleich[1] = 'b';
		LCD_Abgleich[2] = 'g';
		LCD_Abgleich[3] = 'l';
		LCD_Abgleich[4] = 'e';
		LCD_Abgleich[5] = 'i';
		LCD_Abgleich[6] = 'c';
		LCD_Abgleich[7] = 'h';
		LCD_Abgleich[8] = ' ';
		LCD_Abgleich[9] = ' ';
		LCD_Abgleich[10] = '4';
		LCD_Abgleich[11] = 'm';
		LCD_Abgleich[12] = 'A';
		LCD_Abgleich[13] = ' ';
		LCD_Abgleich[14] = '-';
		LCD_Abgleich[15] = '>';
	}
	LCD_Zeile1(LCD_Abgleich);

	if(Abgleichpunkt == ABGLEICHPUNKT_20MA)
		Convert_Int_Bcd(AbgleichPunkt1);
	else
		Convert_Int_Bcd(AbgleichPunkt2);

	Convert_BCD_ASCII();

	LCD_Abgleich[0] = ' ';
	LCD_Abgleich[1] = ' ';
	LCD_Abgleich[2] = ' ';
	LCD_Abgleich[3] = ' ';

	i = NACHKOMMASTELLEN;
	if (i == 0)
	{
		LCD_Abgleich[4] = ' ';	//kein Komma
		i = 5;
		k = 0;
	}
	else
	{
		i = 4;
		k = i + (7 - NACHKOMMASTELLEN);	//Positionszeiger fr Komma
	}

	j = 0;
	while (i < 12)
	{
		if (k == i)
		{
			LCD_Abgleich[i] = ',';		//Komma einfgen
			i++;
		}
		else 
		{
			LCD_Abgleich[i] = BCD_Numb[j];
			i++;
			j++;
		}
	}

	LCD_Abgleich[12] = Einheit[0];
	LCD_Abgleich[13] = Einheit[1];
	LCD_Abgleich[14] = Einheit[2];
	LCD_Abgleich[15] = Einheit[3];

	LCD_Zeile2(LCD_Abgleich);

	Wait_Tasten_Open();

	ReturnValue = Edit_Abgleichwert(&LCD_Abgleich[0]);
	
	return (ReturnValue);
}

signed long int Edit_Abgleichwert(unsigned char * i)
{
	unsigned char j, k;
	unsigned char EditWert[17];
	signed long int ErgebnisWert, Faktor;

	for(j=0; j<16; j++)
	{
		EditWert[j] = *i;
		i++;
	}
	LCD_Zeile2(EditWert);

	j = NACHKOMMASTELLEN;
	if (j == 0)
	{
		j = 5;	//kein Komma, deshalb erste Ziffer bzw. Vorzeichen auf Stelle 5
		k = 0;
	}
	else
	{
		j = 4;
		k = j + (7 - NACHKOMMASTELLEN);	//Positionszeiger fr Komma
	}

	while (Taste_Enter == 0)
	{
		if(Taste_Rechts == 1)
		{
			j++;
			if(j == k)
				j++;
			if(j == 12)
			{
				j = NACHKOMMASTELLEN;
				if (j == 0)
				{
					j = 5;	//kein Komma, deshalb erste Ziffer bzw. Vorzeichen auf Stelle 5
					k = 0;
				}
				else
				{
					j = 4;
					k = j + (7 - NACHKOMMASTELLEN);	//Positionszeiger fr Komma
				}
			}
			Wait_Tasten_Open();
		}
		if((j == 4) || (j == 5 && k == 0))
		{
			if(Taste_Up == 1)
			{
				if(EditWert[j] == ' ')
					EditWert[j] = '-';
				else
					EditWert[j] = ' ';
				LCD_Zeile2(EditWert);
				Wait_Tasten_Open();
			}
		}
		else
		{
			if(Taste_Up == 1)
			{
				EditWert[j]++;
				if(EditWert[j] == ':')
					EditWert[j] = '0';
				LCD_Zeile2(EditWert);
				Wait_Tasten_Open();
			}
		}
	}

	ErgebnisWert = 0;
	Faktor = 1;
	if (k == 0)
	{						//kein Komma vorhanden
		for(j = 11; j > 5; j--)
		{
			ErgebnisWert = ErgebnisWert + ((EditWert[j] - 0x30) * Faktor);
			Faktor = Faktor * 10;
		}
		if(EditWert[j] == '-')
			ErgebnisWert = ErgebnisWert * (-1);
	}
	else
	{
		for(j = 11; j > 4; j--)
		{
			if(j == k)
				j--;		//Komma berspringen
			ErgebnisWert = ErgebnisWert + ((EditWert[j] - 0x30) * Faktor);
			Faktor = Faktor * 10;
		}
		if(EditWert[j] == '-')
			ErgebnisWert = ErgebnisWert * (-1);
	}

	return (ErgebnisWert);
}

void Berechnung_Abgleich(signed long int Abgl20MA, signed int long Abgl4MA)
{
	float X4MA, X20MA;
	float Offset, Steigung;

	X4MA = 640.0;		//Countwert fr 4 mA
	X20MA = 3197.0;		//Countwert fr 20mA

	Steigung = ((float)Abgl20MA - (float)Abgl4MA)/(X20MA - X4MA);
	Offset = Abgl20MA - Steigung * X20MA;

	//Berechnete Werte im EEPROM speichern
	EEPROM_Write_FLOAT(EEPROM_ADR_OFFSET, Offset);
	EEPROM_Write_FLOAT(EEPROM_ADR_STEIGUNG, Steigung);
}


void Pruefung_Grenzwert(signed long int Messwert, signed long int Grenzwert)
{
	if (Messwert > Grenzwert)
		PORTB = 0xFF;			//Grenzwertrelais ein
	else
		PORTB = 0xF7;			//Grenzwertrelais aus
}

signed long int Edit_Grenzwert(signed long int Grenzwert)
{
	unsigned char i, j, k;
	unsigned char LCD_Grenzwert[17];
	signed long int ReturnValue;

	LCD_Grenzwert[0] =  ' ';
	LCD_Grenzwert[1] =  ' ';
	LCD_Grenzwert[2] =  ' ';
	LCD_Grenzwert[3] =  'G';
	LCD_Grenzwert[4] =  'r';
	LCD_Grenzwert[5] =  'e';
	LCD_Grenzwert[6] =  'n';
	LCD_Grenzwert[7] =  'z';
	LCD_Grenzwert[8] =  'w';
	LCD_Grenzwert[9] =  'e';
	LCD_Grenzwert[10] = 'r';
	LCD_Grenzwert[11] = 't';
	LCD_Grenzwert[12] = ':';
	LCD_Grenzwert[13] = ' ';
	LCD_Grenzwert[14] = ' ';
	LCD_Grenzwert[15] = ' ';

	LCD_Zeile1(LCD_Grenzwert);

	Convert_Int_Bcd(Grenzwert);

	Convert_BCD_ASCII();

	LCD_Grenzwert[0] = ' ';
	LCD_Grenzwert[1] = ' ';
	LCD_Grenzwert[2] = ' ';
	LCD_Grenzwert[3] = ' ';

	i = NACHKOMMASTELLEN;
	if (i == 0)
	{
		LCD_Grenzwert[4] = ' ';	//kein Komma
		i = 5;
		k = 0;
	}
	else
	{
		i = 4;
		k = i + (7 - NACHKOMMASTELLEN);	//Positionszeiger fr Komma
	}

	j = 0;
	while (i < 12)
	{
		if (k == i)
		{
			LCD_Grenzwert[i] = ',';		//Komma einfgen
			i++;
		}
		else 
		{
			LCD_Grenzwert[i] = BCD_Numb[j];
			i++;
			j++;
		}
	}

	LCD_Grenzwert[12] = Einheit[0];
	LCD_Grenzwert[13] = Einheit[1];
	LCD_Grenzwert[14] = Einheit[2];
	LCD_Grenzwert[15] = Einheit[3];

	LCD_Zeile2(LCD_Grenzwert);

	Wait_Tasten_Open();

	ReturnValue = Edit_Abgleichwert(&LCD_Grenzwert[0]);
	
	return (ReturnValue);
}


/*	Initialisierung Timer 0	auf 10ms	*/
void Init_Timer0(void)
{
	TCCR0 = 0b00000000;			//Timer 0 Stop
	TIMSK = (1 << TOIE0);		//Timer 0 Interrupt freigeben
	TCNT0 = 256 - TIMER0_10MS;	//Ladewert fr 10ms
	TCCR0 = 0b00000100;			//Timer 0 Starten (Takt = CPU-Takt/256)
	SecTakt = 0;				//Sekundentakt = 0
	Sekunde = 0;
}


/* Funktion zur Erzeugung einer Wartezeit */
/* 2MHz:	i =    1 --> 120s
			i =   10 --> 1,2ms
			i =  100 --> 12ms
			i = 1000 --> 120ms            */
void Wait(unsigned int i)
{
	unsigned char j;
	while (i != 0)
	{
		j = 10;
		while (j != 0)
		{
			j--;
		}
		i--;
	}
}

/* Funktion zur Erzeugung einer Wartezeit im Sekundentakt
   maximale Wartezeit: 255 sec. Steuerung ber Timer 0
   Wenn inerhalb der Wartezeit eine beliebige Taste gedrckt wird
   wird die Wartezeit abgebrochen. */
void Pause_Timer (unsigned char Zeit)
{
	unsigned char Tastenzustand = 0b00000111;

	PausenTakt = 0;				//Pausentaktsteuerung synchronisieren
	PausenTime = Zeit;			//Zeit bernehmen
	while (PausenTime != 0 && Tastenzustand == 0b00000111)
	{
		//Warten bis Zeit abgelaufen ist
		Tastenzustand = PINB;
		Tastenzustand = Tastenzustand & 0b00000111;
		wdt_reset();
	}
}

/* Interrupt Service Routinen	*/

ISR(TIMER0_OVF_vect)
{
	TCNT0 = 256 - TIMER0_10MS;	//Ladewert fr 10ms

	wdt_reset();

	SecTakt++;
	if (SecTakt == 60)
	{
		Sekunde = 1;			//Sekundentakt
		SecTakt = 0;
	}

	PausenTakt++;
	if (PausenTakt == 60)
	{
		PausenTakt = 0;			//Pasuentakt Steuerung
		PausenTime--;
	}

	Tasten_Neu = PINB;			//Tastatur einlesen
	Tasten_Neu = Tasten_Neu & 0b00000111;
	if (Tasten_Neu == Tasten_Alt)
	{
		Prellcounter++;
		if(Prellcounter == ENTPRELLZEIT)
		{
			Prellcounter = 0;
			Tastenzustand = Tasten_Neu;
			if((Tastenzustand & 0b00000001) == 0)
			{
				Taste_Enter = 1;
			}
			else
			{
				Taste_Enter = 0;
			}
			if((Tastenzustand & 0b00000010) == 0)
			{
				Taste_Rechts = 1;
			}
			else
			{
				Taste_Rechts = 0;
			}
			if((Tastenzustand & 0b00000100) == 0)
			{
				Taste_Up = 1;
			}
			else
			{
				Taste_Up = 0;
			}
		}
	}
	else
	{
		Tasten_Alt = Tasten_Neu;
		Prellcounter = 0;
	}
}

void Wait_Tasten_Open(void)
{
	while (Tastenzustand != 0b00000111)
	{
		//auf loslassen der Tasten warten
		wdt_reset();
	}
}


float EEPROM_Read_FLOAT (unsigned int uiAdresse)
{
	typedef union {
		float f;
		unsigned char b[4];
		} FB;

	FB uUebergabe;

	uUebergabe.b[0] = EEPROM_Read_CHAR (uiAdresse + 0);
	uUebergabe.b[1] = EEPROM_Read_CHAR (uiAdresse + 1);
	uUebergabe.b[2] = EEPROM_Read_CHAR (uiAdresse + 2);
	uUebergabe.b[3] = EEPROM_Read_CHAR (uiAdresse + 3);
	return (uUebergabe.f);
}


unsigned char EEPROM_Read_CHAR (unsigned int uiAdresse) 
{
	unsigned char i;

	cli();							//Interrupts sperren

	while (EECR & (1<<EEWE))
	{
		;
	}	

	EEAR = uiAdresse;				//Set Adresse
	EECR |= (1<<EERE);				//Set EERE
	i = EEDR;						//Daten lesen

	sei();							//Interrupts freigeben

	return (i);
}

void EEPROM_Write_FLOAT (unsigned int uiAdresse, float fDaten)
{

	typedef union {
		float f;
		unsigned char b[4];
		} FB;

	FB uUebergabe;

	uUebergabe.f = fDaten;
	EEPROM_Write_CHAR (uiAdresse + 0, uUebergabe.b[0]);
	EEPROM_Write_CHAR (uiAdresse + 1, uUebergabe.b[1]);
	EEPROM_Write_CHAR (uiAdresse + 2, uUebergabe.b[2]);
	EEPROM_Write_CHAR (uiAdresse + 3, uUebergabe.b[3]);
}

void EEPROM_Write_CHAR (unsigned int uiAdresse, unsigned char ucDaten) 
{
	unsigned int iTest = 0;

	#define EECR 28
	#define EEMWE 2
	#define EEWE 1

	while (EECR & (1<<EEWE))
	{
		;
	}	

	EEAR = uiAdresse;
	EEDR = ucDaten;

	cli();							//Interrupts sperren

/*	EECR: Adresse 28	EEMWE: Bit 2	EEWE:	Bit 1	*/
asm volatile (
	"sbi 28,2 \n\t"
	"sbi 28,1 \n\t"
	);
	
	iTest = 0;				//Wartezeit bis EEPROM geschrieben ist
	while (iTest < 10000)
	{
		iTest++;
	}

	sei();							//Interrupts freigeben
}

