/*------------------------------------------------------------

Si571 tx firmware
based on TR2200 project

Written (c) by Wulf-Gerd Traving 2009

Si571 mark: CJC000288G DO8KS817*

PART NUMBER:  	 571CJC000288G
OSCILLATOR SPECIFICATION SUMMARY
9/14/2009 1:03:43 PM
Model Number: 	Si571 (Programmable VCXO)
Output Format:	CMOS
VDD:	3.3V
Output Enable Polarity:	OE active high
Temperature Stability:	+/- 20 ppm
Tuning Slope:	135 ppm/V (positive)
Minimum APR: 	+/- 130 ppm
Frequency Range: 	10 - 160 MHz
Startup Frequency (MHz):	134.800000
I2C Address:	55 hex (85 decimal)
Chip Revision: G

Output power at startup freq: 11.8dBm / 50Ohm

---------------------------------------------------------------*/

#include "defines.h"

#include <ctype.h>
#include <stdint.h>
#include <stdio.h>

#include <avr/eeprom.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>

#include <util/delay.h>

#include "iocompat.h"
#include "si570.h"

int16_t ad_poti1 = -999;
uint8_t transmitting = 0;
int8_t button = -1;

uint32_t freq, rxfreq, txfreq;

typedef enum {
	fixed,
	variable
	} mode_t;

mode_t mode;
uint8_t channel;

uint32_t PROGMEM channel_tab [] PROGMEM = {
	145275000, // OV M02
	144800000, // APRS
	145500000, // Anruf Mobil
	145550000, // Mobil FM
	145700000, // DB0MAR Bderstr.
	145625000, // DB0ZA Aschberg
	145000000, // Alignment (last channel with 1k pot)
	145000000, // Alignment
	145000000, // Alignment
	145000000, // Alignment
	145000000, // Alignment
	145000000, // Alignment
};

#define CHANNELS (sizeof(channel_tab)/sizeof(uint32_t))


/*
 * Do all the startup-time peripheral initializations.
 */
static void ioinit(void)
{

  // enable all pullup to prevent floating inputs
  DDRB  = 0x00; // all input
  PORTB = 0xfb; // all pullups but pb2 = ADC1 enabled

  /* enable ADC, select ADC clock = F_CPU / 8 (i.e. 1MHz kHz) */
  ADCSRA = _BV(ADEN) | _BV(ADPS1) | _BV(ADPS0);
  /* set reference voltage to internal 2.56V source */
  ADMUX |= _BV(REFS1) | _BV(REFS2);

  sei ();
  silabs_init (); /* initialize the Si571, if found */
}


void set_rxtx (void)
{
	if (transmitting) silabs_setNewFreq (txfreq);
	else silabs_setNewFreq (rxfreq);
	_delay_ms (50.0);
}


void set_vfo (uint32_t fhz)
{
	rxfreq = (fhz-ZF);
	// relais shift, if needed
	if (fhz >= 145600000l && fhz < 145800000l)
	{
		txfreq = (fhz - 600000l); // 0.6 MHz shift
	}
	else
	{
		txfreq = fhz;
	}
	set_rxtx ();
}


// handle poti 1 (set frequency or channel)
// read in 1k var. resistor in line with 68 Ohm, 2.2k pullup
// on FA-SY pin 14
void do_poti1 (int16_t new_val)
{
	uint8_t new_channel;
	if ((new_val > ad_poti1+2) || (new_val < ad_poti1-2)) // filter noise
	{
		ad_poti1 = new_val;

		if (mode == variable)
		{
			freq = 144350000l + new_val/5 * 12500L; // abt. 144.5 ... 145.8 MHz range
			set_vfo (freq);
		}

		if (mode == fixed)
		{
    		// Treppenstufen sind wegen vorliegender Beschaltung des Einganges
    		// nicht gleichhoch, daher dieses Konstrukt...
    		if (new_val < 127) new_channel = 0;
    		else if (new_val < 246) new_channel = 1;
    		else if (new_val < 350) new_channel = 2;
    		else if (new_val < 436) new_channel = 3;
    		else if (new_val < 522) new_channel = 4;
    		else if (new_val < 596) new_channel = 5;
    		else if (new_val < 662) new_channel = 6;
    		else if (new_val < 724) new_channel = 7;
    		else if (new_val < 777) new_channel = 8;
    		else if (new_val < 824) new_channel = 9;
    		else if (new_val < 866) new_channel = 10;
    		else new_channel = 11;
			if (new_channel != channel)
			{
				channel = new_channel;
				freq = pgm_read_dword (&channel_tab [channel]);
				set_vfo (freq);
			}
		}
	}
}

void handle_ad (void)
{
    uint16_t temp;

	ADMUX = (ADMUX & ~0x0F) | 1;
	ADCSRA |= _BV(ADSC);
	while (ADCSRA & _BV(ADSC));
	temp = ADCL + (ADCH<<8);
	do_poti1 (temp);
}


// handle vfo / channel mode switch on FA-SY pin 17 (low = vfo mode with 12.5kHz steps)
void handle_switch (void)
{
	mode_t new_mode;
	new_mode = CHANVFO?0:1;
	if (new_mode != mode)
	{
		mode = new_mode;
		ad_poti1 = -999; // force poti evaluation
		if (mode == fixed) channel = 255; // and reset channel then
	}
}

// handle PTT input
void handle_button (void)
{
	uint8_t new_button;
	new_button = PTT?1:0;
	if (new_button != button)
	{
		button = new_button;
		if (button)
		{
			// rx
			transmitting = 0;
		}
		else
		{
			// tx
			transmitting = 1;
		}
		set_vfo (freq);
		// set_rxtx ();
	}
}


/* ---------------------------------------------------------------------------------- */

int main(void)
{
    ioinit();

	_delay_ms (500.0);

	// initialize channel and freq to proper startup value
	mode = fixed;
	channel = 255; // force evaluation of channel
	handle_ad  ();


    for (;;)
    {
		handle_button ();

		if (!transmitting) // don't change operating frequency during transmission
		{
			handle_switch ();
			handle_ad ();
		}

		_delay_ms ((float) UPDATEINTERVAL);

    }
}
