; ------------------------------------------------------------------------------------------------------------
; DCF Uhr/Kalender mit ATMega8L, S65-Display, MMC-Karte und DCF-Modul
; CPU												: ATMega8L
; Quarzfrequenz										: 8,000 MHz
; IRQ's												: Timer1Overflow
; Letzte nderung									: 13.04.2009
; ------------------------------------------------------------------------------------------------------------
.include "m8def.inc"							 
; ------------------------------------------------------------------------------------------------------------
; I/O Definitionen

.equ	SD_Bus_In					= PINB
.equ	SD_Bus_Out					= PORTB
.equ	SD_Sig_CS					= PB2			; (/SS) [Low = SD Karte selected, High = deselected]
.equ	SD_Sig_DataOut				= PB3			; Datenleitung zur Karte (MOSI)
.equ	SD_Sig_DataIn				= PB4			; Datenleitung von Karte (MISO)
.equ	SD_Sig_Clock				= PB5			; Clock fr Karte (SCK) [Low = Idle State]

.equ	Display_Bus_In				= PINC
.equ	Display_Bus_Out				= PORTC
.equ	Display_Sig_RS				= PC0			; Display Register Select (High=Befehl, Low=Daten)
.equ	Display_Sig_Reset			= PC1			; Display Reset (Low = Reset, High = Normal)
.equ	Display_Sig_CS				= PC2			; Low = Display selected, High = deselected
.equ	Display_Sig_Clock			= PC3			; Clock fr Display (Low = Idle State)
.equ	Display_Sig_Data			= PC4			; Datenleitung zum Display (bertragung MSB first)

.equ	DCF_Bus_In					= PIND
.equ	DCF_Bus_Out					= PORTD
.equ	DCF_Sig_Data				= PD0			; Datenleitung DCF (High wenn Signal moduliert[absinkt])
.equ	DCF_Sig_PON					= PD1			; Power-Sig DCF Einheit (Low=Empfnger ein, High=aus)
; ------------------------------------------------------------------------------------------------------------
; Datadirection- und Starteinstellungen fr die Ports

.equ	k_StartUpDirB				= 0b00101111
.equ	k_StartUpOutB				= 0b00010111

.equ	k_StartUpDirC				= 0b00111111
.equ	k_StartUpOutC				= 0b00100101

.equ	k_StartUpDirD				= 0b11111110
.equ	k_StartUpOutD				= 0b11111110
; ------------------------------------------------------------------------------------------------------------
; Konstannten zur Displaysteuerung (Position der Zeichen/Ziffern bei Uhrzeit/Datumsanzeige)
; Position/Koordinatenpaar ist immer der linke obere Punkt des Zeichens
; siehe Bilddatei "Zeichensatz"
; High Byte = X, Low Byte = Y

.equ	Pos_Tag_10er				= 0x1672
.equ	Pos_Tag_1er					= 0x1F72
.equ	Pos_Punkt1					= 0x2872
.equ	Pos_Monat_10er				= 0x3172
.equ	Pos_Monat_1er				= 0x3A72
.equ	Pos_Punkt2					= 0x4372
.equ	Pos_Jahr_1000er				= 0x4C72
.equ	Pos_Jahr_100er				= 0x5572
.equ	Pos_Jahr_10er				= 0x5E72
.equ	Pos_Jahr_1er				= 0x6772
.equ	Pos_Stunde_10er				= 0x2D86
.equ	Pos_Stunde_1er				= 0x3686
.equ	Pos_Doppelpunkt				= 0x3F86
.equ	Pos_Minute_10er				= 0x4886
.equ	Pos_Minute_1er				= 0x5186
; ------------------------------------------------------------------------------------------------------------
; Farbkonstannten zur Displaysteuerung, 16Bit RGB 5-6-5

.equ	Frb_schwarz					= 0x0000
.equ	Frb_grau					= 0xC618
.equ	Frb_dunkelgrau				= 0x8410
.equ	Frb_weiss					= 0xFFFF
.equ	Frb_rot						= 0xF800
.equ	Frb_dunkelrot				= 0x8000
.equ	Frb_gruen					= 0x07E0
.equ	Frb_dunkelgruen				= 0x0400
.equ	Frb_blau					= 0x001F
.equ	Frb_dunkelblau				= 0x0010
.equ	Frb_cyan					= 0x07FF
.equ	Frb_gelb					= 0xFFE0
.equ	Frb_okker					= 0x8400
.equ	Frb_magenta					= 0xF81F
.equ	Frb_lila					= 0x8010

; Bentigt werden nur zwei Farben zur Anzeige von Zeit und Datum... einfach oben aus der Liste was auswhlen

.equ	k_Textfarbe_Zeit			= Frb_schwarz
.equ	k_Textfarbe_Datum			= Frb_rot
; ------------------------------------------------------------------------------------------------------------
; Registerdefinitionen
;									= r0			; wird fr LMP benutzt
.def	r_Offset0					= r1			; Offset fr Beginn
.def	r_Offset1					= r2			; Datenspeicherbereich MMC Karte
.def	r_Offset2					= r3			; Adr0 = Low Byte Adresse
.def	r_Offset3					= r4			; Adr3 = High Byte Adresse
.def	r_tmp						= r16			; Temporrregister
.def	r_Byte						= r17			; zum Byte hin und herschieben (Peripherie)
.def	r_Math0						= r18			; fr Mathematische Funktionen
.def	r_Math1						= r19			; fr Mathematische Funktionen
.def	r_Math2						= r20			; fr Mathematische Funktionen
.def	r_Math3						= r21			; fr Mathematische Funktionen
.def	r_Math4						= r22			; fr Mathematische Funktionen
.def	r_Math5						= r23			; fr Mathematische Funktionen
.def	r_Math6						= r24			; fr Mathematische Funktionen
.def	r_Math7						= r25			; fr Mathematische Funktionen
;		XL							= r26			; wird als Pointer genutzt
;		XH							= r27			; wird als Pointer genutzt
;		ZL							= r30			; wird als Pointer genutzt
;		ZH							= r31			; wird als Pointer genutzt
; ------------------------------------------------------------------------------------------------------------
; IRQ Tabelle
.org 0x00
	rjmp 	RESET 									; Reset Handler
	reti 											; IRQ0 Handler
	reti 											; IRQ1 Handler
	reti 											; Timer2 Compare Handler
	reti 											; Timer2 Overflow Handler
	reti 											; Timer1 Capture Handler
	reti 											; Timer1 CompareA Handler
	reti 											; Timer1 CompareB Handler
	rjmp	TIM1_OVF								; Timer1 Overflow Handler
	reti 											; Timer0 Overflow Handler
	reti 											; SPI Transfer Complete Handler
	reti 											; USART RX Complete Handler
	reti 											; UDR Empty Handler
	reti 											; USART TX Complete Handler
	reti 											; ADC Conversion Complete Handler
	reti 											; EEPROM Ready Handler
	reti 											; Analog Comparator Handler
	reti 											; Two-wire Serial Interface
	reti 											; Store Program Memory Ready
; ------------------------------------------------------------------------------------------------------------
; Timer1Overflow IRQ-Routine
; ein Timer1Overflow tritt bei der Einstellung jede Sekunde auf
;
; Nach einer Sekunde werden die Zeitrelevanten SRAM-Variablen aktualisiert.
; Jedoch nur fr Sekunde, Minute, Stunde
; Wochentag, Tag, Monat, Jahr werden NICHT aktualisiert
; Dies passiert im Hauptprogramm um 00:00 Uhr und nochmals um 02:00 und 03:00 Uhr 
; bei der Synchronisation mit dem DCF Signal

TIM1_OVF:
	push	r_tmp									; r_tmp sichern
	in		r_tmp,SREG								; Statusregister lesen
	push	r_tmp									; und sichern
	lds		r_tmp,Var_Sekunde						; Sekundenzhler aus SRAM laden
	inc		r_tmp									; +1
	sts		Var_Sekunde,r_tmp						; wieder speichern
	cpi		r_tmp,60								; Sekunde = 60 ?
	breq	TIM1_MinuteUm							; Ja, dann Sprung
	rjmp	TIM1_OVF_Ende							; Nein, dann IRQ verlassen
TIM1_MinuteUm:
	ldi		r_tmp,0									; r_tmp = 0
	sts		Var_Sekunde,r_tmp						; Sekundenspeicher auf 0 setzten
	lds		r_tmp,Var_Minute						; Minutenspeicher laden
	inc		r_tmp									; +1
	sts		Var_Minute,r_tmp						; speichern
	cpi		r_tmp,60								; Minute = 60 ?
	breq	TIM1_StundeUm							; Ja, dann Sprung
	rjmp	TIM1_OVF_Ende							; Nein, dann  IRQ Ende
TIM1_StundeUm:
	ldi		r_tmp,0									; r_tmp = 0
	sts		Var_Minute,r_tmp						; Minute = 0 speichern
	lds		r_tmp,Var_Stunde						; Stunde laden
	inc		r_tmp									; +1
	sts		Var_Stunde,r_tmp						; speichern
	cpi		r_tmp,24								; Stundenzhler = 24 ?
	breq	TIM1_TagUm								; Ja, dann Sprung
	rjmp	TIM1_OVF_Ende							; Nein, dann IRQ Ende
TIM1_TagUm:
	ldi		r_tmp,0									; r_tmp = 0
	sts		Var_Stunde,r_tmp						; Stunde auf 0 gesetzt speichern
TIM1_OVF_Ende:
	ldi		r_tmp,High(34286)						; Timer1 Startwert = 34286
	out		TCNT1H,r_tmp
	ldi		r_tmp,Low(34286)
	out		TCNT1L,r_tmp
	pop		r_tmp									; Statusregister vom Stapel
	out		SREG,r_tmp								; und wiederherstellen
	pop		r_tmp									; r_tmp wiederherstellen
	reti											; Ende IRQ Routine
; ------------------------------------------------------------------------------------------------------------
; Startinitialisierungen
Reset:
;	Stack
	ldi		r_tmp,High(RAMEND)						; Stackpointer ans
	out		SPH,r_tmp								; SRAM Ende setzen
	ldi		r_tmp,low(RAMEND)
	out		SPL,r_tmp
;	Ein- Ausgnge	
	ldi		r_tmp,k_StartupDirB						; DataDirection
	out		DDRB,r_tmp								; nach Konstante setzten
	ldi		r_tmp,k_StartUpOutB						; Ausgnge nach
	out		PORTB,r_tmp								; Konstante setzten
	ldi		r_tmp,k_StartupDirC						; DataDirection
	out		DDRC,r_tmp								; nach Konstante setzten
	ldi		r_tmp,k_StartUpOutC						; Ausgnge nach
	out		PORTC,r_tmp								; Konstante setzten
	ldi		r_tmp,k_StartupDirD						; DataDirection
	out		DDRD,r_tmp								; nach Konstante setzten
	ldi		r_tmp,k_StartUpOutD						; Ausgnge nach
	out		PORTD,r_tmp								; Konstante setzten
;	Displayinitialisierung
	rcall	Pause1ms								; 1ms warten (Displayreset)
	sbi		Display_Bus_Out,Display_Sig_Reset		; Resetleitung auf High (Normalfunktion)
	sbi		Display_Bus_Out,Display_Sig_RS			; RS = High = Befehl
	ldi		ZL,Low((2 * Init1))						; Z-Pointer
	ldi		ZH,High((2 * Init1))					; Auf Byte-Tabelle setzten
	ldi		r_tmp,4									; Schleifenzhler = 4
D_Init1:
	lpm		r_Byte,Z+								; Byte aus Tabelle laden
	rcall	ByteToDisplay							; Byte ans Display senden
	dec		r_tmp									; Schleifenzhler +1
	brne	D_Init1									; Noch nicht alle Bytes bertragen, also Rcksprung
	rcall	Pause50ms								; 50ms warten
	ldi		ZL,Low((2 * Init2))						; Z-Pointer
	ldi		ZH,High((2 * Init2))					; Auf Byte-Tabelle setzten
	ldi		r_tmp,20								; Schleifenzhler = 20
D_Init2:
	lpm		r_Byte,Z+								; Byte aus Tabelle laden
	rcall	ByteToDisplay							; Byte ans Display senden
	dec		r_tmp									; Schleifenzhler +1
	brne	D_Init2									; Noch nicht alle Bytes bertragen, also Rcksprung
	rcall	Pause7ms								; warten
	ldi		ZL,Low((2 * Init3))						; Z-Pointer
	ldi		ZH,High((2 * Init3))					; Auf Byte-Tabelle setzten
	ldi		r_tmp,40								; Schleifenzhler = 40
D_Init3:
	lpm		r_Byte,Z+								; Byte aus Tabelle laden
	rcall	ByteToDisplay							; Byte ans Display senden
	dec		r_tmp									; Schleifenzhler +1
	brne	D_Init3									; Noch nicht alle Bytes bertragen, also Rcksprung
	rcall	Pause50ms								; warten
	ldi		ZL,Low((2 * Init4))						; Z-Pointer
	ldi		ZH,High((2 * Init4))					; Auf Byte-Tabelle setzten
	ldi		r_tmp,6									; Schleifenzhler = 6
D_Init4:
	lpm		r_Byte,Z+								; Byte aus Tabelle laden
	rcall	ByteToDisplay							; Byte ans Display senden
	dec		r_tmp									; Schleifenzhler +1
	brne	D_Init4									; Noch nicht alle Bytes bertragen, also Rcksprung
; 	SPI Init fr Zugriff auf Speicherkarte mit geringer Frequenz (125 kHz)
	ldi		r_tmp,0b01010010						; SPI ein, SPI Mode 0,MSB first
	out		SPCR,r_tmp								; Master Mode, Clock/64
; 	SD Karte Init
	ldi		r_tmp,10								; r_tmp = Schleifenzhler = 10
	sbi		SD_Bus_Out,SD_Sig_CS					; Karte CS auf High
SD_Init1:
	ldi		r_Byte,0xFF								; Sendebyte = 0xFF
	rcall	SD_Send_Byte							; Byte senden
	dec		r_tmp									; Schleifenzhler -1
	brne	SD_Init1								; noch keine 10 Durchgnge, dann zurck
SD_Init2:
	ldi		r_Math4,0								; Befehl = CMD0
	ldi		r_Math3,0								; Parameter = 0
	ldi		r_Math2,0								; Parameter = 0
	ldi		r_Math1,0								; Parameter = 0
	ldi		r_Math0,0								; Parameter = 0
	rcall	SD_Send_Command							; Command senden
SD_Read_Response0:
	rcall	SD_Read_Byte							; Antwortbyte lesen
	cpi		r_Byte,0x01								; Antwort = 0x01 ?
	brne	SD_Read_Response0						; Nein, dann Rcksprung
SD_Init3:
	ldi		r_Math4,1								; Befehl = CMD1
	ldi		r_Math3,0								; Parameter = 0
	ldi		r_Math2,0								; Parameter = 0
	ldi		r_Math1,0								; Parameter = 0
	ldi		r_Math0,0								; Parameter = 0
	rcall	SD_Send_Command							; Command senden	
SD_Read_Response1:
	rcall	SD_Read_Byte							; Antwortbyte lesen
	cpi		r_Byte,0xFF								; Antwort = 0xFF ?
	breq	SD_Read_Response1						; Ja, dann Rcksprung
	cpi		r_Byte,0x00								; Antwort = 0 ?
	breq	SD_Init4								; Ja, dann runter zu Init4
	rjmp	SD_Init3								; Weder 0xFF noch 0x00, also Befehl nochmal senden
SD_Init4:
	ldi		r_Math4,16								; Befehl = CMD16
	ldi		r_Math3,0								; Parameter = 0 (Blockgre = 512 Bytes)
	ldi		r_Math2,0								; Parameter = 0 (Blockgre = 512 Bytes)
	ldi		r_Math1,2								; Parameter = 2 (Blockgre = 512 Bytes)
	ldi		r_Math0,0								; Parameter = 0 (Blockgre = 512 Bytes)
	rcall	SD_Send_Command							; Command senden	
SD_Read_Response16:
	rcall	SD_Read_Byte							; Antwortbyte lesen
	cpi		r_Byte,0x00								; Antwort = 0x00 ? 
	brne	SD_Read_Response16						; Nein, dann zurck
SD_Init_End:
	sbi		SD_Bus_Out,SD_Sig_CS					; Karte deselect
;	SPI wird jetzt fr den Kartenzugriff auf 4 MHz "raufgeschraubt"
	ldi		r_tmp,0b01010000						; SPI ein, SPI Mode 0,MSB first
	out		SPCR,r_tmp								; Master Mode, Clock/4
	ldi		r_tmp,0b00000001						; Double Speed Bit setzten
	out		SPSR,r_tmp								; = (8MHz Clock/4)*2 = 4 MHz SPI Frequenz 
;	Bilddatenanfang auf MMC finden (nach Bytefolge "KALENDER" suchen)
;	100 Blocks spter beginnen dann die Bilder
	ldi		r_tmp,0									; Register = 0
	sts		Var_SD_Adr0,r_tmp						; Adresse = 0 
	sts		Var_SD_Adr1,r_tmp						; Adresse = 0
	sts		Var_SD_Adr2,r_tmp						; Adresse = 0
	sts		Var_SD_Adr3,r_tmp						; Adresse = 0
Find_Start:
	rcall	SD_Read_Block							; Block lesen
	rcall	BlockInc								; Block +1
	lds		r_tmp,Var_Datenblock					; 1. Byte laden
	cpi		r_tmp,75								; Byte = K ?
	brne	Find_Start								; Nein, dann zurck
	lds		r_tmp,(Var_Datenblock + 1)				; 2. Byte laden
	cpi		r_tmp,65								; Byte = A ?
	brne	Find_Start								; Nein, dann zurck
	lds		r_tmp,(Var_Datenblock + 2)				; 3. Byte laden
	cpi		r_tmp,76								; Byte = L ?
	brne	Find_Start								; Nein, dann zurck
	lds		r_tmp,(Var_Datenblock + 3)				; 4. Byte laden
	cpi		r_tmp,69								; Byte = E ?
	brne	Find_Start								; Nein, dann zurck
	lds		r_tmp,(Var_Datenblock + 4)				; 5. Byte laden
	cpi		r_tmp,78								; Byte = N ?
	brne	Find_Start								; Nein, dann zurck
	lds		r_tmp,(Var_Datenblock + 5)				; 6. Byte laden
	cpi		r_tmp,68								; Byte = D ?
	brne	Find_Start								; Nein, dann zurck
	lds		r_tmp,(Var_Datenblock + 6)				; 7. Byte laden
	cpi		r_tmp,69								; Byte = E ?
	brne	Find_Start								; Nein, dann zurck
	lds		r_tmp,(Var_Datenblock + 7)				; 8. Byte laden
	cpi		r_tmp,82								; Byte = R ?
	brne	Find_Start								; Nein, dann zurck
	lds		r_Offset0,Var_SD_Adr0					; Offset
	lds		r_Offset1,Var_SD_Adr1					; in den 
	lds		r_Offset2,Var_SD_Adr2					; Registern
	lds		r_Offset3,Var_SD_Adr3					; speichern
	ldi		r_Math0,Byte1(50688)					; Register laden
	ldi		r_Math1,Byte2(50688)
	ldi		r_Math2,Byte3(50688)
	ldi		r_Math3,Byte4(50688)
	add		r_Offset0,r_Math0						; nochmal
	adc		r_Offset1,r_Math1						; 50688
	adc		r_Offset2,r_Math2						; hinzuaddieren	
	adc		r_Offset3,r_Math3						; und der Offset ist nun bekannt
;	DCF-Empfangsteil einschalten
	cbi		DCF_Bus_Out,DCF_Sig_PON					; DCF Empfnger ein
;	SRAM Zeitvariablen auf 0 setzten (Montag 01.01.2000 / 00:00:00 Uhr)
	ldi		r_tmp,0
	sts		Var_Sekunde,r_tmp
	sts		Var_Minute,r_tmp
	sts		Var_Stunde,r_tmp
	sts		Var_Jahr,r_tmp
	ldi		r_tmp,1
	sts		Var_Wochentag,r_tmp
	sts		Var_Tag,r_tmp
	sts		Var_Monat,r_tmp
;	Timer1
	ldi		r_tmp,High(34286)						; TimerStartwert = 34286
	out		TCNT1H,r_tmp							; setzten
	ldi		r_tmp,Low(34286)
	out		TCNT1L,r_tmp
	ldi		r_tmp,0b00000100						; Timer1 luft im Normalmode
	out		TCCR1B,r_tmp							; und Prescal = 256
	ldi		r_tmp,0b00000100						; Timer1OverflowIRQ erlauben
	out		TIMSK,r_tmp								; Overflow-IRQ wird nun jede Sekunde ausgelst
	sei												; Interrupt-Flag mu aber vorher noch gesetzt werden
; ------------------------------------------------------------------------------------------------------------
; Hier im Hauptprogramm wird eigentlich nur geprft wie spt es ist ....
; je nach Zeit wird das "Tagesbild" oder das "Wochentagsbild" mit Zeitanzeige auf dem Display ausgegeben
; oder die DCF Zeit wird empfangen und die Daten werden in die Zeitrelevanten SRAM Vars gesetzt
;
; folgende Dinge passieren zu folgenden Uhrzeiten:
;
; 00:00 Uhr		=	Synchronisation mit DCF Zeit
; 02:00 Uhr		=	Synchronisation mit DCF Zeit
; 03:00 Uhr 	=	Synchronisation mit DCF Zeit
;
; Sekunde  5	=	Anzeige von Wochentagsbild und Zeit
; Sekunde 25	=	Anzeige von Wochentagsbild und Zeit
; Sekunde 45	=	Anzeige von Wochentagsbild und Zeit
; Sekunde 15	=	Anzeige von Tagesbild
; Sekunde 35	=	Anzeige von Tagesbild
; Sekunde 55	=	Anzeige von Tagesbild

Main:
	lds		r_tmp,Var_Minute						; aktuelle Minute ins Register
	cpi		r_tmp,0									; Minute = 0 ?
	brne	Chk_Sekunde								; Nein, dann Sprung
	lds		r_tmp,Var_Stunde						; Minute = 0, also aktuelle Stunde ins Register laden
	cpi		r_tmp,0									; Zeit = 00:00 ? 
	breq	DCFSync									; Ja, dann Sprung
	cpi		r_tmp,2									; Zeit = 02:00 ?
	breq	DCFSync									; Ja, dann Sprung
	cpi		r_tmp,3									; Zeit = 03:00 ?
	breq	DCFSync									; Ja, dann Sprung
Chk_Sekunde:
	lds		r_tmp,Var_Sekunde						; aktuelle Sekunde ins Register
	cpi		r_tmp,5									; Sekunde = 5 ?
	breq	ZeitAnzeige								; Ja, dann Sprung
	cpi		r_tmp,25								; Sekunde = 25 ?
	breq	ZeitAnzeige								; Ja, dann Sprung
	cpi		r_tmp,45								; Sekunde = 45 
	breq	ZeitAnzeige								; Ja, dann Sprung
	cpi		r_tmp,15								; Sekunde = 15 ?
	breq	BildAnzeige								; Ja, dann Sprung
	cpi		r_tmp,35								; Sekunde = 35 ?
	breq	BildAnzeige								; Ja, dann Sprung
	cpi		r_tmp,55								; Sekunde = 55 ?
	breq	BildAnzeige								; Ja, dann Sprung
	rjmp	Main									; Zurck
BildAnzeige:	
	rcall	Calc_Pic_Adr							; Adresse fr Tagesbild berechnen
	rcall	Display_Show_Pic						; Bild anzeigen
	rcall	Pause500ms								; halbe Sekunde warten
	rjmp	Main									; Rcksprung
ZeitAnzeige:
	rcall	Calc_Back_Adr							; Adresse fr Hintergrundbild Zeitanzeige berechnen
	rcall	Display_Show_Pic						; Bild anzeigen
	rcall	DisplayPrintZeit						; Zeitausgabe
	rcall	pause500ms								; halbe Sekunde warten
	rjmp 	Main									; Rcksprung
DCFSync:
	cli												; IRQ sperren
	ldi		r_Math0,Byte1(19404800)					; Startadresse fr Bild "DCF-Sync" berechnen
	ldi		r_Math1,Byte2(19404800)
	ldi		r_Math2,Byte3(19404800)
	ldi		r_Math3,Byte4(19404800)
	add		r_Math0,r_Offset0
	adc		r_Math1,r_Offset1
	adc		r_Math2,r_Offset2
	adc		r_Math3,r_Offset3
	sts		Var_SD_Adr0,r_Math0
	sts		Var_SD_Adr1,r_Math1
	sts		Var_SD_Adr2,r_Math2
	sts		Var_SD_Adr3,r_Math3
	rcall	Display_Show_Pic						; Bild anzeigen
	rcall	DCF_Empfang								; DCF-Signal empfangen und Zeitregister synchronisieren
	ldi		r_tmp,High(34286)						; TimerStartwert = 34286
	out		TCNT1H,r_tmp							; setzten
	ldi		r_tmp,Low(34286)
	out		TCNT1L,r_tmp	
	sei												; IRQ wieder freigeben
	rjmp	Main									; Zurck zu Main
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; +++++++++++++++++++++++++++++++++++++++++++++ Unterprogramme +++++++++++++++++++++++++++++++++++++++++++++++
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; Diese Unterroutine erzeugt eine Verzgerung von 78.400.003 Maschinenzyklen
; Dies entspricht einer Zeit von 9,8 s

; bergaberegister									; Keine
; Rckgaberegister									; Keine
; Verwendete Register								; r_Math0 (wird wiederhergestellt)
													; r_Math1 (wird wiederhergestellt)
													; r_Math2 (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; keine

Pause9_8s:
	push	r_Math0
	push	r_Math1
	push	r_Math2
	ldi		r_Math0,16
	ldi		r_Math1,227
	ldi		r_Math2,239
Loop9_8s:
	nop
	nop
	dec		r_Math0
	brne	Loop9_8s
	dec		r_Math1
	brne	Loop9_8s
	dec		r_Math2
	brne	Loop9_8s
	pop		r_Math2
	pop		r_Math1
	pop		r_Math0
	ret
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine erzeugt eine Verzgerung von ca. 4.000.000 Maschinenzyklen
; Dies entspricht einer Zeit von ca. 500,0 ms
;
; bergaberegister									; Keine
; Rckgaberegister									; Keine
; Verwendete Register								; r_Math0 (wird wiederhergestellt)
													; r_Math1 (wird wiederhergestellt)
													; r_Math2 (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; keine

Pause500ms:
	push	r_Math0									; Sichern
	push	r_Math1									; Sichern
	push	r_Math2									; Sichern
	ldi		r_Math0,189
	ldi		r_Math1,75
	ldi		r_Math2,21
Loop500ms:
	dec		r_Math0
	brne	Loop500ms
	dec		r_Math1
	brne	Loop500ms
	dec		r_Math2
	brne	Loop500ms
	pop		r_Math2									; Wiederherstellen
	pop		r_Math1									; Wiederherstellen
	pop		r_Math0									; Wiederherstellen
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine erzeugt eine Verzgerung von ca. 800.000 Maschinenzyklen
; Dies entspricht einer Zeit von ca. 50,0 ms
;
; bergaberegister									; Keine
; Rckgaberegister									; Keine
; Verwendete Register								; r_Math0 (wird wiederhergestellt)
													; r_Math1 (wird wiederhergestellt)
													; r_Math2 (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; keine

Pause50ms:
	push	r_Math0									; Sichern
	push	r_Math1									; Sichern
	push	r_Math2									; Sichern
	ldi		r_Math0,239
	ldi		r_Math1,15
	ldi		r_Math2,5
Loop50ms:
	dec		r_Math0
	brne	Loop50ms
	dec		r_Math1
	brne	Loop50ms
	dec		r_Math2
	brne	Loop50ms
	pop		r_Math2									; Wiederherstellen
	pop		r_Math1									; Wiederherstellen
	pop		r_Math0									; Wiederherstellen
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine erzeugt eine Verzgerung von ca. 80.000 Maschinenzyklen
; Dies entspricht einer Zeit von ca. 10,0 ms

; bergaberegister									; Keine
; Rckgaberegister									; Keine
; Verwendete Register								; r_Math0 (wird wiederhergestellt)
													; r_Math1 (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; keine
Pause10ms:
	push	r_Math0
	push	r_Math1
	ldi		r_Math0,247
	ldi		r_Math1,78
Loop10ms:
	nop
	dec		r_Math0
	brne	Loop10ms
	dec		r_Math1
	brne	Loop10ms
	pop		r_Math1
	pop		r_Math0
	ret
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine erzeugt eine Verzgerung von ca. 112.000 Maschinenzyklen
; Dies entspricht einer Zeit von ca. 7,0 ms
;
; bergaberegister									; Keine
; Rckgaberegister									; Keine
; Verwendete Register								; r_Math0 (wird wiederhergestellt)
													; r_Math1 (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; keine

Pause7ms:
	push	r_Math0									; Sichern
	push	r_Math1									; Sichern
	ldi		r_Math0,39
	ldi		r_Math1,110
Loop7ms:
	nop
	dec		r_Math0
	brne	Loop7ms
	dec		r_Math1
	brne	Loop7ms
	pop		r_Math1									; Wiederherstellen
	pop		r_Math0									; Wiederherstellen
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine erzeugt eine Verzgerung von ca. 16.000 Maschinenzyklen
; Dies entspricht einer Zeit von ca. 1,0 ms
;
; bergaberegister									; Keine
; Rckgaberegister									; Keine
; Verwendete Register								; r_Math0 (wird wiederhergestellt)
													; r_Math1 (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; keine

Pause1ms:
	push	r_Math0									; Sichern
	push	r_Math1									; Sichern
	ldi		r_Math0,150
	ldi		r_Math1,16
Loop1ms:
	nop
	dec		r_Math0
	brne	Loop1ms
	dec		r_Math1
	brne	Loop1ms
	pop		r_Math1									; Wiederherstellen
	pop		r_Math0									; Wiederherstellen
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine wandelt ein Byte (0-99) in 2 Bytes (Einzelziffern) zur Displayanzeige um

; bergaberegister									; r_Math7 (Byte gefllt mit 0-99) (wird wiederhergestellt)
; Rckgaberegister									; r_Math6 (Zehnerstelle 0-9)
;													; r_Math5 (Einerstelle 0-9)
; verwendete Register								; keine
; von hier aus aufgerufene UPROS					; keine

ByteToZiffern:
	ldi		r_Math6,0								; Rckgaberegister leeren
	ldi		r_Math5,0								; Rckgaberegister leeren
BTZ_Vergleich:	
	cpi		r_Math7,10								; bergabe vergleich mit 10
	brsh	BTZ_Abziehen							; grer, gleich 10, also Sprung
	rjmp	BTZ_Ende								; Kleiner als 10, also Sprung
BTZ_Abziehen:
	subi	r_Math7,10								; 10 von bergabe abziehen
	inc		r_Math6									; Rckgabe 10er Stelle +1
	rjmp	BTZ_Vergleich							; und zurck
BTZ_Ende:
	mov		r_Math5,r_Math7							; Rest aus Berechnung = Einerstelle. ins Rckgaberegister
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine schreibt die Uhrzeit und das Datum ins Display

; bergaberegister									; SRAM-Variablen fr Zeit (Stunde, Minute, Tag, Jahr)
; Rckgaberegister									; keine
; verwendete Register								; r_Math7 (wird wiederhergestellt)
;													; r_Math6 (wird wiederhergestellt)
;													; r_Math5 (wird wiederhergestellt)
;													; r_Math3 (wird wiederhergestellt)
;													; r_Math2 (wird wiederhergestellt)
;													; r_Math1 (wird wiederhergestellt)
;													; r_Math0 (wird wiederhergestellt)
;													; r_Byte (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; DisplayPrintZeichen
;													; ByteToZiffern

DisplayPrintZeit:
	push	r_Math0									; sichern
	push	r_Math1
	push	r_Math2
	push	r_Math3
	push	r_Math5
	push	r_Math6
	push	r_Math7
	push	r_Byte
;	Textfarbe
	ldi		r_Math2,High(k_Textfarbe_Datum)			; Anzeigefarbe aus Konstannte
	ldi		r_Math3,Low(k_Textfarbe_Datum)			; splitten in Low & High
;	Tag 10er Stelle
	ldi		r_Math0,High(Pos_Tag_10er)				; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Tag_10er)				; Y-Koordinate aus Konstannte extrahieren
	lds		r_Math7,Var_Tag							; Tag in Register laden
	rcall	ByteToZiffern							; in 1er und 10er Stelle splitten
	mov		r_Byte,r_Math6							; 10er Stelle kopieren
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Tag 1er Stelle
	ldi		r_Math0,High(Pos_Tag_1er)				; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Tag_1er)				; Y-Koordinate aus Konstannte extrahieren
	lds		r_Math7,Var_Tag							; Tag in Register laden
	rcall	ByteToZiffern							; in 1er und 10er Stelle splitten
	mov		r_Byte,r_Math5							; 1er Stelle kopieren
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Punkt
	ldi		r_Math0,High(Pos_Punkt1)				; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Punkt1)					; Y-Koordinate aus Konstannte extrahieren
	ldi		r_Byte,11								; 11=Kennzeichen fr Punktausgabe
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Monat 10er Stelle
	ldi		r_Math0,High(Pos_Monat_10er)			; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Monat_10er)				; Y-Koordinate aus Konstannte extrahieren
	lds		r_Math7,Var_Monat						; Monat in Register laden
	rcall	ByteToZiffern							; in 1er und 10er Stelle splitten
	mov		r_Byte,r_Math6							; 10er Stelle kopieren
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Monat 1er Stelle
	ldi		r_Math0,High(Pos_Monat_1er)				; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Monat_1er)				; Y-Koordinate aus Konstannte extrahieren
	lds		r_Math7,Var_Monat						; Monat in Register laden
	rcall	ByteToZiffern							; in 1er und 10er Stelle splitten
	mov		r_Byte,r_Math5							; 1er Stelle kopieren
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Punkt
	ldi		r_Math0,High(Pos_Punkt2)				; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Punkt2)					; Y-Koordinate aus Konstannte extrahieren
	ldi		r_Byte,11								; 11=Kennzeichen fr Punktausgabe
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	"2" von Jahr
	ldi		r_Math0,High(Pos_Jahr_1000er)			; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Jahr_1000er)			; Y-Koordinate aus Konstannte extrahieren
	ldi		r_Byte,2								; 2 
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	"0" von Jahr
	ldi		r_Math0,High(Pos_Jahr_100er)			; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Jahr_100er)				; Y-Koordinate aus Konstannte extrahieren
	ldi		r_Byte,0								; 0
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Jahr 10er Stelle
	ldi		r_Math0,High(Pos_Jahr_10er)				; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Jahr_10er)				; Y-Koordinate aus Konstannte extrahieren
	lds		r_Math7,Var_Jahr						; Jahr in Register laden
	rcall	ByteToZiffern							; in 1er und 10er Stelle splitten
	mov		r_Byte,r_Math6							; 10er Stelle kopieren
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Jahr 1er Stelle
	ldi		r_Math0,High(Pos_Jahr_1er)				; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Jahr_1er)				; Y-Koordinate aus Konstannte extrahieren
	lds		r_Math7,Var_Jahr						; Jahr in Register laden
	rcall	ByteToZiffern							; in 1er und 10er Stelle splitten
	mov		r_Byte,r_Math5							; 1er Stelle kopieren
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Textfarbe
	ldi		r_Math2,High(k_Textfarbe_Zeit)			; Anzeigefarbe aus Konstannte
	ldi		r_Math3,Low(k_Textfarbe_Zeit)			; splitten in Low & High
;	Stunde 10er Stelle
	ldi		r_Math0,High(Pos_Stunde_10er)			; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Stunde_10er)			; Y-Koordinate aus Konstannte extrahieren
	lds		r_Math7,Var_Stunde						; Stunde in Register laden
	rcall	ByteToZiffern							; in 1er und 10er Stelle splitten
	mov		r_Byte,r_Math6							; 10er Stelle kopieren
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Stunde 1er Stelle
	ldi		r_Math0,High(Pos_Stunde_1er)			; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Stunde_1er)				; Y-Koordinate aus Konstannte extrahieren
	lds		r_Math7,Var_Stunde						; Stunde in Register laden
	rcall	ByteToZiffern							; in 1er und 10er Stelle splitten
	mov		r_Byte,r_Math5							; 1er Stelle kopieren
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Doppelpunkt
	ldi		r_Math0,High(Pos_Doppelpunkt)			; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Doppelpunkt)			; Y-Koordinate aus Konstannte extrahieren
	ldi		r_Byte,10								; 10=Kennzeichen fr Doppelpunktausgabe
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Minute 10er
	ldi		r_Math0,High(Pos_Minute_10er)			; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Minute_10er)			; Y-Koordinate aus Konstannte extrahieren
	lds		r_Math7,Var_Minute						; Minute in Register laden
	rcall	ByteToZiffern							; in 1er und 10er Stelle splitten
	mov		r_Byte,r_Math6							; 10er Stelle kopieren
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Minute 1er
	ldi		r_Math0,High(Pos_Minute_1er)			; X-Koordinate aus Konstannte extrahieren
	ldi		r_Math1,Low(Pos_Minute_1er)				; Y-Koordinate aus Konstannte extrahieren
	lds		r_Math7,Var_Minute						; Minute in Register laden
	rcall	ByteToZiffern							; in 1er und 10er Stelle splitten
	mov		r_Byte,r_Math5							; 1er Stelle kopieren
	rcall	DisplayPrintZeichen						; Zeichen ausgeben
;	Ende
	pop		r_Byte
	pop		r_Math7
	pop		r_Math6
	pop		r_Math5
	pop		r_Math3
	pop		r_Math2
	pop		r_Math1
	pop		r_Math0
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine setzt eine Ziffer (bestehend aus dem im Code definierten Char) in das Display
; Die Ausgabe erfolgt maskiert (der Hintergrund bleibt erhalten... nur die Pixel der Ziffer werden gesetzt)
; bergeben wird die Ziffer in r_Byte .... bei 0-9 wird die entsprechende Ziffer ausgegeben,
; bei 10 der Doppelpunkt, bei 11 der Punkt
; Weiterhin mu der oebere linke Startpunkt des Zeichens angegeben werden (X-Y-Koordinate)
; und die Farbe mit der gepixelt werden soll
;
; bergaberegister									; r_Math0 Start X-Koordinate (wird wiederhergestellt)
;													; r_Math1 Start-Y-Koordinate (wird wiederhergestellt)
;													; r_Math2 Zeichenfarbe HighByte (bleibt erhalten)
;													; r_Math3 Zeichenfarbe LowByte (bleibt erhalten)
;													; r_Byte (Zeichen/Ziffer0-11) (wird wiederhergestellt)
; Rckgaberegister									; keine
; verwendete Register								; Z-Pointer (wird wiederhergestellt)
													; r_tmp (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; DisplaySetPixel

DisplayPrintZeichen:
	push	ZL										; Sichern
	push	ZH
	push	r_tmp
	push	r_Byte
	push	r_Math0
	push	r_Math1
	ldi		ZL,Low((CharTab * 2))					; Z-Pointer auf Char-Tab
	ldi		ZH,High((CharTab * 2))					; setzten
BerZei:
	cpi		r_Byte,0								; Byte schon 0 ?
	breq	DPZ1									; Ja, dann Sprung
	adiw	ZH:ZL,12								; Nein, dann zum Z-Pointer 12 addieren
	dec		r_Byte									; Byte -1
	rjmp	BerZei									; und zurck
DPZ1:
	ldi		r_tmp,12								; Schleifenzhler = 12 (Bytes aus Flash laden)
DPZ2:
	lpm												; Zeichnsatzbyte aus Flash in r0 laden
DPB7:
	sbrs	r0,7									; Skip, wenn Bit7 gesetzt
	rjmp	DPB6									; Bit7 = 0, dann Sprung
	rcall	DisplaySetPixel							; Bit7 = 1 also ins Display setzten
DPB6:
	inc		r_Math0									; X-Koordinate +1
	sbrs	r0,6									; Skip, wenn Bit6 gesetzt
	rjmp	DPB5									; Bit6 = 0, dann Sprung
	rcall	DisplaySetPixel							; Bit6 = 1 also ins Display setzten
DPB5:
	inc		r_Math0									; X-Koordinate +1
	sbrs	r0,5									; Skip, wenn Bit5 gesetzt
	rjmp	DPB4									; Bit5 = 0, dann Sprung
	rcall	DisplaySetPixel							; Bit5 = 1 also ins Display setzten
DPB4:
	inc		r_Math0									; X-Koordinate +1
	sbrs	r0,4									; Skip, wenn Bit4 gesetzt
	rjmp	DPB3									; Bit4 = 0, dann Sprung
	rcall	DisplaySetPixel							; Bit4 = 1 also ins Display setzten
DPB3:
	inc		r_Math0									; X-Koordinate +1
	sbrs	r0,3									; Skip, wenn Bit3 gesetzt
	rjmp	DPB2									; Bit3 = 0, dann Sprung
	rcall	DisplaySetPixel							; Bit3 = 1 also ins Display setzten
DPB2:
	inc		r_Math0									; X-Koordinate +1
	sbrs	r0,2									; Skip, wenn Bit2 gesetzt
	rjmp	DPB1									; Bit2 = 0, dann Sprung
	rcall	DisplaySetPixel							; Bit2 = 1 also ins Display setzten
DPB1:
	inc		r_Math0									; X-Koordinate +1
	sbrs	r0,1									; Skip, wenn Bit1 gesetzt
	rjmp	DPB0									; Bit1 = 0, dann Sprung
	rcall	DisplaySetPixel							; Bit1 = 1 also ins Display setzten
DPB0:
	inc		r_Math0									; X-Koordinate +1
	sbrs	r0,0									; Skip, wenn Bit0 gesetzt
	rjmp	DPZ_Next_Byte							; Bit0 = 0, dann Sprung
	rcall	DisplaySetPixel							; Bit0 = 1 also ins Display setzten
DPZ_Next_Byte:
	dec		r_tmp									; Schleifenzhler -1
	breq	DPZ_Ende								; schon 12 Bytes durch, dann Sprung
	adiw	ZH:ZL,1									; Nein, dann Z-Pointer +1
	inc		r_Math1									; und Y-Koordinate +1
	subi	r_Math0,7								; und X-Koordinate -7
	rjmp	DPZ2									; und zurck
DPZ_Ende:
	pop		r_Math1									; wiederherstellen
	pop		r_Math0
	pop		r_Byte
	pop		r_tmp
	pop		ZH
	pop		ZL
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine setzt einen Pixel in das Display.
; X-Koordinate, Y-Koordinate, Farbe High-Byte und Farbe Low-Byte mssen bergeben werden

; bergaberegister									; r_Math0 (X-Koordinate) bleibt unverndert
;													; r_Math1 (Y-Koordinate) bleibt unverndert
;													; r_Math2 (Pixelfarbe High-Byte) bleibt unverndert
;													; r_Math3 (Pixelfarbe Low-Byte) bleibt unverndert
; Rckgaberegister									; keine
; verwendete Register								; r_Byte (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; ByteToDisplay

DisplaySetPixel:
	push	r_Byte									; sichern
	sbi		Display_Bus_Out,Display_Sig_RS			; RegisterSelect = High = Befehl
	ldi		r_Byte,0xEF								; Befehlssequenz fr Pixel setzten
	rcall	ByteToDisplay							; mit Koordinaten
	ldi		r_Byte,0x90
	rcall	ByteToDisplay
	ldi		r_Byte,0x05
	rcall	ByteToDisplay
	ldi		r_Byte,0x00
	rcall	ByteToDisplay
	ldi		r_Byte,0x08
	rcall	ByteToDisplay
	mov		r_Byte,r_Math0
	rcall	ByteToDisplay
	ldi		r_Byte,0x09
	rcall	ByteToDisplay
	mov		r_Byte,r_Math0
	rcall	ByteToDisplay
	ldi		r_Byte,0x0A
	rcall	ByteToDisplay
	mov		r_Byte,r_Math1
	rcall	ByteToDisplay
	ldi		r_Byte,0x0B
	rcall	ByteToDisplay
	mov		r_Byte,r_Math1
	rcall	ByteToDisplay
	cbi		Display_Bus_Out,Display_Sig_RS			; RegisterSelect = Low = Daten
	mov		r_Byte,r_Math2							; nur die beiden
	rcall	ByteToDisplay							; Farbbytes
	mov		r_Byte,r_Math3							; als Daten
	rcall	ByteToDisplay							; senden
	pop		r_Byte									; wiederherstellen
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine liest ein komplettes Bild aus der SD Karte und sendet dieses ans Display
; Die Startadresse des Bildes (auf der SD Karte)
; mu voher in den SRAM VARS Var_SD_Adr3...0 gespeichert sein
;
; bergaberegister 									; SRAM VARS Var_SD_Adr3...0 (bleiben NICHT erhalten)
; Rckgaberegister									; keine
; verwendete Register								; r_Byte (wird wiederhergestellt)
;													; r_tmp (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; ByteToDisplay
;													; SD_Read_Block
;													; SD_Read_LastBlock
;													; BlockInc

Display_Show_Pic:
	push	r_tmp									; sichern
	push	r_Byte
	sbi		Display_Bus_Out,Display_Sig_RS			; Display Befehlsmodus
	ldi		r_Byte,0xEF								; Sequenz fr ganzes Display beschreiben senden
	rcall	ByteToDisplay
	ldi		r_Byte,0x90
	rcall	ByteToDisplay
	ldi		r_Byte,0x05
	rcall	ByteToDisplay
	ldi		r_Byte,0x00
	rcall	ByteToDisplay
	ldi		r_Byte,0x08
	rcall	ByteToDisplay
	ldi		r_Byte,0x00
	rcall	ByteToDisplay
	ldi		r_Byte,0x09
	rcall	ByteToDisplay
	ldi		r_Byte,0x83
	rcall	ByteToDisplay
	ldi		r_Byte,0x0A
	rcall	ByteToDisplay
	ldi		r_Byte,0x00
	rcall	ByteToDisplay
	ldi		r_Byte,0x0B
	rcall	ByteToDisplay
	ldi		r_Byte,0xAF
	rcall	ByteToDisplay
	cbi		Display_Bus_Out,Display_Sig_RS			; Display Datenmodus
	ldi		r_tmp,90								; Schleifenzhler = 90
DSP_Loop1:
	rcall	SD_Read_Block							; Datenblock aus SD-Karte lesen und im SRAM speichern
	rcall	Display_Send_Block						; Datenblock an Display senden
	dec		r_tmp									; Schleifenzhler -1
	breq	DSP_Loop2								; schon 90 Blocks ?, dann Sprung
	rcall	BlockInc								; nein, dann Adresse +512
	rjmp	DSP_Loop1								; und zurck
; vom letzten (91.) Datenblock werden nur die ersten 384 Bytes bentigt
DSP_Loop2:
	rcall	BlockInc								; Adresse+512
	rcall	SD_Read_Block							; Block lesen
	rcall	Display_Send_LastBlock					; Block senden
	pop		r_Byte									; wiederherstellen
	pop		r_tmp
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Routine liest den Datenblock aus dem SRAM (Var_Datenblock)
; und sendet ihn ans Display (512 Bytes)
;
; bergaberegister									; Datenblock SRAM (Var_Datenblock)
; Rckgaberegister									; keine
; verwendete Register								; X-Pointer (wird wiederhergestellt)
;													; Z-Pointer (wird wiederhergestellt)
;													; r_Byte (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; ByteToDisplay

Display_Send_Block:
	push	XH										; sichern
	push	XL
	push	ZH
	push	ZL
	push	r_Byte
	ldi		ZL,Low(512)								; Schleifenzhler
	ldi		ZH,High(512)							; = 512
	ldi		XL,Low(Var_Datenblock)					; X-Pointer auf beginn
	ldi		XH,High(Var_Datenblock)					; des Datenblocks
SDBLoop:
	ld		r_Byte,X+								; Byte aus SRAM laden (post increment)
	rcall	ByteToDisplay							; ans Display senden
	sbiw  	ZH:ZL,1									; Schleifenzhler -1
	brne	SDBLoop									; noch keine 512 Bytes, dann zurck
	pop		r_Byte									; wiederherstellen
	pop		ZL
	pop		ZH
	pop		XL
	pop		XH
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Routine liest den Datenblock aus dem SRAM (Var_Datenblock)
; und sendet ihn ans Display (384 Bytes)
;
; bergaberegister									; Datenblock SRAM (Var_Datenblock)
; Rckgaberegister									; keine
; verwendete Register								; X-Pointer (wird wiederhergestellt)
;													; Z-Pointer (wird wiederhergestellt)
;													; r_Byte (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; ByteToDisplay

Display_Send_LastBlock:
	push	XH										; sichern
	push	XL
	push	ZH
	push	ZL
	push	r_Byte
	ldi		ZL,Low(384)								; Schleifenzhler
	ldi		ZH,High(384)							; = 384
	ldi		XL,Low(Var_Datenblock)					; X-Pointer auf beginn
	ldi		XH,High(Var_Datenblock)					; des Datenblocks
SDBLLoop:
	ld		r_Byte,X+								; Byte aus SRAM laden (post increment)
	rcall	ByteToDisplay							; ans Display senden
	sbiw  	ZH:ZL,1									; Schleifenzhler -1
	brne	SDBLLoop								; noch keine 384 Bytes, dann zurck
	pop		r_Byte									; wiederherstellen
	pop		ZL
	pop		ZH
	pop		XL
	pop		XH
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Sendet ein Byte ans Display
; Das Byte mu bergeben werden im Register r_Byte
;
; Routine ist Geschwindigkeitsoptimiert, daher keine Schleife, sondern jedes Bit "zu Fu" bertragen
; ergibt 99 bis 107 Zyklen zur bertragung
;
; bergaberegister									; r_Byte (bleibt unverndert)
; Rckgaberegister									; keine
; genutzte Register									; T-Flag (wird NICHT wiederhergestellt)
; von hier aus aufgerufene UPROS					; UUPRO
ByteToDisplay:
	cbi		Display_Bus_Out,Display_Sig_CS			; Display select
	sbrc	r_Byte,7								
	rjmp	B7_1
B7_0:
	cbi		Display_Bus_Out,Display_Sig_Data		
	rjmp	S7
B7_1:
	sbi		Display_Bus_Out,Display_Sig_Data
S7:
	sbi		Display_Bus_Out,Display_Sig_Clock		; Clock auf High
	nop												; warten
	cbi		Display_Bus_Out,Display_Sig_Clock		; Clock wieder auf Low	
	sbrc	r_Byte,6								
	rjmp	B6_1
B6_0:
	cbi		Display_Bus_Out,Display_Sig_Data		
	rjmp	S6
B6_1:
	sbi		Display_Bus_Out,Display_Sig_Data
S6:
	sbi		Display_Bus_Out,Display_Sig_Clock		; Clock auf High
	nop												; warten
	cbi		Display_Bus_Out,Display_Sig_Clock		; Clock wieder auf Low	
	sbrc	r_Byte,5								
	rjmp	B5_1
B5_0:
	cbi		Display_Bus_Out,Display_Sig_Data		
	rjmp	S5
B5_1:
	sbi		Display_Bus_Out,Display_Sig_Data
S5:
	sbi		Display_Bus_Out,Display_Sig_Clock		; Clock auf High
	nop												; warten
	cbi		Display_Bus_Out,Display_Sig_Clock		; Clock wieder auf Low	
	sbrc	r_Byte,4								
	rjmp	B4_1
B4_0:
	cbi		Display_Bus_Out,Display_Sig_Data		
	rjmp	S4
B4_1:
	sbi		Display_Bus_Out,Display_Sig_Data
S4:
	sbi		Display_Bus_Out,Display_Sig_Clock		; Clock auf High
	nop												; warten
	cbi		Display_Bus_Out,Display_Sig_Clock		; Clock wieder auf Low	
	sbrc	r_Byte,3								
	rjmp	B3_1
B3_0:
	cbi		Display_Bus_Out,Display_Sig_Data		
	rjmp	S3
B3_1:
	sbi		Display_Bus_Out,Display_Sig_Data
S3:
	sbi		Display_Bus_Out,Display_Sig_Clock		; Clock auf High
	nop												; warten
	cbi		Display_Bus_Out,Display_Sig_Clock		; Clock wieder auf Low	
	sbrc	r_Byte,2								
	rjmp	B2_1
B2_0:
	cbi		Display_Bus_Out,Display_Sig_Data		
	rjmp	S2
B2_1:
	sbi		Display_Bus_Out,Display_Sig_Data
S2:
	sbi		Display_Bus_Out,Display_Sig_Clock		; Clock auf High
	nop												; warten
	cbi		Display_Bus_Out,Display_Sig_Clock		; Clock wieder auf Low	
	sbrc	r_Byte,1								
	rjmp	B1_1
B1_0:
	cbi		Display_Bus_Out,Display_Sig_Data		
	rjmp	S1
B1_1:
	sbi		Display_Bus_Out,Display_Sig_Data
S1:
	sbi		Display_Bus_Out,Display_Sig_Clock		; Clock auf High
	nop												; warten
	cbi		Display_Bus_Out,Display_Sig_Clock		; Clock wieder auf Low	
	sbrc	r_Byte,0								
	rjmp	B0_1
B0_0:
	cbi		Display_Bus_Out,Display_Sig_Data		
	rjmp	S0
B0_1:
	sbi		Display_Bus_Out,Display_Sig_Data
S0:
	sbi		Display_Bus_Out,Display_Sig_Clock		; Clock auf High
	nop												; warten
	cbi		Display_Bus_Out,Display_Sig_Clock		; Clock wieder auf Low	
	sbi		Display_Bus_Out,Display_Sig_CS			; Display deselect
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Routine errechnet aus dem aktuellen Tag/Monat das Bild welches fr diesen Tag angezeigt werden soll
; Dazu werden die SRAM VARs Var_Tag und Var_Monat gelesen
; Die Startadresse auf der MMC Karte wird dann in den VARs Var_MMC_Adr0 - Var_MMC_Adr4 gespeichert
;
; Auf der MMC Karte sind die Startadressen fr die Monate immer gleich weit voneinander entfernt
; Ebenfalls die Tage.... somit lt sich die Adresse fr das Tagesbild leicht berechnen
; Die Aufteilung der Bilddaten auf der MMC Karte und die Adressen sind im 
; Textdokument "MMC Speicheraufteilung" zu finden
;
;
; bergaberegister									; SRAM VARs fr Tag und Monat
; Rckgaberegister									; SRAM VARs fr MMC-Adresse
; Genutzte Register									; r_Math0 (wird wiederhergestellt)
;													; r_Math1 (wird wiederhergestellt)
;													; r_Math2 (wird wiederhergestellt)
;													; r_Math3 (wird wiederhergestellt)
;													; r_Math4 (wird wiederhergestellt)
;													; r_Math5 (wird wiederhergestellt)
;													; r_Math6 (wird wiederhergestellt)
;													; r_Math7 (wird wiederhergestellt)
;													; r_tmp (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; keine

Calc_Pic_Adr:
	push	r_Math0									; Alle verwendeten Register sichern
	push	r_Math1
	push	r_Math2
	push	r_Math3
	push	r_Math4
	push	r_Math5
	push	r_Math6
	push	r_Math7
	push	r_tmp
	lds		r_tmp,Var_Monat							; Monat in Register laden
	ldi		r_Math4,Byte1(1587200)					; Register laden
	ldi		r_Math5,Byte2(1587200)					; mit Offset fr einen Monat
	ldi		r_Math6,Byte3(1587200)
	ldi		r_Math7,Byte4(1587200)
	mov		r_Math0,r_Offset0						; Rckgabe / Ergebnisregister
	mov		r_Math1,r_Offset1						; mit Kartenoffset Bilderspeicherbeginn laden
	mov		r_Math2,r_Offset2						
	mov		r_Math3,r_Offset3	
Loop_Monat:
	dec		r_tmp									; r_tmp -1
	cpi		r_tmp,0									; r_tmp = 0 ? 
	breq	Monat_Ende								; Ja, dann Sprung
	add		r_Math0,r_Math4							; Low Bytes addieren
	adc		r_Math1,r_Math5							; Mid Byte addieren
	adc		r_Math2,r_Math6							; Mid-Byte addieren
	adc		r_Math3,r_Math7							; High Bytes addieren
	rjmp	Loop_Monat								; Rcksprung
Monat_Ende:
	lds		r_tmp,Var_Tag							; Tag in Register laden
	ldi		r_Math4,Byte1(51200)					; Register laden
	ldi		r_Math5,Byte2(51200)					; mit Offset fr einen Tag
	ldi		r_Math6,Byte3(51200)
	ldi		r_Math7,Byte4(51200)
Loop_Tag:
	dec		r_tmp									; r_tmp -1
	cpi		r_tmp,0									; r_tmp = 0 ? 
	breq	Tag_Ende								; Ja, dann Sprung
	add		r_Math0,r_Math4							; Low Bytes addieren
	adc		r_Math1,r_Math5							; Mid Byte addieren
	adc		r_Math2,r_Math6							; Mid-Byte addieren
	adc		r_Math3,r_Math7							; High Bytes addieren
	rjmp	Loop_Tag								; Rcksprung
Tag_Ende:
	sts		Var_SD_Adr0,r_Math0						; Adresse in SRAM
	sts		Var_SD_Adr1,r_Math1						; VARs speichern
	sts		Var_SD_Adr2,r_Math2
	sts		Var_SD_Adr3,r_Math3
	pop		r_tmp									; gesicherte Register
	pop		r_Math7									; wiederherstellen
	pop		r_Math6
	pop		r_Math5
	pop		r_Math4
	pop		r_Math3
	pop		r_Math2
	pop		r_Math1
	pop		r_Math0
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Routine errechnet aus dem aktuellen Wochentag das Bild fr den Hintergrund der Zeitanzeige
; Dazu wird die SRAM VAR Var_Wochentag gelesen
; Die Startadresse auf der MMC Karte wird dann in den VARs Var_MMC_Adr0 - Var_MMC_Adr4 gespeichert
;
; Die Aufteilung der Bilddaten auf der MMC Karte und die Adressen sind im 
; Textdokument "MMC Speicheraufteilung" zu finden
;
;
; bergaberegister									; SRAM VAR fr Wochentag
; Rckgaberegister									; SRAM VARs fr MMC-Adresse
; Genutzte Register									; r_Math0 (wird wiederhergestellt)
;													; r_Math1 (wird wiederhergestellt)
;													; r_Math2 (wird wiederhergestellt)
;													; r_Math3 (wird wiederhergestellt)
;													; r_Math4 (wird wiederhergestellt)
;													; r_Math5 (wird wiederhergestellt)
;													; r_Math6 (wird wiederhergestellt)
;													; r_Math7 (wird wiederhergestellt)
;													; r_tmp (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; keine

Calc_Back_Adr:
	push	r_Math0									; Alle verwendeten Register sichern
	push	r_Math1
	push	r_Math2
	push	r_Math3
	push	r_Math4
	push	r_Math5
	push	r_Math6
	push	r_Math7
	push	r_tmp
	lds		r_tmp,Var_Wochentag						; Wochentag in Register laden
	ldi		r_Math4,Byte1(51200)					; Register laden
	ldi		r_Math5,Byte2(51200)					; mit Offset fr ein Bild
	ldi		r_Math6,Byte3(51200)
	ldi		r_Math7,Byte4(51200)
	ldi		r_Math0,Byte1(19046400)					; Rckgabe / Ergebnisregister laden mit
	ldi		r_Math1,Byte2(19046400)					; Speicherplatzbeginn fr 1. Wochentagbild
	ldi		r_Math2,Byte3(19046400)
	ldi		r_Math3,Byte4(19046400)
	add		r_Math0,r_Offset0						; Offset 
	adc		r_Math1,r_Offset1						; Beginn Bildspeicher
	adc		r_Math2,r_Offset2						; dazurechnen
	adc		r_Math3,r_Offset3
Loop_WT:
	dec		r_tmp									; r_tmp -1
	cpi		r_tmp,0									; r_tmp = 0 ? 
	breq	WT_Ende									; Ja, dann Sprung
	add		r_Math0,r_Math4							; Low Bytes addieren
	adc		r_Math1,r_Math5							; Mid Byte addieren
	adc		r_Math2,r_Math6							; Mid-Byte addieren
	adc		r_Math3,r_Math7							; High Bytes addieren
	rjmp	Loop_WT									; Rcksprung
WT_Ende:
	sts		Var_SD_Adr0,r_Math0						; Adresse in SRAM
	sts		Var_SD_Adr1,r_Math1						; VARs speichern
	sts		Var_SD_Adr2,r_Math2
	sts		Var_SD_Adr3,r_Math3
	pop		r_tmp									; gesicherte Register
	pop		r_Math7									; wiederherstellen
	pop		r_Math6
	pop		r_Math5
	pop		r_Math4
	pop		r_Math3
	pop		r_Math2
	pop		r_Math1
	pop		r_Math0
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; In dieser Unterroutine wird zu den Speicheradressen Var_SD-Adr3....0)
; immer 512 dazuaddiert (also ein Block)
; Die 4 Adressbytes werden gelesen, 512 dazuaddiert und zurck ins SRAM geschrieben

; bergaberegister									; SRAM VARs fr SD-Adresse
; Rckgaberegister									; SRAM VARs fr SD-Adresse
; Genutzte Register									; r_Math0 (wird wiederhergestellt)
;													; r_Math1 (wird wiederhergestellt)
;													; r_Math2 (wird wiederhergestellt)
;													; r_Math3 (wird wiederhergestellt)
;													; r_Math4 (wird wiederhergestellt)
;													; r_Math5 (wird wiederhergestellt)
;													; r_Math6 (wird wiederhergestellt)
;													; r_Math7 (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; keine

BlockInc:
	push	r_Math0									; Alle verwendeten Register sichern
	push	r_Math1
	push	r_Math2
	push	r_Math3
	push	r_Math4
	push	r_Math5
	push	r_Math6
	push	r_Math7
	lds		r_Math0,Var_SD_Adr0						; Adresse in Register laden
	lds		r_Math1,Var_SD_Adr1						; Adresse in Register laden
	lds		r_Math2,Var_SD_Adr2						; Adresse in Register laden
	lds		r_Math3,Var_SD_Adr3						; Adresse in Register laden
	ldi		r_Math4,Byte1(512)						; dazuzuaddierenden
	ldi		r_Math5,Byte2(512)						; Wert von
	ldi		r_Math6,Byte3(512)						; 512 in die 
	ldi		r_Math7,Byte4(512)						; Register setzten
	add		r_Math0,r_Math4							; Low Bytes addieren
	adc		r_Math1,r_Math5							; Mid Byte addieren
	adc		r_Math2,r_Math6							; Mid-Byte addieren
	adc		r_Math3,r_Math7							; High Bytes addieren
	sts		Var_SD_Adr0,r_Math0						; neue Adresse in SRAM
	sts		Var_SD_Adr1,r_Math1						; VARs speichern
	sts		Var_SD_Adr2,r_Math2
	sts		Var_SD_Adr3,r_Math3
	pop		r_Math7									; gesicherte Register
	pop		r_Math6									; wiederherstellen	
	pop		r_Math5
	pop		r_Math4
	pop		r_Math3
	pop		r_Math2
	pop		r_Math1
	pop		r_Math0
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine sendet ein Byte via SPI an die SD-Karte und liefert das von dieser gelesene Byte zurck
; Das zu sendende Byte mu in r_Byte bergeben werden. Nach Abschlu der Routine steht in diesem Register
; dann das empfangene Byte

; bergaberegister									; r_Byte (gefllt mit bergebebyte)
; Rckgaberegister									; r_Byte (gefllt mit Rckgabebyte)
; verwendete Register								; keine
; von hier aus aufgerufene UPROS					; keine

SD_Send_Byte:
	out		SPDR,r_Byte								; Byte ins Senderegister schieben 
Wait_Transmit:
	sbis 	SPSR,SPIF								; Warten bis bertragung
	rjmp 	Wait_Transmit							; abgeschlossen
	in		r_Byte,SPDR								; empfangenes Byte in Register lesen
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine sendet ein Command (bestehend aus 5 Bytes) an die SD Karte
; gesendet wird erst das Paramertbyte fr den Befehl (r_Math4)
; dann die Bytes r_Math3...0
; dann eine Checksumme von 0x95 
; (diese wird eigentlich nur fr die Init gebraucht und wird dann von der Karte ignoriert)
;
; bergaberegister:									; r_Math4 (Parameterbyte..1 fr CMD1, 2 fr CMD2...) [bl erhalten]
; 													; r_Math3 (Parameterbyte3) [bleibt erhalten]
;													; r_Math2 (Parameterbyte2) [bleibt erhalten]
;													; r_Math1 (Parameterbyte1) [bleibt erhalten]
;													; r_Math0 (Parameterbyte0) [bleibt erhalten]
; Rckgaberegister									; keine
; verwendete Register								; r_Byte (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; SD_Send_Byte

SD_Send_Command:
	push	r_Byte									; sichern
	sbi		SD_Bus_Out,SD_Sig_CS					; CS auf High = Karte deselect
	ldi		r_Byte,0xFF								; Register laden
	rcall	SD_Send_Byte							; Dummybyte senden
	cbi		SD_Bus_Out,SD_Sig_CS					; CS auf High = Karte select
	mov		r_Byte,r_Math4							; Befehlsbyte (CMD-Befehl) kopieren
	ori		r_Byte,0x40								; aus Byte ein CMD-Byte berechnen
	rcall	SD_Send_BYte							; Byte senden
	mov		r_Byte,r_Math3							; Byte kopieren
	rcall	SD_Send_Byte							; und senden
	mov		r_Byte,r_Math2							; Byte kopieren
	rcall	SD_Send_Byte							; und senden
	mov		r_Byte,r_Math1							; Byte kopieren
	rcall	SD_Send_BYte							; und senden
	mov		r_Byte,r_Math0							; Byte kopieren
	rcall	SD_Send_BYte							; und senden
	ldi		r_Byte,0x95								; CRC Byte ins Register
	rcall	SD_Send_BYte							; und senden
	pop		r_Byte									; Byte wiederherstellen
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine liest ein Byte von der SD-Karte und gibt dieses in r_Byte zurck
;
; bergaberegister									; keine
; Rckgaberegister									; r_Byte (enthlt empfangenes Byte)
; Verwendete Register								; Keine
; von hier aus aufgerufene UPROS					; SD_Send_Byte

SD_Read_Byte:
	ldi		r_Byte,0xFF								; Sendebyte = 0xFF
	rcall	SD_Send_Byte							; Byte senden (und Antwort empfangen)
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Diese Unterroutine liest einen ganzen Datenblock von 512 Bytes (plus 2 CRC Bytes) aus der SD Karte
; Die Adresse mu in den SRAM Vars Var_SD_Adr3...0 stehen
; Die gelesenen Daten stehen nach Aufruf der Unterroutine im SRAM an Position Var_Datenblock
;
; bergaberegister									; SRAM Vars fr SD-Adresse (Var_SD-Adr3..0)
; Rckgaberegister									; Datenblock (514 Bytes) ab (Var_Datenblock)
; Verwendete Register								; r_Math4 (wird wiederhergestellt)
;													; r_Math3 (wird wiederhergestellt)
;													; r_Math2 (wird wiederhergestellt)
;													; r_Math1 (wird wiederhergestellt)
;													; r_Math0 (wird wiederhergestellt)
;													; ZH (wird wiederhergestellt)
;													; ZL (wird wiederhergestellt)
;													; XH (wird wiederhergestellt)
;													; XL (wird wiederhergestellt)
;													; r_Byte (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; SD_Send_Command
;													; SD_Read_Byte


SD_Read_Block:
	push	r_Math4									; Register sichern
	push	r_Math3
	push	r_Math2
	push	r_Math1
	push	r_Math0
	push	ZH
	push	ZL
	push	XH
	push	XL
	push	r_Byte
	ldi		r_Math4,17								; Befehl = CMD17
	lds		r_Math3,Var_SD_Adr3						; Adresse = SRAM Var
	lds		r_Math2,Var_SD_Adr2						; Adresse = SRAM Var
	lds		r_Math1,Var_SD_Adr1						; Adresse = SRAM Var
	lds		r_Math0,Var_SD_Adr0						; Adresse = SRAM Var
	rcall	SD_Send_Command							; Command senden
RB_Resp1:
	rcall	SD_Read_Byte							; Antwortbyte lesen
	cpi		r_Byte,0x00								; Antwort = 0x00 ?
	brne	RB_Resp1								; Nein, dann Rcksprung
RB_Resp2:
	rcall	SD_Read_Byte							; Antwortbyte lesen
	cpi		r_Byte,0xFE								; Antwort = 0xFE (Datatoken)
	brne	RB_Resp2								; Nein, dann Rcksprung
	ldi		ZL,Low(514)								; Bytezhler = 514
	ldi		ZH,High(514)							; Bytezhler = 514
	ldi		XL,Low(Var_Datenblock)					; X-Pointer auf Start des 
	ldi		XH,High(Var_Datenblock)					; Speicherbereiches im SRAM setzten
RB_Read:
	rcall	SD_Read_Byte							; Datenbyte von Karte empfangen
	st		X+,r_Byte								; Byte im SRAM speichern (post increment)
	sbiw	ZH:ZL,1									; Bytezhler -1          
	brne	RB_Read									; noch keine 514 Bytes, also zurck
	sbi		SD_Bus_Out,SD_Sig_CS					; SD Karte deselect
	pop		r_Byte									; Register wiederherstellen
	pop		XL
	pop		XH
	pop		ZL
	pop		ZH
	pop		r_Math0
	pop		r_Math1
	pop		r_Math2
	pop		r_Math3
	pop		r_Math4
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Hier wird das DCF Signal empfangen und die Zeitrelevanten SRAM Variablen werden gesetzt

; bergaberegister									; keine
; Rckgaberegister									; Alle Zeitrelevanten SRAM Vars
; verwendete Register								; r_Byte (wird wiederhergestellt)
;													; r_tmp (wird wiederhergestellt)
;													; r_Math0...5 (werden wiederhergestellt)
; von hier aus aufgerufene UPROS					; DCF_Read_Bit
;													; DCF_Read_NoBit
;													; Pause9_8sec

DCF_Empfang:
	push	r_tmp									; Sichern
	push	r_Byte
	push	r_Math0
	push	r_Math1
	push	r_Math2
	push	r_Math3
	push	r_Math4
	push	r_Math5
DCF_Empfang_Wait:
	ldi		r_Math0,0								; Zwischenspeicher fr Minute = 0
	ldi		r_Math1,0								; Zwischenspeicher fr Stunde = 0
	ldi		r_Math2,0								; Zwischenspeicher fr Tag = 0
	ldi		r_Math3,0								; Zwischenspeicher fr Wochentag = 0
	ldi		r_Math4,0								; Zwischenspeicher fr Monat = 0
	ldi		r_Math5,0								; Zwischenspeicher fr Jahr = 0
	rcall	DCF_Read_NoBit							; Warten auf Low Signalpegel
	cpi		r_Byte,170								; War der Pegel lnger als 1700ms Low ?
	brsh	Time_Telegramm_Start					; Ja, dann Beginn neue Minute... Sprung
	rjmp	DCF_Empfang_Wait						; Nein, dann zurck
Time_Telegramm_Start:
	rcall	pause9_8s								; Start einer neuen Minute
	rcall	pause9_8s								; knappe 20 Sekunden warten (dann kommen die Zeitbits)
	ldi		r_tmp,38								; 38 Bits werden empfangen
Sek20:
	rcall	DCF_Read_Bit							; Bit einlesen
	rcall	Pause500ms								; halbe Sekunde warten
	cpi		r_byte,1								; War Bit = 1 ? (Startbit fr Zeittelegramm MUSS 1 sein)
	breq	LiesSeks								; Ja, dann weiter zum Empfang der restlichen Bits
	rjmp	DCF_Empfang_Wait						; Nein, dann Fehler und von vorn
LiesSeks:
	rcall	DCF_Read_Bit							; Bit lesen
	rcall	Pause500ms								; halbe Sekunde warten
	cpi		r_Byte,255								; Empfangsregister = Fehlercode (255) ?
	breq	DCF_Empfang_Wait						; Ja, dann Fehler und von vorn
	cpi		r_Byte,1								; War Byte = 1 ?
	breq	BerZeit									; Dann Sprung
	rjmp	NextSek									; Byte war 0, dann Sprung hinab
BerZeit:
	cpi		r_tmp,38								; Minutenbit1 ?
	brne	Ber37									; Nein, dann Sprung
	subi	r_Math0,-1								; Ja, dann Minuten +1
	rjmp	NextSek									; und Sprung
Ber37:
	cpi		r_tmp,37								; Minutenbit2 ?
	brne	Ber36									; Nein, dann Sprung
	subi	r_Math0,-2								; Ja, dann Minuten +2
	rjmp	NextSek									; und Sprung
Ber36:
	cpi		r_tmp,36								; Minutenbit3 ?
	brne	Ber35									; Nein, dann Sprung
	subi	r_Math0,-4								; Ja, dann Minuten +4
	rjmp	NextSek									; und Sprung
Ber35:
	cpi		r_tmp,35								; Minutenbit4 ?
	brne	Ber34									; Nein, dann Sprung
	subi	r_Math0,-8								; Ja, dann Minuten +8
	rjmp	NextSek									; und Sprung
Ber34:
	cpi		r_tmp,34								; Minutenbit5 ?
	brne	Ber33									; Nein, dann Sprung
	subi	r_Math0,-10								; Ja, dann Minuten +10
	rjmp	NextSek									; und Sprung
Ber33:
	cpi		r_tmp,33								; Minutenbit6 ?
	brne	Ber32									; Nein, dann Sprung
	subi	r_Math0,-20								; Ja, dann Minuten +20
	rjmp	NextSek									; und Sprung
Ber32:
	cpi		r_tmp,32								; Minutenbit7 ?
	brne	Ber30									; Nein, dann Sprung
	subi	r_Math0,-40								; Ja, dann Minuten +40
	rjmp	NextSek									; und Sprung
Ber30:
	cpi		r_tmp,30								; Stundenbit1 ?
	brne	Ber29									; Nein, dann Sprung
	subi	r_Math1,-1								; Ja, dann Stunde +1
	rjmp	NextSek									; und Sprung
Ber29:
	cpi		r_tmp,29								; Stundenbit2 ?
	brne	Ber28									; Nein, dann Sprung
	subi	r_Math1,-2								; Ja, dann Stunde +2
	rjmp	NextSek									; und Sprung
Ber28:
	cpi		r_tmp,28								; Stundenbit3 ?
	brne	Ber27									; Nein, dann Sprung
	subi	r_Math1,-4								; Ja, dann Stunde +4
	rjmp	NextSek									; und Sprung
Ber27:
	cpi		r_tmp,27								; Stundenbit4 ?
	brne	Ber26									; Nein, dann Sprung
	subi	r_Math1,-8								; Ja, dann Stunde +8
	rjmp	NextSek									; und Sprung
Ber26:
	cpi		r_tmp,26								; Stundenbit5 ?
	brne	Ber25									; Nein, dann Sprung
	subi	r_Math1,-10								; Ja, dann Stunde +10
	rjmp	NextSek									; und Sprung
Ber25:
	cpi		r_tmp,25								; Stundenbit6 ?
	brne	Ber23									; Nein, dann Sprung
	subi	r_Math1,-20								; Ja, dann Stunde +20
	rjmp	NextSek									; und Sprung
Ber23:
	cpi		r_tmp,23								; Tagbit1 ?
	brne	Ber22									; Nein, dann Sprung
	subi	r_Math2,-1								; Ja, dann Tag +1
	rjmp	NextSek									; und Sprung
Ber22:
	cpi		r_tmp,22								; Tagbit2 ?
	brne	Ber21									; Nein, dann Sprung
	subi	r_Math2,-2								; Ja, dann Tag +2
	rjmp	NextSek									; und Sprung
Ber21:
	cpi		r_tmp,21								; Tagbit3 ?
	brne	Ber20									; Nein, dann Sprung
	subi	r_Math2,-4								; Ja, dann Tag +4
	rjmp	NextSek									; und Sprung
Ber20:
	cpi		r_tmp,20								; Tagbit4 ?
	brne	Ber19									; Nein, dann Sprung
	subi	r_Math2,-8								; Ja, dann Tag +8
	rjmp	NextSek									; und Sprung
Ber19:
	cpi		r_tmp,19								; Tagbit5 ?
	brne	Ber18									; Nein, dann Sprung
	subi	r_Math2,-10								; Ja, dann Tag +10
	rjmp	NextSek									; und Sprung
Ber18:
	cpi		r_tmp,18								; Tagbit6 ?
	brne	Ber17									; Nein, dann Sprung
	subi	r_Math2,-20								; Ja, dann Tag +20		
	rjmp	NextSek									; und Sprung
Ber17:
	cpi		r_tmp,17								; Wochentagbit1 ?
	brne	Ber16									; Nein, dann Sprung
	subi	r_Math3,-1								; Ja, dann Wochentag +1
	rjmp	NextSek									; und Sprung
Ber16:
	cpi		r_tmp,16								; Wochentagbit2 ?
	brne	Ber15									; Nein, dann Sprung
	subi	r_Math3,-2								; Ja, dann Wochentag +2
	rjmp	NextSek									; und Sprung
Ber15:
	cpi		r_tmp,15								; Wochentagbit3 ?
	brne	Ber14									; Nein, dann Sprung
	subi	r_Math3,-4								; Ja, dann Wochentag +4
	rjmp	NextSek									; und Sprung
Ber14:
	cpi		r_tmp,14								; Monatsbit1 ?
	brne	Ber13									; Nein, dann Sprung
	subi	r_Math4,-1								; Ja, dann Monat +1
	rjmp	NextSek									; und Sprung
Ber13:
	cpi		r_tmp,13								; Monatsbit2 ?
	brne	Ber12									; Nein, dann Sprung
	subi	r_Math4,-2								; Ja, dann Monat +2
	rjmp	NextSek									; und Sprung
Ber12:
	cpi		r_tmp,12								; Monatsbit3 ?
	brne	Ber11									; Nein, dann Sprung
	subi	r_Math4,-4								; Ja, dann Monat +4
	rjmp	NextSek									; und Sprung
Ber11:
	cpi		r_tmp,11								; Monatsbit4 ?
	brne	Ber10									; Nein, dann Sprung
	subi	r_Math4,-8								; Ja, dann Monat +8
	rjmp	NextSek									; und Sprung
Ber10:
	cpi		r_tmp,10								; Monatsbit5 ?
	brne	Ber9									; Nein, dann Sprung
	subi	r_Math4,-10								; Ja, dann Monat +10
	rjmp	NextSek									; und Sprung
Ber9:
	cpi		r_tmp,9									; Jahresbit1 ?
	brne	Ber8									; Nein, dann Sprung
	subi	r_Math5,-1								; Ja, dann Jahr +1
	rjmp	NextSek									; und Sprung
Ber8:
	cpi		r_tmp,8									; Jahresbit2 ?
	brne	Ber7									; Nein, dann Sprung
	subi	r_Math5,-2								; Ja, dann Jahr +2
	rjmp	NextSek									; und Sprung
Ber7:
	cpi		r_tmp,7									; Jahresbit3 ?
	brne	Ber6									; Nein, dann Sprung
	subi	r_Math5,-4								; Ja, dann Jahr +4
	rjmp	NextSek									; und Sprung
Ber6:
	cpi		r_tmp,6									; Jahresbit4 ?
	brne	Ber5									; Nein, dann Sprung
	subi	r_Math5,-8								; Ja, dann Jahr +8
	rjmp	NextSek									; und Sprung
Ber5:
	cpi		r_tmp,5									; Jahresbit5 ?
	brne	Ber4									; Nein, dann Sprung
	subi	r_Math5,-10								; Ja, dann Jahr +10
	rjmp	NextSek									; und Sprung
Ber4:
	cpi		r_tmp,4									; Jahresbit6 ?
	brne	Ber3									; Nein, dann Sprung
	subi	r_Math5,-20								; Ja, dann Jahr +20
	rjmp	NextSek									; und Sprung
Ber3:
	cpi		r_tmp,3									; Jahresbit7 ?
	brne	Ber2									; Nein, dann Sprung
	subi	r_Math5,-40								; Ja, dann Jahr +40
	rjmp	NextSek									; und Sprung
Ber2:
	cpi		r_tmp,2									; Jahresbit8 ?
	brne	NextSek									; Nein, dann Sprung
	subi	r_Math5,-80								; Ja, dann Jahr +80
NextSek:	
	dec		r_tmp									; Schleifenzhler -1
	breq	PlausiMin								; Alle 38 Bytes durch, dann runter
	rjmp	LiesSeks								; noch nicht alle 38 Bits gelesen also zurck
PlausiMin:
	cpi		r_Math0,61								; Plausibilittsprfung Minute
	brlo	PlausiStunde							; Kleiner als 61, dann OK
	rjmp	DCF_Empfang_Wait						; Nicht kleiner, dann Fehler und von vorn
PlausiStunde:
	cpi		r_Math1,24								; Plausi Stunde
	brlo	PlausiTag								; Keliner als 24, dann OK
	rjmp	DCF_Empfang_Wait						; Nicht kleiner ,dann Fehler und von vorn
PlausiTag:
	cpi		r_Math2,0								; Plausi Tag
	brne	PlausiTag2								; ungleich 0, also Weiter
	rjmp	DCF_Empfang_Wait						; Tag = 0, dann Fehler und von vorn
PlausiTag2:
	cpi		r_Math2,32								; Prfung Obergrenze
	brlo	PlausiWochenTag							; Kleiner als 32, dann OK
	rjmp	DCF_Empfang_Wait						; Nicht kleiner, dann Fehler und von vorn
PlausiWochenTag:
	cpi		r_Math3,0								; Plausi Wochentag
	brne	PlausiWochenTag2						; ungleich 0, also weiter
	rjmp	DCF_Empfang_Wait						; gleich 0, dann Fehler und von vorn
PlausiWochenTag2:
	cpi		r_Math3,8								; Prfung Obergrenze
	brlo	PlausiMonat								; Kleiner als 8, dann OK
	rjmp	DCF_Empfang_Wait						; nicht kleiner, dann Fehler und von vorn
PlausiMonat:
	cpi		r_Math4,0								; Plausi Monat
	brne	PlausiMonat2							; ungleich 0, dann weiter
	rjmp	DCF_Empfang_Wait						; gleich 0, dann Fehler und von vorn
PlausiMonat2:
	cpi		r_Math4,13								; Prfung Obergrenze
	brlo	PlausiJahr								; Kleiner als 13, dann OK
	rjmp	DCF_Empfang_Wait						; nicht kleiner, dann Fehler und von vorn
PlausiJahr:
	cpi		r_Math5,100								; Plausi Jahr
	brlo	PlausiEnde								; unter 100, dann OK
	rjmp	DCF_Empfang_Wait						; nicht kleiner, dann Fehler und von vorn
PlausiEnde:
	sts		Var_Minute,r_Math0						; Empfangene Zeit	
	sts		Var_Stunde,r_Math1						; und Datum
	sts		Var_Tag,r_Math2							; in den SRAM Variablen
	sts		Var_Wochentag,r_Math3					; speichern
	sts		Var_Monat,r_Math4
	sts		Var_Jahr,r_Math5
	ldi		r_Byte,0								; Register = 0
	sts		Var_Sekunde,r_Byte						; Sekunde = 0 speichern
	pop		r_Math5									; wiederherstellen
	pop		r_Math4
	pop		r_Math3
	pop		r_Math2
	pop		r_Math1
	pop		r_Math0
	pop		r_Byte
	pop		r_tmp									
;	Nun nur noch auf die steigende Signalflanke warten, die den Beginn der neuen Minute anzeigt
DCF_Endflanke:
	sbis	DCF_Bus_In,DCF_Sig_Data					; Skip wenn Data = High
	rjmp	DCF_Endflanke							; Low, dann warten auf High
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; Hier wird ein Bit vom DCF Empfnger gelesen....
; Es wird erst mal auf den Beginn des Bits gewartet, danach wird
; der Eingang alle 10 ms (22 Durchgnge lang) abgetastet (Abtastzeit = 220ms)
; Zurckgegeben wird ein Byte bestehend aus Durchgngen die das Eingangssignal High war.....
; Das DCF Signal mu entweder 100 oder 200 ms moduliert sein (High sein) um gltig zu sein.....

; bergaberegister									: keine
; Rckgaberegister									: r_Byte (0 oder 1 oder 255[fr Fehler])
; verwendete Register								: r_tmp (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					: Pause10ms

DCF_Read_Bit:
	push	r_tmp									; sichern
	ldi		r_tmp,22								; Schleifenzhler = 22
	ldi		r_Byte,0								; Rckgaberegister leeren
DCF_RB1:
	sbis	DCF_Bus_In,DCF_Sig_Data					; Auf Beginn des Bits warten
	rjmp	DCF_RB1									; Noch kein Beginn, also zurck
DCF_Bit_Read_Loop:
	sbis	DCF_Bus_In,DCF_Sig_Data					; Skip wenn Data = High
	rjmp	Next_Data								; Sprung
	inc		r_Byte									; Rckgaberegister +1
Next_Data:
	rcall	Pause10ms								; 10ms warten
	dec		r_tmp									; Schleifenzhler -1
	brne	DCF_Bit_Read_Loop						; noch keine 22 Durchgnge, also zurck
DCF_Bit_Ende:
	cpi		r_Byte,8								; unter 80ms 
	brlo	Err										; Fehler
	cpi		r_Byte,13								; 80 - 120 ms
	brlo	Bit0									; Bit = 0
	cpi		r_Byte,18								; 130 - 170 ms
	brlo	Err										; Fehler
	cpi		r_Byte,23								; 180 - 220 ms
	brlo	Bit1									; Bit = 1
Err:
	ldi		r_Byte,255
	rjmp	DCF_Bit_Fertig
Bit0:
	ldi		r_Byte,0
	rjmp	DCF_Bit_Fertig
Bit1:
	ldi		r_Byte,1
DCF_Bit_Fertig:
	pop		r_tmp									; wiederherstellen
	ret												; Ende UPRO
; ------------------------------------------------------------------------------------------------------------
; In dieser Unterroutine wird auf den Beginn einer Bitpause gewartet
; Der Eingang wird wieder alle 10ms abgetastet und geprft ob Low
; Der Abtastvorgang dauert 1750ms....
; Zurckgegeben wird im Register r_Byte wie lange (in 10ms) der Eingang Low war
; Bei 170 bis 175 sollten wir nun den Minutenbeginn erwischt haben

; bergaberegister									; keine
; Rckgaberegister									; r_Byte (0-175)
; verwendete Register								; r_tmp (wird wiederhergestellt)
; von hier aus aufgerufene UPROS					; Pause10ms

DCF_Read_NoBit:
	push	r_tmp									; sichern
	ldi		r_tmp,175								; Schleifenzhler = 175
	ldi		r_Byte,0								; Rckgaberegister leeren
DCFNo_RB1:
	sbic	DCF_Bus_In,DCF_Sig_Data					; Auf Beginn Bitpause warten
	rjmp	DCFNo_RB1								; Noch kein Beginn, also zurck
DCFNo_Bit_Read_Loop:
	sbic	DCF_Bus_In,DCF_Sig_Data					; Skip wenn Data = Low
	rjmp	NextNo_Data								; Sprung
	inc		r_Byte									; Rckgaberegister +1
NextNo_Data:
	rcall	Pause10ms								; 10ms warten
	dec		r_tmp									; Schleifenzhler -1
	brne	DCFNo_Bit_Read_Loop						; noch keine 175 Durchgnge, also zurck
DCFNo_Bit_Ende:
	pop		r_tmp									; wiederherstellen
	ret												; Ende UPRO
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; ++++++++++++++++++++++++++++++++++++++++++++++ Gespeicherte Daten ++++++++++++++++++++++++++++++++++++++++++
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
Init1:
; Daten zur Displayinitialisierung Block 1 (4 Bytes)
.db 0xFD,0xFD,0xFD,0xFD
Init2:
; Daten zur Displayinitialisierung Block 2 (20 Bytes)
.db 0xEF,0x00,0xEE,0x04,0x1B,0x04,0xFE,0xFE,0xFE,0xFE,0xEF,0x90,0x4A,0x04,0x7F,0x3F,0xEE,0x04,0x43,0x06
Init3:
; Daten zur Displayinitialisierung Block 3 (40 Bytes)
.db 0xEF,0x90,0x09,0x83,0x08,0x00,0x0B,0xAF,0x0A,0x00,0x05,0x00,0x06,0x00,0x07,0x00,0xEF,0x00,0xEE,0x0C
.db 0xEF,0x90,0x00,0x80,0xEF,0xB0,0x49,0x02,0xEF,0x00,0x7F,0x01,0xE1,0x81,0xE2,0x02,0xE2,0x76,0xE1,0x83
Init4:
; Daten zur Displayinitialisierung Block 4 (6 Bytes)
.db 0x80,0x01,0xEF,0x90,0x00,0x00
; Zeichensatz fr das Display........ besteht nur aus Ziffern zur Anzeige
; Ein Zeichen ist jeweils 8 Pixel breit, und 12 Pixel hoch
CharTab:
ZS0:
.db 0b00111100, 0b01111110, 0b01100110, 0b11000011, 0b11000011, 0b11000011
.db 0b11000011, 0b11000011, 0b11000011, 0b01100110, 0b01111110, 0b00111100
ZS1:
.db 0b00011000, 0b11111000, 0b11111000, 0b00011000, 0b00011000, 0b00011000
.db 0b00011000, 0b00011000, 0b00011000, 0b00011000, 0b11111111, 0b11111111
ZS2:
.db 0b00111100, 0b01111110, 0b11100111, 0b11000011, 0b00000011, 0b00000110
.db 0b00001100, 0b00011000, 0b00110000, 0b01100000, 0b11111111, 0b11111111
ZS3:
.db 0b00111100, 0b11111110, 0b11000011, 0b00000011, 0b00000111, 0b00111110
.db 0b00111110, 0b00000111, 0b00000011, 0b11000011, 0b11111110, 0b00111100
ZS4:
.db 0b00001110, 0b00001110, 0b00011110, 0b00110110, 0b00110110, 0b01100110
.db 0b11000110, 0b11111111, 0b11111111, 0b00000110, 0b00011111, 0b00011111
ZS5:
.db 0b01111110, 0b01111110, 0b01100000, 0b01100000, 0b01111100, 0b01111110
.db 0b01100111, 0b00000011, 0b00000011, 0b11000111, 0b11111110, 0b01111100
ZS6:
.db 0b00001111, 0b00111111, 0b01110000, 0b01100000, 0b11011100, 0b11111110
.db 0b11100111, 0b11000011, 0b11000011, 0b01100111, 0b01111110, 0b00111100
ZS7:
.db 0b11111111, 0b11111111, 0b11000011, 0b00000111, 0b00000110, 0b00000110
.db 0b00001110, 0b00001100, 0b00001100, 0b00011100, 0b00011000, 0b00011000
ZS8:
.db 0b00111100, 0b01111110, 0b11000011, 0b11000011, 0b11000011, 0b01111110
.db 0b01111110, 0b11000011, 0b11000011, 0b11000011, 0b01111110, 0b00111100
ZS9:
.db 0b00111100, 0b01111110, 0b11100110, 0b11000011, 0b11000011, 0b11100111
.db 0b01111111, 0b00111011, 0b00000110, 0b00001110, 0b11111100, 0b11110000
ZSDoppelPunkt:
.db 0b00000000, 0b00000000, 0b00000000, 0b00011000, 0b00011000, 0b00000000
.db 0b00000000, 0b00011000, 0b00011000, 0b00000000, 0b00000000, 0b00000000
ZSPunkt:
.db 0b00000000, 0b00000000, 0b00000000, 0b00000000, 0b00000000, 0b00000000
.db 0b00000000, 0b00000000, 0b00000000, 0b00000000, 0b00011000, 0b00011000
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; ++++++++++++++++++++++++++++++++++++++++++++++++ SRAM Segment ++++++++++++++++++++++++++++++++++++++++++++++
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.DSEG
; Speicherpltze fr Uhrzeit/Datum im Binrformat
; Diese SRAM-Pltze werden in der Timer1Overflow Routine gefllt
; und beinnhalten immer die aktuellen Werte
Var_Sekunde:
.Byte 1												; 0-59
Var_Minute:
.Byte 1												; 0-59
Var_Stunde:
.Byte 1												; 0-23
Var_Wochentag:
.Byte 1												; 1-7
Var_Tag:
.Byte 1												; 1-28/29/30/31
Var_Monat:
.Byte 1												; 1-12
Var_Jahr:
.Byte 1												; 0-99
; In diesen Speicherpltzen wird fr die SD-Ansteuerung die Speicheradresse abgelegt
Var_SD_Adr0:										; Low Byte der Adresse
.Byte 1
Var_SD_Adr1:										
.Byte 1
Var_SD_Adr2:
.Byte 1
Var_SD_Adr3:										; High Byte der Adresse
.Byte 1
; Hier wird ein kompletter 512 Bytes block (plus 2 CRC Bytes), 
; welcher aus der SD Karte gelesen wurde zwischengespeichert
Var_Datenblock:
.Byte 514
