/*----------------------------------------------------------------------------------------
|
| TR2200GX with Silabs Si570 LO control firmware
|
| Author: (C) Wulf-Gerd Traving, DL1FAC 2008
|
|-----------------------------------------------------------------------------------------
| License:
| This program is free software; you can redistribute it and/or modify it under
| the terms of the GNU General Public License as published by the Free Software
| Foundation; either version 2 of the License, or (at your option) any later
| version.
| This program is distributed in the hope that it will be useful, but
|
| WITHOUT ANY WARRANTY;
|
| without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
| PURPOSE. See the GNU General Public License for more details.
|
| You should have received a copy of the GNU General Public License along with
| this program; if not, write to the Free Software Foundation, Inc., 51
| Franklin St, Fifth Floor, Boston, MA 02110, USA
|
| http://www.gnu.de/gpl-ger.html
`-----------------------------------------------------------------------------------------*/

#include "defines.h"

#include <ctype.h>
#include <stdint.h>
#include <stdio.h>

#include <avr/eeprom.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>

#include <util/delay.h>

#include "iocompat.h"
#include "si570.h"

int16_t ad_poti1 = -999;
uint8_t transmitting = 0;
int8_t button = -1;

uint32_t freq, rxfreq, txfreq;

typedef enum {
	fixed,
	variable
	} mode_t;

mode_t mode;
uint8_t channel;

uint32_t PROGMEM channel_tab [] PROGMEM = {
	145625000, // channel R1
	145650000, // channel R2
	145675000, // channel R3
	145700000, // channel R4
	145725000, // channel R5
	145750000, // channel R6
	145775000, // channel R7
	144800000, // channel R8
	145325000, // channel R9
	144660000, // channel A
	144675000, // channel B
	145500000, // channel C
};

#define CHANNELS (sizeof(channel_tab)/sizeof(uint32_t))

/*
 * Do all the startup-time peripheral initializations.
 */
static void ioinit(void)
{
  // enable all pullup to prevent floating inputs
  DDRB  = 0x00; // all input
  PORTB = 0xfb; // all pullups but pb2 = ADC1 enabled
  // enable ADC, select ADC clock = F_CPU / 8 (i.e. 1MHz kHz)
  ADCSRA = _BV(ADEN) | _BV(ADPS1) | _BV(ADPS0);
  // set reference voltage to internal 2.56V source
  ADMUX |= _BV(REFS1) | _BV(REFS2);
  sei ();
  silabs_init ();
}


void set_rxtx (void)
{
	if (transmitting) silabs_setNewFreq (txfreq);
	else silabs_setNewFreq (rxfreq);
	_delay_ms (50.0);
}


void set_vfo (uint32_t fhz)
{
	rxfreq = (fhz-ZF)/3l;
	// relais shift, if needed
	if (fhz >= 145600000l && fhz < 145800000l)
	{
		txfreq = (fhz - 600000l) / 12l; // 0.6 MHz shift
	}
	else
	{
		txfreq = fhz/12l;
	}
	set_rxtx ();
}


// handle analog input (set frequency or channel)
void do_poti1 (int16_t new_val)
{
	uint8_t new_channel;
	int8_t steps;
	if ((new_val > ad_poti1+3) || (new_val < ad_poti1-3)) // filter noise
	{
		ad_poti1 = new_val;
		// Treppenstufen sind wegen vorliegender Beschaltung des Einganges
		// nicht gleichhoch, daher dieses Konstrukt...
		if (new_val < 127) new_channel = 0;
		else if (new_val < 246) new_channel = 1;
		else if (new_val < 350) new_channel = 2;
		else if (new_val < 436) new_channel = 3;
		else if (new_val < 522) new_channel = 4;
		else if (new_val < 596) new_channel = 5;
		else if (new_val < 662) new_channel = 6;
		else if (new_val < 724) new_channel = 7;
		else if (new_val < 777) new_channel = 8;
		else if (new_val < 824) new_channel = 9;
		else if (new_val < 866) new_channel = 10;
		else new_channel = 11;

		if (mode == variable)
		{
		    steps = new_channel - channel;
			// nun feststellen, ob zurckgedreht wurde
			if (steps > 6) steps -= 12;
			// oder vorwrts?
			if (steps < -6) steps += 12;
			freq -= steps * 12500l; // tuning in 12.5kHz steps
			// TEST to evaluate ad steps by observing freq output
			// freq = 145000000l + new_val * 1000L;
			channel = new_channel;
			set_vfo (freq);
		}

		if (mode == fixed)
		{
			if (new_channel != channel)
			{
				channel = new_channel;
				freq = pgm_read_dword (&channel_tab [channel]);
				set_vfo (freq);
			}
		}
	}
}

void handle_ad (void)
{
   uint16_t temp;

	ADMUX = (ADMUX & ~0x0F) | 1;
	ADCSRA |= _BV(ADSC);
	while (ADCSRA & _BV(ADSC));
	temp = ADCL + (ADCH<<8);
	do_poti1 (temp);
}


void handle_switch (void)
{
	mode_t new_mode;
	new_mode = CHANVFO?0:1;
	if (new_mode != mode)
	{
		mode = new_mode;
		ad_poti1 = -999; // force poti evaluation
		if (mode == fixed) channel = 255; // and reset channel then
	}
}

// handle PTT input
void handle_button (void)
{
	uint8_t new_button;
	new_button = PTT?0:1;
	if (new_button != button)
	{
		button = new_button;
		if (button)
		{
			// rx
			transmitting = 0;
		}
		else
		{
			// tx
			transmitting = 1;
		}
		set_vfo (freq);
		// set_rxtx ();
	}
}


/* ---------------------------------------------------------------------------------- */

int main(void)
{
    ioinit();

	// TR2200GX rx frequency formula (RXfreq - 10.7) / 3
	// TR2200GX tx frequency forumla TXfreq / 12

	_delay_ms (500.0);

	// initialize channel and freq to proper startup value
	mode = fixed;
	channel = 255; // force evaluation of channel
	handle_ad  ();

   for (;;)
   {
		handle_button ();

		if (!transmitting)
		{
			handle_switch ();
			handle_ad ();
		}

		_delay_ms ((float) UPDATEINTERVAL);

   }
}
