/*----------------------------------------------------------------------------------------
|
| TR2200GX with Silabs Si570 LO control firmware
|
| Author: Wulf-Gerd Traving, DL1FAC 2008
| This part is only a port of the Silabs Example Code
|
|-----------------------------------------------------------------------------------------
| License:
| This program is free software; you can redistribute it and/or modify it under
| the terms of the GNU General Public License as published by the Free Software
| Foundation; either version 2 of the License, or (at your option) any later
| version.
| This program is distributed in the hope that it will be useful, but
|
| WITHOUT ANY WARRANTY;
|
| without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
| PURPOSE. See the GNU General Public License for more details.
|
| You should have received a copy of the GNU General Public License along with
| this program; if not, write to the Free Software Foundation, Inc., 51
| Franklin St, Fifth Floor, Boston, MA 02110, USA
|
| http://www.gnu.de/gpl-ger.html
`-----------------------------------------------------------------------------------------*/

#include "defines.h"

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>

#include <avr/io.h>
#include <avr/pgmspace.h>

#include <util/delay.h>

#include <math.h>
#include "i2cmaster.h"
#include "si570.h"


//#define SI570ADDR 0xaa
#define SI570ADDR 0xaa

#define FOUT0 14.999931
#define FOUT0L 14999931

// Si57x's fdco range -- DO NOT EDIT
const float FDCO_MAX = 5670.0; //MHz
const float FDCO_MIN = 4850.0; //MHz
const unsigned long int FDCO_MAXL = 5537109; // * 1024 Hz
const unsigned long int FDCO_MINL = 4736328; // * 1024 Hz
const unsigned char HS_DIV[6] = {11, 9, 7, 6, 5, 4};

unsigned char silabs_detected;
unsigned char reg[6];
unsigned char initial_n1;
unsigned char initial_hsdiv;
unsigned long initial_rfreq_long;
double xtal;

int I2C_ByteWrite (unsigned char reg, unsigned char data)
{
	if (i2c_start (SI570ADDR|I2C_WRITE))
	{
		/* failed to issue start condition, possibly no device found */
		i2c_stop();
		silabs_detected = 0;
		return -1;
	}
	else
	{
		/* issuing start condition ok, device accessible */
		i2c_write(reg);
		i2c_write(data);
		i2c_stop();
	}
	return 0;
}

int I2C_regsWrite (void)
{
    unsigned char i;
	if (i2c_start (SI570ADDR|I2C_WRITE))
	{
		/* failed to issue start condition, possibly no device found */
		i2c_stop();
		silabs_detected = 0;
		return -1;
	}
	else
	{
		/* issuing start condition ok, device accessible */
		i2c_write (7);
	    // load the new values into the device at registers 7 to 12
		for (i=0; i<sizeof(reg); i++) i2c_write (reg[i]);
		i2c_stop();
	}
	return 0;
}

unsigned char I2C_ByteRead (unsigned char reg)
{
	unsigned char ret;
	if (i2c_start (SI570ADDR|I2C_WRITE))
	{
		/* failed to issue start condition, possibly no device found */
		i2c_stop();
		silabs_detected = 0;
		return -1;
	}
	else
	{
		i2c_write (reg);
		i2c_rep_start (SI570ADDR+I2C_READ);       // set device address and read mode
		ret = i2c_readNak ();                     // read one byte
		i2c_stop ();
		return ret;
	}
}

// Reads start-up register contents for RFREQ, HS_DIV, and N1 and calculates the internal crystal frequency (xtal)
void ReadStartUpConfiguration(void)
{
	unsigned char counter;
	// recall the initial state of the device from nvm into ram
	if (I2C_ByteWrite(135, 0x01) < 0)
	{
		silabs_detected = 0;
		return;
	}
	// read registers 7 to 12
    // reg[0] is equivalent to register 7 in the device
    // reg[5] is register 12 in the device
	for(counter=0; counter<6; counter++)
	{
		reg[counter] = I2C_ByteRead(counter+7);
	}
	// HS_DIV conversion
	initial_hsdiv = ((reg[0] & 0xE0) >> 5) + 4; // get reg 7 bits 5, 6, 7
	// initial_hsdiv's value could be verified here to ensure that it is one
	// of the valid HS_DIV values from the datasheet.
	// initial_n1 conversion
	initial_n1 = (( reg[0] & 0x1F ) << 2 ) + // get reg 7 bits 0 to 4
		(( reg[1] & 0xC0 ) >> 6 ); // add with reg 8 bits 7 and 8
	if(initial_n1 == 0)
	{
		initial_n1 = 1;
	}
	else if ((initial_n1 & 1) != 0)
	{
		// add one to an odd number
		initial_n1 = initial_n1 + 1;
	}
	// RFREQ conversion using unsigned long
	initial_rfreq_long = ( reg[1] & 0x3F );
	initial_rfreq_long = (initial_rfreq_long << 8) + ( reg[2] );
	initial_rfreq_long = (initial_rfreq_long << 8) + ( reg[3] );
	initial_rfreq_long = (initial_rfreq_long << 8) + ( reg[4] );
	initial_rfreq_long = (initial_rfreq_long << 6) + ( reg[5] >> 2 ); //ignore lowest two bits to fit into long
	// xtal value is computed only for user information and needed only for output to console
	xtal = (double) FOUT0L * initial_n1 * initial_hsdiv * 67108864.0;
    xtal /= initial_rfreq_long;
}


//
// Program a new frequency
//
void silabs_setNewFreq (unsigned long int newFrequency)
{
	unsigned char counter;
	unsigned char n1;
	unsigned char hsdiv;
	unsigned char validCombo;
	unsigned char reg137, reg135;
	unsigned int divider_max;
	unsigned int curr_div;
	unsigned int rn1;
	float currentFrequency;
	float ratio = 0;
	unsigned long final_rfreq_long;

	if (silabs_detected)
	{
		currentFrequency = (float) newFrequency / 1000000.0;
		// find dividers (get the max and min divider range for the HS_DIV and N1 combo)
		divider_max = FDCO_MAXL / (newFrequency >> 10);
		curr_div = (FDCO_MINL / (newFrequency >> 10)) + 1;
		validCombo = 0;
		n1 = initial_n1;
		hsdiv = initial_hsdiv;
		while (curr_div <= divider_max)
		{
			//check all the HS_DIV values with the next curr_div
			for(counter=0; counter<6; counter++)
			{
				// get the next possible n1 value
				hsdiv = HS_DIV[counter];
				rn1 = curr_div % hsdiv;
				if (rn1 == 0)
				{
					// no remainder, this is a possible n1 value
					n1 = curr_div / hsdiv;
					if ((n1 == 1) || ((n1 & 1) == 0)) goto foundCombo;
				}
			}
			// increment curr_div to find the next divider
			// since the current one was not valid
			curr_div += 1;
		}
	foundCombo:
		// if(validCombo == 0) at this point then there's an error
		// in the calculatio. Check if the provided fout0 and fout1
		// are not valid frequencies
		// calculate RFREQ organizing the float variables to save precision;
		// RFREQ is kept as an unsigned long
		// only 32 bits are available in the long format
		// RFREQ in the device has 34 bits of precision
		// only 34 of the 38 bits are needed since RFREQ is between 42.0 and 50.0 for xtal of 114.285MHz (nominal)
		ratio = currentFrequency / FOUT0;
		ratio *= ((float) n1 / (float) initial_n1) * ((float) hsdiv / (float) initial_hsdiv);
		final_rfreq_long = ratio * (float) initial_rfreq_long;
		reg[0] = 0;
		// set the top 3 bits of reg 13
		reg[0] = (hsdiv-4) << 5;
		// convert new N1 to the binary representation
		if (n1 == 1) n1 = 0;
		else if((n1 & 1) == 0) n1 = n1 - 1; //if n1 is even, subtract one
		// set reg 7 bits 0 to 4
		reg[0] = (reg[0] & 0xE0) | (n1 >> 2);
		// set reg 8 bits 6 and 7
		reg[1] = (n1 & 3) << 6;
		// load new RFREQ into register map
		reg[1] |= final_rfreq_long >> 30;
		reg[2] = final_rfreq_long >> 22;
		reg[3] = final_rfreq_long >> 14;
		reg[4] = final_rfreq_long >> 6;
		reg[5] = final_rfreq_long << 2;
		// Load the new frequency
		// first wait for previous setup beeing thru
		do {
			reg135 = I2C_ByteRead(135);
		} while (reg135 & 0x40);
		// get the current state of register 137
		reg137 = I2C_ByteRead(137);
		// set the Freeze DCO bit in that register
		I2C_ByteWrite(137, reg137 | 0x10);
		// load the new values into the device at registers 7 to 12
		I2C_regsWrite ();
		// get the current state of register 137
		reg137 = I2C_ByteRead(137);
		// clear the FZ_DCO bit in that register
		I2C_ByteWrite(137, reg137 & 0xEF);
		// set the NewFreq bit, bit will clear itself once the device is ready
		I2C_ByteWrite(135, 0x40);
	}
}

void silabs_init (void)
{
    i2c_init();                             // initialize I2C library
    silabs_detected = 1;
    ReadStartUpConfiguration ();
}
