/*
 *    throbtx.c  --  THROB modulator
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "throb.h"
#include "tab.h"
#include "../main/trx.h"
#include "../main/sndmodem.h"
#include "../main/trxctrl.h"
#include "../misc/filter.h"
#include "../misc/misc.h"
#include "../misc/mixer.h"

#ifdef _WIN32_WCE
	#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

static void send_throb(Trx *trx, int symbol)
{
	struct throb   *s		= (struct throb*)trx->modem;
	int				freq	= trx->nTxFrequency;
	int				tone1, tone2;
	int				i;

	if (symbol < 0 || symbol >= s->nChars)
		return;

	tone1 = s->pTonePairs[symbol << 1] - 1;
	tone2 = s->pTonePairs[(symbol << 1) + 1] - 1;

	if (trx->reverse) {
		tone1 = (s->nTones - 1) - tone1;
		tone2 = (s->nTones - 1) - tone2;
	}

	if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202 || g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER)
	{
		trxctrl_ats3_sendtone(&g_TrxCtrl, trx->pModem, tone1, s->symlen >> 2);
		trxctrl_ats3_sendtone(&g_TrxCtrl, trx->pModem, tone2, s->symlen >> 2);
		trxctrl_ats3_sendtone(&g_TrxCtrl, trx->pModem, tone1, s->symlen >> 2);
		trxctrl_ats3_sendtone(&g_TrxCtrl, trx->pModem, tone2, s->symlen >> 2);
	}
	else
	{
		UINT iPhaseDelta1 = trx_freq2phase(freq + s->freqs[tone1]);
		UINT iPhaseDelta2 = trx_freq2phase(freq + s->freqs[tone2]);
		UINT nPhaseAcc1 = 0;
		UINT nPhaseAcc2 = 0;
		int	 v;

		_ASSERT(trx->pModem->nOutBufLen + s->symlen <= trx->pModem->nOutBufSize);
		for (i = 0; i < s->symlen; ++ i) {
			v = (mixer_osc_16(nPhaseAcc1) + mixer_osc_16(nPhaseAcc2)) >> 1;
			trx->pModem->pOutBuf[trx->pModem->nOutBufLen + i] = (s->txpulse[i] * v) >> 16;
			nPhaseAcc1 += iPhaseDelta1;
			nPhaseAcc2 += iPhaseDelta2;
		}

		if (trx->pModem->nOutBufLen == 0) {
			trx->pModem->nOutBufLen = s->symlen;
			trx->pModem->iOutBufPos = (s->symlen > (trx->pModem->nBufSize >> 1)) ? (trx->pModem->nBufSize >> 1) : s->symlen;
			sndmodem_write(trx->pModem, trx->pModem->pOutBuf, trx->pModem->iOutBufPos);
		} else
			trx->pModem->nOutBufLen += s->symlen;
	}
}

int throb_txprocess(Trx *trx)
{
	struct throb *s = (struct throb*)trx->modem;
	int c = -1;
	int i, sym;

	if (trx->pModem->iOutBufPos < trx->pModem->nOutBufLen) {
		int nLen = trx->pModem->nOutBufLen - trx->pModem->iOutBufPos;
		if (nLen > (trx->pModem->nBufSize >> 1))
			nLen = (trx->pModem->nBufSize >> 1);
		sndmodem_write(trx->pModem, trx->pModem->pOutBuf + trx->pModem->iOutBufPos, nLen);
		trx->pModem->iOutBufPos += nLen;
		if (trx->pModem->iOutBufPos == trx->pModem->nOutBufLen) {
			trx->pModem->iOutBufPos = 0;
			trx->pModem->nOutBufLen = 0;
			if (s->bStopping)
				return -1;
		}
		return 0;
	}

	if (s->preamble > 0) {
		send_throb(trx, s->nSymIdle);	// send idle throbs
		s->preamble --;
		throb_flip_syms(trx);
		return 0;
	}

	c = trx_get_tx_char(trx);

	// TX buffer empty
	if (c == -1) {
		// stop if requested to
		if (trx->stopflag) {
			send_throb(trx, s->nSymIdle);
			throb_reset_syms(trx);
			s->bStopping = TRUE;
			return (trx->pModem->iOutBufPos == trx->pModem->nOutBufLen) ? -1 : 0;
		}
		send_throb(trx, s->nSymIdle);	// send idle throbs
		throb_flip_syms(trx);
		return 0;
	}

	if (trx->mode == MODE_THROB) {
		// handle shifts of the old throb mode
		switch (c) {
		case '?':
			send_throb(trx, 5);	// shift
			send_throb(trx, 20);
			trx_put_echo_char(trx, c);
			return 0;
		case '@':
			send_throb(trx, 5);	// shift
			send_throb(trx, 13);
			trx_put_echo_char(trx, c);
			return 0;
		case '=':
			send_throb(trx, 5);	// shift
			send_throb(trx, 9);
			trx_put_echo_char(trx, c);
			return 0;
		case '\r':
		case '\n':
			send_throb(trx, 5);	// shift
			send_throb(trx, 0);
			trx_put_echo_char(trx, c);
			return 0;
		default:
			break;
		}
	}

	// map lower case character to upper case
	if (islower(c))
		c = toupper(c);

	// see if the character can be found in our character set
	for (sym = -1, i = 0; i < s->nChars; ++ i)
		if (c == s->pCharSet[i])
			sym = i;

	if (c == ' ' || sym == -1) {
		// send a space for unknown chars
		send_throb(trx, s->nSymSpace);
		throb_flip_syms(trx);
	} else
		send_throb(trx, sym);

	trx_put_echo_char(trx, c);
	return 0;
}
