/*
 *    throbrx.c  --  THROB demodulator
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifdef _WIN32_WCE
#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

#include <stdio.h>
#include <string.h>

#include "throb.h"
#include "tab.h"
#include "../misc/fixedpoint.h"
#include "../misc/filter.h"
#include "../misc/mixer.h"

// #include "../main/dbgwaterfall.h"
// extern DbgWaterfall s_DbgWaterfall;


static int findtones(cmplx *word, int nTones, int *tone1, int *tone2, BOOL bThrobOld)
{
	int aAmps[NumTones];
	int max1 = 0;
	int max2 = 0;
	int maxtone, i;

	*tone1 = 0;
	*tone2 = 0;

	// Compute amplitudes of the tones
	for (i = 0; i < nTones; ++ i)
		aAmps[i] = ihypot(word[i].re, word[i].im);

	// Find first strongest tone
	for (i = 0; i < nTones; ++ i) {
		if (aAmps[i] > max1) {
			max1 = aAmps[i];
			*tone1 = i;
		}
	}
	maxtone = *tone1;

	for (i = 0; i < nTones; ++ i) {
		if (i == *tone1)
			continue;
		if (aAmps[i] > max2) {
			max2 = aAmps[i];
			*tone2 = i;
		}
	}

	if (bThrobOld && max1 > (max2 << 1))
		*tone2 = *tone1;

	if (*tone1 > *tone2) {
		i = *tone1;
		*tone1 = *tone2;
		*tone2 = i;
	}

	return maxtone;
}

static void decodechar(Trx *trx, int tone1, int tone2)
{
	struct throb *s = (struct throb*)trx->modem;
	int i;

	if (trx->mode == MODE_THROB) {
		if (s->bShift) {
			if (tone1 == 0 && tone2 == 8)
				trx_put_rx_char(trx, '?');
			else if (tone1 == 1 && tone2 == 7)
				trx_put_rx_char(trx, '@');
			else if (tone1 == 2 && tone2 == 6)
				trx_put_rx_char(trx, '=');
			else if (tone1 == 4 && tone2 == 4)
				trx_put_rx_char(trx, '\n');
			s->bShift = FALSE;
			return;
		}
		if (tone1 == 3 && tone2 == 5) {
			s->bShift = TRUE;
			return;
		}
	}

	for (i = 0; i < s->nChars; ++ i) {
		if (s->pTonePairs[i << 1] == tone1 + 1 && s->pTonePairs[(i << 1) + 1] == tone2 + 1) {
			int c = s->pCharSet[i];
			if (trx->mode == MODE_THROBX && (i == s->nSymIdle || i == s->nSymSpace)) {
				if (s->nCharLast != 0 && s->nCharLast != ' ') {
					//FIXME: idle/space sync not perfect, this is a bandaid
					trx_put_rx_char(trx, ' '); 
					s->nCharLast = ' '; 
				} 
				//if (i == spacesym) { trx_put_rx_char(trx, ' '); }
				else
					s->nCharLast = 0;
				throb_flip_syms(trx);
			} else {
				if (c != 0)
					trx_put_rx_char(trx, c);
				s->nCharLast = c;
			}
			break;
		}
	}
}

/// Complex multiply-accumulate.
cmplx cmac(cmplx *a, cmplx *b, int ptr, int len)
{
	cmplx z = { 0, 0 };
	int i;
	ptr = ptr % len;
	for (i = 0; i < len; ++ i) {
		z.re += a[i].re * b[ptr].re - a[i].im * b[ptr].im;
		z.im += a[i].re * b[ptr].im + a[i].im * b[ptr].re;
		ptr = (ptr + 1) % len;
	}
	return z;
}

static void throb_decode(Trx *trx, cmplx in)
{
	struct throb   *s = (struct throb*)trx->modem;
	cmplx			rxword[NumTones];
	int			    i, tone1, tone2, maxtone;

	// store input
	s->symbol[s->symptr] = in;

	// check counter
	if (s->rxcntr > 0)
		return;

	// correlate against all tones
	for (i = 0; i < s->nTones; ++ i) {
		rxword[i] = cmac(s->rxtone[i], s->symbol, s->symptr + 1, s->rxsymlen);
		rxword[i].re >>= 15;
		rxword[i].im >>= 15;
	}

	// find the strongest tones
	maxtone = findtones(rxword, s->nTones, &tone1, &tone2, trx->mode == MODE_THROB);
	// decode
	if (trx->reverse)
		decodechar(trx, s->nTones - 1 - tone2, s->nTones - 1 - tone1);
	else
		decodechar(trx, tone1, tone2);

	if (trx->afcon) {
		cmplx z1	= rxword[maxtone];
		cmplx z2	= cmac(s->rxtone[maxtone], s->symbol, s->symptr + 2, s->rxsymlen);
		int		phase, err;
		z2.re	  >>= 15;
		z2.im	  >>= 15;
		// phase is a 16 bit number from -32768 to +32768
		phase		= iphasediff(&z1, &z2);
//		err  = phase / (2 * DownSample * M_PI / SampleRate) - s->freqs[maxtone];
		err			= ((phase * 250) >> (16 - FREQ_FRAC_BITS)) - s->freqs[maxtone];
		err += 8;
		err >>= 4;
		trx->nRxFrequency -= err;
		if (! trx->bTxLock)
			trx->nTxFrequency = trx->nRxFrequency;
	}

	// done with this symbol, start over
	s->rxcntr    = s->rxsymlen << 16;
	s->bWaitSync = TRUE;
}

static void throb_sync(Trx *trx, cmplx in)
{
	struct throb   *s		= (struct throb*)trx->modem;
	int				maxval	= 0;
	int				maxpos	= 0;
	int				i;

	// "rectify", filter and store input into s->syncbuf[s->symptr]
	filter_I_run(s->syncfilt, ihypot(in.re, in.im), s->syncbuf + s->symptr);

	// check counter if we are waiting for sync
	if (! s->bWaitSync || s->rxcntr > (s->rxsymlen << 15))
		return;

	for (i = 0; i < s->rxsymlen; ++ i) {
		s->dispbuf[i] = s->syncbuf[(i + s->symptr + 1) & (s->rxsymlen - 1)];
		if (s->dispbuf[i] > maxval) {
			maxpos = i;
			maxval = s->dispbuf[i];
		}
	}

	// correct sync
	s->rxcntr   += (maxpos - (s->rxsymlen >> 1)) << (16 - 3);
	s->bWaitSync = FALSE;

	trx_set_scope(s->dispbuf, s->rxsymlen, TRUE);
}

int throb_rxprocess(Trx *trx, short *buf, int len)
{
	struct throb   *s = (struct throb*)trx->modem;
	cmplx			z;
	while (len-- > 0) {
		// Mix with the internal NCO
		mixer_mix(*(buf ++), s->phaseacc, &z);
		s->phaseacc += trx_freq2phase(trx->nRxFrequency);
		z.re = (z.re + 0x3fff) >> 15;
		z.im = (z.im + 0x3fff) >> 15;
		// Lowpass filter
		if (! filter_run(s->rxfilter, z, &z))
			continue;
		z.re >>= 15;
		z.im >>= 15;
//		dbgwaterfall_set_data(&s_DbgWaterfall, &z, 1);
		s->rxcntr -= 1 << 16;
		throb_sync(trx, z);
		throb_decode(trx, z);
		s->symptr = (s->symptr + 1) & (s->rxsymlen - 1);
	}
	return 0;
}
