/*
 *    throb.c  --  THROB modem
 *
 *    Copyright (C) 2001, 2002, 2003, 2004
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>
#include <stdio.h>

#include "throb.h"
#include "tab.h"

#include "../misc/fixedpoint.h"
#include "../misc/filter.h"
#include "../misc/mixer.h"
#include "../main/sndmodem.h"
#include "../main/trxctrl.h"
#include "../resource.h"

#ifdef _WIN32_WCE
	#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

static void throb_txinit(Trx *trx)
{
	struct throb *s = (struct throb*)trx->modem;
	s->preamble		= 4;
	s->bStopping	= FALSE;
	throb_reset_syms(trx);
	if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202 || g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER) {
		int nToneSpacing = s->freqs[1] - s->freqs[0];
		int f0 = (int)trx->nTxFrequency - ((s->nTones - 1) >> 1) * nToneSpacing;
		trxctrl_ats3_start_mfsk(&g_TrxCtrl, trx->pModem, f0, nToneSpacing);
	}
}

static void throb_rxinit(Trx *trx)
{
	struct throb *s = (struct throb*)trx->modem;
	s->rxcntr		= s->rxsymlen;
	s->symptr		= 0;
	s->bWaitSync	= TRUE;
	s->bShift		= FALSE;
	s->nCharLast	= 0;
}

static void throb_free(struct throb *s)
{
	int i;

	if (! s)
		return;

	free(s->txpulse);
	filter_free(s->rxfilter);
	filter_free(s->syncfilt);

	for (i = 0; i < s->nTones; ++ i)
		free(s->rxtone[i]);
	free(s);
}

static void throb_destructor(Trx *trx)
{
	throb_free((struct throb*)trx->modem);
	trx->modem		= NULL;
	trx->txinit		= NULL;
	trx->rxinit		= NULL;
	trx->txprocess	= NULL;
	trx->rxprocess	= NULL;
	trx->destructor = NULL;
}

/// Make a semi raised cosine pulse of length 'len'.
static double *mk_semi_pulse(int len)
{
	double *pulse = (double*)malloc(sizeof(double) * len);
	double	x;
	int		i, j;
	for (i = 0; i < len; ++ i) {
		if (i < len / 5) {
			x = M_PI * i / (len / 5.0);
			pulse[i] = 0.5 * (1.0 - cos(x));
		}
		if (i >= len / 5 && i < len * 4 / 5)
			pulse[i] = 1.0;
		if (i >= len * 4 / 5) {
			j = i - len * 4 / 5;
			x = M_PI * j / (len / 5.0);
			pulse[i] = 0.5 * (1.0 + cos(x));
		}
	}
	return pulse;
}

/// Make a full raised cosine pulse of length 'len'.
static double *mk_full_pulse(int len)
{
	double *pulse = (double*)malloc(sizeof(double) * len);
	int		i;
	for (i = 0; i < len; ++ i)
		pulse[i] = 0.5 * (1.0 - cos(2 * M_PI * i / len));
	return pulse;
}

/// Make a 32 times downsampled complex prototype tone for rx.
static cmplx *mk_rxtone(double freq, double *pulse, int len)
{
	cmplx   *tone	= (cmplx*)malloc(sizeof(cmplx) * len);
	double  *iCoeff = (double*)malloc(sizeof(double) * len);
	double  *qCoeff = (double*)malloc(sizeof(double) * len);
	double   iSum   = 0.0;
	double   qSum   = 0.0;
	double   dValue;
	int      iValue;
	int      i;

	// Calculate matched filter to the tone.
	for (i = 0; i < len; ++ i) {
		dValue = - 2.0 * M_PI * freq * i * DownSample / SampleRate;
		iCoeff[i] = pulse[i] * cos(dValue);
		qCoeff[i] = pulse[i] * sin(dValue);
		iSum     += fabs(iCoeff[i]);
		qSum     += fabs(qCoeff[i]);
	}

	// Normalize the filter, convert to int
	if (qSum > iSum)
		iSum = qSum;
	for (i = 0; i < len; ++ i) {
		dValue = (double)iCoeff[i] * (double)0x7fff / iSum;
		iValue = (int)floor(dValue + 0.5);
		if (iValue > 0x7fff)
			iValue = 0x7fff;
		else if (iValue < -0x7fff)
			iValue = -0x7fff;
		tone[i].re = iValue;
		dValue = (double)qCoeff[i] * (double)0x7fff / qSum;
		iValue = (int)floor(dValue + 0.5);
		if (iValue > 0x7fff)
			iValue = 0x7fff;
		else if (iValue < -0x7fff)
			iValue = -0x7fff;
		tone[i].im = iValue;
	}

	free(iCoeff);
	free(qCoeff);
	return tone;
}

void throb_init(Trx *trx)
{
	struct throb *s		  = 0;
	double	     *rxpulse = 0;
	double		 *txpulse = 0;
	double		  bw;
	int			  i;
	double		  dSum	  = 0.0;

	s = calloc(sizeof(struct throb), 1);
	s->nSymIdle  = 0;

	if (trx->submode == 0)
		trx->submode = ID_SUBMODE_TPS2;

	if (trx->mode == MODE_THROB) {
		s->nTones     = 9;
		s->nChars     = 45;
		s->pCharSet   = ThrobCharSet;
		s->pTonePairs = ThrobTonePairs;
		s->nSymSpace  = 44;
	} else {
		s->nTones     = 11;
		s->nChars     = 55;
		s->pCharSet	  = ThrobXCharSet;
		s->pTonePairs = ThrobXTonePairs;
		s->nSymSpace  = 1;
	}

	switch (trx->submode) {
	case ID_SUBMODE_TPS1:
		s->symlen	= SymbolLen1;
		txpulse		= mk_semi_pulse(SymbolLen1);
		rxpulse		= mk_semi_pulse(SymbolLen1 / DownSample);
		break;
	case ID_SUBMODE_TPS2:
		s->symlen	= SymbolLen2;
		txpulse		= mk_semi_pulse(SymbolLen2);
		rxpulse		= mk_semi_pulse(SymbolLen2 / DownSample);
		break;
	case ID_SUBMODE_TPS4:
		// nonstandard for THROBX
		s->symlen	= SymbolLen4;
		txpulse		= mk_full_pulse(SymbolLen4);
		rxpulse		= mk_full_pulse(SymbolLen4 / DownSample);
		break;
	default:
		throb_free(s);
		return;
	}

	if (trx->mode == MODE_THROB) {
		if (trx->submode == ID_SUBMODE_TPS4) {
			s->freqs	= ThrobToneFreqsWid;
			bw			= 72.0 / SampleRate;
		} else {
			_ASSERT(trx->submode == ID_SUBMODE_TPS1 || ID_SUBMODE_TPS2);
			s->freqs	= ThrobToneFreqsNar;
			bw			= 36.0 / SampleRate;
		}
	} else {
		if (trx->submode == ID_SUBMODE_TPS4) {
			// NONSTANDARD
			s->freqs	= ThrobXToneFreqsWid;
			bw			= 94.0 / SampleRate;
		} else {
			_ASSERT(trx->submode == ID_SUBMODE_TPS1 || ID_SUBMODE_TPS2);
			s->freqs	= ThrobXToneFreqsNar;
			bw			= 47.0 / SampleRate;
		}
	}

	s->rxsymlen = s->symlen / DownSample;
	s->txpulse  = filter_double2int(txpulse, s->symlen);

	for (i = 0; i < s->nTones; ++ i) {
		s->rxtone[i] = mk_rxtone((double)(s->freqs[i] >> FREQ_FRAC_BITS), rxpulse, s->rxsymlen);
		if (!s->rxtone[i]) {
			throb_free(s);
			return;
		}
	}

	// normalize rxpulse before it is used to build a filter
	for (i = 0; i < s->rxsymlen; ++ i)
		dSum += fabs(rxpulse[i]);
	for (i = 0; i < s->rxsymlen; ++ i)
		rxpulse[i] /= dSum;

	if ((s->rxfilter = filter_init_lowpass(127, DownSample, bw)) == 0 ||
		(s->syncfilt = filter_init_double(s->rxsymlen, 1, rxpulse, NULL)) == 0) {
		throb_free(s);
		return;
	}

	free(rxpulse);
	free(txpulse);

	trx->modem		= s;
	trx->txinit		= throb_txinit;
	trx->rxinit		= throb_rxinit;
	trx->txprocess	= throb_txprocess;
	trx->rxprocess	= throb_rxprocess;
	trx->destructor = throb_destructor;

	trx->reverse	= FALSE;
	trx->bandwidth  = (int)(bw * FREQ_FRAC_VALUE + 0.5);

/*
	trx->samplerate = SampleRate;
	trx->fragmentsize = s->symlen;
	trx->bandwidth = s->freqs[8] - s->freqs[0];
	trx->syncpos	= 0.5;
*/
}

void throb_flip_syms(Trx *trx)
{
	if (trx->mode == MODE_THROBX) {
		struct throb *s = (struct throb*)trx->modem;
		if (s->nSymIdle) {
			s->nSymIdle  = 0;
			s->nSymSpace = 1;
		} else {
			s->nSymIdle  = 1;
			s->nSymSpace = 0;
		}
	}
}

void throb_reset_syms(Trx *trx)
{
	struct throb *s = (struct throb*)trx->modem;
	if (trx->mode == MODE_THROB) {
		s->nSymIdle  = 0;
		s->nSymSpace = 44;
	} else {
		s->nSymIdle  = 0;
		s->nSymSpace = 1;
	}
}
