/*
 *    rttytx.c  --  RTTY modulator
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "../main/trx.h"
#include "../main/sndmodem.h"
#include "../main/trxctrl.h"
#include "../misc/fixedpoint.h"
#include "../misc/mixer.h"

#include "rtty.h"
#include "baudot.h"
#include "rttypar.h"


static void send_symbol(Trx *trx, BOOL symbol)
{
	struct rtty *s			 = (struct rtty*)trx->modem;
	BOOL		 sym		 = trx->reverse ? (! symbol) : symbol;

	if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202 || g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER)
	{
		trxctrl_ats3_sendtone(&g_TrxCtrl, trx->pModem, sym ? 0 : 1, s->symbollen);
	}
	else
	{
		int			 iFrequency  = trx->nTxFrequency + ((sym ? - s->shift : s->shift) >> 1);
		unsigned	 iPhaseDelta = trx_freq2phase(iFrequency);
		int			 i;

		for (i = 0; i < s->symbollen; ++ i) {
			trx->pModem->pOutBuf[i] = mixer_osc_16(s->phaseacc) >> 1;
			s->phaseacc += iPhaseDelta;
		}

		sndmodem_write(trx->pModem, trx->pModem->pOutBuf, s->symbollen);
	}
}

static void send_stop(Trx *trx)
{
	struct rtty *s = (struct rtty *) trx->modem;

	if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202 || g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER)
	{
		trxctrl_ats3_sendtone(&g_TrxCtrl, trx->pModem, trx->reverse ? 1 : 0, s->stoplen);
	}
	else
	{
		int			 iFrequency  = trx->nTxFrequency + ((trx->reverse ? s->shift : - s->shift) >> 1);
		unsigned	 iPhaseDelta = trx_freq2phase(iFrequency);
		int			 i;

		for (i = 0; i < s->stoplen; ++ i) {
			trx->pModem->pOutBuf[i] = mixer_osc_16(s->phaseacc) >> 1;
			s->phaseacc += iPhaseDelta;
		}

		sndmodem_write(trx->pModem, trx->pModem->pOutBuf, s->stoplen);
	}
}

static void send_char(Trx *trx, int c)
{
	struct rtty *s = (struct rtty *) trx->modem;
	int			 i, j;

	if (s->nbits == 5) {
		if (c == BAUDOT_LETS)
			c = 0x1F;
		else if (c == BAUDOT_FIGS)
			c = 0x1B;
	}

	/* start bit */
	send_symbol(trx, 0);

	/* data bits */
	for (i = 0; i < s->nbits; i++) {
		j = (s->msb) ? (s->nbits - 1 - i) : i;
		send_symbol(trx, (c >> j) & 1);
	}

	/* parity bit */
	if (s->parity != RTTY_PARITY_NONE)
		send_symbol(trx, rtty_parity(c, s->nbits, s->parity));

	/* stop bit(s) */
	send_stop(trx);

	if (s->nbits == 5)
		c = baudot_dec(&s->nBaudotRxMode, c, s->bUnshiftOnSpace);

	if (c > -1)
		trx_put_echo_char(trx, c);
}

static void send_idle(Trx *trx)
{
	struct rtty *s = (struct rtty *) trx->modem;

	if (s->nbits == 5)
		send_char(trx, BAUDOT_LETS);
	else
		send_char(trx, 0);
}

int rtty_txprocess(Trx *trx)
{
	struct rtty *s = (struct rtty *) trx->modem;
	int			 c = -1;

	if (s->preamble > 0) {
		send_symbol(trx, 1);
		s->preamble--;
		return 0;
	}

	c = trx_get_tx_char(trx);

	/* TX buffer empty */
	if (c == -1) {
		/* stop if requested to... */
		if (trx->stopflag)
			return -1;
		/* send idle character */
		send_idle(trx);
		s->nBaudotTxMode = BAUDOT_LETS;
		return 0;
	}

	if (s->nbits != 5) {
		send_char(trx, c);
		return 0;
	}

	/* unshift-on-space */
	if (c == ' ') {
		if (s->bUnshiftOnSpace) {
			send_char(trx, BAUDOT_LETS);
			s->nBaudotTxMode = BAUDOT_LETS;
		}
		send_char(trx, 0x04);
		return 0;
	}

	if ((c = baudot_enc(c)) < 0)
		return 0;

	/* switch case if necessary */
	if ((c & s->nBaudotTxMode) == 0) {
		if (s->nBaudotTxMode == BAUDOT_FIGS) {
			send_char(trx, BAUDOT_LETS);
			s->nBaudotTxMode = BAUDOT_LETS;
		} else {
			send_char(trx, BAUDOT_FIGS);
			s->nBaudotTxMode = BAUDOT_FIGS;
		}
	}

	send_char(trx, c & 0x1F);

	return 0;
}
