/*
 *    rttyrx.c  --  RTTY demodulator
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <math.h>

#include "../main/trx.h"
#include "../misc/fixedpoint.h"
#include "../misc/filter.h"
#include "../misc/mixer.h"
#include "../misc/misc.h"

#include "rtty.h"
#include "baudot.h"
#include "rttypar.h"

#ifdef _WIN32_WCE
	#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

static unsigned char bitreverse(unsigned char in, int n)
{
	unsigned char out = 0;
	int i;

	for (i = 0; i < n; i++)
		out = (out << 1) | ((in >> i) & 1);

	return out;
}

static int decode_char(struct rtty *s)
{
	unsigned int parbit, par, data;

	parbit = (s->rxdata >> s->nbits) & 1;
	par = rtty_parity(s->rxdata, s->nbits, s->parity);

	if (s->parity != RTTY_PARITY_NONE && parbit != par) {
//		fprintf(stderr, "P");
		return 0;
	}

	data = s->rxdata & ((1 << s->nbits) - 1);

	if (s->msb)
		data = bitreverse(data, s->nbits);

	if (s->nbits == 5)
		return baudot_dec(&s->nBaudotRxMode, data, s->bUnshiftOnSpace);

	return data;
}

static int rttyrxbuf(Trx *trx)
{
	struct rtty *s = (struct rtty *) trx->modem;
	int          i = s->symbufpos;
	int          j, k;
	int			 symlen = 16;
	BOOL		 prev   = 0;
	
	// waiting for start bit
	if (s->symbuf[i]) // start bit
		return FALSE;

	// skip to the middle of start bit
	i += (symlen >> 1);
	if (i >= s->symbufsz)
		i -= s->symbufsz;

	if (s->symbuf[i]) // start bit?
		return FALSE;

	s->rxdata = 0;
	for (j = 0; j < s->nbits; ++ j) {
		// DPLL on symbol change
		for (k = 1; k <= symlen; ++ k) {
			if (++ i == s->symbufsz)
				i = 0;
			if (s->symbuf[i] != prev) {
				// signum change
				i += symlen - k;
				if ((k << 1) == symlen) {
					// right on time
				} else if ((k << 1) < symlen) {
					// early
					i -= 1;
				} else {
					// late
					i += 1;
				}
				break;
			}
		}
		if (i >= s->symbufsz)
			i -= s->symbufsz;
		prev = s->symbuf[i];
		if (prev)
			s->rxdata |= 1 << j;
	}

	i += symlen;
	if (i >= s->symbufsz)
		i -= s->symbufsz;

	// Stop bit. Ignore first framing error.
	if (s->symbuf[i]) {
		s->bFramingError = FALSE;
		return TRUE;
	}
	if (s->bFramingError)
		return FALSE;
	s->bFramingError = TRUE;
	return TRUE;
}

int rtty_rxprocess(Trx *trx, short *buf, int len)
{
	struct rtty *s	    = (struct rtty*)trx->modem;
	const int    nShift = s->shift >> 1;
	cmplx		 z, z2;
	int			 iPhase;
	int			 iPhaseFiltered;
	int          iAmplitude, iAmplitude2;
	int			 bit;

	while (len-- > 0) {
		// Mix with the internal NCO
		mixer_mix(*buf, s->phaseacc,  &z);
		mixer_mix(*buf, s->phaseacc2, &z2);
		++ buf;
		s->phaseacc  += trx_freq2phase(trx->nRxFrequency + nShift);
		s->phaseacc2 += trx_freq2phase(trx->nRxFrequency - nShift);
		z.re  = z.re  >> 15;
		z.im  = z.im  >> 15;
		z2.re = z2.re >> 15;
		z2.im = z2.im >> 15;

		// Lowpass filter
		filter_run(s->rxfilter, z, &z);
		if (! filter_run(s->rxfilter2, z2, &z2))
			continue;
		z.re  >>= 16;
		z.im  >>= 16;
		z2.re >>= 16;
		z2.im >>= 16;

		iAmplitude  = isqrt(z.re * z.re + z.im * z.im);
		iAmplitude2 = isqrt(z2.re * z2.re + z2.im * z2.im);
		_ASSERT(iAmplitude <= z.re * z.re + z.im * z.im);
		_ASSERT(iAmplitude2 <= z2.re * z2.re + z2.im * z2.im);
		iAmplitude -= iAmplitude2;
		filter_I_run(s->rxbitfir, iAmplitude, &iAmplitude);
		bit	= trx->reverse ? (iAmplitude > 0) :  (iAmplitude < 0);

		s->symbuf[s->symbufpos] = bit;
		if (++ s->symbufpos == s->symbufsz)
			s->symbufpos = 0;

		if (++ s->symbuflen == s->symbufsz) {
			// symbol received
			if (rttyrxbuf(trx)) {
				int c = decode_char(s);
				if (c > -1) // printable
					trx_put_rx_char(trx, c);
				s->symbufpos = 0;
				s->symbuflen = 0;
			} else
				-- s->symbuflen;
		}
	}

	return 0;
}
