/*
 *    rtty.c  --  RTTY modem
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>
#include <stdio.h>

#include "../main/trx.h"
#include "../misc/fixedpoint.h"
#include "../misc/filter.h"
#include "../misc/mixer.h"
#include "../main/sndmodem.h"
#include "../main/trxctrl.h"
#include "../resource.h"
#include "rtty.h"
#include "baudot.h"

static void rtty_txinit(Trx *trx)
{
    struct rtty *r = (struct rtty *) trx->modem;

	r->nBaudotRxMode = BAUDOT_LETS;
	r->nBaudotTxMode = BAUDOT_LETS;

	/* start each transmission with 440ms of MARK tone */
	r->preamble   = 20;

	if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202 || g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER) {
		int f0 = (int)trx->nTxFrequency - (r->shift >> 1);
		if (f0 < 0)
			f0 = 0;
		trxctrl_ats3_start_mfsk(&g_TrxCtrl, trx->pModem, f0, r->shift);
	}
}

static void rtty_free(struct rtty *s)
{
	filter_free(s->rxfilter);
	filter_free(s->rxfilter2);
	filter_free(s->rxbitfir);
	free(s);
}

static void rtty_rxinit(Trx *trx)
{
    struct rtty *r = (struct rtty*)trx->modem;
	r->nBaudotRxMode = BAUDOT_LETS;
	r->nBaudotTxMode = BAUDOT_LETS;
	r->symbufpos  = 0;
	r->symbuflen  = 0;
	r->bFramingError = FALSE;
}

static void rtty_destructor(Trx *trx)
{
	struct rtty  *s = (struct rtty*)trx->modem;
	rtty_free(s);
    trx->modem		= 0;
    trx->txinit		= 0;
    trx->rxinit		= 0;
    trx->txprocess	= 0;
    trx->rxprocess	= 0;
    trx->destructor = 0;
}

void rtty_init(Trx *trx)
{
	struct rtty *s;
	double		 dBaud = 45.45;
	double		 fhi;
	int			 nSubsampling;

	if ((s = calloc(1, sizeof(struct rtty))) == NULL)
		return;

	s->nbits = 5;
	s->parity = RTTY_PARITY_NONE;

	switch (trx->submode) {
	case ID_SUBMODE_RTTY_HAM_45_170:
		s->shift			= 170 << FREQ_FRAC_BITS;
		s->bUnshiftOnSpace	= TRUE;
		nSubsampling		= 11;
		break;
	case ID_SUBMODE_RTTY_HAM_45_200:
		s->shift			= 200 << FREQ_FRAC_BITS;
		s->bUnshiftOnSpace	= TRUE;
		nSubsampling		= 11;
		break;
	case ID_SUBMODE_RTTY_DWD_50_85:
		dBaud				= 50.0;
		s->shift			= 85 << FREQ_FRAC_BITS;
		nSubsampling		= 10;
		s->bUnshiftOnSpace	= FALSE;
		break;
	case ID_SUBMODE_RTTY_DWD_50_450:
		dBaud				= 50.0;
		s->shift			= 450 << FREQ_FRAC_BITS;
		nSubsampling		= 10;
		s->bUnshiftOnSpace	= FALSE;
		break;
	default:
		trx->submode		= ID_SUBMODE_RTTY_HAM_45_170;
		s->shift			= 170 << FREQ_FRAC_BITS;
		nSubsampling		= 11;
		s->bUnshiftOnSpace	= TRUE;
		break;
	}

	s->symbollen	= (int) ((double)SampleRate / dBaud + 0.5);
	s->stoplen		= (int) (1.5 * (double)SampleRate / dBaud + 0.5);
	s->msb			= FALSE;
	s->symbufsz     = s->symbollen * (s->nbits + 2) / nSubsampling;

	fhi = (dBaud * 0.6) / (float)SampleRate;
	s->rxfilter  = filter_init_lowpass(127, nSubsampling, fhi);
	s->rxfilter2 = filter_init_lowpass(127, nSubsampling, fhi);
	s->rxbitfir  = filter_init_lowpass(64,  1, 0.6 * dBaud * nSubsampling / (float)SampleRate);
	trx->bandwidth  = s->shift;

    trx->modem		= s;
	trx->txinit		= rtty_txinit;
	trx->rxinit		= rtty_rxinit;
	trx->txprocess	= rtty_txprocess;
	trx->rxprocess	= rtty_rxprocess;
	trx->destructor = rtty_destructor;
}
