/*
 *    psk31tx.c  --  PSK31 modulator
 *
 *    Copyright (C) 2001, 2002, 2003, 2004
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <ctype.h>

#include "psk31.h"
#include "varicode.h"
#include "../misc/mixer.h"
#include "../main/sndmodem.h"
#include "../main/trxctrl.h"

static void send_symbol_raw(Trx *trx, int re1, int im1, int re2, int im2)
{
	struct psk31 *s			  = (struct psk31*)trx->modem;
	unsigned	  iPhaseDelta = trx_freq2phase(trx->nTxFrequency);
	int			  i;

	for (i = 0; i < s->symbollen; i++) {
		if (1) {
			int ival = (s->txshape[i] * re1 + (32767 - s->txshape[i]) * re2);
			int qval = (s->txshape[i] * im1 + (32767 - s->txshape[i]) * im2);
			trx->pModem->pOutBuf[i] = mixer_mix2(ival, qval, s->phaseacc) >> 16;
		} else {
			int ival = (re1 == re2) ? (32767 * re1) : 
				(s->txshape[i] * re1 + s->txshape2[i] * re2);
	//			(s->txshape[i] * re1 + (32767 - s->txshape[i]) * re2);
			int qval = (im1 == im2) ? (32767 * im1) : 
				(s->txshape[i] * im1 + s->txshape2[i] * im2);
	//			(s->txshape[i] * im1 + (32767 - s->txshape[i]) * im2);
			trx->pModem->pOutBuf[i] = mixer_mix2(ival, qval, s->phaseacc) >> 16;
		}
		s->phaseacc += iPhaseDelta;
	}

	sndmodem_write(trx->pModem, trx->pModem->pOutBuf, s->symbollen);
}

static void send_symbol(Trx *trx, int sym)
{
	struct psk31 *s = (struct psk31 *) trx->modem;

	if (s->qpsk && trx->reverse)
		sym = (4 - sym) & 3;

	if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202)
	{
		if (sym == 2) { // 0 degrees
			sndmodem_write_bell202_pause(trx->pModem, s->symbollen);
		} else {
			int iSpace = (s->symbollen * 3 - 200) >> 1;
			// add phase difference
			s->prevsymbol.re += (sym == 0) ? 2 : ((sym == 1) ? 3 : 1);
			s->prevsymbol.re &= 3;
			sndmodem_write_bell202_async(trx->pModem, 0x01, 3);	// TX off, 100 thirds
			sndmodem_write_bell202_pause_thirds(trx->pModem, iSpace);
			sndmodem_write_bell202_async(trx->pModem, 0x02 + s->prevsymbol.re, 3); // TX on, 100 thirds
			sndmodem_write_bell202_pause_thirds(trx->pModem, iSpace);
		}
	} else if (g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER) {
		if (sym == 2) { // 0 degrees
			sndmodem_write_manchester_pause(trx->pModem, s->symbollen);
		} else {
			int iSpace = (s->symbollen - 40) >> 1;
			// add phase difference
			s->prevsymbol.re += (sym == 0) ? 2 : ((sym == 1) ? 3 : 1);
			s->prevsymbol.re &= 3;
			sndmodem_write_manchester_async(trx->pModem, 0x01, 3);	// TX off, 5 bits * 4 = 20 samples
			sndmodem_write_manchester_pause(trx->pModem, iSpace);
			sndmodem_write_manchester_async(trx->pModem, 0x02 + s->prevsymbol.re, 3); // TX on, 5 bits * 4 = 20 samples
			sndmodem_write_manchester_pause(trx->pModem, iSpace);
		}
	} else
	{
		cmplx         z, symbol;

		/* differential QPSK modulation - top bit flipped */
		switch (sym) {
		case 0:
			c_re(z) = -1;	/* 180 degrees */
			c_im(z) =  0;
			break;
		case 1:
			c_re(z) =  0;	/* 270 degrees */
			c_im(z) = -1;
			break;
		case 2:
			c_re(z) =  1;	/*   0 degrees */
			c_im(z) =  0;
			break;
		case 3:
			c_re(z) =  0;	/*  90 degrees */
			c_im(z) =  1;
			break;
		}

		c_re(symbol) = c_re(s->prevsymbol) * c_re(z) - c_im(s->prevsymbol) * c_im(z);
		c_im(symbol) = c_re(s->prevsymbol) * c_im(z) + c_im(s->prevsymbol) * c_re(z);

		send_symbol_raw(trx, s->prevsymbol.re, s->prevsymbol.im, symbol.re, symbol.im);

		/* save the current symbol */
		s->prevsymbol = symbol;
	}
}

static void send_bit(Trx *trx, int bit)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	unsigned int sym;

	if (s->qpsk)
		sym = encoder_encode(s->enc, bit);
	else
		sym = bit << 1;

	send_symbol(trx, sym);
}

static void send_char(Trx *trx, unsigned char c)
{
	char *code;

	code = psk_varicode_encode(c);

	while (*code) {
		send_bit(trx, (*code - '0'));
		code++;
	}

	send_bit(trx, 0);
	send_bit(trx, 0);
}

static void flush_tx(Trx *trx)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	int i;

	if (s->qpsk) {
		// flush the encoder (QPSK only)
		send_bit(trx, 0);
		send_bit(trx, 0);
		send_bit(trx, 0);
		send_bit(trx, 0);
		send_bit(trx, 0);
	}

	// DCD off sequence (unmodulated carrier)
	for (i = 0; i < 31; i++)
		send_symbol(trx, 2);

	if (g_TrxCtrl.nType != TRXCTRL_ATS3_BELL202 && g_TrxCtrl.nType != TRXCTRL_ATS3_MANCHESTER)
		send_symbol_raw(trx, s->prevsymbol.re, s->prevsymbol.im, 0, 0); // output value to 0 without click
}

int psk31_txprocess(Trx *trx)
{
	struct psk31   *s = (struct psk31 *) trx->modem;
	int				c = -1;

	if (s->preamble > 0) {
		s->preamble --;
		if (s->preamble == 31)
			send_symbol_raw(trx, 0, 0, 1, 0);
		else
			send_symbol(trx, 0);	/* send phase reversals */
		return 0;
	}

	c = trx_get_tx_char(trx);

	/* TX buffer empty or TX was killed */
	if (c == -1) {
		/* stop if requested to... */
		if (trx->stopflag) {
			flush_tx(trx);
			return -1;	/* we're done */
		}

		send_symbol(trx, 0);	/* send phase reversals */
		send_symbol(trx, 0);
		return 0;
	}

	send_char(trx, c);
	trx_put_echo_char(trx, c);

	return 0;
}
