// MFSK trasnmitter and receiver code, Pawel Jalocha, December 2004

#include <stdlib.h>
#include <stdio.h>

#include "../main/trx.h"
#include "../misc/cmplx.h"
#include "../misc/fixedpoint.h"
#include "../misc/mixer.h"
#include "../main/sndmodem.h"
#include "../main/trxctrl.h"
#include "../misc/misc.h"

#include "olivia.h"
#include "mfsk.h"
#include "rttym.h"

// Inverse Fast Hadamard Transform
static void IFHT(char *Data, size_t Len)
{  
	char Bit1, Bit2, NewBit1, NewBit2;
	for (size_t Step = (Len >> 1); Step; Step >>= 1) {
		for (size_t Ptr = 0; Ptr < Len; Ptr += (Step << 1)) { 
			for (size_t Ptr2 = Ptr; Ptr2 - Ptr < Step; ++ Ptr2) {
				Bit1			= Data[Ptr2];
				Bit2			= Data[Ptr2 + Step];
				NewBit1			= Bit1;
				NewBit1		   -= Bit2;
				NewBit2			= Bit1;
				NewBit2		   += Bit2;
				Data[Ptr2]		= NewBit1;
				Data[Ptr2+Step]	= NewBit2;
			}
		}
	}
}

int MFSK_Transmitter::Preset()
{
    // preset the input character buffer
    Input.Len = 1024;
	if (Input.Preset() < 0)
		goto Error;
    Monitor.Len = 256;
	if (Monitor.Preset() < 0)
		goto Error;

    // preset the encoder
	if (ReallocArray(&FHT_Buffer, m_pModem->nSymbolsPerBlock) < 0)
		goto Error;
    if (ReallocArray(&OutputBlock, m_pModem->nSymbolsPerBlock) < 0)
		goto Error;

    // preset the modulator
	SymbolShape = olivia_symbolshape_int(m_pModem->nSymbolLen, 0.7);
	if (SymbolShape == 0)
		goto Error;
    if (ReallocArray(&OutTap, m_pModem->nSymbolLen) < 0)
		goto Error;
	memset(OutTap, 0, sizeof(int*) * m_pModem->nSymbolLen);
	TapPtr		= 0;
    SymbolPhase	= 0;

    // reset the state logic
    SymbolPtr	= 0;
    State		= 0;
    return 0;

Error: 
	Free(); 
	return -1;
}

void MFSK_Transmitter::Reset()
{ 
	Input.Reset();
	Monitor.Reset();
	SymbolPtr=0;
    State=0;
}

void MFSK_Transmitter::Free()
{ 
	Input.Free();
    Monitor.Free();
	free(SymbolShape);
	SymbolShape = 0;
	free(OutTap);
	OutTap = 0;
	free(FHT_Buffer);  
	FHT_Buffer  = 0;
	free(OutputBlock);
	OutputBlock = 0;
}

int MFSK_Transmitter::Output(Trx *trx)
{ 
	if (SymbolPtr == 0) { 
		if ((State & State_StopReq) && Input.Empty()) { 
			State = 0;
		} else if (State & State_Running) {
			for (size_t i = 0; i < m_pModem->nBitsPerSymbol; ++ i) {
				UCHAR Char;
				if (Input.Read(Char) <= 0)
					break;
				InputBlock[i] = Char;
				if (m_pModem->nBitsPerCharacter == 5) {
					int c = rttym_dec(&m_pModem->nBaudotRxMode, Char);
					if (c > -1) {
						Char = (UCHAR)c;
						Monitor.Write(Char);
					}
				} else
					Monitor.Write(Char);				
			}
			if (i < m_pModem->nBitsPerSymbol) {
				// fill in idle symbols
				m_pModem->nBaudotTxMode = RTTYM_LETS;
				for (; i < m_pModem->nBitsPerSymbol; ++ i)
					InputBlock[i] = 0;
			}
			EncodeBlock(InputBlock);
		}
	}

	if (State & State_Running) {
		ModulatorSend(trx, OutputBlock[SymbolPtr]);
		if (++ SymbolPtr == m_pModem->nSymbolsPerBlock)
			SymbolPtr = 0;
	}

	size_t SymbolSepar = m_pModem->nSymbolLen >> 1;
	if (g_TrxCtrl.nType != TRXCTRL_ATS3_BELL202 && g_TrxCtrl.nType != TRXCTRL_ATS3_MANCHESTER) {
		// send to the sound card
		for (size_t i = 0; i < SymbolSepar; ++ i) {
			trx->pModem->pOutBuf[i] = OutTap[TapPtr];
			OutTap[TapPtr] = 0;
			++ TapPtr;
			TapPtr &= (m_pModem->nSymbolLen - 1);
		}
		sndmodem_write(trx->pModem, trx->pModem->pOutBuf, SymbolSepar);
	}
    return SymbolSepar;
}

void MFSK_Transmitter::ModulatorSend(Trx *trx, UCHAR Symbol)
{
	int iTone		= Symbol ^ (Symbol >> 1); // gray encoding
	if (trx->reverse)
		iTone = m_pModem->nTones - iTone - 1;

	if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202 || g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER)
	{
		trxctrl_ats3_sendtone(&g_TrxCtrl, trx->pModem, iTone, m_pModem->nSymbolLen >> 1);
	}
	else
	{
		int iFreqLow = (int)trx->nTxFrequency - 4000 * CarrierSepar * ((m_pModem->nTones - 1) << FREQ_FRAC_BITS) / m_pModem->nSymbolLen;
		if (iFreqLow < 0)
			iFreqLow = 0;
		UINT iPhaseDelta = trx_freq2phase(iFreqLow + 8000 * CarrierSepar * (iTone << FREQ_FRAC_BITS) / m_pModem->nSymbolLen);
		for (int Time = 0; Time < m_pModem->nSymbolLen; ++ Time) {
			OutTap[TapPtr ++] += (mixer_osc_16(SymbolPhase) * SymbolShape[Time]) >> 16;
			TapPtr &= m_pModem->nSymbolLen - 1;
			SymbolPhase  += iPhaseDelta;
		}
		SymbolPhase += 0x40000000; // + PI/2
		iRand = 69069 * iRand + 362437;
		if (iRand & 1)
			SymbolPhase += 0x80000000; // + PI
	}
}

void MFSK_Transmitter::EncodeCharacter(UCHAR Char)
{
	if (m_pModem->nBitsPerCharacter == 7) {
		// Olivia
		if (Char > 0x7f)
			Char = '?';
	} else if (m_pModem->nBitsPerCharacter == 6) {
		// Contestai
		// lower case to upper case
		if (Char >= 'a' && Char <= 'z')
			Char += 'A' - 'a';
		if (Char == ' ')		// space
			Char = 59;
		else if (Char == '\r')	// return
			Char = 60;
		else if (Char >= 33 && Char <= 90)
			Char -= 32;
		else if (Char == 8)		// backspace
			Char = 61;
		else if (Char == 0) // idle character
			Char = 0; // encode idle character as a space
		else
			Char = '?' - 32;
	} else if (m_pModem->nBitsPerCharacter == 5) {
		// RTTYM
		// Nothing to do, the BAUDOT encoding has been done in olivia_txprocess already.
	}
	memset(FHT_Buffer, 0, m_pModem->nSymbolsPerBlock);
    if (Char < m_pModem->nSymbolsPerBlock)
		FHT_Buffer[Char] = 1;
    else
		FHT_Buffer[Char - m_pModem->nSymbolsPerBlock] = -1;
	IFHT(FHT_Buffer, m_pModem->nSymbolsPerBlock);
}

void MFSK_Transmitter::ScrambleFHT(size_t CodeOffset)
{
    size_t CodeWrap = m_pModem->nSymbolsPerBlock - 1;
    size_t CodeBit  = CodeOffset & CodeWrap;
	unsigned __int64 nScramblingCode = (m_pModem->nBitsPerCharacter == 7) ? ScramblingCodeOlivia : ScramblingCodeContestia;
    for (int TimeBit = 0; TimeBit < m_pModem->nSymbolsPerBlock; ++ TimeBit) {
		unsigned __int64 CodeMask = 1; CodeMask <<= CodeBit;
        if (nScramblingCode & CodeMask)
			FHT_Buffer[TimeBit] = - FHT_Buffer[TimeBit];
		++ CodeBit;
		CodeBit &= CodeWrap;
	}
}

void MFSK_Transmitter::EncodeBlock(UCHAR *InputBlock)
{
	memset(OutputBlock, 0, m_pModem->nSymbolsPerBlock);
	size_t nShift = (m_pModem->nBitsPerCharacter == 7) ? 13 : 5; // Olivia or Contestia/RTTYM
	for (size_t FreqBit = 0; FreqBit < m_pModem->nBitsPerSymbol; ++ FreqBit) {
		EncodeCharacter(InputBlock[FreqBit]);
        ScrambleFHT(FreqBit * nShift);
		size_t Rotate = 0;
		for (int TimeBit = 0; TimeBit < m_pModem->nSymbolsPerBlock; ++ TimeBit) {
			if (FHT_Buffer[TimeBit] < 0) {
				size_t Bit = FreqBit + Rotate;
				if (Bit >= m_pModem->nBitsPerSymbol)
					Bit -= m_pModem->nBitsPerSymbol;
				OutputBlock[TimeBit] |= 1 << Bit;
			}
			if (++ Rotate >= m_pModem->nBitsPerSymbol)
				Rotate -= m_pModem->nBitsPerSymbol;
		}
	}
}
