// MFSK trasnmitter and receiver code, Pawel Jalocha, December 2004

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <float.h>

#include "../main/trx.h"
#include "../misc/cmplx.h"
#include "../misc/fixedpoint.h"
#include "../misc/mixer.h"
#include "../misc/misc.h"
#include "../misc/int_fft.h"

#include "olivia.h"
#include "mfsk.h"
#include "struc.h"
#include "lowpass3.h"
#include "fifo.h"
#include "rttym.h"

#ifdef _WIN32_WCE
	#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

/********************************************************* MFSK_Receiver *********************************************************/

MFSK_Receiver::MFSK_Receiver(struct olivia *s) :
	m_pModem(s),
	Decoder(0), 
	DecodePipe(0),
    SyncMargin(8),
    SyncIntegLen(4),
	SyncThreshold(3),
	InpTap(0),
	SymbolShape(0),
    FirstCarrier(32),
	DecodeWidth(0),
	bReverse(FALSE)
{
    Energy[0]=0;
	Energy[1]=0;
}

void MFSK_Receiver::Free()
{ 
	delete[] Decoder;
	Decoder = 0;

    if (DecodePipe) { 
		for(size_t i = 0; i < BlockPhases; ++ i)
           DecodePipe[i].Free();
			free(DecodePipe); 
			DecodePipe = 0;
	}

    InputBuffer.Free();
    SyncSignal.Free();
    SyncNoiseEnergy.Free();
    Output.Free();

	free(InpTap);		InpTap	  = 0;
	free(SymbolShape);	SymbolShape = 0;
	free(Energy[0]);	Energy[0] = 0;
    free(Energy[1]);	Energy[1] = 0;
}

int MFSK_Receiver::Preset()
{ 
	size_t i;

    if(SyncMargin<2) SyncMargin=2;

    if(SyncIntegLen<2) SyncIntegLen=2;

    if (SyncThreshold < 3) SyncThreshold = 3;

	// Demodulator
	FirstCarrier	= (m_pModem->nSymbolLen >> 4) + 1;
	int Carriers	= 1 << m_pModem->nBitsPerSymbol;
    
    if (ReallocArray(&InpTap, m_pModem->nSymbolLen) < 0)
		goto Error;
    ClearArray(InpTap, m_pModem->nSymbolLen);
    InpTapPtr = 0;

	SymbolShape = olivia_symbolshape_int(m_pModem->nSymbolLen, 0.0);
	if (SymbolShape == 0)
		goto Error;

    DecodeWidth = (Carriers * CarrierSepar - 1) + (SyncMargin << 1);
    if (ReallocArray(&Energy[0], DecodeWidth) < 0)
		goto Error;
    if (ReallocArray(&Energy[1], DecodeWidth) < 0)
		goto Error;

    WindowLen	= 16 * m_pModem->nSymbolLen;

    if (Decoder)
		for (i = 0; i < SlicesPerSymbol * FreqOffsets; ++ i)
			Decoder[i].Free();

    if (DecodePipe) { 
		for(i=0; i<BlockPhases; i++)
           DecodePipe[i].Free();
    }

    FreqOffsets = 2 * SyncMargin + 1;
    BlockPhases = SlicesPerSymbol * m_pModem->nSymbolsPerBlock;

	Decoder = new MFSK_SoftDecoder[SlicesPerSymbol * FreqOffsets];
    if (Decoder == 0)
		goto Error;
	for (i = 0; i < SlicesPerSymbol * FreqOffsets; ++ i)
		if (Decoder[i].Preset(m_pModem) < 0)
			goto Error;

    if (ReallocArray(&DecodePipe,BlockPhases) < 0)
		goto Error;
    for (i = 0; i < BlockPhases; ++ i)
		DecodePipe[i].Init();
    for (i = 0; i < BlockPhases; ++ i) {
		DecodePipe[i].Width=FreqOffsets;
        DecodePipe[i].Len=SyncIntegLen;
        if(DecodePipe[i].Preset()<0) 
			goto Error;
        DecodePipe[i].Clear(); 
	}

    SyncSignal.Width=FreqOffsets;
    SyncSignal.Len=BlockPhases;
    if(SyncSignal.Preset()<0) 
		goto Error;
    SyncSignal.Clear();

    SyncNoiseEnergy.Width=FreqOffsets;
    SyncNoiseEnergy.Len=BlockPhases;
    if(SyncNoiseEnergy.Preset()<0) 
		goto Error;
    SyncNoiseEnergy.Clear();

//    SyncFilterWeight = 1.0f / SyncIntegLen;
    BlockPhase=0;
    SyncBestSignal=0;
    SyncBestBlockPhase=0;
    SyncBestFreqOffset=0;
    SyncSNR=0;

    if (InputBuffer.EnsureSpace(WindowLen + 2048) < 0)
		goto Error;

    Output.Len = 1024;
    if (Output.Preset() < 0)
		goto Error;

    return 0;

Error:
	Free();
	return -1;
}

void MFSK_Receiver::Reset()
{ 
	size_t i;

    InputBuffer.Clear();

    for (i=0; i<(SlicesPerSymbol*FreqOffsets); i++)
		Decoder[i].Reset();

    for (i=0; i<BlockPhases; i++)
		DecodePipe[i].Clear();

	SyncSignal.Clear();
    SyncNoiseEnergy.Clear();

    BlockPhase=0;
    SyncBestSignal=0;
    SyncBestBlockPhase=0;
    SyncBestFreqOffset=0;
    SyncSNR=0;
    Output.Reset();
}

/*
void MFSK_Receiver::PrintParameters()
{
	printf("MFSK_Receiver parameters:\n");

    float FreqBin = (float)SampleRate/Demodulator.SymbolLen;
    printf("%d bits/symbol(tone), %d tones, %4.2f Hz/tone, %4.1f Hz bandwidth\n",
		Demodulator.BitsPerSymbol,
        Demodulator.Carriers,
        FreqBin*CarrierSepar,
        FreqBin*CarrierSepar*Demodulator.Carriers);

    printf("%4.2f baud, FEC block: %d char. => %dx%d bits, %3.1f sec\n",
        SampleRate/Demodulator.SymbolSepar,
        BitsPerSymbol, BitsPerSymbol, SymbolsPerBlock,
        (SymbolsPerBlock*Demodulator.SymbolSepar)/SampleRate);

    printf("Audio band: %3.1f - %3.1f Hz, %3.1f Hz total\n",
		FreqBin*Demodulator.FirstCarrier,
        FreqBin*(Demodulator.FirstCarrier+CarrierSepar*Demodulator.Carriers),
        FreqBin*CarrierSepar*Demodulator.Carriers );

	printf("Tuning tolerance = +/- %3.1f Hz, Sync. S/N threshold = %3.1f\n",
		FreqBin*SyncMargin, SyncThreshold);
}
*/

int MFSK_Receiver::Process(short *Input, size_t InputLen)
{
	InputBuffer.EnsureSpace(InputBuffer.Len + InputLen);
	for (UINT i = 0; i < InputLen;)
		InputBuffer.Elem[InputBuffer.Len ++] = Input[i ++];
	ProcessInputBuffer();
	return 0; 
}

void MFSK_Receiver::Flush()
{ 
	ProcessInputBuffer();
    size_t i;

	for (i = InputBuffer.Len; i < WindowLen; ++ i)
		InputBuffer[i] = 0;
	InputBuffer.Len = WindowLen;
    ProcessInputBuffer();

	for (i = 0; i < WindowLen; ++ i)
		InputBuffer[i] = 0;
	size_t FlushLen = m_pModem->nSymbolLen * m_pModem->nSymbolsPerBlock * SyncIntegLen;
    for (i = 0; i < FlushLen; i += WindowLen) {
		InputBuffer.Len = WindowLen;
	    ProcessInputBuffer();
	}
}

void MFSK_Receiver::ProcessInputBuffer()
{
	while (InputBuffer.Len >= WindowLen) {
        for (size_t i = 0; i < WindowLen; i += (m_pModem->nSymbolLen >> 1))
			ProcessSymbol(InputBuffer.Elem + i);
        InputBuffer.Delete(0, WindowLen);
	}
}

void MFSK_Receiver::ProcessSymbol(short *Input)
{
	// fft of two consequent halfsymbols, compute energies
	Demodulate(Input);
 
	MFSK_SoftDecoder *DecoderPtr = Decoder;

	for (size_t Slice = 0; Slice < SlicesPerSymbol; ++ Slice, ++ BlockPhase) {
		AvgFilter				*NoiseEnergyPtr  = SyncNoiseEnergy.AbsPtr(BlockPhase);
		AvgFilter				*SignalPtr		 = SyncSignal.AbsPtr(BlockPhase);
		unsigned __int64		*DecodeBlockPtr  = DecodePipe[BlockPhase].CurrPtr();
		int						 BestSliceSignal = 0;
		size_t					 BestSliceOffset = 0;
		short					 Symbol[8];

		for (size_t Offset = 0; Offset < FreqOffsets; ++ Offset, ++ DecoderPtr, ++ DecodeBlockPtr, ++ NoiseEnergyPtr, ++ SignalPtr) {
			SoftDecode(Symbol, Slice, Offset - (FreqOffsets >> 1));

			DecoderPtr->Input(Symbol);
			DecoderPtr->Process();
			DecoderPtr->Output(*DecodeBlockPtr);

			NoiseEnergyPtr->Process(DecoderPtr->NoiseEnergy); //, SyncFilterWeight);
			SignalPtr->Process(DecoderPtr->Signal); //, SyncFilterWeight);

			if (SignalPtr->Output > BestSliceSignal) {
				BestSliceSignal = SignalPtr->Output;
				BestSliceOffset = Offset; 
			}
		}
		DecodePipe[BlockPhase] += 1;

		if (BlockPhase == SyncBestBlockPhase) {
			SyncBestSignal		= BestSliceSignal;
			SyncBestFreqOffset	= BestSliceOffset;
		} else if(BestSliceSignal > SyncBestSignal) {
			SyncBestSignal		= BestSliceSignal;
			SyncBestBlockPhase	= BlockPhase;
			SyncBestFreqOffset	= BestSliceOffset;
		}

		int Dist = (int)BlockPhase - (int)SyncBestBlockPhase;
		if (Dist < 0)
			Dist += BlockPhases;
		if (Dist == (int)(BlockPhases/2)) {
			int 		BestNoise	= SyncNoiseEnergy.AbsPtr(SyncBestBlockPhase)[SyncBestFreqOffset].Output;
			const int	MinNoise	= 1;
			BestNoise = (BestNoise > 0) ? isqrt(BestNoise) : 0;
			if (BestNoise < MinNoise)
				BestNoise = MinNoise;
			SyncSNR = SyncBestSignal / BestNoise;
			if (SyncSNR >= SyncThreshold) {
				unsigned __int64 *BestBlockPtr = DecodePipe[SyncBestBlockPhase].CurrPtr();
				unsigned __int64  Block		 = BestBlockPtr[SyncBestFreqOffset];
				for (size_t Byte = 0; Byte < m_pModem->nBitsPerSymbol; ++ Byte) {
					UCHAR Char	=(UCHAR)(Block&0x7F);
					BOOL  bIdle = FALSE;
					if (m_pModem->nBitsPerCharacter == 6) {
						// Contestia
						if (Char == 59)
							Char = ' ';
						else if (Char == 60)
							Char = '\r';
						else if (Char == 61)
							Char = 8; // backspace
						else if (Char == 0)
							bIdle = TRUE;
						else
							Char += 32;
					} else if (m_pModem->nBitsPerCharacter == 5) {
						// RTTYM
						int c = rttym_dec(&m_pModem->nBaudotRxMode, Char);
						if (c < 0)
							bIdle = TRUE;
						else
							Char = (UCHAR)c;
					}
					if (! bIdle)
						Output.Write(Char);
					Block >>= 8;
				}
/*
				{
					char bufout[256];
					sprintf(bufout, "SyncSNR: %d, SyncThreshold: %d\r\n", SyncSNR, SyncThreshold);
					OutputDebugString(bufout);
				}
*/
			}
		}
	}

	_ASSERT(BlockPhase >= 0 && BlockPhase <= BlockPhases);
	if (BlockPhase == BlockPhases)
		BlockPhase = 0;
}

void MFSK_Receiver::Demodulate(short *Input)
{ 
	short *fr = (short*)malloc(sizeof(short) * m_pModem->nSymbolLen * 2);
	short *fi = fr + m_pModem->nSymbolLen;
    size_t WrapMask = m_pModem->nSymbolLen - 1;

    for (int i = 0; i < (m_pModem->nSymbolLen >> 2); ++ i) {
		InpTap[InpTapPtr ++] = Input[i];
        InpTapPtr &= WrapMask;
	}
    for (int Time = 0; Time < m_pModem->nSymbolLen; ++ Time) {
		fr[Time] = ((int)InpTap[InpTapPtr ++] * SymbolShape[Time]) >> 15;
		InpTapPtr &= WrapMask;
	}

    for(; i < (m_pModem->nSymbolLen >> 1); ++ i) {
		InpTap[InpTapPtr ++] = Input[i];
        InpTapPtr &= WrapMask;
	}
    for (Time = 0; Time < m_pModem->nSymbolLen; ++ Time) {
		fi[Time] = ((int)InpTap[InpTapPtr ++] * SymbolShape[Time]) >> 15;
		InpTapPtr &= WrapMask;
	}

	fix_fft(fr, fi, ilog2(m_pModem->nSymbolLen), 0);

	// split the two real FFTs and compute energies of the selected bins.
	size_t iFreq = FirstCarrier - SyncMargin;
	int    re, im;
//	_ASSERT(iFreq > 0);
	for (i = 0; i < DecodeWidth; ++ i, ++ iFreq) {
		int iBin = bReverse ? (DecodeWidth - i - 1) : i;
		re =  (int)fr[iFreq] + (int)fr[m_pModem->nSymbolLen - iFreq];
		im =  (int)fi[iFreq] - (int)fi[m_pModem->nSymbolLen - iFreq];
		Energy[0][iBin] = re * re + im * im;
		re =  (int)fi[iFreq] + (int)fi[m_pModem->nSymbolLen - iFreq];
		im = -(int)fr[iFreq] + (int)fr[m_pModem->nSymbolLen - iFreq];
		Energy[1][iBin] = re * re + im * im;
	}

	free(fr);
}

void MFSK_Receiver::SoftDecode(short *Symbol, size_t Slice, int FreqOffset)
{
	int    *EnergyPtr	= Energy[Slice] + (SyncMargin + FreqOffset);
    int  	TotalEnergy	= 0;
    size_t	Freq		= 0;
	int		aiSymbol[8];
	memset(aiSymbol, 0, sizeof(int) * m_pModem->nBitsPerSymbol);

	int		Carriers	= 1 << m_pModem->nBitsPerSymbol;
    for (int i = 0; i < Carriers; ++ i) {
		UCHAR SymbIdx = grayencode(i);
        int   Energy  = EnergyPtr[Freq] >> m_pModem->nBitsPerSymbol;
        UCHAR Mask    = 1;
        TotalEnergy  += Energy;
        for (size_t Bit = 0; Bit < m_pModem->nBitsPerSymbol; ++ Bit) {
			aiSymbol[Bit] += (SymbIdx & Mask) ? - Energy : Energy;
			Mask <<= 1;
		}
        Freq += CarrierSepar;
	}

	if (TotalEnergy == 0) {
		memset(Symbol, 0, sizeof(short) * m_pModem->nBitsPerSymbol);
		return; // nothing to normalize
	}

	// Normalize the bit energies
	_ASSERT(TotalEnergy > 0);
	int iShift = 15; // Right shift before division.
	if (TotalEnergy & 0xffff0000) {
		do {
			-- iShift;
			TotalEnergy >>= 1;
		} while (TotalEnergy & 0xffff0000);
	} else {
		while ((TotalEnergy & 0x8000) == 0) {
			++ iShift;
			TotalEnergy <<= 1;
		}
	}
	for (size_t Bit = 0; Bit < m_pModem->nBitsPerSymbol; ++ Bit)
		Symbol[Bit] = (aiSymbol[Bit] << iShift) / TotalEnergy;
}

/********************************************************* MFSK_SoftDecoder *********************************************************/

MFSK_SoftDecoder::MFSK_SoftDecoder() :
	m_pModem(0),
	InputBuffer(0),
    FHT_Buffer(0),
    OutputBlock(0)
{
}

void MFSK_SoftDecoder::Free()
{ 
	free(InputBuffer); 
	InputBuffer=0;
	free(FHT_Buffer); 
	FHT_Buffer=0;
	free(OutputBlock);
	OutputBlock=0;
}

void MFSK_SoftDecoder::Reset()
{ 
	memset(InputBuffer, 0, sizeof(short) * InputBufferLen);
	InputPtr=0; 
}

int MFSK_SoftDecoder::Preset(struct olivia *s)
{
	m_pModem = s;
	InputBufferLen = m_pModem->nSymbolsPerBlock * m_pModem->nBitsPerSymbol;
    if (ReallocArray(&InputBuffer, InputBufferLen) < 0)
		goto Error;
	if (ReallocArray(&FHT_Buffer, m_pModem->nSymbolsPerBlock) < 0)
		goto Error;
    if (ReallocArray(&OutputBlock, m_pModem->nBitsPerSymbol) < 0)
		goto Error;
    Reset();
    return 0;
    
Error: 
	Free(); 
	return -1;
}

void MFSK_SoftDecoder::Input(short *Symbol)
{ 
	for (size_t FreqBit = 0; FreqBit < m_pModem->nBitsPerSymbol; ++ FreqBit)
		InputBuffer[InputPtr ++] = Symbol[FreqBit];
	if (InputPtr == InputBufferLen)
		InputPtr = 0;
}

// Forward Fast Hadamard Transform
static void FHT(int *Data, size_t Len)
{ 
	int Bit1, Bit2, NewBit1, NewBit2;
	for (size_t Step = 1; Step < Len; Step <<= 1) {
		for (size_t Ptr = 0; Ptr < Len; Ptr += (Step << 1)) { 
			for (size_t Ptr2 = Ptr; Ptr2 - Ptr < Step; ++ Ptr2) {
				Bit1	   = Data[Ptr2];  
				Bit2	   = Data[Ptr2+Step];
				NewBit1	   = Bit2; 
				NewBit1   += Bit1;
				NewBit2    = Bit2; 
				NewBit2   -= Bit1;
				Data[Ptr2] = NewBit1;
				Data[Ptr2+Step]=NewBit2;
			}
		}
	}
}

void MFSK_SoftDecoder::DecodeCharacter(size_t FreqBit)
{
	size_t Ptr		= InputPtr;
	size_t Rotate	= FreqBit;
	size_t CodeWrap	= m_pModem->nSymbolsPerBlock - 1;
	size_t nShift	= (m_pModem->nBitsPerCharacter == 7) ? 13 : 5; // Olivia or Contestia/RTTYM
	size_t CodeBit	= (FreqBit * nShift) & CodeWrap;
	unsigned __int64 nScramblingCode = (m_pModem->nBitsPerCharacter == 7) ? ScramblingCodeOlivia : ScramblingCodeContestia;

	for (int TimeBit = 0; TimeBit < m_pModem->nSymbolsPerBlock; ++ TimeBit) {
		int Bit = InputBuffer[Ptr + Rotate];
		unsigned __int64 CodeMask = 1; CodeMask <<= CodeBit;
		FHT_Buffer[TimeBit] = (nScramblingCode & CodeMask) ? -Bit : Bit;
        ++ CodeBit;
		CodeBit &= CodeWrap;
		if (++ Rotate >= m_pModem->nBitsPerSymbol)
			Rotate -= m_pModem->nBitsPerSymbol;
		Ptr += m_pModem->nBitsPerSymbol;
        if (Ptr >= InputBufferLen)
			Ptr -= InputBufferLen;
	}

	FHT(FHT_Buffer, m_pModem->nSymbolsPerBlock);

	int  	Peak	= 0;
	size_t	PeakPos	= 0;
	int 	SqrSum	= 0;
	for (TimeBit = 0; TimeBit < m_pModem->nSymbolsPerBlock; ++ TimeBit) {
		int Signal = FHT_Buffer[TimeBit] >> m_pModem->nBitsPerCharacter;
		_ASSERT(Signal >= -32767 && Signal <= 32767);
		SqrSum += (Signal * Signal) >> (m_pModem->nBitsPerCharacter - 1);
		if (ABS(Signal) > ABS(Peak)) {
			Peak	= Signal;
			PeakPos = TimeBit;
		}
	}

	OutputBlock[FreqBit] = PeakPos + ((Peak < 0) ? m_pModem->nSymbolsPerBlock : 0);
	SqrSum		-= (Peak * Peak) >> (m_pModem->nBitsPerCharacter - 1);
	NoiseEnergy += SqrSum; // (float)SqrSum/(m_pModem->nSymbolsPerBlock - 1);
	Signal		+= ABS(Peak);
}

void MFSK_SoftDecoder::Process()
{
    Signal		 = 0;
	NoiseEnergy  = 0;
	for (size_t FreqBit = 0; FreqBit < m_pModem->nBitsPerSymbol; ++ FreqBit)
		DecodeCharacter(FreqBit);
}

size_t MFSK_SoftDecoder::Output(UCHAR *Buffer)
{ 
	for (size_t FreqBit = 0; FreqBit < m_pModem->nBitsPerSymbol; ++ FreqBit)
		Buffer[FreqBit] = OutputBlock[FreqBit];
	return m_pModem->nBitsPerSymbol;
}

size_t MFSK_SoftDecoder::Output(unsigned __int64 &PackedBuffer)
{ 
    PackedBuffer = 0;
    for (size_t FreqBit = m_pModem->nBitsPerSymbol; FreqBit > 0; ) { 
		PackedBuffer <<= 8;
        -- FreqBit;
		PackedBuffer |= OutputBlock[FreqBit];
	}
    return m_pModem->nBitsPerSymbol;
}
