#pragma once

#include "../main/sndmodem.h"
#include "fifo.h"

extern const double MFSK_SymbolFreqShape[];
extern const size_t MFSK_SymbolFreqShapeLen;

class MFSK_Transmitter
{ 
private:
	enum {
		  State_Running = 0x0001,
		  State_StopReq = 0x0010,
	};

	int				State;

	FIFO<UCHAR>		Input;			// buffer(queue) for the characters to be encoded
	UCHAR			InputBlock[8];  // FEC code block buffer
	FIFO<UCHAR>		Monitor;		// buffer for monitoring the characters being sent

public:
	MFSK_Transmitter(struct olivia *s) : 
		m_pModem(s), SymbolShape(0), OutTap(0), iRand(0), FHT_Buffer(0), OutputBlock(0) {}
	~MFSK_Transmitter()	{ Free(); }

	void	Free();
	int		Preset();	// preset internal arrays according to primary paramaters
	void	Reset();

	// start the transmission
	void	Start()		{ State|=State_Running; }

	// request to stop (and complete) the transmission
	// but the transmitter will only stop after transmitting all the data
	void	Stop()		{ State|=State_StopReq; }

	// check if the transmission is still running (not complete)
	int		Running()		{ return State&State_Running; }

	// put the character into the transmitter input queue
	int		PutChar(UCHAR Char)	{ return Input.Write(Char); }

	// get one character from the monitor buffer
	int		GetChar(UCHAR &Char)	{ return Monitor.Read(Char); }

	size_t	GetReadReady()	{ return Input.ReadReady(); }

	// get out the transmitter output (audio)
	int		Output(Trx *trx);

protected:
	void	ModulatorSend(Trx *trx, UCHAR Symbol);
	void	EncodeCharacter(UCHAR Char);
	void	ScrambleFHT(size_t CodeOffset);
	void	EncodeBlock(UCHAR *InputBlock);

	struct olivia	*m_pModem;

	int		*SymbolShape;		// the shape of the symbol
	UINT	 SymbolPhase;		// the phase of the tone being transmitted
	int     *OutTap;			// output tap (buffer)
	size_t   TapPtr;
	int		 iRand;				// random number for toggling phase

	size_t	 SymbolPtr;
	char	*FHT_Buffer;
	UCHAR	*OutputBlock;
};

/// Allocate olivia symbol of SymbolLen.
/// Scale it by ShapeScale if it is greater than 0.
/// Do not scale if ShapeScale == 0.
/// Normalize and scale by (-ShapeScale) if ShapeScale < 0.
extern double*	olivia_symbolshape_double(size_t SymbolLen, double ShapeScale);
/// Scale coefficients from <-1;1> to <-32767;32767>
extern int*		olivia_symbolshape_int(size_t SymbolLen, double ShapeScale);
