/*
 *    olivia.cc  --  OLIVIA modem
 *
 *    Copyright (C) 2005
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "../main/trx.h"
#include "../main/sndmodem.h"
#include "../main/trxctrl.h"
#include "../misc/cmplx.h"
#include "../misc/filter.h"
#include "../resource.h"

#include "olivia.h"
#include "mfsk.h"
#include "rttym.h"

#ifdef _WIN32_WCE
	#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

extern "C" void olivia_init(Trx *trx);

// the symbol shape described in frequency domain
const double MFSK_SymbolFreqShape[]		= { +1.0000000000, +1.1913785723, -0.0793018558, -0.2171442026, -0.0014526076 } ;
const size_t MFSK_SymbolFreqShapeLen	= sizeof(MFSK_SymbolFreqShape) / sizeof(double);

static void olivia_txinit(Trx *trx)
{
	struct olivia *s = (struct olivia *)trx->modem;
	UCHAR c;
//FIXME
//	s->Rx->Flush();

	while (s->Rx->GetChar(c) > 0)
		trx_put_rx_char(trx, c);
	s->nBaudotRxMode = RTTYM_LETS;
	s->nBaudotTxMode = RTTYM_LETS;
	s->Tx->Start();
	s->escape = 0;

	if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202 || g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER) {
		int iToneSpacing = 8000 * (CarrierSepar << FREQ_FRAC_BITS) / s->nSymbolLen;
		int iFreqLow = (int)trx->nTxFrequency - (s->nTones >> 1) * iToneSpacing + (iToneSpacing >> 1);
		if (iFreqLow < 0)
			iFreqLow = 0;
		trxctrl_ats3_start_mfsk(&g_TrxCtrl, trx->pModem, iFreqLow, iToneSpacing);
	}
}

static void olivia_rxinit(Trx *trx)
{
	struct olivia *s = (struct olivia *) trx->modem;
	s->Rx->Reset();
	s->nBaudotRxMode = RTTYM_LETS;
	s->nBaudotTxMode = RTTYM_LETS;
	s->escape = 0;
}

static void olivia_free(struct olivia *s)
{
	if (s) {
		delete s->Tx;
		delete s->Rx;
		free(s);
	}
}

static void olivia_destructor(Trx *trx)
{
	struct olivia *s = (struct olivia*)trx->modem;
	olivia_free(s);
	trx->modem = NULL;
	trx->txinit = NULL;
	trx->rxinit = NULL;
	trx->txprocess = NULL;
	trx->rxprocess = NULL;
	trx->destructor = NULL;
}

static int olivia_unescape(Trx *trx, int c)
{
	struct olivia *s = (struct olivia*)trx->modem;

	if (s->olivia_esc == 0)
		return c;

	if (s->escape) {
		s->escape = 0;
		return c + 128;
	}

	if (c == 127) {
		s->escape = 1;
		return -1;
	}

	return c;
}

static int olivia_txprocess(Trx *trx)
{
	struct olivia *s = (struct olivia *) trx->modem;
	int c;
	UCHAR ch;

	/*
	 * The encoder works with BitsPerSymbol length blocks. If the
	 * modem already has that many characters buffered, don't try
	 * to read any more. If stopflag is set, we will always read 
	 * whatever there is.
	 */
	if (trx->stopflag || (s->Tx->GetReadReady() < s->nBitsPerSymbol)) {
		if ((c = trx_get_tx_char(trx)) == -1) {
			if (trx->stopflag)
				s->Tx->Stop();
		} else {
			// Replace un-representable characters with a dot
			if (c > (s->olivia_esc ? 255 : 127))
				c = '.';

			if (c > 127) {
				c &= 127;
				s->Tx->PutChar(127);
			}

			if (s->nBitsPerCharacter == 7) { // Olivia
				s->Tx->PutChar(c);
			} else if (s->nBitsPerCharacter == 6) { // Contestia
				if (c != '\n')
					s->Tx->PutChar(c);
			} else {
				_ASSERT(s->nBitsPerCharacter == 5); // RTTYM
				if (c == ' ') { // unshift-on-space
					s->Tx->PutChar(0x00); // RTTYM_LETS
					s->Tx->PutChar(0x04); // space
					s->nBaudotTxMode = RTTYM_LETS;
				} else if ((c = rttym_enc(c)) >= 0) {
					if ((c & s->nBaudotTxMode) == 0) {
						if (s->nBaudotTxMode == RTTYM_FIGS) {
							s->Tx->PutChar(0x00);
							s->nBaudotTxMode = RTTYM_LETS;
						} else {
							s->Tx->PutChar(27);
							s->nBaudotTxMode = RTTYM_FIGS;
						}
					}
					s->Tx->PutChar(c & 0x1F);
				}
			}
		}
	}

	if (s->Tx->GetChar(ch) > 0)
		if ((c = olivia_unescape(trx, ch)) != -1)
			trx_put_echo_char(trx, c);
	s->Tx->Output(trx);

	return s->Tx->Running() ? 0 : -1;
}

static int olivia_rxprocess(Trx *trx, short *buf, int len)
{
	struct olivia *s = (struct olivia *) trx->modem;
	int c;
	UCHAR ch = 0;

//      fprintf(stderr, "olivia_rxprocess(%d)\n", len);

	s->Rx->SyncThreshold = trx->squelchon ? trx->iSquelchThreshold : 0;
	s->Rx->FirstCarrier  = 
		(((s->nSymbolLen >> 4) * (trx->nRxFrequency - (s->nBandwidth << (FREQ_FRAC_BITS - 1))) / 500) >> FREQ_FRAC_BITS) + 1;
	if (s->Rx->FirstCarrier < s->Rx->SyncMargin)
		s->Rx->FirstCarrier = s->Rx->SyncMargin;
	s->Rx->bReverse = trx->reverse;
	s->Rx->Process(buf, len);

/*
	float snr = s->Rx->SignalToNoiseRatio();
	if (snr > 99.9)
		snr = 99.9f;
 	trx_set_metric(snr);
*/

/*
	msg = g_strdup_printf("SNR: %4.1f | Freq: %+4.1f/%4.1f Hz | Time: %5.3f/%5.3f sec",
			      s->Rx->SignalToNoiseRatio(),
			      s->Rx->FrequencyOffset(),
			      s->Rx->TuneMargin(),
			      s->Rx->TimeOffset(),
			      s->Rx->BlockPeriod());

	statusbar_set_main(msg);
	free(msg);
*/

	while (s->Rx->GetChar(ch) > 0)
		if ((c = olivia_unescape(trx, ch)) != -1 && c > 7)
			trx_put_rx_char(trx, c);

	return 0;
}

void olivia_init(Trx *trx)
{
	struct olivia *s = (struct olivia*)calloc(sizeof(struct olivia), 1);

	switch (trx->submode) {
	case ID_SUBMODE_4_250:
		s->nTones		= 4;
		s->nBandwidth	= 250;
		break;
	case ID_SUBMODE_4_500:
		s->nTones		= 4;
		s->nBandwidth	= 500;
		break;
	case ID_SUBMODE_16_500:
		s->nTones		= 16;
		s->nBandwidth	= 500;
		break;
	case ID_SUBMODE_16_1K:
		s->nTones		= 16;
		s->nBandwidth	= 1000;
		break;
	case ID_SUBMODE_8_250:
		s->nTones		= 8;
		s->nBandwidth	= 250;
		break;
	case ID_SUBMODE_8_500:
		s->nTones		= 8;
		s->nBandwidth	= 500;
		break;
	case ID_SUBMODE_32_1K:
		s->nTones		= 32;
		s->nBandwidth	= 1000;
		break;
	default:
		trx->submode	= ID_SUBMODE_16_500;
		s->nTones		= 16;
		s->nBandwidth	= 500;
		break;
	}

	s->olivia_smargin	= 8;	// synchronizer integration period [FEC blocks]
	s->olivia_sinteg	= 4;	// synchronizer integration period [FEC blocks]

	s->nBitsPerSymbol		= ilog2(s->nTones);
    _ASSERT(s->nTones == (1 << s->nBitsPerSymbol));
	s->nBitsPerCharacter	= (trx->mode == MODE_OLIVIA) ? 7 : ((trx->mode == MODE_CONTESTIA) ? 6 : 5); // Contestia, RTTYM
	s->nSymbolsPerBlock		= 1 << (s->nBitsPerCharacter - 1);
    s->nSymbolLen			= 1 << (s->nBitsPerSymbol + 7 - ilog2(s->nBandwidth / 125));

	s->Tx = new MFSK_Transmitter(s);
	s->Rx = new MFSK_Receiver(s);
	if (s->Tx == 0 || s->Rx == 0 || s->Tx->Preset() < 0) {
// 		g_warning("olivia_init: transmitter preset failed!");
		olivia_free(s);
		return;
	}

	s->Rx->SyncMargin	= s->olivia_smargin;
	s->Rx->SyncIntegLen = s->olivia_sinteg;
	s->Rx->SyncThreshold = trx->squelchon ? trx->iSquelchThreshold : 0;

	if (s->Rx->Preset() < 0) {
// 		g_warning("olivia_init: receiver preset failed!");
		olivia_free(s);
		return;
	}

	trx->modem			= s;
	trx->txinit			= olivia_txinit;
	trx->rxinit			= olivia_rxinit;
	trx->txprocess		= olivia_txprocess;
	trx->rxprocess		= olivia_rxprocess;
	trx->destructor		= olivia_destructor;
	trx->bandwidth		= s->nBandwidth << FREQ_FRAC_BITS;
}

double*	olivia_symbolshape_double(size_t SymbolLen, double ShapeScale)
{
    double *pSymbolShapeDbl = (double*)malloc(sizeof(double) * SymbolLen);
	if (pSymbolShapeDbl == 0)
		return 0;

	double dAmpl = MFSK_SymbolFreqShape[0];
	for (size_t i = 0; i < SymbolLen; ++ i)
	    pSymbolShapeDbl[i] = dAmpl;

	size_t iFreq;
	for (iFreq = 1; iFreq < MFSK_SymbolFreqShapeLen; ++ iFreq) { 
        dAmpl = MFSK_SymbolFreqShape[iFreq];
        if (iFreq & 1)
			dAmpl = - dAmpl;
		size_t iPhase = 0;
        for (i = 0; i < SymbolLen; ++ i) { 
			pSymbolShapeDbl[i] += dAmpl * cos((2.0 * M_PI * iPhase) / SymbolLen);
			iPhase += iFreq;
			if (iPhase >= SymbolLen)
				iPhase -= SymbolLen;
		}
    }

	if (ShapeScale > 0.0) {
		for (size_t Time = 0; Time < SymbolLen; ++ Time)
			pSymbolShapeDbl[Time] *= ShapeScale;
	} else if (ShapeScale < 0.0) {
		double dSum = 0.0;
		for (size_t Time = 0; Time < SymbolLen; ++ Time)
			dSum += fabs(pSymbolShapeDbl[Time]);
		ShapeScale = - ShapeScale / dSum;
		for (Time = 0; Time < SymbolLen; ++ Time)
			pSymbolShapeDbl[Time] *= ShapeScale;
	}

	return pSymbolShapeDbl;
}

extern "C" int*	filter_double2int(double *aFloats, int nFloats);

int* olivia_symbolshape_int(size_t SymbolLen, double ShapeScale)
{
	double *afSymbolShape = olivia_symbolshape_double(SymbolLen, ShapeScale);
	if (afSymbolShape == 0)
		return 0;
	int    *aiSymbolShape = filter_double2int(afSymbolShape, SymbolLen);
	free(afSymbolShape);
	return aiSymbolShape;	
}
