// MFSK trasnmitter and receiver code, Pawel Jalocha, December 2004

#pragma once

// =====================================================================

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <float.h>

#include "../misc/misc.h"
#include "../misc/fixedpoint.h"

#include "struc.h"
#include "lowpass3.h"
#include "fifo.h"

#define	SampleRate			8000
#define SlicesPerSymbol			2	/// FFT slices per symbol
#define CarrierSepar			2
#define ScramblingCodeOlivia	0xE257E6D0291574ECL
#define ScramblingCodeContestia	0xEDB88320L
#define ScramblingCodeRTTYM		0xEDB88320L

class MFSK_Transmitter;
class MFSK_Receiver;

struct olivia {
	MFSK_Transmitter *Tx;
	MFSK_Receiver    *Rx;

	int		escape;

	BOOL	olivia_esc;
	int		olivia_smargin;
	int		olivia_sinteg;

	int		nBitsPerCharacter;
	int		nSymbolsPerBlock;

	size_t	nTones;				// number of tones: 4, 8, 16, 32, 64, 128, 256
	int  	nBandwidth;			// bandwidth: 125, 250, 500, 1000, 2000

	size_t	nBitsPerSymbol;		// number of bits per symbol
	int 	nSymbolLen;			// length of the symbol, must be a power of 2

	/// RTTYM_LETS or  RTTYM_FIGS
	int					nBaudotRxMode;
	int					nBaudotTxMode;
};

// A circular buffer to store history of data.
// Data may come as single numbers or in batches
// of fixed size (-> Width)
template <class Type>
 class CircularBuffer
{ public:

   size_t Width;  // input/output data width (row width)
   size_t Len;    // buffer length (column height)

  public:

   size_t Size;   // total size of the storage in the buffer
   size_t Ptr;    // current pointer (counts rows)
   Type  *Data;   // allocated storage

  public:
   CircularBuffer()
     { Init(); }

   ~CircularBuffer()
     { free(Data); }

   void Init(void)
     { Data=0; Size=0; Width=1; }

   void Free(void)
     { free(Data); Data=0; Size=0; }

   // reset: set pointer to the beginning of the buffer
   void Reset(void)
     { Ptr=0; }

   // preset for given length and width
   int Preset(void)
     { Size=Width*Len;
       if(ReallocArray(&Data,Size)<0) return -1;
       Reset(); return 0; }

   // set all elements to given value
   void Set(Type &Value)
     { size_t Idx;
	   for(Idx=0; Idx<Size; Idx++)
	     Data[Idx]=Value;
	 }

   // set all elements to zero
   void Clear(void)
     { Type Zero; Zero=0; Set(Zero); }

   // increment the pointer (with wrapping around)
   void IncrPtr(size_t &Ptr, size_t Step=1)
     { Ptr+=Step; if(Ptr>=Len) Ptr-=Len; }

   // decrement the pointer (with wrapping around)
   void DecrPtr(size_t &Ptr, size_t Step=1)
     { if(Ptr>=Step) Ptr-=Step;
	            else Ptr+=(Len-Step); }

   // synchronize current pointer with another circular buffer
   template <class SrcType>
    void operator |= (CircularBuffer<SrcType> Buffer)
     { Ptr=Buffer.Ptr; }

   // advance (increment) current pointer
   void operator += (size_t Step)
     { IncrPtr(Ptr,Step); }

   // decrement current pointer
   void operator -= (size_t Step)
     { DecrPtr(Ptr,Step); }

   // index operator to get the absolute data pointer
   Type *operator [] (size_t Idx)
     { return Data+(Idx*Width); }

   // get storage pointer corresponding to an absolute pipe pointer
   Type *AbsPtr(size_t Ptr)
     { return Data+(Ptr*Width); }

   // get storage pointer corresponding to current pipe pointer
   Type *CurrPtr(void)
     { return Data+(Ptr*Width); }

   // get storage pointer corresponding to current pointer +/- offset
   Type *OffsetPtr(int Offset)
     { Offset+=Ptr; Offset*=Width;
	   if(Offset<0) Offset+=Size;
	   else if(Offset>=(int)Size) Offset-=Size;
	   return Data+Offset; }
};

class MFSK_SoftDecoder
{ 
public:
	int     Signal, NoiseEnergy;
	UCHAR  *OutputBlock;

public:
	MFSK_SoftDecoder();
	~MFSK_SoftDecoder() { Free(); }

	void	Free();
	void	Reset();
	int		Preset(struct olivia *s);
	void	Input(short *Symbol);
	void	DecodeCharacter(size_t FreqBit);
	void	Process();
	size_t	Output(UCHAR *Buffer);
	size_t	Output(unsigned __int64 &PackedBuffer);
//	void	PrintOutputBlock(FILE *File=stdout);

private:
	struct olivia	*m_pModem;
	size_t			 InputBufferLen;
	short			*InputBuffer;
	size_t			 InputPtr;
	int				*FHT_Buffer;
};

/*

How to use the MFSK_Receiver class:

1. create an object like:

   #include "mfsk.h"

   MFSK_Receiver<float> Receiver;

2. Set the parameters, for example:

   Receiver.Tones           = 32;     // number of tones (symbols)
   Receiver.Bandwidth       = 1000;   // bandwidth [Hz]
   Receiver.SyncMargin      = 8;      // synchronizer tune margin [tone freq. spacing]
   Receiver.SyncIntegLen    = 4;      // synchronizer integration period [FEC blocks]
   Receiver.SyncThreshold   = 3.2;    // S/N threshold for printing

   You don't need to set all the parameters, as upon creation
   of the Receiver object they are already given certain default
   values.

   If you changed parameters at one time and want later to go back
   to the default values you can call: Receiver.Default();

3. Preset the Receiver internal arrays for the parameters you just set:

   if(Receiver.Preset()<0)
     printf("Not enough RAM or another problem\n");

   Each time you change the parameters you need to call Preset()
   in order to resize the internal arrays. Preset() will as well
   destroy all data being in the process of decoding, if you need
   this data then call first Receiver.Flush()

4. Read back the parameters you set in point 1., they could have been adjusted
   by Preset() to their closest allowed values.

5. Feed the audio into the Receiver:

   Receiver.Process(AudioBuffer, BufferLength);

   AudioBuffer can be an array of short (16-bit signed integers)
   that you fill with the data from the soundcard. I suggest you feed
   the receiver with batches of 512 or 1024 samples, but in can be any number
   of samples at a time.

6. Call GetChar(Char) to get decoded characters. Note, that
   characters come in batches, and so, you need to call GetChar()
   possibly several times. GetChar() returns 0 when the decoder FIFO
   is empty or 1 when not empty. In the latter case the argument
   contains the character read form the FIFO. The loop could be like:
   
   for( ; ; )
   { UCHAR Char;
     if(Receiver.GetChar(Char)==0) break;
     printf("%c",Char);
   }

   Keep in mind that you may see (random) control code characters here,
   and thus you must be able to deal with them. I suggest to process
   only carriage return (code=13) and Backspace (code=8). NUL (code=0)
   is the idle character: it is being sent when there is no text to be sent.

7. At any time you can read the signal-to-noise ratio of the
   incoming signal by calling Receiver.SignalToNoiseRatio() or frequency offset
   by calling Receiver.FrequencyOffset()

8. When the user decides to turn off the receiver and switch over
   to transmitt you may still call Receiver.Flush()in order to flush
   the data still being buffered in the decoder pipeplines.
*/

class MFSK_Receiver
{
public:
	// primary parameters: set by the user
	int			SyncMargin;     // synchronizer search margin, frequency-wide
	size_t		SyncIntegLen;   // synchronizer integration period
	int			SyncThreshold;  // synchronizer S/N threshold (if below, output is suppressed)
	int			FirstCarrier;	// first carrier in terms of FFT freq. bins
	BOOL		bReverse;

private:
	Seq<short>	InputBuffer;

	size_t		FreqOffsets;   // number of possible frequency offsets
	size_t		BlockPhases;   // number of possible time-phases within the FEC block
	MFSK_SoftDecoder *Decoder;   // array of decoders
	size_t		BlockPhase;    // current running time-phase

	CircularBuffer<AvgFilter>  SyncNoiseEnergy;	// FEC noise integrators
	CircularBuffer<AvgFilter>  SyncSignal;		// FEC signal integrators
	int			SyncFilterWeight;									// weight for the integrators

	int			SyncBestSignal;			/// best signal
	size_t		SyncBestBlockPhase;		/// time-phase of the best signal
	size_t		SyncBestFreqOffset;		/// frequency offset of the best signal
	int			SyncSNR;				/// S/N corresponding to the SyncBestSignal

	CircularBuffer<unsigned __int64> *DecodePipe; // pipeline for decoded FEC blocks

	FIFO<UCHAR> Output;      // buffer for decoded characters

public:
	MFSK_Receiver(struct olivia *s);
	~MFSK_Receiver() { Free(); }

	void	Free();
	// resize internal arrays according the parameters
	int		Preset();
	void	Reset();

/*
	float	BaudRate()				const { return SampleRate/Demodulator.SymbolSepar; }
	float	TuneMargin()			const { return SyncMargin*(SampleRate/Demodulator.SymbolLen); }
	float	BlockPeriod()			const { return (SymbolsPerBlock*Demodulator.SymbolSepar)/SampleRate; }
	float	CharactersPerSecond()	const { return BitsPerSymbol*SampleRate/(SymbolsPerBlock*Demodulator.SymbolSepar); }
	float	SignalToNoiseRatio()	const { return SyncSNR; }
	float	FrequencyOffset()		const { return ((int)SyncBestFreqOffset-(int)FreqOffsets/2)*(SampleRate/Demodulator.SymbolLen); }
	float	TimeOffset()			const { return ((float)SyncBestBlockPhase/SlicesPerSymbol)*(Demodulator.SymbolSepar/SampleRate); }
	void	PrintParameters();
*/

	// process an audio batch: first the input processor, then the demodulator
	int		Process(short *Input, size_t InputLen);
	void	Flush();

	// get one character from the output buffer
	int		GetChar(UCHAR &Char)	{ return Output.Read(Char); }

private:
	// process the input buffer: first the input processor, then the demodulator
    void	ProcessInputBuffer();

	// process (through the demodulator) an audio batch corresponding to one symbol
	// (demodulator always works with audio batches corresponding to one symbol period)
    void	ProcessSymbol(short *Input);

	void	Demodulate(short *Input);
	void	SoftDecode(short *Symbol, size_t Slice, int FreqOffset);

	size_t	WindowLen;

	struct olivia	*m_pModem;

	int				 DecodeWidth;
	int				*SymbolShape;				// the shape of the symbol and the FFT window
	int				*Energy[SlicesPerSymbol];

	short			*InpTap;                    // input buffer
	size_t			 InpTapPtr;
};
