/*
 *    sfft.c  --  Sliding FFT
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

#include "sfft.h"
#include "misc.h"

//#define	STABCOEFF	0.997
#define	STABCOEFF	0.995

/* ---------------------------------------------------------------------- */

struct sfft *sfft_init(int len, int first, int last)
{
	struct sfft *s;
	int i;

	s = malloc(sizeof(struct sfft));
	memset(s, 0, sizeof(struct sfft));

	s->twiddles = malloc(sizeof(dblcomplex) * len);
	s->history  = malloc(sizeof(dblcomplex) * len);
	s->bins     = malloc(sizeof(dblcomplex) * len);
	memset(s->history, 0, sizeof(dblcomplex) * len);
	memset(s->bins, 0, sizeof(dblcomplex) * len);

	s->fftlen = len;
	s->first = first;
	s->last = last;

	for (i = 0; i < len; i++) {
		s->twiddles[i].re = cos(i * 2.0 * M_PI / len) * STABCOEFF;
		s->twiddles[i].im = sin(i * 2.0 * M_PI / len) * STABCOEFF;
	}

	s->corr = pow(STABCOEFF, len);

	return s;
}

void sfft_free(struct sfft *s)
{
	if (s) {
		free(s->twiddles);
		free(s->history);
		free(s->bins);
		free(s);
	}
}

/*
 * Sliding FFT, complex input, complex output
 */
dblcomplex *sfft_run(struct sfft *s, dblcomplex new)
{
	dblcomplex old, z;
	int i;

	/* restore the sample fftlen samples back */
	old = s->history[s->ptr];
	old.re *= s->corr;
	old.im *= s->corr;

	/* save the new sample */
	s->history[s->ptr] = new;

	/* advance the history pointer */
	s->ptr = (s->ptr + 1) % s->fftlen;

	/* calculate the wanted bins */
	for (i = s->first; i < s->last; i++) {
		z = s->bins[i];
		z.re -= old.re;
		z.im -= old.im;
		z.re += new.re;
		z.im -= new.im;
		s->bins[i].re = z.re * s->twiddles[i].re - z.im * s->twiddles[i].im;
		s->bins[i].im = z.re * s->twiddles[i].im + z.im * s->twiddles[i].re;
	}

	return s->bins;
}

/* ---------------------------------------------------------------------- */
