#pragma once

#include "cmplx.h"

#ifdef MIN
#undef MIN
#endif
#define MIN(a,b)        ((a)<(b)?(a):(b))

#ifdef MAX
#undef MAX
#endif
#define MAX(a,b)        ((a)>(b)?(a):(b))

#ifdef ABS
#undef ABS
#endif
#define ABS(a)          ((a) >= 0 ? (a) : -(a))

#ifndef MAXINT
#define MAXINT          ((int)(~(unsigned)0 >> 1))
#endif

#ifndef MININT
#define MININT          (-MAXINT-1)
#endif

/**
 * Unrolled C implementation of the CORDIC algorithm by Simon Hosie.
 * posted to comp.dsp on 21-03-04
 *
 * Example with 65536 units per circle, leaving optimisation to the
 * compiler. The maximum error should be less than 2 units. Accuracy can
 * be reduced or increased by changing the preprocessor directives. Note
 * that there will be some error from the rounding of the result
 * adjustment, and so adding stages is not sufficient to improve accuracy
 * beyond a certain point. Also note that the vector can grow in length
 * very quickly, and the input range shouldn't be limited.
 */
int iatan2(int y, int x);

/// Compute phase difference of c1 and c2.
/// The returned difference lies between -32768 and 32768
__inline int iphasediff(cmplx *c1, cmplx *c2)
{
	return iatan2(c1->re * c2->im - c1->im * c2->re,
				  c1->re * c2->re + c1->im * c2->im);
}

/**
 * ARM inversed square routines: 
 * http://www.finesse.demon.co.uk/steven/sqrt.html
 */
extern __inline unsigned int isqrt(unsigned int n)
{
    unsigned int root = 0, uitry;
#define iter1(N) \
    uitry = root + (1 << (N)); \
    if (n >= uitry << (N))   \
    {   n -= uitry << (N);   \
        root |= 2 << (N); \
    }
    iter1 (15);    iter1 (14);    iter1 (13);    iter1 (12);
    iter1 (11);    iter1 (10);    iter1 ( 9);    iter1 ( 8);
    iter1 ( 7);    iter1 ( 6);    iter1 ( 5);    iter1 ( 4);
    iter1 ( 3);    iter1 ( 2);    iter1 ( 1);    iter1 ( 0);
#undef iter1
    return root >> 1;
}

extern __inline unsigned int ihypot(int i1, int i2)
{
	return isqrt(i1 * i1 + i2 * i2);
}

extern __inline unsigned int ilog2(unsigned int x)
{
	unsigned int y = 0;
	for (; x > 1; x >>= 1)
		++ y;
	return y;
}
