/*
 *    filter.c  --  FIR filter
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "cmplx.h"
#include "filter.h"

#ifdef _WIN32_WCE
	#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

static int mac(const int *a, const int *b, unsigned int len)
{ 
	int	sum = 0;
	unsigned int i;
	for (i = 0; i < len; ++ i)
		sum += (*a++) * (*b++);
	return sum;
}

/// Create a band pass FIR filter with 6 dB corner frequencies
/// of 'f1' and 'f2'. (0 <= f1 < f2 <= 0.5)
static double *mk_filter(int len, int hilbert, double f1, double f2, BOOL bNormalize)
{
	double *fir	= (double*)malloc(sizeof(double) * len);
	double  sum	= 0.0;
	int		i;

	for (i = 0; i < len; ++ i) {
		double t = (double)i - 0.5 * ((double)len - 1.0);
		double h = (double)i / ((double)len - 1.0);
		double x = hilbert ? 
			/*
			 * The actual filter code assumes the impulse response
			 * is in time reversed order. This will be anti-
			 * symmetric so the minus sign handles that for us.
			 */
			(f1 * cosc(2.0 * f1 * t) - f2 * cosc(2.0 * f2 * t)) :
			(f2 * sinc(2.0 * f2 * t) - f1 * sinc(2.0 * f1 * t));
		x = 2.0 * x * hamming(h);
		sum += fabs(x);
		fir[i] = x;
	}

	// normalize the filter
	if (bNormalize)
		for (i = 0; i < len; ++ i)
			fir[i] /= sum;

	return fir;
}

int* filter_float2int(float *aFloats, int nFloats)
{
	int *aInts = malloc(nFloats * sizeof(int));
	int  i;
	if (aInts == 0)
		return 0;
	for (i = 0; i < nFloats; ++ i) {
		double dValue = (double)aFloats[i] * (double)0x7fff;
		int    iValue = (int)floor(dValue + 0.5);
		if (iValue > 0x7fff)
			iValue = 0x7fff;
		else if (iValue < -0x7fff)
			iValue = -0x7fff;
		aInts[i] = iValue;
	}
	return aInts;
}

int* filter_double2int(double *aFloats, int nFloats)
{
	int *aInts = malloc(nFloats * sizeof(int));
	int  i;
	if (aInts == 0)
		return 0;
	for (i = 0; i < nFloats; ++ i) {
		double dValue = aFloats[i] * (double)0x7fff;
		int    iValue = (int)floor(dValue + 0.5);
		if (iValue > 0x7fff)
			iValue = 0x7fff;
		else if (iValue < -0x7fff)
			iValue = -0x7fff;
		aInts[i] = iValue;
	}
	return aInts;
}

struct filter* filter_init_float(int len, int dec, float *itaps, float *qtaps)
{
	struct filter *f = calloc(1, sizeof(struct filter));
	if (f == 0)
		return 0;

	f->length		 = len;
	f->decimateratio = dec;
	f->pointer		 = len;
	f->counter		 = 0;

	if ((itaps && (f->ifilter = filter_float2int(itaps, len)) == 0) ||
		(qtaps && (f->qfilter = filter_float2int(qtaps, len)) == 0)) {
		filter_free(f);
		return 0;
	}

	return f;
}


struct filter* filter_init_double(int len, int dec, double *itaps, double *qtaps)
{
	struct filter *f = calloc(1, sizeof(struct filter));
	if (f == 0)
		return 0;

	f->length		 = len;
	f->decimateratio = dec;
	f->pointer		 = len;
	f->counter		 = 0;

	if ((itaps && (f->ifilter = filter_double2int(itaps, len)) == 0) ||
		(qtaps && (f->qfilter = filter_double2int(qtaps, len)) == 0)) {
		filter_free(f);
		return 0;
	}

	return f;
}

struct filter *filter_init_lowpass(int len, int dec, double freq)
{
	double			*fir  = mk_filter(len, FALSE, 0.0, freq, TRUE);
	struct filter   *fltr = filter_init_double(len, dec, fir, fir);
	free(fir);
	return fltr;
}

struct filter *filter_init_bandpass(int len, int dec, double fLow, double fHigh)
{
	double			*fir  = mk_filter(len, FALSE, fLow, fHigh, TRUE);
	struct filter   *fltr = filter_init_double(len, dec, fir, fir);
	free(fir);
	return fltr;
}

struct filter *filter_init_hilbert(int len, int dec)
{
	double *iCoeff = mk_filter(len, FALSE, 0.05, 0.45, FALSE);
	double *qCoeff = mk_filter(len, TRUE,  0.05, 0.45, FALSE);
	double  iSum   = 0.0;
	double  qSum   = 0.0;
	struct  filter *fltr;
	int     i;

	// normalize the filters
	for (i = 0; i < len; ++ i) {
		iSum += fabs(iCoeff[i]);
		qSum += fabs(qCoeff[i]);
	}
	if (qSum > iSum)
		iSum = qSum;
	for (i = 0; i < len; ++ i) {
		iCoeff[i] /= iSum;
		qCoeff[i] /= iSum;
	}

	fltr = filter_init_double(len, dec, iCoeff, qCoeff);
	free(iCoeff);
	free(qCoeff);
	return fltr;
}

void filter_free(struct filter *f)
{
	if (f) {
		free(f->ifilter);
		free(f->qfilter);
		free(f);
	}
}

BOOL filter_run(struct filter *f, cmplx in, cmplx *out)
{
	int *iptr = f->ibuffer + f->pointer;
	int *qptr = f->qbuffer + f->pointer;

	_ASSERT(in.re < 32768 && in.re > - 32768);
	_ASSERT(in.im < 32768 && in.im > - 32768);

	++ f->pointer;
	++ f->counter;

	*iptr = in.re;
	*qptr = in.im;

	if (f->counter == f->decimateratio) {
		out->re = mac(iptr - f->length, f->ifilter, f->length);
		out->im = mac(qptr - f->length, f->qfilter, f->length);
	}

	if (f->pointer == BufferLen) {
		iptr = f->ibuffer + BufferLen - f->length;
		qptr = f->qbuffer + BufferLen - f->length;
		memcpy(f->ibuffer, iptr, f->length * sizeof(int));
		memcpy(f->qbuffer, qptr, f->length * sizeof(int));
		f->pointer = f->length;
	}

	if (f->counter == f->decimateratio) {
		f->counter = 0;
		return TRUE;
	}

	return FALSE;
}

BOOL filter_I_run(struct filter *f, int in, int *out)
{
	int *iptr = f->ibuffer + f->pointer;

	_ASSERT(in < 32768 && in > - 32768);

	++ f->pointer;
	++ f->counter;

	*iptr = in;

	if (f->counter == f->decimateratio)
		*out = mac(iptr - f->length, f->ifilter, f->length);

	if (f->pointer == BufferLen) {
		iptr = f->ibuffer + BufferLen - f->length;
		memcpy(f->ibuffer, iptr, f->length * sizeof(int));
		f->pointer = f->length;
	}

	if (f->counter == f->decimateratio) {
		f->counter = 0;
		return TRUE;
	}

	return FALSE;
}
