/*
 *    mfsk.c  --  MFSK modem
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>

#include "mfsk.h"
#include "../main/trx.h"
#include "../misc/filter.h"
#include "../misc/viterbi.h"
#include "../main/sndmodem.h"
#include "../main/trxctrl.h"
#include "interleave.h"

static void mfsk_txinit(Trx *trx)
{
	Mfsk *m = (Mfsk*) trx->modem;
	m->txstate	= TX_STATE_PREAMBLE;
	m->bitstate = 0;
	m->counter	= 0;
	m->phaseacc = 0;
	encoder_reset(m->enc);
	interleave_reset(m->txinlv);

	if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202 || g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER) {
		int f0 = (int)trx->nTxFrequency - 250 * 8 + 125;
		if (f0 < 0)
			f0 = 0;
		trxctrl_ats3_start_mfsk(&g_TrxCtrl, trx->pModem, f0, 250 /* 15.625 */);
	}
}

static void mfsk_rxinit(Trx *trx)
{
	Mfsk *m = (Mfsk*) trx->modem;
	m->rxstate = RX_STATE_DATA;
	m->synccounter = 0;
	m->symcounter = 0;
	m->met1 = 0;
	m->met2 = 0;
	m->counter = 0;
	m->pipeptr = 0;
}

static void mfsk_free(Mfsk *s)
{
	if (! s)
		return;
	free(s->pipe);
	free(s->dftcoeffs);
	encoder_free(s->enc);
	viterbi_free(s->dec1);
	viterbi_free(s->dec2);
	filter_free(s->filt);
	free(s);
}

static void mfsk_destructor(Trx *trx)
{
	Mfsk *s = (Mfsk*)trx->modem;
	mfsk_free(s);
	memset(trx, 0, sizeof(trx));
}

void mfsk_init_dft(Mfsk *s)
{
	int		 i, j;
	int		 fftlen  = s->symlen >> MFSK_UNDERSAMPLING;
	double	*aCoeffs = malloc(2 * sizeof(double) * s->numtones * fftlen);
	double  *aWindow = malloc(sizeof(double) * fftlen);
	double	 dSumMax = 0.0;
	cmplx   *z;
	double  *c;

	for (i = 0; i < (fftlen >> 1); ++ i) {
		aWindow[i] = (float)(0.54 - 0.46 * cos(M_PI * (double)i / (double)((fftlen >> 1) - 1)));
		//FIXME
		aWindow[i] = 1.0;
		aWindow[fftlen - i - 1] = aWindow[i];
	}

	for (i = 0; i < s->numtones; ++ i) {
		double  tone   = s->basetone + i;
		double *coeff  = aCoeffs + i * fftlen * 2;
		double  dSumRe = 0.0;
		double  dSumIm = 0.0;
		for (j = 0; j < fftlen; ++ j) {
			double phase = - (double)j * tone * 2.0 * M_PI / (double)fftlen;
			double dCos  = cos(phase) * aWindow[j];
			double dSin  = sin(phase) * aWindow[j];
			*coeff ++ = dCos;
			*coeff ++ = dSin;
			dSumRe += fabs(dCos);
			dSumIm += fabs(dSin);
		}
		if (dSumRe > dSumMax)
			dSumMax = dSumRe;
		if (dSumIm > dSumMax)
			dSumMax = dSumIm;
	}

	s->dftcoeffs = (cmplx*)malloc(sizeof(cmplx) * s->numtones * fftlen);
	z = s->dftcoeffs;
	c = aCoeffs;
	for (i = 0; i < s->numtones; ++ i) {
		for (j = 0; j < fftlen; ++ j, ++ z) {
			double dValue = (*c ++) * (double)0x7fff / dSumMax;
			int    iValue;
			if (dValue > 0.0) {
				iValue = (int)floor(dValue + 0.5);
				if (iValue > 0x7fff)
					iValue = 0x7fff;
			} else {
				iValue = (int)floor(dValue - 0.5);
				if (iValue < -0x7fff)
					iValue = -0x7fff;
			}
			z->re = iValue;

			dValue = (*c ++) * (double)0x7fff / dSumMax;
			if (dValue > 0.0) {
				iValue = (int)floor(dValue + 0.5);
				if (iValue > 0x7fff)
					iValue = 0x7fff;
			} else {
				iValue = (int)floor(dValue - 0.5);
				if (iValue < -0x7fff)
					iValue = -0x7fff;
			}
			z->im = iValue;
		}
	}

	free(aCoeffs);
	free(aWindow);
}

void mfsk_init(Trx *trx)
{
	Mfsk *s;
	double bw, fhi;

	s = (Mfsk*)malloc(sizeof(Mfsk));
	memset(s, 0, sizeof(Mfsk));

	switch (trx->mode) {
	case MODE_MFSK16:
		s->symlen   = 512;
		s->symbits  = 4;
		s->basetone = -7;
        break;

	case MODE_MFSK8:
		s->symlen   = 1024;
		s->symbits  = 5;
		s->basetone = 128 >> MFSK_UNDERSAMPLING;		/* 1000 Hz / MFSK_UNDERSAMPLING */
        break;

	default:
		mfsk_free(s);
		return;
	}

	s->numtones = 1 << s->symbits;
	s->tonespacing = (SampleRate << FREQ_FRAC_BITS) / s->symlen;

	s->pipe = calloc(sizeof(cmplx), 3 * (s->symlen >> MFSK_UNDERSAMPLING));

	if (!(s->enc = encoder_init(K, POLY1, POLY2))) {
//		g_warning("mfsk_init: encoder_init failed\n");
		mfsk_free(s);
		return;
	}
	if (!(s->dec1 = viterbi_init(K, POLY1, POLY2))) {
//		g_warning("mfsk_init: viterbi_init failed\n");
		mfsk_free(s);
		return;
	}
	if (!(s->dec2 = viterbi_init(K, POLY1, POLY2))) {
//		g_warning("mfsk_init: viterbi_init failed\n");
		mfsk_free(s);
		return;
	}
	viterbi_set_traceback(s->dec1, 45);
	viterbi_set_traceback(s->dec2, 45);
	viterbi_set_chunksize(s->dec1, 1);
	viterbi_set_chunksize(s->dec2, 1);

	if (!(s->txinlv = interleave_init(s->symbits, INTERLEAVE_FWD))) {
//		g_warning("mfsk_init: interleave_init failed\n");
		mfsk_free(s);
		return;
	}
	if (!(s->rxinlv = interleave_init(s->symbits, INTERLEAVE_REV))) {
//		g_warning("mfsk_init: interleave_init failed\n");
		mfsk_free(s);
		return;
	}

	bw = (s->numtones - 1) * s->tonespacing / 16.0; // >> FREQ_FRAC_BITS
	fhi = 0.9 * (bw + 1) / SampleRate;

	if ((s->filt = filter_init_lowpass(127, 1 << MFSK_UNDERSAMPLING, (float)fhi)) == NULL) {
//		g_warning("mfsk_init: filter_init failed\n");
		mfsk_free(s);
		return;
	}

	// set DFT coefficients
	mfsk_init_dft(s);

	trx->bandwidth = s->tonespacing * 15;

	trx->modem = s;

	trx->txinit = mfsk_txinit;
	trx->rxinit = mfsk_rxinit;

	trx->txprocess = mfsk_txprocess;
	trx->rxprocess = mfsk_rxprocess;

	trx->destructor = mfsk_destructor;

//	trx->samplerate = SampleRate;
//	trx->fragmentsize = s->symlen;
//	trx->bandwidth = (s->numtones - 1) * s->tonespacing;
}
