/*
 *    trx.c  --  Modem engine
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <windows.h>
#include <tchar.h>

#include "trx.h"
#include "sndmodem.h"
#include "trxctrl.h"
#include "qsodata.h"
#include "ascii.h"
#include "../misc/misc.h"
#include "../misc/mixer.h"
#include "../resource.h"


#ifdef _WIN32_WCE
#ifdef WIN32_PLATFORM_PSPC
#include "Aygshell.h"
#endif /* WIN32_PLATFORM_PSPC */
#else /* _WIN32_WCE */
#include <stdio.h>
#endif /* _WIN32_WCE */

extern void mfsk_init(Trx *trx);
extern void rtty_init(Trx *trx);
extern void throb_init(Trx *trx);
extern void psk31_init(Trx *trx);
extern void cw_init(Trx *trx);
extern void mt63_init(Trx *trx);
extern void feld_init(Trx *trx);

/* ---------------------------------------------------------------------- */

TCHAR *trx_mode_names[] = {
	_T("BPSK31"),
	_T("QPSK31"),
	_T("PSK63"),
	_T("RTTY"),
	_T("CW"),
	_T("MFSK16"),
	_T("THROB"),
	_T("THROBX"),
	_T("FELDHELL"),
	_T("OLIVIA"),
	_T("CONTESTIA"),
	_T("RTTYM"),
	_T("AX25"),
	_T("MFSK8"),
	_T("MT63"),
	_T("FMHELL")
};

TCHAR *trx_state_names[] = {
	_T("PAUSED"),
	_T("RECEIVE"),
	_T("TRANSMIT"),
	_T("TUNING"),
	_T("ABORTED"),
	_T("FLUSHING")
};

TCHAR *submode_throb_names[] = {
	_T("1 tps"),
	_T("2 tps"),
	_T("4 tps"),
};

TCHAR *submode_olivia_names[] = {
	_T("4-250"),
	_T("4-500"),
	_T("16-500"),
	_T("16-1K"),
	_T("8-250"),
	_T("8-500"),
	_T("32-1K"),
};

TCHAR *submode_rtty_names[] = {
	_T("HAM 45/170"),
	_T("HAM 45/200"),
	_T("DWD 50/85"),
	_T("DWD 50/450"),
};

/// Forward reference to the trx window procedure
LRESULT CALLBACK trx_wndproc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK rx_wndproc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK tx_wndproc(HWND, UINT, WPARAM, LPARAM);

static const TBYTE	WIN32CLASS_TRX[]	= _T("USR_TRX");
static const TBYTE	WIN32CLASS_RX[]		= _T("USR_RX");
static const TBYTE	WIN32CLASS_TX[]		= _T("USR_TX");
static WNDPROC		edit_wndproc		= 0;

BOOL trx_register_classes(HINSTANCE hInstance)
{
  WNDCLASS	wc, wcEdit;

  wc.style			= CS_HREDRAW | CS_VREDRAW;
  wc.lpfnWndProc	= (WNDPROC) trx_wndproc;
  wc.cbClsExtra		= 0;
  wc.cbWndExtra		= 0;
  wc.hInstance		= hInstance;
  wc.hIcon			= 0;
  wc.hCursor		= 0;
  wc.hbrBackground	= (HBRUSH) GetStockObject(WHITE_BRUSH);
  wc.lpszMenuName	= 0;
  wc.lpszClassName	= WIN32CLASS_TRX;

  if (RegisterClass(&wc) == 0)
	  return FALSE;

  if (! GetClassInfo(hInstance, _T("EDIT"), &wcEdit)) 
	  return FALSE;

  memcpy(&wc, &wcEdit, sizeof(wc));
  wc.lpszClassName  = WIN32CLASS_RX;
  edit_wndproc		= wc.lpfnWndProc;
  wc.lpfnWndProc	= (WNDPROC)rx_wndproc;
  if (RegisterClass(&wc) == 0)
	  return FALSE;

  wc.lpszClassName  = WIN32CLASS_TX;
  wc.lpfnWndProc	= (WNDPROC)tx_wndproc;
  if (RegisterClass(&wc) == 0)
	  return FALSE;

  return TRUE;
}

Trx* trx_new()
{
	Trx *trx = (Trx*)malloc(sizeof(Trx));
	if (trx == 0)
		return 0;
	if (! trx_init(trx)) {
		trx_destroy(trx);
		return 0;
	}
	trx_initmodem(trx);
	return trx;
}

void trx_free(Trx *trx)
{
	if (trx == 0)
		return;
	trx_destroy(trx);
	free(trx);
}

BOOL trx_init(Trx *trx)
{
	memset(trx, 0, sizeof(Trx));
	trx->mode		= MODE_BPSK31;
	trx->state		= TRX_STATE_RX;
//	trx.samplerate	= 8000;
//	trx->afcon		= TRUE;
	trx->squelchon	= FALSE;
	trx->iSquelchThreshold = 50;
	trx->nSyncSpeed	= 1; // normal
	trx->reverse	= FALSE;
	trx->nRxFrequency = 1500 << FREQ_FRAC_BITS;
	trx->nTxFrequency = trx->nRxFrequency;
	trx->cw_rxspeed   = 4; // 10 WPM
	trx->cw_txspeed   = 18;
	trx->cw_bandwidth = 3; // 70 Hz
	trx->cw_firtaps   = 0; // 64 taps

	trx->pQso	 = qsodata_new();
	mixer_initialize();
	return TRUE;
}

BOOL trx_initmodem(Trx *trx)
{
	trx->state = TRX_STATE_RX;

	if (trx->destructor != 0) {
		trx->destructor(trx);
		trx->txinit		= 0;
		trx->rxinit		= 0;
		trx->txprocess	= 0;
		trx->rxprocess	= 0;
		trx->destructor = 0;
	}

	switch (trx->mode) {
	case MODE_RTTY:
		rtty_init(trx);
		break;

	case MODE_BPSK31:
	case MODE_QPSK31:
	case MODE_PSK63:
		psk31_init(trx);
		break;

	case MODE_CW:
		cw_init(trx);
		break;

	case MODE_MFSK16:
	case MODE_MFSK8:
		mfsk_init(trx);
		break;

	case MODE_FELDHELL:
//	case MODE_FMHELL:
		feld_init(trx);
		break;

	case MODE_THROB:
	case MODE_THROBX:
		throb_init(trx);
		break;

	case MODE_OLIVIA:
	case MODE_CONTESTIA:
	case MODE_RTTYM:
		olivia_init(trx);
		break;

	case MODE_AX25:
		ax25_init(trx);
		break;

/*
	case MODE_MT63:
		mt63_init(trx);
		break;
*/
	}

	trx->rxinit(trx);

	return TRUE;
}

BOOL trx_create(Trx *trx, HWND hWndParent, HANDLE hInstance, SndModem *pModem)
{
	trx->hWnd = CreateWindow(WIN32CLASS_TRX, NULL, WS_CHILD | WS_CLIPCHILDREN | WS_CLIPSIBLINGS | WS_VISIBLE,
		0, 0, 10, 10, hWndParent, NULL, hInstance, (void*)trx);
	if (! trx->hWnd)
		return FALSE;
#ifdef USE_RXVIEW
	trx->pRxView = rxview_new();
	rxview_create(trx->pRxView, WS_CHILD | WS_VISIBLE | WS_VSCROLL, 0, 0, 10, 10, trx->hWnd, hInstance);	
#else /* USE_RXVIEW */
	trx->hWndReceive = CreateWindow(WIN32CLASS_RX, NULL, 
		WS_BORDER | WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE | WS_VSCROLL | 
		ES_AUTOVSCROLL | ES_LEFT | ES_MULTILINE,
		0, 0, 10, 10, trx->hWnd, NULL, hInstance, (void*)trx);
#endif /* USE_RXVIEW */
	trx->hWndSend    = CreateWindow(WIN32CLASS_TX, NULL, 
		WS_BORDER | WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE | WS_VSCROLL |
		ES_AUTOVSCROLL | ES_LEFT | ES_MULTILINE | ES_WANTRETURN,
		0, 0, 10, 10, trx->hWnd, NULL, hInstance, (void*)trx);

	trx->pModem = pModem;
	trx->iLineReceivedIdx = 0;
	trx->nLineReceivedLen = 1024 * 64;

	{
		LOGFONT logfont;
		memset(&logfont, 0, sizeof(LOGFONT));
		logfont.lfHeight  = -11;
		logfont.lfWeight  = FW_NORMAL;
		logfont.lfCharSet = ANSI_CHARSET;
		_tcscpy(logfont.lfFaceName, _T("Tahoma"));
		trx->hFont = CreateFontIndirect(&logfont);
	}

	SendMessage(trx->hWnd,			WM_SETFONT, (WPARAM)trx->hFont, FALSE);
#ifdef USE_RXVIEW
#else /* USE_RXVIEW */
	SendMessage(trx->hWndReceive,	WM_SETFONT, (WPARAM)trx->hFont, FALSE);
	SendMessage(trx->hWndReceive,	EM_SETLIMITTEXT, 64000, FALSE);
	SendMessage(trx->hWndReceive,	EM_SETREADONLY, (WPARAM)FALSE, 0);
#endif /* USE_RXVIEW */
	SendMessage(trx->hWndSend,		WM_SETFONT, (WPARAM)trx->hFont, FALSE);
	SendMessage(trx->hWndSend,		EM_SETLIMITTEXT, 64000, FALSE);
	
	SetWindowLong(trx->hWnd, GWL_USERDATA, (long)trx);

	return TRUE;
}

void trx_destroy(Trx *trx)
{
	SetWindowLong(trx->hWnd, GWL_USERDATA, 0l);
	DeleteObject(trx->hFont);
#ifdef USE_RXVIEW
	rxview_free(trx->pRxView);
#else /* USE_RXVIEW */
	DestroyWindow(trx->hWndReceive);
#endif /* USE_RXVIEW */
	DestroyWindow(trx->hWndSend);
	DestroyWindow(trx->hWnd);
	trx->destructor(trx);
	qsodata_destroy(trx->pQso);
	log_to_file_activate(&trx->log, FALSE, 0);
	memset(trx, 0, sizeof(Trx));
}

BOOL trx_layout(Trx *trx, layout_t nLayout, RECT *rect)
{
	HDC			hDC		 = GetWindowDC(trx->hWnd);
	HGDIOBJ		hFontOld = SelectObject(hDC, trx->hFont);
	int		    nWidth   = rect->right - rect->left;
	int		    nHeight  = rect->bottom - rect->top;
	TEXTMETRIC  tm;

	trx->nLayout = nLayout;

	GetTextMetrics(hDC, &tm);

	if (nLayout == LAYOUT_SINGLELINE) {
		rect->bottom = rect->top + tm.tmHeight;
#ifdef USE_RXVIEW
		ShowWindow(trx->pRxView->hWnd, SW_HIDE);
#else /* USE_RXVIEW */
		ShowWindow(trx->hWndReceive, SW_HIDE);
#endif /* USE_RXVIEW */
		ShowWindow(trx->hWndSend, SW_HIDE);
	} else if (nLayout == LAYOUT_RCV) {
		ShowWindow(trx->hWndSend, SW_HIDE);
#ifdef USE_RXVIEW
		MoveWindow(trx->pRxView->hWnd, -1, 0, nWidth + 2, nHeight + 1, TRUE);
		ShowWindow(trx->pRxView->hWnd, SW_SHOW);
#else /* USE_RXVIEW */
		MoveWindow(trx->hWndReceive, -1, 0, nWidth + 2, nHeight + 1, TRUE);
		ShowWindow(trx->hWndReceive, SW_SHOW);
#endif /* USE_RXVIEW */
	} else if (nLayout == LAYOUT_TRX) {
		int nHalf = nHeight / 2;
#ifdef USE_RXVIEW
		MoveWindow(trx->pRxView->hWnd, -1, 0, nWidth + 2, nHalf + 1, TRUE);
		ShowWindow(trx->pRxView->hWnd, SW_SHOW);
#else /* USE_RXVIEW */
		MoveWindow(trx->hWndReceive, -1, 0, nWidth + 2, nHalf + 1, TRUE);
		ShowWindow(trx->hWndReceive, SW_SHOW);
#endif /* USE_RXVIEW */
//		UpdateWindow(trx->hWndReceive);
		MoveWindow(trx->hWndSend, -1, nHalf, nWidth + 2, nHeight - nHalf + 1, TRUE);
		ShowWindow(trx->hWndSend, SW_SHOW);
//		SetWindowPos(trx->hWndSend, HWND_TOP, 0, nHalf, nWidth, nHeight - nHalf, SWP_SHOWWINDOW);
//		UpdateWindow(trx->hWndSend);
//		SetFocus(trx->hWndSend);
	}

	SelectObject(hDC, hFontOld);
	ReleaseDC(trx->hWnd, hDC);
	return TRUE;
}

#ifndef USE_RXVIEW
static void trx_display_rxchar(Trx *trx, UINT data, BOOL bIsEcho)
{
	int nLength = SendMessage(trx->hWndReceive, WM_GETTEXTLENGTH, 0, 0);

	if (GetFocus() == trx->hWndReceive) {
		// While having focus, do not push received data into the edit box. This would invalidate selection, scroll bars, cursor position.
		if (trx->iLineReceivedIdx < trx->nLineReceivedLen) {
			trx->aLineReceived[trx->iLineReceivedIdx ++] = bIsEcho;
			trx->aLineReceived[trx->iLineReceivedIdx ++] = data;
			if (trx->iLineReceivedIdx == trx->nLineReceivedLen)
				SetFocus(s_aChannels[s_iChannelActive]->hWndSend);

		}
		return;
	}

/*
#ifndef _WIN32_WCE
	{
		char bufout[256];
		sprintf(bufout, "Char received: %d: '%c'\n", data, data);
		OutputDebugString(bufout);
	}
#endif /* _WIN32_WCE */

/*
	if (trx->bPrevCharEcho != bIsEcho &&
		trx->cPrevChar != '\r' && trx->cPrevChar != '\n' && data != '\r' && data != '\n') {
		// Switching between RX/TX. Add crlf if it has not been 
		TCHAR buf[2] = { '\r', 0 };
		SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE, (LPARAM)&buf);
	}
	
	trx->bPrevCharEcho = bIsEcho;
*/

	if (data == '\r' || (trx->cPrevChar != '\r' && data == '\n')) {
		// send crlf
		TCHAR buf[3] = { '\r', '\n', 0 };
		SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE, (LPARAM)&buf);
		trx->cPrevChar = data;
		trx->bPrevCharEcho = bIsEcho;
		return;
	}

	if (trx->cPrevChar == '\r' && data == '\n') {
		// crlf was already entered into the receive window
		trx->cPrevChar     = '\n';
		trx->bPrevCharEcho = bIsEcho;
		return;
	}

	if (trx->bPrevCharEcho != bIsEcho && trx->cPrevChar != '\r' && trx->cPrevChar != '\n') {
		// Switching between RX/TX. Add crlf if it has not been added yet.
		TCHAR buf[3] = { '\r', '\n', 0 };
		SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE, (LPARAM)&buf);
	}

	trx->bPrevCharEcho = bIsEcho;
	trx->cPrevChar = data;

	if (data == '\b') {
		if (nLength > 0) {
			UINT empty = 0;
			SendMessage(trx->hWndReceive, EM_SETSEL, nLength - 1, nLength);
			SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)&empty);
		}
	} else {
		SendMessage(trx->hWndReceive, EM_SETSEL, SendMessage(trx->hWndReceive, WM_GETTEXTLENGTH, 0, 0), -1);
		if (data < 32) {
			if (0)
				//FIXME make it configurable
				SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)ascii[data]);
		} else {
			TCHAR buf[2] = { data, 0 };
			SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)&buf);
		}
	}

	if (trx->nLayout == LAYOUT_SINGLELINE) {
		HDC			hDC      = GetDC(trx->hWnd);
		HGDIOBJ		hFontOld = SelectObject(hDC, trx->hFont);
		RECT		rect, rectWnd, rectClip;
		TCHAR       label[3] = _T("0:");
		TCHAR		newchar[2] = _T("0");
		int         nChannelLabelWidth, nNewCharWidth, nShift;

		// Get window rectangle
		GetClientRect(trx->hWnd, &rectWnd);
		memcpy(&rect, &rectWnd, sizeof(rect));

		// Compute width of the channel label
		label[0] += trx->nChannelID;
		DrawText(hDC, label, 2, &rect, DT_CALCRECT | DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);
		nChannelLabelWidth = rect.right - rect.left;

		// Compute width of the new character
		memcpy(&rect, &rectWnd, sizeof(rect));
		newchar[0] = data;
		DrawText(hDC, newchar, 1, &rect, DT_CALCRECT | DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);
		nNewCharWidth = rect.right - rect.left;

		nShift = rectWnd.right - rectWnd.left - (trx->nLineOutPos + nNewCharWidth);
		if (nShift < 0) {
			// Scroll the text (without the channel label)
			memcpy(&rect, &rectWnd, sizeof(rect));
			memcpy(&rectClip, &rectWnd, sizeof(rect));
			rect.left     = nChannelLabelWidth;
			rectClip.left = nChannelLabelWidth;
			ScrollWindowEx(trx->hWnd, nShift, 0, 0, &rectClip, 0, 0, 0);
			trx->nLineOutPos = rectWnd.right - rectWnd.left - nNewCharWidth;
		}
		// Draw the new character
		memcpy(&rect, &rectWnd, sizeof(rect));
		rect.left = trx->nLineOutPos;
		DrawText(hDC, newchar, 1, &rect, DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);
		trx->nLineOutPos += nNewCharWidth;

		SelectObject(hDC, hFontOld);
		ReleaseDC(trx->hWnd, hDC);
	}

/*
	static int iOldPosRight = 0;
	HDC hDC = ::GetDC(s_hWndMain);
	bool bScroll = false;
	RECT rectLine;
	rectLine.left   = iOldPosRight;
	rectLine.right  = 300;
	rectLine.bottom = 320 - 2 * MENU_HEIGHT;
	rectLine.top    = rectLine.bottom - 20;
	if (data == 13 || data == 10)
		bScroll = true;
	else {
		::DrawText(hDC, (TCHAR*)&data, 1, &rectLine, DT_BOTTOM | DT_LEFT | DT_CALCRECT | DT_SINGLELINE);
		if (rectLine.right > 240) {
			rectLine.right	= rectLine.right - rectLine.left;
			rectLine.left	= 0;
			bScroll			= true;
		}
	}
	if (bScroll) {
		RECT rect;
		rect.left   = 0;
		rect.top    = 180;
		rect.right  = 240;
		rect.bottom = 320 - 2 * MENU_HEIGHT;
		::ScrollWindowEx(s_hWndMain, 0, -20, &rect, 0, 0, 0, SW_ERASE | SW_INVALIDATE);
		::UpdateWindow(s_hWndMain);
	}
	if (data != 13 && data != 10)
		::DrawText(hDC, (TCHAR*)&data, 1, &rectLine, DT_BOTTOM | DT_LEFT | DT_SINGLELINE);
	::ReleaseDC(s_hWndMain, hDC);

	iOldPosRight = rectLine.right;
*/
}
#endif /* USE_RXVIEW */

void trx_put_rx_char(Trx *trx, UINT c)
{
#ifdef USE_RXVIEW
	rxview_add_text(trx->pRxView, 0, &c, 1);
#else /* USE_RXVIEW */	
	trx_display_rxchar(trx, c, FALSE);
#endif /* USE_RXVIEW */
	if (c < 32) {
		log_to_file(&trx->log, LOG_RX, ascii[c]);
	} else {
		TCHAR buf[2] = { c, 0 };
		log_to_file(&trx->log, LOG_RX, buf);
	}
}

void trx_put_echo_char(Trx *trx, UINT c)
{
#ifdef USE_RXVIEW
	rxview_add_text(trx->pRxView, 1, &c, 1);
#else /* USE_RXVIEW */	
	trx_display_rxchar(trx, c, TRUE);
#endif /* USE_RXVIEW */
	if (c < 32) {
		log_to_file(&trx->log, LOG_TX, ascii[c]);
	} else {
		TCHAR buf[2] = { c, 0 };
		log_to_file(&trx->log, LOG_TX, buf);
	}
}

void trx_send_char(Trx *trx, TCHAR c)
{
	int   nLength = SendMessage(trx->hWndSend, WM_GETTEXTLENGTH, 0, 0);
	TCHAR data[2] = { c, 0 };
	SendMessage(trx->hWndSend, EM_SETSEL, SendMessage(trx->hWndSend, WM_GETTEXTLENGTH, 0, 0), -1);
	SendMessage(trx->hWndSend, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)data);
}

void trx_send_string(Trx *trx, TCHAR *str)
{
	int nLength = SendMessage(trx->hWndSend, WM_GETTEXTLENGTH, 0, 0);
	SendMessage(trx->hWndSend, EM_SETSEL, SendMessage(trx->hWndSend, WM_GETTEXTLENGTH, 0, 0), -1);
	SendMessage(trx->hWndSend, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)str);
}

void trx_clearrx(Trx *trx)
{
#ifdef USE_RXVIEW
#else /* USE_RXVIEW */
	SetWindowText(trx->hWndReceive, _T(""));
#endif /* USE_RXVIEW */
	trx->cPrevChar		= 0;
	trx->bPrevCharEcho	= FALSE;
}

void trx_cleartx(Trx *trx)
{
	SetWindowText(trx->hWndSend, _T(""));
	trx->nCharsSent = 0;
}

BOOL trx_is_sending(Trx *trx)
{
	return trx != 0 && trx->state != TRX_STATE_PAUSE && trx->state != TRX_STATE_RX;
}

BOOL trx_txloop(Trx *trx)
{
	BOOL bStop = FALSE;
	while (sndmodem_tx_buffer_underflowed(trx->pModem)) {
		if (trx->state == TRX_STATE_TUNE) {
			if (trx->stopflag)
				bStop = TRUE;
			else {
				// fill one buffer with the center tone
				int symbollen = 256;
				if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202) {
					sndmodem_write_bell202_async(trx->pModem, 0x02, 3); // TX on
					sndmodem_write_bell202_pause_thirds(trx->pModem, symbollen * 3);
				} else if (g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER) {
					sndmodem_write_manchester_async(trx->pModem, 0x02, 3); // TX on
					sndmodem_write_manchester_pause(trx->pModem, symbollen);
				} else {
					static UINT iPhase	    = 0;
					UINT		iPhaseDelta = trx_freq2phase(trx->nTxFrequency);
					int		    i;
					for (i = 0; i < symbollen; i ++) {
						trx->pModem->pOutBuf[i] = MIXER_OSC_SHORT(iPhase);
						iPhase += iPhaseDelta;
					}
					sndmodem_write(trx->pModem, trx->pModem->pOutBuf, symbollen);
				}
			}
		} else
			bStop = trx->txprocess(trx) < 0;
		if (bStop) {
			trx->state = TRX_STATE_FLUSH;
			if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202 || g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER)
				trxctrl_ats3_endtx(&g_TrxCtrl, trx->pModem);
			sndmodem_tx_flush(trx->pModem);
			break;
		}
	}
	return TRUE;
}

void trx_txend(Trx *trx)
{
	trx->backspaces = 0;
	trx->stopflag	= FALSE;
	trx->state		= TRX_STATE_RX;
}

int trx_get_tx_char(Trx *trx)
{
	//FIXME
/*
	static int iChar = 'a';
	int iRes = iChar;
	if (iChar == 'z' || iChar == -1)
		iChar = -1;
	else
		++ iChar;
	return iChar;
*/
	int     nLength = SendMessage(trx->hWndSend, WM_GETTEXTLENGTH, 0, 0);
	LPCTSTR lpszText;
	HANDLE  hData;
	TCHAR   iChar   = -1;
	if (nLength <= trx->nCharsSent) {
		trx->nCharsSent = nLength;
		return -1;
	}
	SendMessage(trx->hWndSend, EM_SETSEL, trx->nCharsSent, trx->nCharsSent + 1);
	SendMessage(trx->hWndSend, WM_COPY, 0, 0);
	SendMessage(trx->hWndSend, EM_SETSEL, nLength, -1);
	OpenClipboard(NULL);
	hData = GetClipboardData(
#ifdef _WIN32_WCE
		CF_UNICODETEXT
#else /* _WIN32_WCE */
		CF_TEXT
#endif /* _WIN32_WCE */
		);
	if (hData != 0) {
		lpszText = (LPTSTR)GlobalLock(hData);
		iChar = lpszText[0];
		GlobalUnlock(hData);
    }
	EmptyClipboard();
	CloseClipboard();
	++ trx->nCharsSent;
	if (iChar == TRX_RX_CMD || iChar == TRX_CLEAR_CMD) {
		s_aChannels[s_iChannelActive]->stopflag = TRUE;
		UpdateMainMenuBar(2);
		if (iChar == TRX_CLEAR_CMD)
			trx_cleartx(trx);
		return -1;
	}
	return iChar;
}

static void draw(Trx *trx, HDC hDC)
{
#ifndef USE_RXVIEW
	if (trx->nLayout == LAYOUT_SINGLELINE) {
		HGDIOBJ		hFontOld = SelectObject(hDC, trx->hFont);
		RECT		rectWnd, rect;
		TCHAR       label[3] = _T("0:");
		int         nTextLength = SendMessage(trx->hWndReceive, WM_GETTEXTLENGTH, 0, 0);
		TCHAR      *pText = malloc(sizeof(TCHAR) * nTextLength + 1);
		int         nChannelLabelWidth;

		// Get window rectangle
		GetClientRect(trx->hWnd, &rectWnd);
		memcpy(&rect, &rectWnd, sizeof(rect));

		// Compute width of the channel label
		label[0] += trx->nChannelID;
		DrawText(hDC, label, 2, &rect, DT_CALCRECT | DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);
		nChannelLabelWidth = rect.right - rect.left;

		SendMessage(trx->hWndReceive, WM_GETTEXT, (WPARAM)(nTextLength + 1), (LPARAM)pText); 
		memcpy(&rect, &rectWnd, sizeof(rect));
		DrawText(hDC, pText, nTextLength, &rect, DT_CALCRECT | DT_NOCLIP | DT_NOPREFIX | DT_RIGHT | DT_SINGLELINE);
		if (rect.right - rect.left < rectWnd.right - rectWnd.left - nChannelLabelWidth) {
			trx->nLineOutPos = rect.right - rect.left;
			memcpy(&rect, &rectWnd, sizeof(rect));
			rect.left = nChannelLabelWidth;
			DrawText(hDC, pText, nTextLength, &rect, DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);
		} else {
			trx->nLineOutPos = rectWnd.right;
			memcpy(&rect, &rectWnd, sizeof(rect));
			rect.left = nChannelLabelWidth;
			DrawText(hDC, pText, nTextLength, &rect, DT_NOCLIP | DT_NOPREFIX | DT_RIGHT | DT_SINGLELINE);
		}
		free(pText);

		memcpy(&rect, &rectWnd, sizeof(rect));
		SetTextColor(hDC, RGB(255, 255, 255));
		SetBkColor(hDC, RGB(0, 0, 0));
		DrawText(hDC, label, 2, &rect, DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);

		SelectObject(hDC, hFontOld);
	}
#endif /* USE_RXVIEW */
}

LRESULT CALLBACK trx_wndproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	Trx *trx = (Trx*)GetWindowLong(hWnd, GWL_USERDATA);
	if (trx == 0)
		return DefWindowProc(hWnd, message, wParam, lParam);

	if (message == WM_PAINT) {
		PAINTSTRUCT paint;
		draw(trx, BeginPaint(hWnd, &paint));
		EndPaint(hWnd, &paint);
		return 0;
	}
	if (message == WM_LBUTTONDOWN) {
#ifdef _WIN32_WCE
#ifdef WIN32_PLATFORM_PSPC
		SHRGINFO shrg;
		shrg.cbSize		= sizeof(SHRGINFO);
		shrg.dwFlags	= SHRG_RETURNCMD;
		shrg.hwndClient = hWnd;
		shrg.ptDown.x	= LOWORD(lParam);
		shrg.ptDown.y	= HIWORD(lParam);
		if (SHRecognizeGesture(&shrg) == GN_CONTEXTMENU) {
			trx_show_channel_context_menu(hWnd, shrg.ptDown.x, shrg.ptDown.y, trx->nChannelID);
		} else
#endif /* WIN32_PLATFORM_PSPC */
#endif /* _WIN32_WCE */
			SendMessage(GetParent(hWnd), WM_COMMAND, (WPARAM)(ID_CHANNEL0 + trx->nChannelID), 0);
		return 0;
	}
	if (message == WM_RBUTTONDOWN) {
		trx_show_channel_context_menu(hWnd, LOWORD(lParam), HIWORD(lParam), trx->nChannelID);
		return 0;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

static void ShowContextMenu(HWND hWnd, int x, int y, BOOL bIsTX)
{
	HMENU hMenu = CreatePopupMenu();
	int   iCmd;
	POINT pt;
	int   nStartChar, nEndChar;
	int	  nLength		  = GetWindowTextLength(hWnd);
	UINT  uiFlagsEmpty    = MF_ENABLED | MF_STRING;
	UINT  uiFlagsEmptySel = MF_ENABLED | MF_STRING;
	Trx  *trx			  = (Trx*)GetWindowLong(GetParent(hWnd), GWL_USERDATA);
	HANDLE hClipboardData = 0;

	pt.x = x;
	pt.y = y;
	ClientToScreen(hWnd, &pt);

	SendMessage(hWnd, EM_GETSEL, (WPARAM)&nStartChar, (LPARAM)&nEndChar);

	OpenClipboard(NULL);
	hClipboardData = GetClipboardData(
#ifdef _WIN32_WCE
		CF_UNICODETEXT
#else /* _WIN32_WCE */
		CF_TEXT
#endif /* _WIN32_WCE */
		);
	CloseClipboard();

	if (nLength == 0)
		uiFlagsEmpty    |= MF_GRAYED;
	if (nStartChar >= nEndChar)
		uiFlagsEmptySel |= MF_GRAYED;

	AppendMenu(hMenu, uiFlagsEmpty,				ID_CLEAR,		_T("Clear Window"));
	AppendMenu(hMenu, uiFlagsEmptySel,			ID_COPY,		_T("Copy"));
	AppendMenu(hMenu, uiFlagsEmptySel,			ID_CUT,			_T("Cut"));
	AppendMenu(hMenu, MF_ENABLED | MF_STRING | (hClipboardData ? 0 : MF_GRAYED),	
												ID_PASTE,		_T("Paste"));
	AppendMenu(hMenu, uiFlagsEmpty,				ID_SELECT_ALL,	_T("Select All"));

	if (! bIsTX && nStartChar < nEndChar) {
		AppendMenu(hMenu, MF_ENABLED | MF_SEPARATOR, 0, 0);
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_CALL,		_T("Call"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_NAME,		_T("Name"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_QTH,			_T("QTH"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_LOC,			_T("Loc"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_RST_SENT,	_T("TX RST"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_RST_RCVD,	_T("RX RST"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_NOTES,		_T("Notes"));
	}

	iCmd = TrackPopupMenuEx(hMenu, TPM_LEFTALIGN | TPM_RETURNCMD, pt.x, pt.y, hWnd, 0);
	DestroyMenu(hMenu);

	switch (iCmd) {
	case ID_CLEAR:
		if (bIsTX)
			trx_cleartx(trx);
		else
			trx_clearrx(trx);
		break;
	case ID_COPY:
		SendMessage(hWnd, WM_COPY, 0, 0);
		break;
	case ID_CUT:
		SendMessage(hWnd, WM_CUT, 0, 0);
		break;
	case ID_PASTE:
		SendMessage(hWnd, WM_PASTE, 0, 0);
		break;
	case ID_SELECT_ALL:
		SendMessage(hWnd, EM_SETSEL, (WPARAM)(INT)0, (LPARAM)(INT)-1);
		break;
	case ID_COPY_CALL:
	case ID_COPY_NAME:
	case ID_COPY_QTH:
	case ID_COPY_LOC:
	case ID_COPY_RST_SENT:
	case ID_COPY_RST_RCVD:
	case ID_COPY_NOTES:
	{
		SendMessage(hWnd, EM_SETSEL, nStartChar, nEndChar);
		SendMessage(hWnd, WM_COPY, 0, 0);
		OpenClipboard(NULL);
		hClipboardData = GetClipboardData(
#ifdef _WIN32_WCE
		CF_UNICODETEXT
#else /* _WIN32_WCE */
		CF_TEXT
#endif /* _WIN32_WCE */
		);
		if (hClipboardData != 0) {
			LPTSTR pString = (LPTSTR)GlobalLock(hClipboardData);
			int    nLen    = _tcslen(pString);
			int    iLine   = iCmd - ID_COPY_CALL;
			QsoInfo *info  = trx->pQso;
			GlobalUnlock(hClipboardData);
			QSOINFO_ARRAY[iLine] = (TCHAR*)realloc(
				QSOINFO_ARRAY[iLine], (nLen + 1) * sizeof(TCHAR));
			memcpy(QSOINFO_ARRAY[iLine], pString, (nLen + 1) * sizeof(TCHAR));
		}
		CloseClipboard();
		if (iCmd == ID_COPY_CALL)
			UpdateMenuChannels();
		break;
	}
	default:
		break;
	}
}

void trx_show_channel_context_menu(HWND hWnd, int ptx, int pty, int iChannel)
{
	HMENU hMenu = CreatePopupMenu();
//	TCHAR buf[256];
	POINT pt;
	int   iCmd;

	pt.x = ptx;
	pt.y = pty;
	ClientToScreen(hWnd, &pt);

//	_stprintf(buf, _T("Channel %d"), iChannel);
//	AppendMenu(hMenu, MF_STRING, 0, buf);
//	AppendMenu(hMenu, MF_SEPARATOR, 0, 0);
	AppendMenu(hMenu, MF_ENABLED  | MF_STRING,    ID_CHANNEL0 + iChannel,		_T("Activate"));
	AppendMenu(hMenu, MF_ENABLED  | MF_STRING,    ID_CHANNEL_CLOSE0 + iChannel,	_T("Close"));
	iCmd = TrackPopupMenuEx(hMenu, TPM_LEFTALIGN | TPM_RETURNCMD, pt.x, pt.y, hWnd, 0);
	DestroyMenu(hMenu);
	if (iCmd != 0)
		PostMessage(s_hWndMain, WM_COMMAND, iCmd, 0);
}

LRESULT CALLBACK rx_wndproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	Trx *trx = (Trx*)GetWindowLong(GetParent(hWnd), GWL_USERDATA);

#ifdef _WIN32_WCE
#ifdef WIN32_PLATFORM_PSPC
	if (message == WM_LBUTTONDOWN) {
		SHRGINFO shrg;
		shrg.cbSize		= sizeof(SHRGINFO);
		shrg.dwFlags	= SHRG_RETURNCMD;
		shrg.hwndClient = hWnd;
		shrg.ptDown.x	= LOWORD(lParam);
		shrg.ptDown.y	= HIWORD(lParam);
		if (SHRecognizeGesture(&shrg) == GN_CONTEXTMENU) {
			ShowContextMenu(hWnd, shrg.ptDown.x, shrg.ptDown.y, FALSE);
			return 0;
		}
	} else 
#endif /* WIN32_PLATFORM_PSPC */
#endif /* _WIN32_WCE */
	if (message == WM_RBUTTONDOWN) {
		ShowContextMenu(hWnd, LOWORD(lParam), HIWORD(lParam), FALSE);
		return 0;
	} else if (message == WM_ACTIVATE || message == WM_HSCROLL || message == WM_VSCROLL) {
		SetFocus(hWnd);
	} else if (message == WM_KILLFOCUS) {
		if (trx != 0) {
			int i;
			for (i = 0; i < trx->iLineReceivedIdx; i += 2)
				trx_display_rxchar(trx, trx->aLineReceived[i + 1], trx->aLineReceived[i]);
			trx->iLineReceivedIdx = 0;
		}
	}
	return CallWindowProc(edit_wndproc, hWnd, message, wParam, lParam);
}

LRESULT CALLBACK tx_wndproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
#ifdef _WIN32_WCE
#ifdef WIN32_PLATFORM_PSPC
	if (message == WM_LBUTTONDOWN) {
		SHRGINFO shrg;
		shrg.cbSize		= sizeof(SHRGINFO);
		shrg.dwFlags	= SHRG_RETURNCMD;
		shrg.hwndClient = hWnd;
		shrg.ptDown.x	= LOWORD(lParam);
		shrg.ptDown.y	= HIWORD(lParam);
		if (SHRecognizeGesture(&shrg) == GN_CONTEXTMENU) {
			ShowContextMenu(hWnd, shrg.ptDown.x, shrg.ptDown.y, TRUE);
			return 0;
		}
	}
	else
#endif /* WIN32_PLATFORM_PSPC */
#endif /* _WIN32_WCE */
	if (message == WM_RBUTTONDOWN) {
		ShowContextMenu(hWnd, LOWORD(lParam), HIWORD(lParam), TRUE);
		return 0;
	}
	return CallWindowProc(edit_wndproc, hWnd, message, wParam, lParam);
}

void trx_set_phase(int phase, BOOL highlight)
{
	// phase preview, empty
}
