#ifndef __SNDMODEM_H
#define __SNDMODEM_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include <windows.h>
#include <windowsx.h>

#ifndef _WIN32_WCE
#include <Mmsystem.h>
#endif /* _WIN32_WCE */

#include "main.h"
#include "waterfall.h"

struct _SndModem
{
	/// Hande of the window, which receives callbacks
	HWND				hWnd;
	/// Recording handle
	HWAVEIN				hWaveIn;
	/// Transmit handle
	HWAVEOUT			hWaveOut;

	/// Receive/Transmit channels
	Trx				  **aChannels;
	int					nChannels;
	/// TX volume - 0..65536
	int					iVolume;

	Waterfall		   *pWaterfall;

	/// Number of input buffers
	int					 nBuffersIn;
	/// Number of output buffers
	int					 nBuffersOut;
	/// Size of one input or output buffer
	int					 nBufSize;
	/// Number of output buffers, that will be kept full if possible.
	int					 nBufOutPrefetch;
	/// Cummulative buffer for recording
	char				*pBufferIn;
	/// Cummulative buffer for transmitting
	char				*pBufferOut;
	/// Windows soundsystem recording headers
	WAVEHDR				*pHeadersIn;
	/// Windows soundsystem transmitting headers
	WAVEHDR				*pHeadersOut;
	/// Index of the active recording buffer
	int					iBufInActive;
	/// First input buffer after input opening
	BOOL				bBufInFirst;
	/// Index of the active output buffer (the last not yet sent to the driver)
	int					iBufOutActive;
	/// Number of buffers sent to the driver
	int					nBufOutUsed;
	/// Chars written to the active buffer
	int					nBufLenLast;
	/// Set to true by sndmodem_destroy to true.
	/// recording callback will not put new buffers to the recording callback
	BOOL				bDestroying;

	/// Buffer for caching one output symbol, which was not sent to the sound card driver yet
	short			   *pOutBuf;
	int					iOutBufPos;
	int					nOutBufLen;
	int					nOutBufSize;

	/// History buffer for replying
	char				*pHistory;
	int					nHistorySeconds;
	BOOL				bHistoryEnabled;
	int					nHistorySize;
	int					nHistoryCount;
	int					nHistoryLast;

	/// Sound output is open for transceiver control
	BOOL				bControl;

	BOOL				bAutoCQ;
	int					nAutoCQPause;
	int					nAutoCQMacro;
};

typedef struct _SndModem		SndModem;

/// Initialize pModem, allocate buffers.
extern BOOL sndmodem_init(SndModem *pModem, HWND hWnd, Trx **aChannels, int nChannels, Waterfall *pWaterfall);
/// Close sound input/output, release buffers.
extern void sndmodem_destroy(SndModem *pModem);

/// Load settings from registry
extern void	sndmodem_load(SndModem *pModem);
/// Initialize by default values
extern void	sndmodem_load_defaults(SndModem *pModem);
/// Store settings to registry
extern void	sndmodem_store(SndModem *pModem);

/// Write buffer to the device. Correct volume by pModem->iVolume.
/// pBuffer will be modified!
extern void sndmodem_write(SndModem *pModem, short *pBuffer, int nLength);
/// Write buffer to the device. Do not correct volume.
extern void sndmodem_write_raw(SndModem *pModem, char *pBuffer, int nLength);
/// Write the last buffer to the device even if it is not full yet.
extern void sndmodem_tx_flush(SndModem *pModem);

/// Return true if the tx buffers are underrun and another chunk has to be written to the output driver
extern BOOL sndmodem_tx_buffer_underflowed(SndModem *pModem);

/// Open input device and start sampling.
extern BOOL sndmodem_start_recording(SndModem *pModem);
/// Process full buffers and close device.
extern BOOL	sndmodem_stop_recording(SndModem *pModem, BOOL bProcessRecorded);

/// Replay last xx seconds
extern void sndmodem_replay(SndModem *pModem);
extern void sndmodem_set_replay(SndModem *pModem, BOOL bEnabled, int nSeconds);

/// Open output device.
extern BOOL sndmodem_start_playing(SndModem *pModem, BOOL bControl);
/// End of playing. It kills the TX immediately, buffers sent to the sound driver will be lost.
extern BOOL	sndmodem_stop_playing(SndModem *pModem);

extern void	sndmodem_set_autocq(SndModem *pModem, int iDelay, int iMacro);
extern void	sndmodem_kill_autocq(SndModem *pModem);

/// callbacks called by window function
void sndmodem_wim_close(SndModem *pModem);
void sndmodem_wim_data(SndModem *pModem);
void sndmodem_wom_close(SndModem *pModem);
void sndmodem_wom_done(SndModem *pModem);


/// Digital modem over sound card
/// First the 1200Bd Bell202 modem
extern void sndmodem_write_bell202_tone(SndModem *pModem, BOOL bToneLow, UINT nSampleThirds);
extern void sndmodem_write_bell202_async(SndModem *pModem, USHORT nValue, int nBits /* = 8 */);
extern void sndmodem_write_bell202_pause_thirds(SndModem *pModem, UINT nSampleThirds);
extern void sndmodem_write_bell202_pause(SndModem *pModem, UINT nSamples);
extern void sndmodem_write_bell202_frame(SndModem *pModem, UCHAR cmd, UCHAR *data, int len);

/// Manchester modem. Baud rate 2000Bd.
extern void sndmodem_write_manchester_async(SndModem *pModem, USHORT nValue, int nBits /* = 8 */);
extern void sndmodem_write_manchester_pause(SndModem *pModem, UINT nSamples);
extern void sndmodem_write_manchester_frame(SndModem *pModem, UCHAR cmd, UCHAR *data, int len);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __SNDMODEM_H */
