#include "sndmodem.h"

#include <tchar.h>
#include <shellapi.h>
#include <shlobj.h>
#include <stdio.h>

#include "macro.h"
#include "trxctrl.h"
#include "../resource.h"
#include "../misc/mixer.h"
#include "../misc/winsupport.h"

#ifdef _WIN32_WCE
	#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

extern int s_iChannelActive;

void sndmodem_wim_close(SndModem *pModem)
{
	free(pModem->pBufferIn);
	pModem->pBufferIn		= 0;	
	pModem->hWaveIn			= 0;
	pModem->iBufInActive	= 0;
}

static TCHAR *strPathAutoFile = 0;

static BOOL ConfigureAutoFile(SndModem *pModem)
{
	HKEY  hKeyRoot = 0;
	TCHAR szPath[MAX_PATH];
	DWORD dwType  = REG_DWORD;
	DWORD dwValue = 0;
	DWORD dwSize  = 4;

	if (RegOpenKeyEx(HKEY_CURRENT_USER, _T("Software\\OK1IAK\\PocketDigi"), 0,
		KEY_ENUMERATE_SUB_KEYS | KEY_QUERY_VALUE | KEY_READ, &hKeyRoot) != ERROR_SUCCESS)
		return FALSE;

	if (RegQueryValueEx(hKeyRoot, _T("AutoSend"), 0, &dwType, (LPBYTE)&dwValue, &dwSize) != ERROR_SUCCESS ||
		dwType != REG_DWORD || dwValue == 0) {
		RegCloseKey(hKeyRoot);
		return FALSE;
	}
	RegCloseKey(hKeyRoot);

	if (! GetPocketDigiDir(szPath))
		return FALSE;

	_tcscat(szPath, _T("\\AutoSend.txt"));
	strPathAutoFile = _tcsdup(szPath);
	DeleteFile(strPathAutoFile);
	return TRUE;
}

static BOOL SendAutoFile(SndModem *pModem)
{
	int   i;
	TCHAR rxcmd[2] = { TRX_CLEAR_CMD, 0 };

	// allowed only on the first channel
	if (s_iChannelActive != 0 || strPathAutoFile == 0)
		return FALSE;

	// other channels shall be off
	for (i = 1; i < pModem->nChannels; ++ i)
		if (pModem->aChannels[i] != 0)
			return FALSE;

	if (! macro_sendfile(pModem->aChannels[0], strPathAutoFile, TRUE, TRUE))
		return FALSE;

	DeleteFile(strPathAutoFile);
	trx_send_string(pModem->aChannels[0], rxcmd);
	SendMessage(pModem->hWnd, WM_COMMAND, ID_STARTTX, 0);
	return TRUE;
}

void sndmodem_wim_data(SndModem *pModem)
{
	waveInUnprepareHeader(pModem->hWaveIn, pModem->pHeadersIn + pModem->iBufInActive, sizeof(WAVEHDR));

	if (pModem->bBufInFirst) {
		pModem->bBufInFirst = FALSE;
	} else if (! pModem->bDestroying) {
		int nBufPrev = (pModem->iBufInActive == 0) ? (pModem->nBuffersIn - 1) : (pModem->iBufInActive - 1);
		WAVEHDR *hdr = pModem->pHeadersIn + nBufPrev;
		hdr->dwBufferLength	= pModem->nBufSize;
		hdr->lpData			= pModem->pBufferIn + pModem->nBufSize * nBufPrev;
		hdr->dwFlags		= 0;
		hdr->reserved		= 0;
		hdr->dwLoops		= 0;
		hdr->lpNext			= 0;
		waveInPrepareHeader(pModem->hWaveIn, hdr, sizeof(WAVEHDR));
		waveInAddBuffer(pModem->hWaveIn, hdr, sizeof(WAVEHDR));
	}

	if (! pModem->bDestroying) {
		int i;
		short *ptr = (short*)(pModem->pBufferIn + pModem->nBufSize * pModem->iBufInActive);
		if (++ pModem->iBufInActive == pModem->nBuffersIn)
			pModem->iBufInActive = 0;
		if (pModem->bHistoryEnabled) {
			memcpy(pModem->pHistory + pModem->nBufSize * pModem->nHistoryLast, ptr, pModem->nBufSize);
			if (++ pModem->nHistoryLast == pModem->nHistorySize)
				pModem->nHistoryLast = 0;
			if (pModem->nHistoryCount < pModem->nHistorySize)
				++ pModem->nHistoryCount;
		}
		waterfall_set_data(pModem->pWaterfall, ptr, pModem->nBufSize >> 1);
		if (! SendAutoFile(pModem)) {
			for (i = 0; i < pModem->nChannels; ++ i)
				if (pModem->aChannels[i] != 0)
					pModem->aChannels[i]->rxprocess(pModem->aChannels[i], ptr, pModem->nBufSize >> 1);
		}
	}

//FIXME
/* #ifdef _WIN32_WCE
	if (GetForegroundWindow() == s_hWndMain)
		SystemIdleTimerReset();
#endif /* _WIN32_WCE */
}

void sndmodem_wom_close(SndModem *pModem)
{
	free(pModem->pBufferOut);
	pModem->pBufferOut		= 0;
	pModem->hWaveOut	    = 0;
	pModem->iBufOutActive	= 0;
	pModem->nBufOutUsed		= 0;
	pModem->nBufLenLast		= 0;
}

void sndmodem_wom_done(SndModem *pModem)
{
	Trx *trx		 = pModem->aChannels[s_iChannelActive];
	int iBufReleased = pModem->iBufOutActive - pModem->nBufOutUsed;
	if (iBufReleased < 0)
		iBufReleased += pModem->nBuffersOut;
	waveOutUnprepareHeader(pModem->hWaveOut, pModem->pHeadersOut + iBufReleased, sizeof(WAVEHDR));
	-- pModem->nBufOutUsed;

	if (! pModem->bDestroying && ! pModem->bControl && trx_is_sending(trx) &&
		trx->state != TRX_STATE_FLUSH && 
		sndmodem_tx_buffer_underflowed(pModem))
		trx_txloop(trx);
	else if (pModem->nBufOutUsed == 0 && (pModem->bControl || trx->stopflag)) {
		pModem->iBufOutActive = 0;
		pModem->nBufOutUsed   = 0;
		pModem->nBufLenLast   = 0;
		if (pModem->bControl)
			pModem->bControl = FALSE;
		else {
			trx_txend(trx);
			trxctrl_ptt(&g_TrxCtrl, FALSE); // PTT off
		}
		if (! pModem->bDestroying) {
			sndmodem_stop_playing(pModem);
			sndmodem_start_recording(pModem);
			if (pModem->bAutoCQ && pModem->nAutoCQMacro != -1 && pModem->nAutoCQPause > 0)
				SetTimer(pModem->hWnd, 1, pModem->nAutoCQPause, 0);
		}
	}

//FIXME
/* #ifdef _WIN32_WCE
	if (GetForegroundWindow() == s_hWndMain)
		SystemIdleTimerReset();
#endif /* _WIN32_WCE */
//	SystemIdleTimerReset();
}

BOOL sndmodem_start_recording(SndModem *pModem)
{
	WAVEINCAPS	 wic;
	WAVEFORMATEX wfx;
	MMRESULT	 res;
	UINT	     nDevices = waveInGetNumDevs();
	int  		 i;
	if (nDevices < 1) {
		MessageBox(pModem->hWnd, _T("No sound input device"), _T("PocketDigi - Error"), MB_ICONSTOP | MB_OK);
		return FALSE;
	}

	res = waveInGetDevCaps(0, &wic, sizeof(wic));
	// WAVE_FORMAT_1M16 11.025 kHz, mono, 16-bit 
	wfx.wFormatTag		= WAVE_FORMAT_PCM;
	wfx.nChannels		= 1;
	wfx.nSamplesPerSec  = 8000;
	wfx.nAvgBytesPerSec = 16000;
	wfx.nBlockAlign		= 2;
	wfx.wBitsPerSample	= 16;
	wfx.cbSize = 0;
	res = waveInOpen(&pModem->hWaveIn, WAVE_MAPPER, &wfx, 
		(DWORD)pModem->hWnd, (DWORD)pModem, CALLBACK_WINDOW);
	if (res != MMSYSERR_NOERROR) {
		TCHAR buf[256];
		_stprintf(buf, _T("Cannot open sound input\r\nError code: %d"), res);
		MessageBox(pModem->hWnd, buf, _T("PocketDigi - Error"), MB_ICONSTOP | MB_OK);
		return FALSE;
	}

	pModem->pBufferIn = (char*)malloc(pModem->nBuffersIn * pModem->nBufSize);
	for (i = 0; i < pModem->nBuffersIn; ++ i) {
		WAVEHDR *hdr = pModem->pHeadersIn + i;
		hdr->dwBufferLength	= pModem->nBufSize;
		hdr->lpData			= pModem->pBufferIn + pModem->nBufSize * i;
		hdr->dwFlags		= 0;
		hdr->reserved		= 0;
		hdr->dwLoops		= 0;
		hdr->lpNext			= 0;
		res = waveInPrepareHeader(pModem->hWaveIn, hdr, sizeof(WAVEHDR));
		res = waveInAddBuffer(pModem->hWaveIn, hdr, sizeof(WAVEHDR));
	}
	pModem->iBufInActive = 0;
	pModem->bBufInFirst  = TRUE;
	res = waveInStart(pModem->hWaveIn);
	if (res != MMSYSERR_NOERROR) {
		TCHAR buf[256];
		_stprintf(buf, _T("Cannot start sound input\r\nError code: %d"), res);
		MessageBox(pModem->hWnd, buf, _T("PocketDigi - Error"), MB_ICONSTOP | MB_OK);
		return FALSE;
	}
	return res == MMSYSERR_NOERROR;
}

BOOL sndmodem_stop_recording(SndModem *pModem, BOOL bProcessRecorded)
{
	MSG msg;

	if (pModem->hWaveIn == 0)
		return FALSE;

	pModem->bDestroying = TRUE;
	if (bProcessRecorded) {
		waveInStop(pModem->hWaveIn);
		while (PeekMessage(&msg, pModem->hWnd, MM_WIM_OPEN, MM_WIM_DATA, PM_REMOVE)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}

	waveInReset(pModem->hWaveIn);
	while (PeekMessage(&msg, pModem->hWnd, MM_WIM_OPEN, MM_WIM_DATA, PM_REMOVE)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	waveInClose(pModem->hWaveIn);
	while (PeekMessage(&msg, pModem->hWnd, MM_WIM_OPEN, MM_WIM_DATA, PM_REMOVE)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	free(pModem->pBufferIn);
	pModem->hWaveIn			= 0;
	pModem->pBufferIn		= 0;
	pModem->iBufInActive	= 0;
	pModem->bDestroying		= FALSE;
	pModem->nHistoryCount   = 0;
	pModem->nHistoryLast    = 0;
	return TRUE;
}

BOOL sndmodem_start_playing(SndModem *pModem, BOOL bControl)
{
	WAVEOUTCAPS  woc;
	WAVEFORMATEX wfx;
	MMRESULT	 res;
	UINT		 nDevices = waveInGetNumDevs();
	if (nDevices < 1) {
		MessageBox(pModem->hWnd, _T("No sound output device"), _T("PocketDigi - Error"), MB_ICONSTOP | MB_OK);
		return FALSE;
	}

	res = waveOutGetDevCaps(0, &woc, sizeof(woc));
	wfx.wFormatTag		= WAVE_FORMAT_PCM;
	wfx.nChannels		= 1;
	wfx.nSamplesPerSec  = 8000;
	wfx.nAvgBytesPerSec = 16000;
	wfx.nBlockAlign		= 2;
	wfx.wBitsPerSample	= 16;
	wfx.cbSize = 0;
	res = waveOutOpen(&pModem->hWaveOut, WAVE_MAPPER, &wfx, 
		(DWORD)pModem->hWnd, (DWORD)pModem, CALLBACK_WINDOW);
	if (res != MMSYSERR_NOERROR) {
		TCHAR buf[256];
		_stprintf(buf, _T("Cannot open sound output\r\nError code: %d"), res);
		MessageBox(pModem->hWnd, buf, _T("PocketDigi - Error"), MB_ICONSTOP | MB_OK);
		return FALSE;
	}
	_ASSERT(pModem->pBufferOut == 0);
	pModem->pBufferOut		= (char*)malloc(pModem->nBuffersOut * pModem->nBufSize);
	pModem->nBufOutUsed		= 0;
	pModem->iBufOutActive	= 0;
	pModem->nBufLenLast		= 0;
	pModem->bControl		= bControl;
	return TRUE;
}

BOOL sndmodem_stop_playing(SndModem *pModem)
{
	MSG msg;

	if (pModem->hWaveOut == 0)
		return FALSE;

	pModem->bDestroying = TRUE;
	waveOutReset(pModem->hWaveOut);
	while (PeekMessage(&msg, pModem->hWnd, MM_WOM_OPEN, MM_WOM_DONE, PM_REMOVE)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	waveOutClose(pModem->hWaveOut);
	while (PeekMessage(&msg, pModem->hWnd, MM_WOM_OPEN, MM_WOM_DONE, PM_REMOVE)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	free(pModem->pBufferOut);
	pModem->pBufferOut		= 0;
	pModem->hWaveOut		= 0;
	pModem->iBufOutActive	= 0;
	pModem->nBufOutUsed		= 0;
	pModem->nBufLenLast		= 0;
	pModem->bDestroying		= FALSE;
	pModem->bControl		= FALSE;
	return TRUE;
}

void sndmodem_replay(SndModem *pModem)
{
	int  i;
	int  iBuffer;
	Trx *trx = pModem->aChannels[s_iChannelActive];
	if (! pModem->bHistoryEnabled || pModem->nHistoryCount == 0)
		return;
	iBuffer = pModem->nHistoryLast - pModem->nHistoryCount;
	if (iBuffer < 0)
		iBuffer += pModem->nHistorySize;
	for (i = 0; i < pModem->nHistoryCount; ++ i) {
		trx->rxprocess(trx, (short*)(pModem->pHistory + iBuffer * pModem->nBufSize), pModem->nBufSize >> 1);
		if (++ iBuffer == pModem->nHistorySize)
			iBuffer = 0;
	}
}

void sndmodem_set_replay(SndModem *pModem, BOOL bEnabled, int nSeconds)
{
	pModem->bHistoryEnabled = bEnabled;

	if (nSeconds < 5)
		pModem->nHistorySeconds = 5;
	else if (nSeconds > 60)
		pModem->nHistorySeconds = 60;
	else
		pModem->nHistorySeconds = nSeconds;

	if (! pModem->bHistoryEnabled) {
		free(pModem->pHistory);
		pModem->pHistory	  = 0;
		pModem->nHistorySize  = 0;
		pModem->nHistoryCount = 0;
		pModem->nHistoryLast  = 0;
		return;
	}

	// number of receive buffers
	pModem->nHistorySize  = (int)((float)pModem->nHistorySeconds * 16000.0f / (float)pModem->nBufSize + 0.5f);
	pModem->nHistoryCount = 0;
	pModem->nHistoryLast  = 0;
	pModem->pHistory	  = malloc(pModem->nBufSize * pModem->nHistorySize);
}

BOOL sndmodem_init(SndModem *pModem, HWND hWnd, Trx **aChannels, int nChannels, Waterfall *pWaterfall)
{
	memset(pModem, 0, sizeof(SndModem));
	pModem->hWnd		= hWnd;
	pModem->aChannels	= aChannels;
	pModem->nChannels	= nChannels;
	pModem->pWaterfall	= pWaterfall;
	pModem->nOutBufSize	= 32768;
	pModem->pOutBuf		= (short*)malloc(sizeof(short) * pModem->nOutBufSize);
	pModem->nBuffersIn	= 8;
	pModem->nBuffersOut	= 16;
	pModem->nBufSize	= 4000; // 1/4 of a second
	pModem->nBufOutPrefetch = 2;
	pModem->pHeadersIn	= (WAVEHDR*)malloc(sizeof(WAVEHDR) * pModem->nBuffersIn);
	pModem->pHeadersOut	= (WAVEHDR*)malloc(sizeof(WAVEHDR) * pModem->nBuffersOut);
	ConfigureAutoFile(pModem);
	sndmodem_load(pModem);
	sndmodem_set_replay(pModem, pModem->bHistoryEnabled, pModem->nHistorySeconds);
	return TRUE;
}

void sndmodem_destroy(SndModem *pModem)
{
	sndmodem_stop_playing(pModem);
	sndmodem_stop_recording(pModem, FALSE);
	free(pModem->pOutBuf);
	free(pModem->pHeadersIn);
	free(pModem->pHeadersOut);
	free(pModem->pHistory);
	memset(pModem, 0, sizeof(SndModem));
}

void sndmodem_load(SndModem *pModem)
{
	HKEY  hKey   = 0;
	DWORD dwType = REG_DWORD;
	DWORD dwSize = 4;

	sndmodem_load_defaults(pModem);
	if (RegOpenKeyEx(HKEY_CURRENT_USER, _T("Software\\OK1IAK\\PocketDigi\\SndModem"), 0,
		KEY_ENUMERATE_SUB_KEYS | KEY_QUERY_VALUE | KEY_READ, &hKey) != ERROR_SUCCESS)
		return;

	dwType = REG_DWORD; dwSize = 4;
	RegQueryValueEx(hKey, _T("VolumeTX"),	  0, &dwType, (LPBYTE)&pModem->iVolume,			&dwSize);
	if (pModem->iVolume < 100 || pModem->iVolume > 65536)
		pModem->iVolume = 32768;
	dwType = REG_DWORD; dwSize = 4;
	RegQueryValueEx(hKey, _T("PrefetchBufs"), 0, &dwType, (LPBYTE)&pModem->nBufOutPrefetch, &dwSize);
	if (pModem->nBufOutPrefetch < 2 || pModem->nBufOutPrefetch > 10)
		pModem->nBufOutPrefetch = 2;
	dwType = REG_DWORD; dwSize = 4;
	RegQueryValueEx(hKey, _T("HistoryEnabled"), 0, &dwType, (LPBYTE)&pModem->bHistoryEnabled, &dwSize);
	dwType = REG_DWORD; dwSize = 4;
	RegQueryValueEx(hKey, _T("HistorySeconds"), 0, &dwType, (LPBYTE)&pModem->nHistorySeconds, &dwSize);
	if (pModem->nHistorySeconds < 5 || pModem->nHistorySeconds > 60)
		pModem->nHistorySeconds = 20;

	RegCloseKey(hKey);
}

void sndmodem_load_defaults(SndModem *pModem)
{
	pModem->iVolume			= 32768; // one half of maximum
	pModem->nBufOutPrefetch = 2;
	pModem->bHistoryEnabled = TRUE;
	pModem->nHistorySeconds = 20;
}

void sndmodem_store(SndModem *pModem)
{
	HKEY  hKey = 0;
	DWORD dwDisposition = 0;

	if (RegCreateKeyEx(HKEY_CURRENT_USER, _T("Software\\OK1IAK\\PocketDigi\\SndModem"), 0, 0,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, 0, &hKey, &dwDisposition) != ERROR_SUCCESS)
		return;

	RegSetValueEx(hKey, _T("VolumeTX"),		  0, REG_DWORD, (LPBYTE)&pModem->iVolume,			4);
	RegSetValueEx(hKey, _T("PrefetchBufs"),   0, REG_DWORD, (LPBYTE)&pModem->nBufOutPrefetch, 4);
	RegSetValueEx(hKey, _T("HistoryEnabled"), 0, REG_DWORD, (LPBYTE)&pModem->bHistoryEnabled, 4);
	RegSetValueEx(hKey, _T("HistorySeconds"), 0, REG_DWORD, (LPBYTE)&pModem->nHistorySeconds, 4);

	RegCloseKey(hKey);
}

void sndmodem_write(SndModem *pModem, short *pBuffer, int nLength)
{
	int i;
	for (i = 0; i < nLength; ++ i)
		pBuffer[i] = ((int)pBuffer[i] * pModem->iVolume) >> 16;
	sndmodem_write_raw(pModem, (char*)pBuffer, nLength << 1);
}

void sndmodem_write_raw(SndModem *pModem, char *pBuffer, int nLength)
{
	int nLength1 = (nLength <= pModem->nBufSize - pModem->nBufLenLast) ?
		nLength : (pModem->nBufSize - pModem->nBufLenLast);

	_ASSERT(pModem->nBufOutUsed < pModem->nBuffersOut);
	_ASSERT(pModem->nBufLenLast < pModem->nBufSize);
	_ASSERT(pModem->iBufOutActive >= 0 && pModem->iBufOutActive < pModem->nBuffersOut);

	_ASSERT(nLength <= pModem->nBufSize);
	nLength -= nLength1;
	memcpy(pModem->pBufferOut + pModem->nBufSize * pModem->iBufOutActive + pModem->nBufLenLast,
		pBuffer, nLength1);
	pModem->nBufLenLast += nLength1;
	_ASSERT(pModem->nBufLenLast <= pModem->nBufSize);
	if (pModem->nBufLenLast == pModem->nBufSize)
		sndmodem_tx_flush(pModem);

	if (nLength > 0) {
		memcpy(pModem->pBufferOut + pModem->nBufSize * pModem->iBufOutActive + pModem->nBufLenLast,
			pBuffer + nLength1, nLength);
		pModem->nBufLenLast += nLength;
	}
	_ASSERT(pModem->nBufLenLast < pModem->nBufSize);
}

void sndmodem_tx_flush(SndModem *pModem)
{
	WAVEHDR *hdr		= 0;
	MMRESULT res;

	if (pModem->nBufLenLast == 0)
		return;

 	_ASSERT(pModem->nBufOutUsed < pModem->nBuffersOut);

	hdr					= pModem->pHeadersOut + pModem->iBufOutActive;
	hdr->dwBufferLength	= pModem->nBufLenLast;
	hdr->lpData			= pModem->pBufferOut + pModem->nBufSize * pModem->iBufOutActive;
	hdr->dwFlags		= 0;
	hdr->reserved		= 0;
	hdr->dwLoops		= 0;
	hdr->lpNext			= 0;
	res = waveOutPrepareHeader(pModem->hWaveOut, pModem->pHeadersOut + pModem->iBufOutActive, sizeof(WAVEHDR));
	if (res != MMSYSERR_NOERROR) {
		TCHAR buf[256];
		_stprintf(buf, _T("Sound output - waveOutPrepareHeader\r\nError code: %d"), res);
		MessageBox(pModem->hWnd, buf, _T("PocketDigi - Error"), MB_ICONSTOP | MB_OK);
		return;
	}

	res = waveOutWrite(pModem->hWaveOut, pModem->pHeadersOut + pModem->iBufOutActive, sizeof(WAVEHDR));
	if (res != MMSYSERR_NOERROR) {
		TCHAR buf[256];
		_stprintf(buf, _T("Sound output - waveOutWrite\r\nError code: %d"), res);
		MessageBox(pModem->hWnd, buf, _T("PocketDigi - Error"), MB_ICONSTOP | MB_OK);
		return;
	}

	pModem->nBufLenLast = 0;
	++ pModem->nBufOutUsed;
	++ pModem->iBufOutActive;
	if (pModem->iBufOutActive == pModem->nBuffersOut)
		pModem->iBufOutActive = 0;
}

BOOL sndmodem_tx_buffer_underflowed(SndModem *pModem)
{
	return pModem->nBufOutUsed < pModem->nBufOutPrefetch;
}

void sndmodem_set_autocq(SndModem *pModem, int iDelay, int iMacro)
{
	pModem->bAutoCQ      = TRUE;
	pModem->nAutoCQMacro = iMacro;
	pModem->nAutoCQPause = iDelay;
}

void sndmodem_kill_autocq(SndModem *pModem)
{
	pModem->bAutoCQ      = FALSE;
	pModem->nAutoCQMacro = -1;
	pModem->nAutoCQPause = 0;
	KillTimer(pModem->hWnd, 1);
}

void sndmodem_write_bell202_tone(SndModem *pModem, BOOL bToneLow, UINT nSampleThirds)
{
	// PI corresponds to 2^31
	static UINT iPhase		= 0;
	// Symbol length of 1200bd at 8kSamples/sec is 20=6+1/3.
	// Store modulo here, 0, 1 or 2.
	static UINT	iSamplePos	= 0;
	UINT		iPhaseInc	= bToneLow ? 214748365 :  393705335;
	UINT		iPhaseInc3	= bToneLow ? 644245094 : 1181116006;
	UINT		nSamples	= nSampleThirds - iSamplePos;
	DWORD		nWritten	= 0;

	iPhase += iSamplePos * iPhaseInc;
	pModem->pOutBuf[nWritten ++] = mixer_osc_16(iPhase);
	while (nSamples > 3) {
		iPhase	 += iPhaseInc3;
		nSamples -= 3;
		pModem->pOutBuf[nWritten ++] = mixer_osc_16(iPhase);
	}

	iPhase	  += nSamples * iPhaseInc;
	iSamplePos = 3 - nSamples;
	sndmodem_write(pModem, pModem->pOutBuf, nWritten);
}

void sndmodem_write_bell202_async(SndModem *pModem, USHORT nValue, int nBits /* = 8 */)
{
	int i;
	sndmodem_write_bell202_tone(pModem, FALSE, 20); // start bit
	for (i = 0; i < nBits; ++ i) {
		sndmodem_write_bell202_tone(pModem, nValue & 1, 20);
		nValue >>= 1;
	}
	sndmodem_write_bell202_tone(pModem, TRUE, 20); // stop bit
}

void sndmodem_write_bell202_pause_thirds(SndModem *pModem, UINT nSampleThirds)
{
	sndmodem_write_bell202_tone(pModem, TRUE, nSampleThirds);
}

void sndmodem_write_bell202_pause(SndModem *pModem, UINT nSamples)
{
	sndmodem_write_bell202_tone(pModem, TRUE, nSamples * 3);
}

void sndmodem_write_bell202_frame(SndModem *pModem, UCHAR cmd, UCHAR *data, int len)
{
	int i;
	sndmodem_write_bell202_async(pModem, 0xfe, 8); // start of the frame
	sndmodem_write_bell202_async(pModem, cmd,  8); // command
	for (i = 0; i < len; ++ i)
		sndmodem_write_bell202_async(pModem, data[i],  8); // data
	sndmodem_write_bell202_async(pModem, 0xfd, 8); // end of the frame
}

static BOOL bManchesterPrevValueOut = FALSE;

void sndmodem_write_manchester_bit(SndModem *pModem, UINT bValue)
{
	int   sign = bManchesterPrevValueOut ? -1 : 1;
	if (bValue) {
		// insert halfwave of 1000Hz
		pModem->pOutBuf[0] = 0;
		pModem->pOutBuf[1] = sign * 23162; // sin(PI/4)
		pModem->pOutBuf[2] = sign * 32767; // sin(PI/2)
		pModem->pOutBuf[3] = sign * 23162; // sin(PI/4)
		bManchesterPrevValueOut = ! bManchesterPrevValueOut;
	} else {
		// insert two halfwaves of 2000Hz
		pModem->pOutBuf[0] = 0;
		pModem->pOutBuf[1] = sign * 32767; // sin(PI/2)
		pModem->pOutBuf[2] = 0;
		pModem->pOutBuf[3] = - sign * 32767; // sin(3*PI/2)
	}
	sndmodem_write(pModem, pModem->pOutBuf, 4);
}

void sndmodem_write_manchester_async(SndModem *pModem, USHORT nValue, int nBits /* = 8 */)
{
	int i;
	sndmodem_write_manchester_bit(pModem, FALSE); // start bit
	for (i = 0; i < nBits; ++ i) {
		sndmodem_write_manchester_bit(pModem, nValue & 1);
		nValue >>= 1;
	}
	sndmodem_write_manchester_bit(pModem, TRUE); // stop bit
}

void sndmodem_write_manchester_pause(SndModem *pModem, UINT nSamples)
{
	UINT nBits = nSamples >> 2;
	UINT i;
	for (i = 0; i < nBits; ++ i)
		sndmodem_write_manchester_bit(pModem, TRUE);
}

void sndmodem_write_manchester_frame(SndModem *pModem, UCHAR cmd, UCHAR *data, int len)
{
	int i;
	sndmodem_write_manchester_async(pModem, 0xfe, 8); // start of the frame
	sndmodem_write_manchester_async(pModem, cmd,  8); // command
	for (i = 0; i < len; ++ i)
		sndmodem_write_manchester_async(pModem, data[i],  8); // data
	sndmodem_write_manchester_async(pModem, 0xfd, 8); // end of the frame
}
