/*
 * Receive Widget
 * Copyright (C) 2006 Vojtech Bubnik <bubnikv@seznam.cz>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "rxview.h"

#include <memory.h>
#include <stdio.h>
#include <tchar.h>
#include <crtdbg.h>
#include <zmouse.h>

/// Forward reference to the rxview window procedure
LRESULT CALLBACK rxview_wndproc(HWND, UINT, WPARAM, LPARAM);

const TBYTE WIN32CLASS_RXVIEW[] = _T("RXVIEW");

/// Register class of the rxview window
ATOM rxview_register_class(HINSTANCE hInstance)
{
	WNDCLASS	wc;

	wc.style			= CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc		= (WNDPROC) rxview_wndproc;
	wc.cbClsExtra		= 0;
	wc.cbWndExtra		= 0;
	wc.hInstance		= hInstance;
	wc.hIcon			= 0;
	wc.hCursor			= 0;
	wc.hbrBackground	= (HBRUSH) GetStockObject(WHITE_BRUSH);
	wc.lpszMenuName		= 0;
	wc.lpszClassName	= WIN32CLASS_RXVIEW;

	return RegisterClass(&wc);
}

RxView* rxview_new()
{
	RxView *pThis = (RxView*)malloc(sizeof(RxView));
	if (pThis == 0)
		return 0;
	if (! rxview_init(pThis)) {
		rxview_destroy(pThis);
		return 0;
	}
	return pThis;
}

void rxview_free(RxView *pThis)
{
	if (pThis == 0)
		return;
	rxview_destroy(pThis);
	free(pThis);
}

BOOL rxview_init(RxView *pThis)
{
	memset(pThis, 0, sizeof(RxView));

	pThis->nChunksSize		= 128;
	pThis->pChunks			= (RxViewChunk*)calloc(pThis->nChunksSize, sizeof(RxViewChunk));

	pThis->nDataMax			= 1024 * 64;
	pThis->nDataSize		= 1024 *  8;
	pThis->pData			= (UCHAR*)calloc(pThis->nDataSize, 1);

	pThis->nFormatSize		= 1024;
	pThis->pFormat			= (UCHAR*)calloc(pThis->nFormatSize, 1);

	pThis->nTapeRows		= 14;
	pThis->nTapeWrapDuplicatedColumns = 5;
	pThis->bTapeDoubleWidth	= TRUE;
//	pThis->bTapeDoubleWidth	= FALSE;
//	pThis->bTapeTwoTimes	= FALSE;
	pThis->bTapeTwoTimes	= TRUE;
	pThis->bTapeDrawSeparator = TRUE;

	pThis->bIsAtEnd			= TRUE;

	return TRUE;
}

BOOL rxview_create(RxView *pThis, DWORD dwStyle,
	int x, int y, int nWidth, int nHeight, HWND hWndParent, HANDLE hInstance)
{
	pThis->hWnd = CreateWindow(WIN32CLASS_RXVIEW, NULL, dwStyle,
		x, y, nWidth, nHeight, hWndParent, NULL, hInstance, (void*)pThis);
	if (! pThis->hWnd)
		return FALSE;
	SetWindowLong(pThis->hWnd, GWL_USERDATA, (long)pThis);
	{
		SCROLLINFO si;
		memset(&si, 0, sizeof(si));
		si.cbSize = sizeof(si);
		si.fMask  = SIF_DISABLENOSCROLL | SIF_ALL;
		SetScrollInfo(pThis->hWnd, SB_VERT, &si, FALSE);
	}
	{
		LOGFONT logfont;
		memset(&logfont, 0, sizeof(LOGFONT));
		logfont.lfHeight  = -9;
		logfont.lfWeight  = FW_NORMAL;
		logfont.lfCharSet = ANSI_CHARSET;
		_tcscpy(logfont.lfFaceName, _T("Tahoma"));
		pThis->hFont = CreateFontIndirect(&logfont);
	}
	{
		HDC			hDC			= GetWindowDC(pThis->hWnd);
		HGDIOBJ		hPrevFont	= SelectObject(hDC, pThis->hFont);
		TEXTMETRIC	tm;
		GetTextMetrics(hDC, &tm);
		pThis->nFontHeight = tm.tmHeight + tm.tmExternalLeading;
		pThis->nFontAscent = tm.tmAscent;
		SelectObject(hDC, hPrevFont);
		ReleaseDC(pThis->hWnd, hDC);
	}

	return TRUE;
}

void rxview_destroy(RxView *pThis)
{
	DeleteObject(pThis->hFont);
	DestroyWindow(pThis->hWnd);
	DeleteObject(pThis->hTapeBitmap);
	memset(pThis, 0, sizeof(RxView));
}

/// Add a new chunk.
/// If the preallocated sockets are full, grow it.
/// Return index of the new chunk.
static int AddChunk2(RxView *pThis)
{
	_ASSERT(pThis->nChunksLen <= pThis->nChunksSize);
	if (pThis->nChunksLen != pThis->nChunksSize)
		return (pThis->iChunkFirst + pThis->nChunksLen ++) & (pThis->nChunksSize - 1);

	if (pThis->iChunkFirst == 0) {
		pThis->nChunksSize <<= 1;
		pThis->pChunks = realloc(pThis->pChunks, pThis->nChunksSize * sizeof(RxViewChunk));
		return pThis->nChunksLen ++;
	}
	{
		int			 nSizeNew = pThis->nChunksSize << 1;
		int          nLen1    = sizeof(RxViewChunk) * (pThis->nChunksSize - pThis->iChunkFirst);
		RxViewChunk *pNew	  = calloc(nSizeNew, sizeof(RxViewChunk));
		memcpy(pNew, pThis->pChunks + pThis->iChunkFirst, nLen1);
		memcpy(pNew + nLen1, pThis->pChunks, sizeof(RxViewChunk) * pThis->iChunkFirst);
		pThis->iChunkFirst = 0;
		pThis->nChunksSize = nSizeNew;
		return pThis->nChunksLen ++;
	}
}

static RxViewChunk* AddChunk(RxView *pThis, RxViewChunkType nType)
{
	int			 iChunk = AddChunk2(pThis);
	RxViewChunk *pChunk = pThis->pChunks + iChunk;
	if (pThis->nChunksLen == 1)  {
		_ASSERT(pThis->nDataLen == 0);
		pChunk->iData = 0;
	} else {
		pChunk->iData = (pThis->pChunks[pThis->iChunkFirst].iData + pThis->nDataLen) & (pThis->nDataSize - 1);
	}
	pChunk->nType	= nType;
	pChunk->iRow	= 0;
	pChunk->iFormat = 0;
	return pChunk;
}

/// Release first chunk, make place for a new data
static void ReleaseFirstChunk(RxView *pThis)
{
	if (pThis->nChunksLen == 0)
		return;

	if (pThis->nChunksLen == 1) {
		pThis->iChunkFirst	= 0;
		pThis->nChunksLen	= 0;
		pThis->nDataLen		= 0;
		pThis->nFormatLen	= 0;
		pThis->nRows		= 0;
	} else {
		int iNext		= (pThis->iChunkFirst + 1) & (pThis->nChunksSize - 1);
		int nDataLen	= (pThis->pChunks[iNext].iData   - pThis->pChunks[pThis->iChunkFirst].iData)   & (pThis->nDataSize - 1);
		int nFormatLen	= (pThis->pChunks[iNext].iFormat - pThis->pChunks[pThis->iChunkFirst].iFormat) & (pThis->nFormatSize - 1);
		int nRowsLen	= pThis->pChunks[iNext].iRow    - pThis->pChunks[pThis->iChunkFirst].iRow;

		pThis->iChunkFirst	= iNext;
		-- pThis->nChunksLen;
		pThis->nDataLen	   -= nDataLen;
		pThis->nFormatLen  -= nFormatLen;
		pThis->nRows	   -= nRowsLen;
	}
}

static void ReleaseFirstLineText(RxView *pThis)
{
	RxViewChunk *pChunk		= pThis->pChunks + pThis->iChunkFirst;
	int			 iDataThis	= pChunk->iData;
	int			 iDataNext;
	int			 iFormatNext;
	int			 iType;

	_ASSERT(pThis->nChunksLen > 0);

	if (pThis->nChunksLen == 1) {
		// The only chunk
		iDataNext    = (iDataThis + pThis->nDataLen) & (pThis->nDataSize - 1);
		iFormatNext  = (pChunk->iFormat + pThis->nFormatLen) & (pThis->nFormatSize - 1);
	} else {
		// There is more chunks in the buffer
		int iNext	= (pThis->iChunkFirst + 1) & (pThis->nChunksSize - 1);
		iDataNext   = pThis->pChunks[iNext].iData;
		iFormatNext = pThis->pChunks[iNext].iFormat;
	}

	iType = *((TCHAR*)(pThis->pData + iDataThis));

	if (pThis->nColumnsScreen == 0) {
		// not word wrapped yet
		ReleaseFirstChunk(pThis);
	} else if (pChunk->iFormat == iFormatNext) {
		// Chunk is shorter than one line.
		ReleaseFirstChunk(pThis);
	} else {
		// Line wrapped chunk. Release the first line.
		int nSkip = ((int*)(pThis->pFormat + pChunk->iFormat))[1];
		iDataThis = (iDataThis + nSkip) & (pThis->nDataSize - 1);
		pThis->nDataLen -= nSkip;
		*(TCHAR*)(pThis->pFormat + iDataThis) = iType;
		pChunk->iData = iDataThis;
		pChunk->iFormat = (pChunk->iFormat + (sizeof(int) << 1)) & (pThis->nFormatSize - 1);
		pThis->nFormatLen -= sizeof(int) << 1;
		pChunk->iRow += pThis->nFontHeight;
		pThis->nRows -= pThis->nFontHeight;
	}
}

static void ReleaseFirstLineTape(RxView *pThis)
{
	RxViewChunk *pChunk		= pThis->pChunks + pThis->iChunkFirst;
	int			 iDataThis	= pChunk->iData;
	int			 iDataNext;

	_ASSERT(pThis->nChunksLen > 0);

	if (pThis->nChunksLen == 1) {
		// The only chunk
		iDataNext = (iDataThis + pThis->nDataLen) & (pThis->nDataSize - 1);
	} else {
		// There is more chunks in the buffer
		int iNext = (pThis->iChunkFirst + 1) & (pThis->nChunksSize - 1);
		iDataNext = pThis->pChunks[iNext].iData;
	}

	// remove one line
	{
		int iType			= *((UCHAR*)(pThis->pData + iDataThis));
		int nLength			= iDataNext - iDataThis - 1 + ((iDataNext >= iDataThis + 1) ? 0 : pThis->nDataSize);
		int nColumns		= (nLength + pThis->nTapeRows - 1) / pThis->nTapeRows;
		int nColumnsScreen  = (pThis->nColumnsScreen ? pThis->nColumnsScreen : 240) >> (pThis->bTapeDoubleWidth ? 1 : 0);
		int nLengthWrapped	= nColumnsScreen - pThis->nTapeWrapDuplicatedColumns;
		int nRows			= nColumns / nLengthWrapped;
		int nRest			= nColumns % nLengthWrapped;
		int nSkip;

		if (nRows == 0 || nRest > pThis->nTapeWrapDuplicatedColumns)
			++ nRows;

		if (nRows == 1) {
			ReleaseFirstChunk(pThis);
			return;
		}

		// Line wrapped chunk. Release the first line.
		nSkip = nLengthWrapped * pThis->nTapeRows;
		iDataThis = (iDataThis + nSkip) & (pThis->nDataSize - 1);
		pThis->nDataLen -= nSkip;
		*(UCHAR*)(pThis->pFormat + iDataThis) = iType;
		pChunk->iData = iDataThis;

		pChunk->iRow += pThis->nTapeRows;
		pThis->nRows -= pThis->nTapeRows;
		if (pThis->bTapeTwoTimes) {
			pChunk->iRow += pThis->nTapeRows;
			pThis->nRows -= pThis->nTapeRows;
		}
		if (pThis->bTapeDrawSeparator) {
			++ pChunk->iRow;
			-- pThis->nRows;
		}
	}
}

static void ReleaseFirstLineImage(RxView *pThis)
{
	_ASSERT(FALSE);
}

static void AddData(RxView *pThis, UCHAR *data, int len, BOOL *bOlderFreed)
{
	int nSizeNew = pThis->nDataSize;
	while (pThis->nDataLen + len > nSizeNew && nSizeNew < pThis->nDataMax)
		nSizeNew <<= 1;

	if (nSizeNew > pThis->nDataSize) {
		pThis->pData	 = realloc(pThis->pData, nSizeNew);
		pThis->nDataSize = nSizeNew;
	}

	while (len + pThis->nDataLen > pThis->nDataSize) {
		switch (pThis->pChunks[pThis->iChunkFirst].nType) {
		case RXVIEW_CHUNK_TEXT:
			ReleaseFirstLineText(pThis);
			break;
		case RXVIEW_CHUNK_TAPE:
			ReleaseFirstLineTape(pThis);
			break;
		case RXVIEW_CHUNK_IMAGE:
			ReleaseFirstLineImage(pThis);
			break;
		default:
			_ASSERT(FALSE);
		}
		*bOlderFreed = TRUE;
	}

	_ASSERT(len + pThis->nDataLen <= pThis->nDataSize);
	{
		int iDataFirst = pThis->pChunks[pThis->iChunkFirst].iData + pThis->nDataLen;
		if (iDataFirst >= pThis->nDataSize)
			iDataFirst -= pThis->nDataSize;
		if (iDataFirst + len <= pThis->nDataSize)
			memcpy(pThis->pData + iDataFirst, data, len);
		else {
			int nFirst = pThis->nDataSize - iDataFirst;
			memcpy(pThis->pData + iDataFirst, data, nFirst);
			memcpy(pThis->pData, data + nFirst, len - nFirst);
		}
		pThis->nDataLen += len;
	}
}

/// Collapse format vector to a smallest power of 2.
/// Shift the circular buffer to start position.
static void CollapseFormat(RxView *pThis)
{
	// 1) Compute size of format buffer as a lowest power of 2
	int		nChunks = pThis->nChunksLen;
	int		iChunk  = pThis->iChunkFirst;
	int		nSize  = 1024;
	UCHAR  *pNew;
	int     iFirst;

	while (nSize < pThis->nFormatLen)
		nSize <<= 1;

	// 2) Copy formats to the new buffer
	pNew   = (UCHAR*)malloc(nSize);
	iFirst = pThis->pChunks[pThis->iChunkFirst].iFormat;
	if (iFirst + pThis->nFormatLen <= pThis->nFormatSize) {
		memcpy(pNew, pThis->pFormat + iFirst, pThis->nFormatLen);
	} else {
		int nFirst = pThis->nFormatSize - iFirst;
		memcpy(pNew, pThis->pFormat + iFirst, nFirst);
		memcpy(pNew + nFirst, pThis->pFormat, pThis->nFormatLen - nFirst);
	}

	// 3) Shift format indices in chunks
	while (nChunks > 0) {
		RxViewChunk *pChunk = pThis->pChunks + iChunk;
		pChunk->iFormat = (pChunk->iFormat - iFirst) & (pThis->nFormatSize - 1);
		if (++ iChunk == pThis->nChunksSize)
			iChunk = 0;
	}

	// 4) Store the new buffer to pThis
	free(pThis->pFormat);
	pThis->pFormat		= pNew;
	pThis->nFormatSize	= nSize;
}

/// Add formating block
static void AddFormat(RxView *pThis, UCHAR *data, int len)
{
	// 1) Make sure there is space for new format values
	int iLast;
	int nSizeNew = pThis->nFormatSize;
	while (pThis->nFormatLen + len > nSizeNew)
		nSizeNew <<= 1;

	if (nSizeNew > pThis->nFormatSize) {
		// make the buffer bigger
		if (pThis->pChunks[pThis->iChunkFirst].iFormat != 0)
			CollapseFormat(pThis);
		pThis->pFormat	   = realloc(pThis->pFormat, nSizeNew);
		pThis->nFormatSize = nSizeNew;
	}

	// 2) Copy the formats
	iLast = (pThis->pChunks[pThis->iChunkFirst].iFormat + pThis->nFormatLen) & (pThis->nFormatSize - 1);
	if (iLast + len <= pThis->nFormatSize) {
		memcpy(pThis->pFormat + iLast, data, len);
	} else {
		int nFirst = pThis->nFormatSize - iLast;
		memcpy(pThis->pFormat + iLast, data, nFirst);
		memcpy(pThis->pFormat, data + nFirst, len - nFirst);
	}

	pThis->nFormatLen += len;
}

static int WordWrapChunkText(RxView *pThis, int iChunk, int iDataNext, HDC hDC)
{
	int		iDataThis = pThis->pChunks[iChunk].iData;
	TCHAR  *pString   = 0;
	UCHAR  *pCopy     = 0;
	int     len		  = 0;
	int		nRows	  = 0;
	int		aFormat[2];

	iDataThis += sizeof(TCHAR);
	if (iDataThis >= pThis->nDataSize)
		iDataThis -= pThis->nDataSize;

	if (iDataThis < iDataNext) {
		len		= iDataNext - iDataThis;
		pString = (TCHAR*)(pThis->pData + iDataThis);
	} else {
		int nFirst = pThis->nDataSize - iDataThis;
		len		= iDataNext + pThis->nDataSize - iDataThis;
		pCopy   = malloc(len);
		pString = (TCHAR*)pCopy;
		memcpy(pString, pThis->pData + iDataThis, nFirst);
		memcpy(pString + nFirst, pThis->pData, len - nFirst);
	}

#ifdef _UNICODE
	len >>= 1;
#endif

	while (len > 0) {
		int  iEnd;
		int	 nChars;
		SIZE szString; 
		// How many characters pass into nWidth pixels?
		GetTextExtentExPoint(hDC, (TCHAR*)pString, len, pThis->nColumnsScreen, &nChars, 0, &szString);
		nRows += pThis->nFontHeight;
		if (nChars == len)
			break; // the rest of the line will not be split
		// The string has to be split.
		// Find start of the word that has to be split.
		iEnd = nChars - 1;
		while (iEnd > 0 && ! _istspace(pString[iEnd]))
			-- iEnd;

		if (iEnd == 0) {
			if (_istspace(*pString)) {
				// first character is space, wrap after the first space
				iEnd   = 1;
				nChars = 1;
			} else {
				// break the word
				iEnd   = nChars;
			}
		} else {
			_ASSERT(_istspace(pString[iEnd]));
			nChars = iEnd + 1;
		}
		_ASSERT(iEnd == nChars || iEnd + 1 == nChars);
		aFormat[0] = iEnd;		// number of characters to display
		aFormat[1] = nChars;	// number of characters to skip
		AddFormat(pThis, (UCHAR*)aFormat, sizeof(aFormat));
		len		  -= nChars;
		pString   += nChars;
	}

	free(pCopy);
	return nRows;
}

static int WordWrapChunkTape(RxView *pThis, int iChunk, int iDataNext, HDC hDC)
{
	int nLength			= iDataNext - pThis->pChunks[iChunk].iData - 1 + ((iDataNext >= pThis->pChunks[iChunk].iData + 1) ? 0 : pThis->nDataSize);
	int nColumns		= (nLength + pThis->nTapeRows - 1) / pThis->nTapeRows;
	int nColumnsScreen  = pThis->nColumnsScreen >> (pThis->bTapeDoubleWidth ? 1 : 0);
	int nLengthWrapped	= nColumnsScreen - pThis->nTapeWrapDuplicatedColumns;
	int nRows			= nColumns / nLengthWrapped;
	int nRest			= nColumns % nLengthWrapped;

	if (nRows == 0 || nRest > pThis->nTapeWrapDuplicatedColumns)
		++ nRows;

	return nRows * ((pThis->nTapeRows << (pThis->bTapeTwoTimes ? 1 : 0)) + (pThis->bTapeDrawSeparator ? 1 : 0));
}

static int WordWrapChunkImage(RxView *pThis, int iChunk, int iDataNext, HDC hDC)
{
	_ASSERT(FALSE);
	return 0;
}

static int WordWrapChunk(RxView *pThis, int iChunk, int iDataNext, HDC hDC)
{
	switch (pThis->pChunks[iChunk].nType) {
	case RXVIEW_CHUNK_TEXT:
		return WordWrapChunkText(pThis, iChunk, iDataNext, hDC);
	case RXVIEW_CHUNK_TAPE:
		return WordWrapChunkTape(pThis, iChunk, iDataNext, hDC);
	case RXVIEW_CHUNK_IMAGE:
		return WordWrapChunkImage(pThis, iChunk, iDataNext, hDC);
	}
	_ASSERT(FALSE);
	return 0;
}

static void SetScrollbar(RxView *pThis)
{
	SCROLLINFO si;
	memset(&si, 0, sizeof(si));
	si.cbSize = sizeof(si);
	si.fMask  = SIF_DISABLENOSCROLL | SIF_ALL;
	si.nMin   = 0;
	si.nMax   = pThis->nRows - 1;
	si.nPage  = pThis->nRowsScreen;
	si.nPos   = pThis->iRowFirst;
	if (pThis->nChunksLen > 0)
		si.nPos -= pThis->pChunks[pThis->iChunkFirst].iRow;
	SetScrollInfo(pThis->hWnd, SB_VERT, &si, FALSE);
}

/// Recalculate word and tape wrapping of all the chunks
void RecalculateWordWrap(RxView *pThis)
{
	int		nChunks		= pThis->nChunksLen;
	int		iChunk		= pThis->iChunkFirst;
	HDC		hDC			= GetWindowDC(pThis->hWnd);
	HGDIOBJ hPrevFont	= SelectObject(hDC, pThis->hFont);

	pThis->nFormatLen = 0;
	pThis->nRows      = 0;

	while (nChunks > 0) {
		RxViewChunk *pChunk = pThis->pChunks + iChunk;
		int          iNext  = (iChunk + 1) & (pThis->nChunksSize - 1);
		int			 iDataNext;

		if (-- nChunks == 0) {
			// last chunk
			iDataNext = pThis->pChunks[pThis->iChunkFirst].iData;
			iDataNext += pThis->nDataLen;
			if (iDataNext >= pThis->nDataSize)
				iDataNext -= pThis->nDataSize;
		} else {
			iDataNext = pThis->pChunks[iNext].iData;
		}

		pChunk->iFormat = pThis->nFormatLen;
		pChunk->iRow    = pThis->nRows;
		pThis->nRows   += WordWrapChunk(pThis, iChunk, iDataNext, hDC);
		iChunk			= iNext;
	}

	SelectObject(hDC, hPrevFont);
	ReleaseDC(pThis->hWnd, hDC);
	SetScrollbar(pThis);
}

void rxview_add_text(RxView *pThis, int type, TCHAR *data, int len, BOOL bUpdate)
{
	RxViewChunk *pChunkLast  = 0;
	int          iChunkLast  = (pThis->iChunkFirst + pThis->nChunksLen - 1) & (pThis->nChunksSize - 1);
	int			 i;
	BOOL		 bOlderFreed	= FALSE;
	int			 iRowFirstPrev  = (pThis->nChunksLen == 0) ? 0 : pThis->pChunks[pThis->iChunkFirst].iRow;
	int			 iRowLastPrev   = iRowFirstPrev + pThis->nRows;
	int			 nRowsPrev		= pThis->nRows;

	if (pThis->nChunksLen > 0)
		pChunkLast = pThis->pChunks + iChunkLast;
	if (pThis->nChunksLen == 0 || pChunkLast->nType != RXVIEW_CHUNK_TEXT ||
		((TCHAR*)pThis->pData)[pChunkLast->iData] != type) {
		pThis->bLastChunkClosed = TRUE;
	}

	while (len > 0) {
		int iNext = len;
		if (pThis->bLastChunkClosed) {
			AddChunk(pThis, RXVIEW_CHUNK_TEXT);
			// add text type ID
			AddData(pThis, (UCHAR*)&type, sizeof(TCHAR), &bOlderFreed);
			pThis->bLastChunkClosed = FALSE;
		}
		for (i = 0; i < len; ++ i) {
			if (data[i] == '\r') {
				iNext = i + 1;
				if (iNext < len && data[iNext] == '\n')
					++ iNext;
				pThis->bLastChunkClosed = TRUE;
				break;
			} else if (data[i] == '\n') {
				iNext = i + 1;
				pThis->bLastChunkClosed = TRUE;
				break;
			}
		}
		AddData(pThis, data, i, &bOlderFreed);
		len  -= iNext;
		data += iNext;
	}

	if (pThis->bIsAtEnd && pThis->nRows > pThis->nRowsScreen) {
		/// scroll the text if it is needed
		int	iRowFirstThis = pThis->pChunks[pThis->iChunkFirst].iRow;
		int iRowLastThis  = iRowFirstThis + pThis->nRows;
		if (iRowFirstThis != iRowFirstPrev || iRowLastThis != iRowLastPrev) {
			// refresh scrollbar, scroll the screen
			SetScrollbar(pThis);
			if (iRowLastThis != iRowLastPrev) {
				RECT rect = { 0, 0, pThis->nColumnsScreen, pThis->nRowsScreen };
				ScrollWindowEx(pThis->hWnd, 0, iRowLastThis - iRowLastPrev, &rect, 0, 0, 0, SW_INVALIDATE);
			}
		}
	}

//	if (pThis->nRows > 0x3fffffff) {
		// recalculate row indices
//	}
}

void rxview_add_tape(RxView *pThis, int type, UCHAR *data, int len, BOOL bUpdate)
{
	RxViewChunk *pChunkLast  = 0;
	int          iChunkLast  = (pThis->iChunkFirst + pThis->nChunksLen - 1) & (pThis->nChunksSize - 1);
	BOOL		 bOlderFreed = FALSE;
	int			 nRowsPrev   = pThis->nRows;

	if (pThis->nChunksLen > 0)
		pChunkLast = pThis->pChunks + iChunkLast;
	if (pThis->nChunksLen == 0 || pChunkLast->nType != RXVIEW_CHUNK_TAPE ||
		((UCHAR*)pThis->pData)[pChunkLast->iData] != type) {
		pThis->bLastChunkClosed = TRUE;
	}

	if (pThis->bLastChunkClosed) {
		AddChunk(pThis, RXVIEW_CHUNK_TAPE);
		// add text type ID
		AddData(pThis, (UCHAR*)&type, sizeof(TCHAR), &bOlderFreed);
		pThis->bLastChunkClosed = FALSE;
	}
	AddData(pThis, data, len, &bOlderFreed);

	if (nRowsPrev != pThis->nRows || bOlderFreed)
		// refresh scrollbar, scroll the screen
		SetScrollbar(pThis);

//	if (pThis->nRows > 0x3fffffff) {
		// recalculate row indices
//	}
}

void RenderChunkText(RxView *pThis, int iChunk, int iRowFirst, int iRowLast, int iDataNext, int iFormatNext, HDC hDC)
{
	RxViewChunk *pChunk    = pThis->pChunks + iChunk;
	int			 iDataThis = pChunk->iData;
	int			 iFormat   = pChunk->iFormat;
	TCHAR		*pString   = 0;
	UCHAR		*pCopy     = 0;
	int			 len	   = 0;
	int			 iRow	   = pChunk->iRow;

	// extract the type, set color
	if (iDataThis >= pThis->nDataSize)
		iDataThis -= pThis->nDataSize;
	{
		int iType = *((TCHAR*)(pThis->pData + iDataThis));
		SetTextColor(hDC, (iType == 0) ? RGB(255, 0, 0) : RGB(0, 0, 0));
	}

	iDataThis += sizeof(TCHAR);
	if (iDataThis >= pThis->nDataSize)
		iDataThis -= pThis->nDataSize;

	// Get the string. If it is wrapped around the pData boundary, make a copy.
	if (iDataThis < iDataNext) {
		len		= iDataNext - iDataThis;
		pString = (TCHAR*)(pThis->pData + iDataThis);
	} else {
		int nFirst = pThis->nDataSize - iDataThis;
		len		= iDataNext + pThis->nDataSize - iDataThis;
		pCopy   = malloc(len);
		pString = (TCHAR*)pCopy;
		memcpy(pString, pThis->pData + iDataThis, nFirst);
		memcpy(pString + nFirst, pThis->pData, len - nFirst);
	}

#ifdef _UNICODE
	len >>= 1;
#endif

	while (iRow + pThis->nFontHeight <= iRowFirst) {
		int nSkip = ((int*)(pThis->pFormat + iFormat))[1];
		pString += nSkip;
		len     -= nSkip;
		iRow    += pThis->nFontHeight;
		iFormat  = (iFormat + (sizeof(int) << 1)) & (pThis->nFormatSize - 1);
	}

	for (;;) {
		int  nCount     = len;
		int  nSkip		= 0;
		int  iRowScreen = iRow - pThis->iRowFirst;
		RECT rect;
		if (iFormat != iFormatNext) { // not the last line in the chunk
			int *pFormat = (int*)(pThis->pFormat + iFormat);
			nCount = pFormat[0];
			nSkip  = pFormat[1];
		}
		rect.left   = 0;
		rect.right  = pThis->nColumnsScreen;
		rect.top    = iRowScreen;
		rect.bottom = iRowScreen + pThis->nFontHeight;
		ExtTextOut(hDC, 0, iRowScreen /*  + pThis->nFontAscent */,
			/* ETO_CLIPPED | */ ETO_OPAQUE, &rect, pString, nCount, 0);
		iRow += pThis->nFontHeight;
		if (! nSkip || iRow >= iRowLast)
			break;
		len		  -= nSkip;
		pString   += nSkip;
		iFormat    = (iFormat + (sizeof(int) << 1)) & (pThis->nFormatSize - 1);
	}

	free(pCopy);
}

static void AllocateTapeBitmap(RxView *pThis)
{
	HDC				hDC		= GetWindowDC(0);
	BITMAPINFO		bmi;

	DeleteObject(pThis->hTapeBitmap);
	pThis->hTapeBitmap = 0;

	pThis->nTapeRowInc = ((pThis->nColumnsScreen * 24 + 31) & ~31) >> 3;

	memset(&bmi, 0, sizeof(BITMAPINFO));
	bmi.bmiHeader.biSize		= sizeof(bmi.bmiHeader);
	bmi.bmiHeader.biWidth		= pThis->nColumnsScreen;
	bmi.bmiHeader.biHeight		= pThis->nTapeRows;
	bmi.bmiHeader.biPlanes		= 1;
	bmi.bmiHeader.biBitCount	= 24;
	bmi.bmiHeader.biSizeImage   = pThis->nTapeRowInc * pThis->nTapeRows;
	bmi.bmiHeader.biCompression = BI_RGB;
	bmi.bmiHeader.biXPelsPerMeter = 3780;
	bmi.bmiHeader.biYPelsPerMeter = 3780;

	pThis->hTapeBitmap = CreateDIBSection(hDC, &bmi, DIB_RGB_COLORS, (void**)&pThis->pTapeBitmapBits, NULL, 0);
	memset(pThis->pTapeBitmapBits, 255, bmi.bmiHeader.biSizeImage);

	ReleaseDC(0, hDC);
}

void RenderChunkTape(RxView *pThis, int iChunk, int iRowFirst, int iRowLast, int iDataNext, int iFormatNext, HDC hDC)
{
	RxViewChunk *pChunk    = pThis->pChunks + iChunk;
	int			 iDataThis = pChunk->iData;
	UCHAR		*pData	   = 0;
	UCHAR		*pCopy     = 0;
	int			 nLength   = 0;
	int			 iRow	   = pChunk->iRow;
	int			 nColumns;
	int			 nLines;
	int			 nLengthWrapped;
	int			 nRest;
	int			 nTapeRows = pThis->nTapeRows;
	int			 nRowSize;
	HDC			 hBmpDC    = CreateCompatibleDC(hDC);
	int			 nColumnsScreen = pThis->nColumnsScreen >> (pThis->bTapeDoubleWidth ? 1 : 0);

	if (pThis->bTapeTwoTimes)
		nTapeRows <<= 1;
	if (pThis->bTapeDrawSeparator)
		++ nTapeRows;


	// extract the type, set color
	if (iDataThis >= pThis->nDataSize)
		iDataThis -= pThis->nDataSize;
	{
		int iType = *((TCHAR*)(pThis->pData + iDataThis));
		SetTextColor(hDC, (iType == 0) ? RGB(255, 0, 0) : RGB(0, 0, 0));
	}

	++ iDataThis;
	if (iDataThis >= pThis->nDataSize)
		iDataThis -= pThis->nDataSize;

	// Get the string. If it is wrapped around the pData boundary, make a copy.
	if (iDataThis < iDataNext) {
		nLength	= iDataNext - iDataThis;
		pData   = pThis->pData + iDataThis;
	} else {
		nLength = iDataNext + pThis->nDataSize - iDataThis;
		pCopy   = malloc(nLength);
		pData   = pCopy;
		memcpy(pData, pThis->pData + iDataThis, pThis->nDataLen - iDataThis);
		memcpy(pData + pThis->nDataLen - iDataThis, pThis->pData, nLength - pThis->nDataLen + iDataThis);
	}

	// allocate windows device independent bitmap
	if (pThis->hTapeBitmap == 0)
		AllocateTapeBitmap(pThis);

	nColumns		= (nLength + pThis->nTapeRows - 1) / pThis->nTapeRows;
	nLengthWrapped	= nColumnsScreen - pThis->nTapeWrapDuplicatedColumns;
	nLines			= nColumns / nLengthWrapped;
	nRest			= nColumns % nLengthWrapped;

	if (nLines == 0 || nRest > pThis->nTapeWrapDuplicatedColumns)
		++ nLines;

	nRowSize = nLengthWrapped * pThis->nTapeRows;

	// skip invisible rows
	while (iRow + nTapeRows <= iRowFirst) {
		_ASSERT(nLines > 0);
		pData   += nRowSize;
		nLength -= nRowSize;
		iRow    += nTapeRows;
		-- nLines;
	}

	while (nLines > 0) {
		UCHAR  *bmp = pThis->pTapeBitmapBits;
		int		len = nColumnsScreen * pThis->nTapeRows;
		int		i   = pThis->nTapeRows;
		int     j   = 0;
		BOOL	bRes = FALSE;
		HGDIOBJ hObjOld;
		POINT	aPts[2];
		if (len > nLength)
			len = nLength;
		/// clear the tape by a white color
		memset(pThis->pTapeBitmapBits, 255, pThis->nTapeRowInc * pThis->nTapeRows);
		while (len > 0) {
			memset(bmp, *pData, pThis->bTapeDoubleWidth ? 6 : 3);
			if (-- i > 0) {
				bmp += pThis->nTapeRowInc;
			} else {
				bmp = pThis->pTapeBitmapBits + (++ j) * (pThis->bTapeDoubleWidth ? 6 : 3);
				i   = pThis->nTapeRows;
			}
			-- len;
			++ pData;
		}
		hObjOld = SelectObject(hBmpDC, pThis->hTapeBitmap);
		bRes = BitBlt(hDC, 0, iRow - pThis->iRowFirst, pThis->nColumnsScreen, pThis->nTapeRows, hBmpDC, 0, 0, SRCCOPY);
		iRow += pThis->nTapeRows;
		if (pThis->bTapeTwoTimes) {
			bRes = BitBlt(hDC, 0, iRow - pThis->iRowFirst, pThis->nColumnsScreen, pThis->nTapeRows, hBmpDC, 0, 0, SRCCOPY);
			iRow += pThis->nTapeRows;
		}
		SelectObject(hBmpDC, hObjOld);
		if (pThis->bTapeDrawSeparator) {
			aPts[0].x = 0;
			aPts[1].x = ++ j;
			if (pThis->bTapeDoubleWidth)
				aPts[1].x <<= 1;
			aPts[0].y = aPts[1].y = iRow ++ - pThis->iRowFirst;
			Polyline(hDC, aPts, 2);
		}
		nLength -= nRowSize;
		-- nLines;
		pData -= pThis->nTapeWrapDuplicatedColumns * pThis->nTapeRows;
	}

	DeleteDC(hBmpDC);
	free(pCopy);
}

void RenderChunkImage(RxView *pThis, int iChunk, int iRowFirst, int iRowLast, int iDataNext, int iFormatNext, HDC hDC)
{
	_ASSERT(FALSE);
}

void RenderChunk(RxView *pThis, int iChunk, int iRowFirst, int iRowLast, int iDataNext, int iFormatNext, HDC hDC)
{
	switch (pThis->pChunks[iChunk].nType) {
	case RXVIEW_CHUNK_TEXT:
		RenderChunkText(pThis, iChunk, iRowFirst, iRowLast, iDataNext, iFormatNext, hDC);
		break;
	case RXVIEW_CHUNK_TAPE:
		RenderChunkTape(pThis, iChunk, iRowFirst, iRowLast, iDataNext, iFormatNext, hDC);
		break;
	case RXVIEW_CHUNK_IMAGE:
		RenderChunkImage(pThis, iChunk, iRowFirst, iRowLast, iDataNext, iFormatNext, hDC);
		break;
	default:
		_ASSERT(FALSE);
	}
}

int FindChunkOfRow(RxView *pThis, int iRowFirst)
{
	int iChunkFirst = pThis->iChunkFirst;
	int iChunkLast  = (iChunkFirst + pThis->nChunksLen - 1) & (pThis->nChunksSize - 1);

	if (iChunkFirst >= iChunkLast) {
		if (pThis->pChunks[0].iRow <= iRowFirst)
			iChunkFirst = 0;
		else
			iChunkLast = pThis->nChunksSize - 1;
	}
	
	// binary search in the list of chunks
	while (iChunkFirst + 1 < iChunkLast) {
		int iMiddle = (iChunkFirst + iChunkLast) >> 1;
		if (iRowFirst < pThis->pChunks[iMiddle].iRow)
			iChunkLast = iMiddle;
		else
			iChunkFirst = iMiddle;
	}

	return (iRowFirst < pThis->pChunks[iChunkLast].iRow) ? iChunkFirst : iChunkLast;
}

static void draw(RxView *pThis, HDC hDC, BOOL bRefresh)
{
	int     iRowFirst, iRowLast;
	int		iChunk;
	int     iChunkLast = (pThis->iChunkFirst + pThis->nChunksLen - 1) & (pThis->nChunksSize - 1);

	iRowFirst	= pThis->iRowFirst;
	iRowLast	= iRowFirst + pThis->nRowsScreen;
	if (iRowLast > iRowFirst + pThis->nRows)
		iRowLast = iRowFirst + pThis->nRows;
	iChunk		= FindChunkOfRow(pThis, iRowFirst);

	for (;;) {
		RxViewChunk *pChunk		= pThis->pChunks + iChunk;
		int          iChunkNext	= (iChunk + 1) & (pThis->nChunksSize - 1);
		RxViewChunk	*pChunkNext = pThis->pChunks + iChunkNext;

		if (iChunk == iChunkLast) {
			// last chunk ever
			RxViewChunk	*pChunkFirst = pThis->pChunks + pThis->iChunkFirst;
			int			 iDataEnd	 = pChunkFirst->iData   + pThis->nDataLen;
			int			 iFormatEnd  = (pChunkFirst->iFormat + pThis->nFormatLen) & (pThis->nFormatSize - 1);
			if (iDataEnd >= pThis->nDataSize)
				iDataEnd -= pThis->nDataSize;
			RenderChunk(pThis, iChunk, iRowFirst, iRowLast, iDataEnd, iFormatEnd, hDC);
			break;
		}

		if (iRowLast <= pChunkNext->iRow) {
			// last chunk on the screen
			RenderChunk(pThis, iChunk, iRowFirst, iRowLast, pChunkNext->iData, pChunkNext->iFormat, hDC);
			break;
		}

		// next chunk
		RenderChunk(pThis, iChunk, iRowFirst, pChunkNext->iRow, pChunkNext->iData, pChunkNext->iFormat, hDC);
		iChunk	  = iChunkNext;
		iRowFirst = pChunkNext->iRow;
	}
}

LRESULT CALLBACK rxview_wndproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	RxView *pThis = (RxView*)GetWindowLong(hWnd, GWL_USERDATA);
	if (message == WM_PAINT) {
		PAINTSTRUCT paint;
		HDC			hDC		  = BeginPaint(hWnd, &paint);
		HGDIOBJ		hPrevFont = SelectObject(hDC, pThis->hFont);
		draw(pThis, hDC, TRUE);
		SelectObject(hDC, hPrevFont);
		EndPaint(hWnd, &paint);
	} else if (message == WM_ERASEBKGND) {
		// ignore background erasing
		return TRUE;
//		return FALSE;
	} else if (message == WM_LBUTTONDOWN) {
	} else if (message == WM_SIZE) {
		if (pThis != 0) {
			RECT rect;
			int nColumnsPrev = pThis->nColumnsScreen;
			GetClientRect(hWnd, &rect);
			pThis->nColumnsScreen = rect.right - rect.left;
			pThis->nRowsScreen    = rect.bottom - rect.top;
			if (nColumnsPrev != pThis->nColumnsScreen) {
				DeleteObject(pThis->hTapeBitmap);
				pThis->hTapeBitmap = 0;
				RecalculateWordWrap(pThis);
			}
		}
	} else if (message == WM_MOUSEMOVE) {
	} else if (message == WM_LBUTTONUP) {
	} else if (message == WM_VSCROLL) {
	    int nScrollCode = (int)LOWORD(wParam);
		int iRowPrev    = pThis->iRowFirst;
		int iScroll;
		SCROLLINFO si = { sizeof(SCROLLINFO),
			SIF_PAGE|SIF_POS|SIF_RANGE|SIF_TRACKPOS, 0, 0, 0, 0, 0};
		GetScrollInfo(hWnd, SB_VERT, &si);
		switch (nScrollCode) {
		case SB_BOTTOM:
			si.nPos = pThis->nRows;
			break;
		case SB_TOP:
			si.nPos = 0;
		case SB_ENDSCROLL:
			break;
		case SB_LINEDOWN:
			si.nPos += pThis->nFontHeight;
			break;
		case SB_LINEUP:
			si.nPos -= pThis->nFontHeight;
			break;
		case SB_PAGEDOWN:
			si.nPos += si.nPage;
			break;
		case SB_PAGEUP:
			si.nPos -= si.nPage;
			break;
		case SB_THUMBPOSITION:
			break;
		case SB_THUMBTRACK:
			si.nPos = si.nTrackPos;
			break;
		}
		pThis->iRowFirst = si.nPos;
		if (pThis->iRowFirst < 0)
			pThis->iRowFirst = 0;
		else if (pThis->iRowFirst > pThis->nRows - pThis->nRowsScreen)
			pThis->iRowFirst = pThis->nRows - pThis->nRowsScreen;
	    si.fMask = SIF_POS;
		si.nPos  = pThis->iRowFirst;
		SetScrollInfo(hWnd, SB_VERT, &si, TRUE);
		pThis->iRowFirst += (pThis->nChunksLen == 0) ? 0 : pThis->pChunks[pThis->iChunkFirst].iRow;
		iScroll  = iRowPrev - pThis->iRowFirst;
		if (iScroll > - (pThis->nRowsScreen >> 1) && iScroll < (pThis->nRowsScreen >> 1)) {
			RECT rect = { 0, 0, pThis->nColumnsScreen, pThis->nRowsScreen };
			ScrollWindowEx(pThis->hWnd, 0, iScroll, &rect, 0, 0, 0, SW_INVALIDATE);
		} else
			InvalidateRect(hWnd, 0, FALSE);
		return 0;
	} else if (message == WM_MOUSEWHEEL) {
		SCROLLINFO si;
		int zDelta = HIWORD(wParam);
		pThis->iRowFirst += zDelta * pThis->nFontHeight / WHEEL_DELTA;
		if (pThis->iRowFirst < 0)
			pThis->iRowFirst = 0;
		else if (pThis->iRowFirst >= pThis->nRows)
			pThis->iRowFirst = pThis->nRows - 1;
	    si.fMask = SIF_POS;
		si.nPos  = pThis->iRowFirst;
		SetScrollInfo(hWnd, SB_VERT, &si, TRUE);
		InvalidateRect(hWnd, 0, FALSE);
		return 0;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}
