/*
 *    macro.c  --  Fixtexts and macros
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
$$  	  - The letter '$'
$tx 	  - Push the TX button.
$rx 	  - Switch to receive.
$mycall   - Callsign as defined in preferences.
$myname   - Name as defined in preferences.
$myqth    - QTH as defined in preferences.
$myloc    - Locator as defined in preferences.
$myemail  - Email address as defined in preferences.
$time     - Local time.
$utctime  - Universal Coordinated Time.
$date	  - Local date.
$utcdate  - UTC date.
$call     - Other partys callsign taken from QSO data.
$band	  - Band taken from QSO data.
$rxrst	  - Received RST taken from QSO data.
$txrst	  - Transmitted RST taken from QSO data.
$name	  - Other partys name taken from QSO data.
$qth	  - Other partys QTH taken from QSO data.
$notes	  - Notes taken from QSO data.
$soft	  - Software version.
$startqso - Set QSO start time to current time.
$logqso   - Send QSO data to Xlog
$clearqso - Clear qso information
$mode	  - Currently active mode name.
$pic	  - Send grayscale picture.
$picc	  - Send color picture
*/

#include <windows.h>
#include <commdlg.h>
#include <tchar.h>

#include <string.h>
#include <stdio.h>
#include <ctype.h>

#ifdef WIN32_PLATFORM_PSPC
#include "Aygshell.h"
#endif /* WIN32_PLATFORM_PSPC */

#include "macro.h"
#include "stninfo.h"
#include "qsodata.h"
#include "dlggen.h"
#include "sndmodem.h"
#include "../resource.h"

#define ID_LABEL	101
#define ID_NAME		102
#define ID_VALUE	103

typedef struct _Macro Macro;

struct _Macro {
	TCHAR *name;
	TCHAR *text;
};

static int   s_iMacroEdited = 0;
static Macro s_aMacros[NUMMACROS];

extern TCHAR *g_strSoftwareVersion;


INT_PTR CALLBACK macroconfig_dlgproc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	int res;
	
	if (uMsg == WM_INITDIALOG) {
		HWND hWndName  = GetDlgItem(hwnd, ID_NAME);
		HWND hWndValue = GetDlgItem(hwnd, ID_VALUE);
		if (s_aMacros[s_iMacroEdited].name != 0)
			SetWindowText(hWndName, s_aMacros[s_iMacroEdited].name);
		if (s_aMacros[s_iMacroEdited].text != 0)
			SetWindowText(hWndValue, s_aMacros[s_iMacroEdited].text);
	}

    switch(uMsg)
    {

	case WM_SIZE:
	{
		RECT rectWnd, rectName, rectValue;
		HWND hWndName  = GetDlgItem(hwnd, ID_NAME);
		HWND hWndValue = GetDlgItem(hwnd, ID_VALUE);
		GetClientRect(hwnd,		 &rectWnd);
		GetWindowRect(hWndName,  &rectName);
		GetWindowRect(hWndValue, &rectValue);
		ScreenToClient(hwnd, (LPPOINT)(&rectName.left));
		ScreenToClient(hwnd, (LPPOINT)(&rectName.right));
		ScreenToClient(hwnd, (LPPOINT)(&rectValue.left));
		ScreenToClient(hwnd, (LPPOINT)(&rectValue.right));
		MoveWindow(hWndName,  rectName.left, rectName.top,
			rectWnd.right - rectName.left - rectValue.left,
			rectName.bottom - rectName.top, TRUE);
#ifdef WIN32_PLATFORM_PSPC
		MoveWindow(hWndValue, -1, rectValue.top, rectWnd.right + 2,
			rectWnd.bottom - rectValue.top + 1, TRUE);
#else /* WIN32_PLATFORM_PSPC */
		MoveWindow(hWndValue, rectValue.left, rectValue.top,
			rectWnd.right  - (rectValue.left << 1), 
			rectWnd.bottom - rectValue.top - rectValue.left - 35, TRUE);
		{
			HWND hWndOK		= GetDlgItem(hwnd, IDOK);
			HWND hWndCancel = GetDlgItem(hwnd, IDCANCEL);
			RECT rectButton;
			GetWindowRect(hWndOK, &rectButton);
			ScreenToClient(hwnd, (LPPOINT)(&rectButton.left));
			ScreenToClient(hwnd, (LPPOINT)(&rectButton.right));
			MoveWindow(hWndCancel, 
				rectWnd.right - rectValue.left - (rectButton.right - rectButton.left), 
				rectWnd.bottom - rectValue.left - (rectButton.bottom - rectButton.top),
				rectButton.right - rectButton.left, 
				rectButton.bottom - rectButton.top, TRUE);
			MoveWindow(hWndOK, 
				rectWnd.right - 3 * rectValue.left - 2 * (rectButton.right - rectButton.left),
				rectWnd.bottom - rectValue.left - (rectButton.bottom - rectButton.top),
				rectButton.right - rectButton.left, 
				rectButton.bottom - rectButton.top, TRUE);
		}
#endif /* WIN32_PLATFORM_PSPC */
		return 0;
	}

	case WM_COMMAND:
        if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) {
			if (LOWORD(wParam) == IDOK) {
				HWND hWndName  = GetDlgItem(hwnd, ID_NAME);
				HWND hWndValue = GetDlgItem(hwnd, ID_VALUE);
				int  nLenName  = GetWindowTextLength(hWndName);
				int  nLenValue = GetWindowTextLength(hWndValue);
				if (nLenName > 0) {
					s_aMacros[s_iMacroEdited].name = (TCHAR*)realloc(
						s_aMacros[s_iMacroEdited].name, (nLenName + 1) * sizeof(TCHAR));
					GetWindowText(hWndName, s_aMacros[s_iMacroEdited].name, nLenName + 1);
				} else {
					free(s_aMacros[s_iMacroEdited].name);
					s_aMacros[s_iMacroEdited].name = 0;
				}
				if (nLenValue > 0) {
					s_aMacros[s_iMacroEdited].text = (TCHAR*)realloc(
						s_aMacros[s_iMacroEdited].text, (nLenValue + 1) * sizeof(TCHAR));
					GetWindowText(hWndValue, s_aMacros[s_iMacroEdited].text, nLenValue + 1);
				} else {
					free(s_aMacros[s_iMacroEdited].text);
					s_aMacros[s_iMacroEdited].text = 0;
				}
			}
            EndDialog(hwnd, LOWORD(wParam) == IDOK);
            return 0;
        }
        break;

    default:
        break;
	}


	res = dlggen_dlgproc(hwnd, uMsg, wParam, lParam);
	if (res >= 0)
		return res;

    return FALSE;
}

BOOL macro_edit(int n)
{
	DlgTemplate tmpl;
	dlggen_create(&tmpl, _T("Macro Editor"), DLGGEN_WINDOW_STYLE,
		CW_USEDEFAULT, CW_USEDEFAULT, 137, 170, _T("Tahoma"), 8);
	dlggen_add_static(&tmpl, _T("Name"),
		WS_CHILD | WS_VISIBLE, 0, 2, 2, 30, 10, ID_LABEL);
	dlggen_add_edit_box(&tmpl, _T(""), DLGGEN_ENTRY_STYLE, DLGGEN_ENTRY_STYLE_EX,
		34, 2, 101, 10, ID_NAME);
	dlggen_add_static(&tmpl, _T("Macro text"),
		WS_CHILD | WS_VISIBLE, 0, 2, 17, 134, 12, ID_LABEL);
	dlggen_add_edit_box(&tmpl, _T(""), DLGGEN_EDITBOX_STYLE, DLGGEN_EDITBOX_STYLE_EX,
		2, 30, 133, 100, ID_VALUE);
	dlggen_add_okcancel(&tmpl);
	s_iMacroEdited = n;
	tmpl.nEntries      = 1;
	tmpl.nEntryFirstID = ID_NAME;
	return dlggen_dialog_box(&tmpl, g_hInst, s_hWndMain, macroconfig_dlgproc) == IDOK;
}

extern UINT	s_iTXMode;

void macro_update_menu(HMENU hMenu)
{
	HMENU hMenuEdit = CreatePopupMenu();
	int   nEnabledOnRX = (s_iTXMode == 0) ? MF_ENABLED : MF_GRAYED;
	int i;

	// Remove all items
	for (i = 0; i < NUMMACROS + 5; ++ i)
		if (! DeleteMenu(hMenu, 0, MF_BYPOSITION))
			break;

	// Create edit items
	for (i = 0; i < NUMMACROS; ++ i) {
		TCHAR buf[256];
		_stprintf(buf, _T("%d: %s"), i + 1, (s_aMacros[i].name == 0) ? _T("No name") : s_aMacros[i].name);
		AppendMenu(hMenuEdit, nEnabledOnRX | MF_STRING, ID_MACRO_EDIT1 + i, buf);
	}

	for (i = 0; i < NUMMACROS; ++ i) {
		TCHAR buf[256];
		if (s_aMacros[i].name == 0 && s_aMacros[i].text == 0)
			continue;
		_stprintf(buf, _T("%d: %s"), i + 1, (s_aMacros[i].name == 0) ? _T("No name") : s_aMacros[i].name);
		AppendMenu(hMenu, MF_ENABLED | MF_STRING, ID_MACRO_EXECUTE1 + i, buf);
	}

	AppendMenu(hMenu, MF_ENABLED | MF_SEPARATOR, 0, 0);
	AppendMenu(hMenu, nEnabledOnRX | MF_STRING | MF_POPUP,  (UINT_PTR)hMenuEdit, _T("Edit"));
	AppendMenu(hMenu, MF_ENABLED | MF_STRING, ID_MACRO_SENDFILE, _T("Send file"));
	AppendMenu(hMenuEdit, nEnabledOnRX | MF_STRING, ID_MACRO_DEFAULTS, _T("Load defaults"));
}

void macro_load()
{
	HKEY  hKeyMacros = 0;
	int   i;

	memset(s_aMacros, 0, sizeof(s_aMacros));

	if (RegOpenKeyEx(HKEY_CURRENT_USER, _T("Software\\OK1IAK\\PocketDigi\\Macros"), 0, 
		KEY_ENUMERATE_SUB_KEYS | KEY_QUERY_VALUE | KEY_READ, &hKeyMacros) != ERROR_SUCCESS) {
		macro_load_defaults();
		return;
	}

	for (i = 0; i < NUMMACROS; ++ i) {
		TCHAR strKey[4];
		HKEY hKeyThis	 = 0;
		DWORD dwType	 = REG_SZ;
		DWORD dwSizeName = 0;
		DWORD dwSizeText = 0;
		_stprintf(strKey, _T("%d"), i + 1);
		if (RegOpenKeyEx(hKeyMacros, strKey, 0, 
			KEY_ENUMERATE_SUB_KEYS | KEY_QUERY_VALUE | KEY_READ, &hKeyThis) != ERROR_SUCCESS)
			continue;
		if (RegQueryValueEx(hKeyThis, _T("Name"), 0, &dwType, 0, &dwSizeName) == ERROR_SUCCESS) {
			s_aMacros[i].name = malloc(dwSizeName);
			RegQueryValueEx(hKeyThis, _T("Name"), 0, &dwType, (LPBYTE)s_aMacros[i].name, &dwSizeName);
		}
		if (RegQueryValueEx(hKeyThis, _T("Text"), 0, &dwType, 0, &dwSizeText) == ERROR_SUCCESS) {
			s_aMacros[i].text = malloc(dwSizeText);
			RegQueryValueEx(hKeyThis, _T("Text"), 0, &dwType, (LPBYTE)s_aMacros[i].text, &dwSizeText);
		}
		RegCloseKey(hKeyThis);
	}

	RegCloseKey(hKeyMacros);
}

void macro_load_defaults()
{
	int i;
	for (i = 0; i < NUMMACROS; ++ i) {
		free(s_aMacros[i].name);
		free(s_aMacros[i].text);
	}
	memset(s_aMacros, 0, sizeof(s_aMacros));

	if (_tcscmp(MACRODEF1NAME, _T("")) != 0) {
		s_aMacros[0].name = _tcsdup(MACRODEF1NAME);
		s_aMacros[0].text = _tcsdup(MACRODEF1TEXT);
	}
	if (_tcscmp(MACRODEF2NAME, _T("")) != 0) {
		s_aMacros[1].name = _tcsdup(MACRODEF2NAME);
		s_aMacros[1].text = _tcsdup(MACRODEF2TEXT);
	}
	if (_tcscmp(MACRODEF3NAME, _T("")) != 0) {
		s_aMacros[2].name = _tcsdup(MACRODEF3NAME);
		s_aMacros[2].text = _tcsdup(MACRODEF3TEXT);
	}
	if (_tcscmp(MACRODEF4NAME, _T("")) != 0) {
		s_aMacros[3].name = _tcsdup(MACRODEF4NAME);
		s_aMacros[3].text = _tcsdup(MACRODEF4TEXT);
	}
	if (_tcscmp(MACRODEF5NAME, _T("")) != 0) {
		s_aMacros[4].name = _tcsdup(MACRODEF5NAME);
		s_aMacros[4].text = _tcsdup(MACRODEF5TEXT);
	}
	if (_tcscmp(MACRODEF6NAME, _T("")) != 0) {
		s_aMacros[5].name = _tcsdup(MACRODEF6NAME);
		s_aMacros[5].text = _tcsdup(MACRODEF6TEXT);
	}
	if (_tcscmp(MACRODEF7NAME, _T("")) != 0) {
		s_aMacros[6].name = _tcsdup(MACRODEF7NAME);
		s_aMacros[6].text = _tcsdup(MACRODEF7TEXT);
	}
	if (_tcscmp(MACRODEF8NAME, _T("")) != 0) {
		s_aMacros[7].name = _tcsdup(MACRODEF8NAME);
		s_aMacros[7].text = _tcsdup(MACRODEF8TEXT);
	}
	if (_tcscmp(MACRODEF9NAME, _T("")) != 0) {
		s_aMacros[8].name = _tcsdup(MACRODEF9NAME);
		s_aMacros[8].text = _tcsdup(MACRODEF9TEXT);
	}
	if (_tcscmp(MACRODEF10NAME, _T("")) != 0) {
		s_aMacros[9].name = _tcsdup(MACRODEF10NAME);
		s_aMacros[9].text = _tcsdup(MACRODEF10TEXT);
	}
	if (_tcscmp(MACRODEF11NAME, _T("")) != 0) {
		s_aMacros[10].name = _tcsdup(MACRODEF11NAME);
		s_aMacros[10].text = _tcsdup(MACRODEF11TEXT);
	}
	if (_tcscmp(MACRODEF12NAME, _T("")) != 0) {
		s_aMacros[11].name = _tcsdup(MACRODEF12NAME);
		s_aMacros[11].text = _tcsdup(MACRODEF12TEXT);
	}
}

void macro_store()
{
	HKEY  hKeyMacros    = 0;
	DWORD dwDisposition = 0;
	int   i;

	if (RegCreateKeyEx(HKEY_CURRENT_USER, _T("Software\\OK1IAK\\PocketDigi\\Macros"), 0, 0,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, 0, &hKeyMacros, &dwDisposition) != ERROR_SUCCESS)
		return;

	for (i = 0; i < NUMMACROS; ++ i) {
		TCHAR strKey[4];
		HKEY hKeyThis	 = 0;
		DWORD dwType	 = REG_SZ;
		DWORD dwSizeName = 0;
		DWORD dwSizeText = 0;
		if (s_aMacros[i].name == 0 && s_aMacros[i].text == 0)
			continue;
		_stprintf(strKey, _T("%d"), i + 1);
		if (RegCreateKeyEx(hKeyMacros, strKey, 0, 0, 0, KEY_SET_VALUE, 0, &hKeyThis, &dwDisposition) != ERROR_SUCCESS)
			continue;
		if (s_aMacros[i].name != 0) 
			RegSetValueEx(hKeyThis, _T("Name"), 0, REG_SZ, 
				(LPBYTE)s_aMacros[i].name, (_tcslen(s_aMacros[i].name) + 1) * sizeof(TCHAR));
		if (s_aMacros[i].text != 0)
			RegSetValueEx(hKeyThis, _T("Text"), 0, REG_SZ, 
				(LPBYTE)s_aMacros[i].text, (_tcslen(s_aMacros[i].text) + 1) * sizeof(TCHAR));
		RegCloseKey(hKeyThis);
	}

	RegCloseKey(hKeyMacros);
}

static TCHAR *getword(TCHAR **ptr, BOOL bLineStart)
{
	TCHAR *word, *p;

	switch (**ptr) {
	case '(':
		if ((p = _tcschr(*ptr, ')')) == NULL)
			return NULL;
		++ p;
		break;
	case '{':
		if ((p = _tcschr(*ptr, '}')) == NULL)
			return NULL;
		++ p;
		break;
	default:
		for (p = *ptr; *p && isalnum(*p); ++ p)
			;
		break;
	}

	word = _tcsdup(*ptr /*, p - *ptr + 1 */);
	word[p - *ptr] = 0;

	if (bLineStart) {
		TCHAR *p2 = p;
		// if the macro stays at the start of the line and there is no other text on the same line,
		// ignore all whitespaces and enter
		// 
		// skip whitespaces
		for (; *p2 && (*p2 == ' ' || *p2 == '\t'); ++ p2) ;
		// if enter or end of macro, we can safely ignore all the whitespaces
		if (*p2 == '\r') {
			p = ++ p2;
			if (*p2 == '\n')
				p = ++ p2;
		} else if (*p2 == '\n')
			p = ++ p2;
		else if (*p2 == 0)
			p = p2;
	}

	*ptr = p;
	return word;
}

void macro_send(Trx *trx, int iMacro)
{
//	_ASSERT(iMacro >= 0 && iMacro < NUMMACROS);
	macro_sendtext(trx, s_aMacros[iMacro].text, iMacro, FALSE);
}

typedef struct _MacroBuffer MacroBuffer;

struct _MacroBuffer
{
	Trx   *trx;
	int    nBufOutSize;
	int    nBufOutSend;
	int    nBufOutLen;
	TCHAR *pBufOut;
	TCHAR  cPrev;
};

void BufOutChar(MacroBuffer *pBuffer, TCHAR c)
{
	if (c == '\r') {
		pBuffer->pBufOut[pBuffer->nBufOutLen ++] = '\r';
		pBuffer->pBufOut[pBuffer->nBufOutLen ++] = '\n';
		pBuffer->cPrev = '\r';
	} else if (c == '\n') {
		if (pBuffer->cPrev == '\r') {
			//CRLF, ignore the second line feed
			pBuffer->cPrev = 0;
			return;
		}
		pBuffer->pBufOut[pBuffer->nBufOutLen ++] = '\r';
		pBuffer->pBufOut[pBuffer->nBufOutLen ++] = '\n';
		pBuffer->cPrev = '\n';
	} else {
		pBuffer->cPrev = 0;
		pBuffer->pBufOut[pBuffer->nBufOutLen ++] = c;
	}
	if (pBuffer->nBufOutLen > pBuffer->nBufOutSend) {
		pBuffer->pBufOut[pBuffer->nBufOutLen ++] = 0;
		trx_send_string(pBuffer->trx, pBuffer->pBufOut);
		pBuffer->nBufOutLen = 0;
	}
}

void BufOutString(MacroBuffer *pBuffer, LPCTSTR str)
{
	UINT i = 0;
	UINT len = _tcslen(str);
	for (i = 0; i < len; ++ i)
		BufOutChar(pBuffer, str[i]);
}

static void send_date(MacroBuffer *pBuffer, BOOL utc)
{
	TCHAR buf[256];
	SYSTEMTIME tm;

	if (utc)
		GetSystemTime(&tm);
	else
		GetLocalTime(&tm);

	_stprintf(buf, _T("%02d.%02d.%02d"), tm.wDay, tm.wMonth, tm.wYear);
	BufOutString(pBuffer, buf);
}

static void send_time(MacroBuffer *pBuffer, BOOL utc)
{
	TCHAR buf[256];
	SYSTEMTIME tm;

	if (utc)
		GetSystemTime(&tm);
	else
		GetLocalTime(&tm);

	_stprintf(buf, _T("%02d:%02d:%02d"), tm.wHour, tm.wMinute, tm.wSecond);
	BufOutString(pBuffer, buf);
}

void macro_sendtext(Trx *trx, LPCTSTR strMacro, int iMacro, BOOL bRaw)
{
	TCHAR	   *p, *str, *word;
	TCHAR		c;
	MacroBuffer bufOut;
	BOOL	    bSendTX = FALSE;

	bufOut.trx			= trx;
	bufOut.nBufOutSize	= 17408;
	bufOut.nBufOutSend	= 16384;
	bufOut.nBufOutLen	= 0;
	bufOut.pBufOut		= (TCHAR*)malloc(sizeof(TCHAR) * bufOut.nBufOutSize);
	bufOut.cPrev		= 0;

	str = _tcsdup(strMacro);
	p   = str;

	while (p && *p) {
		c = *p;
//		p = &_tcsnextc(p);
		++ p;

		if (c == '$') {
			TCHAR *cmd = NULL;
			TCHAR *arg = NULL;

			word = getword(&p, p - 1 == str || p[-2] == '\r' || p[-2] == '\n');

			if (word == NULL)
				continue;

			switch (*word) {
			case '(':
//				run_command(word);
				free(word);
				continue;
			case '{':
				cmd = word + 1;
				word[_tcslen(word) - 1] = 0;
				break;
			default:
				cmd = word;
				break;
			}

			if ((arg = _tcschr(cmd, ':')) != NULL)
				*arg++ = 0;

			if (!_tcsicmp(cmd, _T("$")))
				BufOutChar(&bufOut, '$');

			/*
			 * Version string
			 */
			if (!_tcsicmp(cmd, _T("ver")))
				BufOutString(&bufOut, g_strSoftwareVersion);

			/*
			 * Buttons
			 */
			if (!_tcsicmp(cmd, _T("tx")))
				bSendTX = TRUE;

			if (!_tcsicmp(cmd, _T("rx")))
				BufOutChar(&bufOut, TRX_RX_CMD);

			if (!_tcsicmp(cmd, _T("txtoggle")) && ! trx->stopflag)
				if (trx_is_sending(trx)) {
					BufOutChar(&bufOut, TRX_RX_CMD);
				} else
					SendMessage(s_hWndMain, WM_COMMAND, ID_SENDBUF, 0);

			if (!_tcsicmp(cmd, _T("rxandclear")))
				BufOutChar(&bufOut, TRX_CLEAR_CMD);

			if (!_tcsicmp(cmd, _T("clearrx")))
				trx_clearrx(trx);

			/*
			 * My station info
			 */
			if (!_tcsicmp(cmd, _T("mycall")))
				BufOutString(&bufOut, g_StnInfo.strCall);

			if (!_tcsicmp(cmd, _T("myname")))
				BufOutString(&bufOut, g_StnInfo.strName);

			if (!_tcsicmp(cmd, _T("myqth")))
				BufOutString(&bufOut, g_StnInfo.strQth);

			if (!_tcsicmp(cmd, _T("myloc")))
				BufOutString(&bufOut, g_StnInfo.strLoc);

			if (!_tcsicmp(cmd, _T("myemail")))
				BufOutString(&bufOut, g_StnInfo.strEmail);

			/*
			 * Time and date
			 */
			if (!_tcsicmp(cmd, _T("time")))
				send_time(&bufOut, FALSE);

			if (!_tcsicmp(cmd, _T("utctime")))
				send_time(&bufOut, TRUE);

			if (!_tcsicmp(cmd, _T("date")))
				send_date(&bufOut, FALSE);

			if (!_tcsicmp(cmd, _T("utcdate")))
				send_date(&bufOut, TRUE);

			/*
			 * QSO data
			 */
			if (!_tcsicmp(cmd, _T("call")))
				BufOutString(&bufOut, trx->pQso->strCall);

			if (!_tcsicmp(cmd, _T("band")))
				BufOutString(&bufOut, trx->pQso->strBand);

			if (!_tcsicmp(cmd, _T("rxrst")))
				BufOutString(&bufOut, trx->pQso->strRXRst);

			if (!_tcsicmp(cmd, _T("txrst")))
				BufOutString(&bufOut, trx->pQso->strTXRst);

			if (!_tcsicmp(cmd, _T("name")))
				BufOutString(&bufOut, trx->pQso->strName);

			if (!_tcsicmp(cmd, _T("qth")))
				BufOutString(&bufOut, trx->pQso->strQth);

			if (!_tcsicmp(cmd, _T("notes")))
				BufOutString(&bufOut, trx->pQso->strNotes);

			if (!_tcsicmp(cmd, _T("autocq")) && iMacro != -1) {
				bSendTX = TRUE;
				sndmodem_set_autocq(trx->pModem, 1000 * ((arg == 0 || _ttoi(arg) == 0) ? 5 : _ttoi(arg)), iMacro);
			}

			/*
			 * QSO logging
			 */
//			if (!_tcsicmp(cmd, _T("startqso")))
//				qsodata_set_starttime(TRUE);

//			if (!_tcsicmp(cmd, _T("logqso")))
//				qsodata_log();

//			if (!_tcsicmp(cmd, _T("clearqso")))
//				qsodata_clear();

			/*
			 * Mode
			 */
//			if (!_tcsicmp(cmd, _T("mode")))
//				BufOutString(&bufOut, trx_get_mode_name());

			/*
			 * Pictures
			 */
/*
			if (!_tcsicmp(cmd, _T("pic")))
				picture_send(arg, FALSE);

			if (!_tcsicmp(cmd, _T("picc")))
				picture_send(arg, TRUE);
*/

			/* an unknown macro gets ignored */

			free(word);
			continue;
		}

		BufOutChar(&bufOut, c);
	}

	if (bufOut.nBufOutLen > 0) {
		bufOut.pBufOut[bufOut.nBufOutLen ++] = 0;
		trx_send_string(trx, bufOut.pBufOut);
	}

	free(str);
	free(bufOut.pBufOut);

	if (bSendTX)
		SendMessage(s_hWndMain, WM_COMMAND, ID_SENDBUF, 0);
}

BOOL macro_sendfile(Trx *trx, LPCTSTR szFilePath, BOOL bRaw, BOOL bLock)
{
	HANDLE hFile = CreateFile(szFilePath, bLock ? (GENERIC_READ | GENERIC_WRITE) : GENERIC_READ, 
		bLock ? 0 : FILE_SHARE_READ, 0, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
	DWORD  nRead = 0;
	UCHAR *buf   = 0;
	BOOL   bRes  = FALSE;

	if (hFile == INVALID_HANDLE_VALUE)
		return FALSE;

	buf = malloc(65536);
	if (buf != 0 && ReadFile(hFile, buf, 65534, &nRead, 0)) {
		UINT sum = 0;
#ifdef _WIN32_WCE
		int  i;
		for (i = 1; i < (int)nRead; i += 2)
			sum += buf[i];
		if (sum > nRead * 10) {
			// ascii, convert to unicode
			if (nRead > 32767)
				nRead = 32767;
			for (i = (int)(nRead - 1); i >= 0; -- i) {
				buf[i << 1]		  = buf[i];
				buf[(i << 1) + 1] = 0;
			}
			nRead <<= 1;
		}
#endif /* _WIN32_WCE */
		buf[nRead ++] = 0;
		buf[nRead]    = 0;
		macro_sendtext(trx, (LPCTSTR)((void*)buf), -1, bRaw);
		bRes = TRUE;
	}
	CloseHandle(hFile);
	free(buf);
	return bRes;
}

#define SZ_FILEFILTER	_T("Text Files (*.txt)\0*.txt\0All Files (*.*)\0*.*\0")
#define SZ_DEFEXT		_T("txt")

BOOL macro_chooseandsendfile(Trx *trx)
{
	OPENFILENAME ofn;
	TCHAR		 szFilePath[1024];
	HANDLE		 hFile = 0;
	
	memset(&ofn, 0, sizeof(ofn));
	szFilePath[0]	 = _T('\0');
    ofn.lStructSize	 = sizeof(OPENFILENAME);
	ofn.hwndOwner	 = s_hWndMain;
    ofn.lpstrFilter	 = SZ_FILEFILTER;
    ofn.lpstrDefExt	 = SZ_DEFEXT;
    ofn.lpstrFile	 = szFilePath;
    ofn.nMaxFile	 = sizeof(szFilePath) / sizeof(TCHAR);
	ofn.Flags		 = OFN_PATHMUSTEXIST;
	
	return GetOpenFileName(&ofn) && macro_sendfile(trx, szFilePath, /* raw */ FALSE, /* lock */ FALSE);
}
