/*
 *    log.c  --  Received text logging
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <windows.h>
#include <stdio.h>
#include <tchar.h>
#include <shellapi.h>
#include <shlobj.h>

#include "log.h"
#include "../misc/winsupport.h"

#ifdef _WIN32_WCE
#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

static TCHAR *lognames[] = {
	_T("RX"),
	_T("TX")
};

#define	TIMESTR_BUF_LEN		64

static void format_time(TCHAR *pBuffer)
{
	SYSTEMTIME tm;
	GetSystemTime(&tm);
//	strftime(timestr, TIMESTR_BUF_LEN, "%Y-%m-%d %H:%M", tm);
	_stprintf(pBuffer, _T("%02d-%02d-%02d %02d:%02d"/*:%02d"*/), 
		tm.wYear, tm.wMonth, tm.wDay, tm.wHour, tm.wMinute);
}

BOOL WriteString8Bit(HANDLE hFile, const TCHAR *str)
{
	DWORD  nWritten;
#ifdef _UNICODE
	UCHAR  buf[256];
	UCHAR *c    = (UCHAR*)str;
	int	   nLen = _tcslen(str);
	int	   i;
	for (i = 0; i <= nLen; ++ i, c += 2)
		buf[i] = *c;
	return WriteFile(hFile, (void*)buf, nLen, &nWritten, 0) != 0;
#else /* _UNICODE */
	return WriteFile(hFile, (void*)str, _tcslen(str) * sizeof(TCHAR), &nWritten, 0) != 0;
#endif /* _UNICODE */
}

void log_to_file(Log *log, log_t type, const TCHAR *s)
{	
	if (! log->hFile)
		return;

	// ignore lf of the crlf sequence
	if (log->bLogTypeLast == type && log->nCharPrev == '\r' && *s == '\n') {
		log->nCharPrev = '\n';
		return;
	}

	do {
		TCHAR timestr[TIMESTR_BUF_LEN];
		TCHAR buf[1024];
		int   nLen		  = _tcslen(s);
		TCHAR cLast		  = s[nLen - 1];
		BOOL  bAddEnter   = FALSE;
		BOOL  bAddHeader  = FALSE;

		if (log->nCharPrev == '\r' || log->nCharPrev == '\n')
			bAddHeader = TRUE;
		else if (log->bLogTypeLast != type) {
			bAddEnter  = TRUE;
			bAddHeader = TRUE;
		}

		if (bAddEnter) {
			TCHAR buf[3] = { '\r', '\n', 0 };
			if (! WriteString8Bit(log->hFile, buf)) break;
		}

		if (bAddHeader) {
			format_time(timestr);
			_stprintf(buf, _T("%s (%sZ): "), lognames[type], timestr);
			if (! WriteString8Bit(log->hFile, buf)) break;
		}

		if (*s == '\r' || *s == '\n') {
			TCHAR buf[3] = { '\r', '\n', 0 };
			if (! WriteString8Bit(log->hFile, buf)) break;
		} else {
			if (! WriteString8Bit(log->hFile, s)) break;
		}
		if (FlushFileBuffers(log->hFile) == 0) break;

		log->nCharPrev    = cLast;
		log->bLogTypeLast = type;
		return;
	} while (0);

//	errmsg(_("Error writing to log file: %m"));
	CloseHandle(log->hFile);
	log->hFile = 0;
}

BOOL log_to_file_activate(Log *log, BOOL start, int index)
{
	TCHAR timestr[TIMESTR_BUF_LEN];	
	TCHAR szPath[MAX_PATH];
	TCHAR buf[1024];

	if (start) {
		if (! log->hFile) {
			log->nCharPrev	  = '\n';
			log->bLogTypeLast = LOG_RX;
			if (! GetPocketDigiDir(szPath))
				return FALSE;
			_stprintf(buf, _T("\\Channel%d.log"), index + 1);
			_tcscat(szPath, buf);
			log->hFile = CreateFile(szPath, GENERIC_WRITE, FILE_SHARE_READ, 0,
				OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0);
			if (log->hFile == 0)
				return FALSE;
			SetFilePointer(log->hFile, 0, 0, FILE_END);
			format_time(timestr);
			_stprintf(buf, _T("\r\n--- Logging started at %s UTC ---\r\n"), timestr);
			WriteString8Bit(log->hFile, buf);
		}
		return TRUE;
	}
	
	if (log->hFile) {
		format_time(timestr);
		_stprintf(buf, _T("\r\n--- Logging stopped at %s UTC ---\r\n"), timestr);
		WriteString8Bit(log->hFile, buf);
		CloseHandle(log->hFile);
		log->hFile = 0;
	}

	return TRUE;
}
