/*
 *    devinfo.c  --  Device info
 *
 *    Copyright (C) 2006
 *      Vojtech Bubnik OK1IAK (bubnikv@seznam.cz)
 *
 *    This file is part of PocketDigi.
 *
 *    PocketDigi is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    PocketDigi is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <windows.h>
#include <tchar.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <shellapi.h>
#include <shlobj.h>

#ifdef _WIN32_WCE
#ifdef WIN32_PLATFORM_PSPC
#include "Aygshell.h"
#endif /* WIN32_PLATFORM_PSPC */
#endif /* _WIN32_WCE */

#include "devinfo.h"
#include "dlggen.h"
#include "../resource.h"
#include "../misc/winsupport.h"

extern HINSTANCE	g_hInst;
extern HWND			s_hWndMain;

#define ID_EDITBOX	102

static void ExpandSoundInfo(HWND hWnd);
static void ExpandSoundInputInfo(HWND hwnd, int iDevice);
static void ExpandSoundOutputInfo(HWND hwnd, int iDevice);
#ifndef _WIN32_WCE
static void ExpandMixerInfo(HWND hwnd, int iDevice);
#endif /* _WIN32_WCE */
static void ExpandSysInfo(HWND hwnd);
static void ExpandDeviceCaps(HWND hwnd);
static BOOL devinfo_write_to_file(HWND hWnd, TCHAR *szPath);

INT_PTR CALLBACK devinfo_dlgproc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	int res;
	
	if (uMsg == WM_INITDIALOG) {
		ExpandSysInfo(GetDlgItem(hwnd, ID_EDITBOX));
		ExpandDeviceCaps(GetDlgItem(hwnd, ID_EDITBOX));
		ExpandSoundInfo(GetDlgItem(hwnd, ID_EDITBOX));
	}

	switch(uMsg)
    {

	case WM_SIZE:
	{
		RECT rectWnd, rectValue;
		HWND hWndValue;
		GetClientRect(hwnd, &rectWnd);
		hWndValue = GetDlgItem(hwnd, ID_EDITBOX);
		GetWindowRect(hWndValue, &rectValue);
		ScreenToClient(hwnd, (LPPOINT)(&rectValue.left));
		ScreenToClient(hwnd, (LPPOINT)(&rectValue.right));
#ifndef WIN32_PLATFORM_PSPC
		{
			HWND hWndOK	= GetDlgItem(hwnd, IDOK);
			RECT rectButton;
			GetWindowRect(hWndOK, &rectButton);
			ScreenToClient(hwnd, (LPPOINT)(&rectButton.left));
			ScreenToClient(hwnd, (LPPOINT)(&rectButton.right));
			MoveWindow(hWndOK,
				rectWnd.right - DLG_MARGIN - (rectButton.right - rectButton.left),
				rectWnd.bottom - DLG_MARGIN - (rectButton.bottom - rectButton.top),
				rectButton.right - rectButton.left, 
				rectButton.bottom - rectButton.top, TRUE);
			
			hWndOK = GetDlgItem(hwnd, ID_EXPORTTOFILE);
			GetWindowRect(hWndOK, &rectButton);
			ScreenToClient(hwnd, (LPPOINT)(&rectButton.left));
			ScreenToClient(hwnd, (LPPOINT)(&rectButton.right));
			MoveWindow(hWndOK,
				rectWnd.right - DLG_MARGIN - (rectButton.right - rectButton.left) - 80,
				rectWnd.bottom - DLG_MARGIN - (rectButton.bottom - rectButton.top),
				rectButton.right - rectButton.left,
				rectButton.bottom - rectButton.top, TRUE);

			rectWnd.bottom -= rectButton.bottom - rectButton.top + 2 * DLG_MARGIN;
		}
#endif /* WIN32_PLATFORM_PSPC */
		MoveWindow(hWndValue, rectValue.left, rectValue.top,
			rectWnd.right - rectWnd.left - rectValue.left - 2, 
			rectWnd.bottom - rectWnd.top - rectValue.top, TRUE);
		return 0;
	}

    case WM_COMMAND:
        if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) {
            EndDialog(hwnd, LOWORD(wParam) == IDOK);
			return 0;
		} else if (LOWORD(wParam) == ID_EXPORTTOFILE) {
			HWND	hEditBox = GetDlgItem(hwnd, ID_EDITBOX);
			TCHAR	szPath[MAX_PATH];
			if (devinfo_write_to_file(hEditBox, szPath)) {
				TCHAR buf[MAX_PATH + 100] = _T("Log was saved to a file\r\n");
				_tcscat(buf, szPath);
				MessageBox(hwnd, buf, _T("PocketDigi"), MB_ICONINFORMATION | MB_OK);
			} else
				MessageBox(hwnd, _T("Cannot create log file."), _T("PocketDigi"), MB_ICONERROR | MB_OK);
			return 0;
		}
        break;

	default:
        break;
	}

	res = dlggen_dlgproc(hwnd, uMsg, wParam, lParam);
	if (res >= 0)
		return res;

    return FALSE;
}

void devinfo_show()
{
	DlgTemplate tmpl;

	dlggen_create(&tmpl, _T("Device Info"), DLGGEN_WINDOW_STYLE,
		CW_USEDEFAULT, CW_USEDEFAULT, 110, 90, _T("Tahoma"), 8);
	tmpl.nMenuID = IDR_MENUBAR_DEVINFO;

	dlggen_add_edit_box(&tmpl, _T(""),
		WS_BORDER | WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE | WS_VSCROLL | ES_AUTOVSCROLL | ES_MULTILINE, 
		0, 2, 2, 100, 100, ID_EDITBOX);

#ifndef WIN32_PLATFORM_PSPC
	dlggen_add_button(&tmpl, _T("Close"),
		WS_CHILD | WS_VISIBLE | WS_TABSTOP, 0,
		2, 140, 50, 15, IDOK);
	dlggen_add_button(&tmpl, _T("Export to file"),
		WS_CHILD | WS_VISIBLE | WS_TABSTOP, 0,
		50, 140, 50, 15, ID_EXPORTTOFILE);
#endif /* WIN32_PLATFORM_PSPC */

	dlggen_dialog_box(&tmpl, g_hInst, s_hWndMain, devinfo_dlgproc);
}

void AddString(HWND hwnd, TCHAR *str)
{
	int nLength = SendMessage(hwnd, WM_GETTEXTLENGTH, 0, 0);
	SendMessage(hwnd, EM_SETSEL, SendMessage(hwnd, WM_GETTEXTLENGTH, 0, 0), -1);
	_tcscat(str, _T("\r\n"));
	SendMessage(hwnd, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)str);
}

static int aWaveFormats[] = {
	WAVE_FORMAT_1M08, WAVE_FORMAT_1M16, WAVE_FORMAT_1S08, WAVE_FORMAT_1S16,
	WAVE_FORMAT_2M08, WAVE_FORMAT_2M16, WAVE_FORMAT_2S08, WAVE_FORMAT_2S16,
	WAVE_FORMAT_4M08, WAVE_FORMAT_4M16, WAVE_FORMAT_4S08, WAVE_FORMAT_4S16,
#ifndef _WIN32_WCE
	WAVE_FORMAT_48M08, WAVE_FORMAT_48S08, WAVE_FORMAT_48M16, WAVE_FORMAT_48S16,
	WAVE_FORMAT_96M08, WAVE_FORMAT_96S08, WAVE_FORMAT_96M16, WAVE_FORMAT_96S16
#endif /* _WIN32_WCE */
};

static const TCHAR* GetWaveFormatName(int iFormat)
{
	switch (iFormat) {
	case WAVE_FORMAT_1M08:  return _T("11.025 kHz, mono, 8-bit");
	case WAVE_FORMAT_1M16:  return _T("11.025 kHz, mono, 16-bit");
	case WAVE_FORMAT_1S08:  return _T("11.025 kHz, stereo, 8-bit");
	case WAVE_FORMAT_1S16:  return _T("11.025 kHz, stereo, 16-bit");
	case WAVE_FORMAT_2M08:  return _T("22.05 kHz, mono, 8-bit");
	case WAVE_FORMAT_2M16:  return _T("22.05 kHz, mono, 16-bit");
	case WAVE_FORMAT_2S08:  return _T("22.05 kHz, stereo, 8-bit");
	case WAVE_FORMAT_2S16:  return _T("22.05 kHz, stereo, 16-bit");
	case WAVE_FORMAT_4M08:  return _T("44.1 kHz, mono, 8-bit");
	case WAVE_FORMAT_4M16:  return _T("44.1 kHz, mono, 16-bit");
	case WAVE_FORMAT_4S08:  return _T("44.1 kHz, stereo, 8-bit");
	case WAVE_FORMAT_4S16:  return _T("44.1 kHz, stereo, 16-bit");
#ifndef _WIN32_WCE
	case WAVE_FORMAT_48M08: return _T("48.0 kHz, mono, 8-bit");
	case WAVE_FORMAT_48S08: return _T("48.0 kHz, stereo, 8-bit");
	case WAVE_FORMAT_48M16: return _T("48.0 kHz, mono, 16-bit");
	case WAVE_FORMAT_48S16: return _T("48.0 kHz, stereo, 16-bit");
	case WAVE_FORMAT_96M08: return _T("96.0 kHz, mono, 8-bit");
	case WAVE_FORMAT_96S08: return _T("96.0 kHz, stereo, 8-bit");
	case WAVE_FORMAT_96M16: return _T("96.0 kHz, mono, 16-bit");
	case WAVE_FORMAT_96S16: return _T("96.0 kHz, stereo, 16-bit");
#endif /* _WIN32_WCE */
	default: return _T("Unknown");
	}
}

static int aWaveCaps[] = {
	WAVECAPS_LRVOLUME, WAVECAPS_PITCH, WAVECAPS_PLAYBACKRATE, 
	WAVECAPS_SYNC, WAVECAPS_VOLUME, WAVECAPS_SAMPLEACCURATE
};

static const TCHAR* GetWaveCapsName(int iCaps)
{
	switch (iCaps) {
	case WAVECAPS_LRVOLUME:			return _T("LRVOLUME");		// Supports separate left and right volume control. 
	case WAVECAPS_PITCH:			return _T("PITCH");			// Supports pitch control.
	case WAVECAPS_PLAYBACKRATE:		return _T("PLAYBACKRATE");	// Supports playback rate control. 
	case WAVECAPS_SYNC:				return _T("SYNC");			// The driver is synchronous and will block while playing a buffer. 
	case WAVECAPS_VOLUME:			return _T("VOLUME");		// Supports volume control.
	case WAVECAPS_SAMPLEACCURATE:	return _T("SAMPLEACCURATE");// Returns sample-accurate position information. 
	default: return _T("Unknown");
	}
}

void ExpandSoundInfo(HWND hwnd)
{
	UINT nDevicesIn  = waveInGetNumDevs();
	UINT nDevicesOut = waveOutGetNumDevs();
#ifndef _WIN32_WCE
	UINT nMixers     = mixerGetNumDevs();
#endif /* _WIN32_WCE */
	UINT i;

	for (i = 0; i < nDevicesIn; ++ i)
		ExpandSoundInputInfo(hwnd, i);
	for (i = 0; i < nDevicesOut; ++ i)
		ExpandSoundOutputInfo(hwnd, i);
#ifndef _WIN32_WCE
	for (i = 0; i < nMixers; ++ i)
		ExpandMixerInfo(hwnd, i);
#endif /* _WIN32_WCE */
}

void ExpandSoundInputInfo(HWND hwnd, int iDevice)
{
	WAVEINCAPS	 wc;
	TCHAR        buf[2048];
	int          i;
	MMRESULT	 res = waveInGetDevCaps(iDevice, &wc, sizeof(wc));
	if (res != MMSYSERR_NOERROR)
		return;
	_stprintf(buf, _T(""), iDevice);
	AddString(hwnd, buf);
	_stprintf(buf, _T("Sound Input Device %d"), iDevice);
	AddString(hwnd, buf);
	_stprintf(buf, _T("manufacturer: %d, product: %d, ver: %d.%d, name: %s"), wc.wMid, wc.wPid, 
		(wc.vDriverVersion & 0xff00) >> 8, wc.vDriverVersion & 0xff, wc.szPname);
	AddString(hwnd, buf);

	for (i = 0; i < sizeof(aWaveFormats) / sizeof(int); ++ i) {
		if ((wc.dwFormats & aWaveFormats[i]) == 0)
			continue;
		_tcscpy(buf, GetWaveFormatName(aWaveFormats[i]));
		AddString(hwnd, buf);
	}
}

void ExpandSoundOutputInfo(HWND hwnd, int iDevice)
{
	WAVEOUTCAPS	 wc;
	TCHAR        buf[2048];
	int          i;
	MMRESULT	 res = waveOutGetDevCaps(iDevice, &wc, sizeof(wc));
	if (res != MMSYSERR_NOERROR)
		return;

	_stprintf(buf, _T(""), iDevice);
	AddString(hwnd, buf);
	_stprintf(buf, _T("Sound Output Device %d"), iDevice);
	AddString(hwnd, buf);
	_stprintf(buf, _T("manufacturer: %d, product: %d, ver: %d.%d, name: %s"), wc.wMid, wc.wPid, 
		(wc.vDriverVersion & 0xff00) >> 8, wc.vDriverVersion & 0xff, wc.szPname);
	AddString(hwnd, buf);

	for (i = 0; i < sizeof(aWaveFormats) / sizeof(int); ++ i) {
		if ((wc.dwFormats & aWaveFormats[i]) == 0)
			continue;
		_tcscpy(buf, GetWaveFormatName(aWaveFormats[i]));
		AddString(hwnd, buf);
	}

	for (i = 0; i < sizeof(aWaveCaps) / sizeof(int); ++ i) {
		if ((wc.dwSupport & aWaveCaps[i]) == 0)
			continue;
		_tcscpy(buf, GetWaveCapsName(aWaveCaps[i]));
		AddString(hwnd, buf);
	}
}

#ifndef _WIN32_WCE
void ExpandMixerInfo(HWND hwnd, int iDevice)
{
	MIXERCAPS    mc;
	TCHAR        buf[2048];
	MMRESULT	 res = mixerGetDevCaps(iDevice, &mc, sizeof(mc));
	if (res != MMSYSERR_NOERROR)
		return;
	_stprintf(buf, _T(""), iDevice);
	AddString(hwnd, buf);
	_stprintf(buf, _T("Mixer Device %d"), iDevice);
	AddString(hwnd, buf);
	_stprintf(buf, _T("manufacturer: %d, product: %d, ver: %d.%d, name: %s, destinations: %d"), mc.wMid, mc.wPid, 
		(mc.vDriverVersion & 0xff00) >> 8, mc.vDriverVersion & 0xff, mc.szPname, mc.cDestinations);
	AddString(hwnd, buf);
}
#endif /* _WIN32_WCE */

// structure for the lookup tables. 
typedef struct tagLookupEntry{
	int     Value;
	TCHAR    String[100];
} LookupEntry;

//GetDeviceCaps() codes from WINGDI.H 
#define NINDEX  28
LookupEntry  DevCaps[NINDEX] =
{{  0 , _T("DRIVERVERSION ")},
{  2 , _T("TECHNOLOGY    ")},
{  4 , _T("HORZSIZE      ")},
{  6 , _T("VERTSIZE      ")},
{  8 , _T("HORZRES       ")},
{  10, _T("VERTRES       ")},
{  12, _T("BITSPIXEL     ")},
{  14, _T("PLANES        ")},
{  16, _T("NUMBRUSHES    ")},
{  18, _T("NUMPENS       ")},
{  20, _T("NUMMARKERS    ")},
{  22, _T("NUMFONTS      ")},
{  24, _T("NUMCOLORS     ")},
{  26, _T("PDEVICESIZE   ")},
{  28, _T("CURVECAPS     ")},
{  30, _T("LINECAPS      ")},
{  32, _T("POLYGONALCAPS ")},
{  34, _T("TEXTCAPS      ")},
{  36, _T("CLIPCAPS      ")},
{  38, _T("RASTERCAPS    ")},
{  40, _T("ASPECTX       ")},
{  42, _T("ASPECTY       ")},
{  44, _T("ASPECTXY      ")},
{  88, _T("LOGPIXELSX    ")},
{  90, _T("LOGPIXELSY    ")},
{ 104, _T("SIZEPALETTE   ")},
{ 106, _T("NUMRESERVED   ")},
{ 108, _T("COLORRES      ")}};

/* Device Technologies */
#define NDevice 7
LookupEntry  Device[NDevice] =
{{ 0 , _T("DT_PLOTTER   ")},
{ 1 , _T("DT_RASDISPLAY")},
{ 2 , _T("DT_RASPRINTER")},
{ 3 , _T("DT_RASCAMERA ")},
{ 4 , _T("DT_CHARSTREAM")},
{ 5 , _T("DT_METAFILE  ")},
{ 6 , _T("DT_DISPFILE  ")}};


/* Curve Capabilities */
#define NCurveCaps 9
LookupEntry  CurveCaps[NCurveCaps] =
{{ 0   , _T("CC_NONE      ")},
{ 1   , _T("CC_CIRCLES   ")},
{ 2   , _T("CC_PIE       ")},
{ 4   , _T("CC_CHORD     ")},
{ 8   , _T("CC_ELLIPSES  ")},
{ 16  , _T("CC_WIDE      ")},
{ 32  , _T("CC_STYLED    ")},
{ 64  , _T("CC_WIDESTYLED")},
{ 128 , _T("CC_INTERIORS ")}};


/* Line Capabilities */
#define NLineCaps 8
LookupEntry  LineCaps[NLineCaps] =
{{ 0   , _T("LC_NONE       ")},
{ 2   , _T("LC_POLYLINE   ")},
{ 4   , _T("LC_MARKER     ")},
{ 8   , _T("LC_POLYMARKER ")},
{ 16  , _T("LC_WIDE       ")},
{ 32  , _T("LC_STYLED     ")},
{ 64  , _T("LC_WIDESTYLED ")},
{ 128 , _T("LC_INTERIORS  ")}};


/* Polygonal Capabilities */
#define NPolygonalCaps 10
LookupEntry  PolygonalCaps[NPolygonalCaps] =
{{ 0   , _T("PC_NONE       ")},
{ 1   , _T("PC_POLYGON    ")},
{ 2   , _T("PC_RECTANGLE  ")},
{ 4   , _T("PC_WINDPOLYGON")},
{ 4   , _T("PC_TRAPEZOID  ")},   // HACK   two 4's ???
{ 8   , _T("PC_SCANLINE   ")},
{ 16  , _T("PC_WIDE       ")},
{ 32  , _T("PC_STYLED     ")},
{ 64  , _T("PC_WIDESTYLED ")},
{ 128 , _T("PC_INTERIORS  ")}};


/* Clip Capabilities */
#define NClipCaps 2
LookupEntry  ClipCaps[NClipCaps] =
{{ 0   , _T("CP_NONE     ")},
{ 1   , _T("CP_RECTANGLE")}};


/* Text Capabilities */
#define NTextCaps 16
LookupEntry  TextCaps[NTextCaps] =
{{ 0x0001 , _T("TC_OP_CHARACTER")},
{ 0x0002 , _T("TC_OP_STROKE   ")},
{ 0x0004 , _T("TC_CP_STROKE   ")},
{ 0x0008 , _T("TC_CR_90       ")},
{ 0x0010 , _T("TC_CR_ANY      ")},
{ 0x0020 , _T("TC_SF_X_YINDEP ")},
{ 0x0040 , _T("TC_SA_DOUBLE   ")},
{ 0x0080 , _T("TC_SA_INTEGER  ")},
{ 0x0100 , _T("TC_SA_CONTIN   ")},
{ 0x0200 , _T("TC_EA_DOUBLE   ")},
{ 0x0400 , _T("TC_IA_ABLE     ")},
{ 0x0800 , _T("TC_UA_ABLE     ")},
{ 0x1000 , _T("TC_SO_ABLE     ")},
{ 0x2000 , _T("TC_RA_ABLE     ")},
{ 0x4000 , _T("TC_VA_ABLE     ")},
{ 0x8000 , _T("TC_RESERVED    ")}};


/* Raster Capabilities */
#define NRasterCaps 12
LookupEntry  RasterCaps[NRasterCaps] =
{{  1     , _T("RC_BITBLT       ")},
{  2     , _T("RC_BANDING      ")},
{  4     , _T("RC_SCALING      ")},
{  8     , _T("RC_BITMAP64     ")},
{  0x0010, _T("RC_GDI20_OUTPUT ")},
{  0x0080, _T("RC_DI_BITMAP    ")},
{  0x0100, _T("RC_PALETTE      ")},
{  0x0200, _T("RC_DIBTODEV     ")},
{  0x0400, _T("RC_BIGFONT      ")},
{  0x0800, _T("RC_STRETCHBLT   ")},
{  0x1000, _T("RC_FLOODFILL    ")},
{  0x2000, _T("RC_STRETCHDIB   ")}};

/* GetSysColor() codes from WINUSER.H */
#define NSYSCOLORS 32
LookupEntry  SysColors[NSYSCOLORS] =
    {{ COLOR_SCROLLBAR          , _T("COLOR_SCROLLBAR        \t%06lx")},
     { COLOR_BACKGROUND         , _T("COLOR_BACKGROUND       \t%06lx")},
     { COLOR_ACTIVECAPTION      , _T("COLOR_ACTIVECAPTION    \t%06lx")},
     { COLOR_INACTIVECAPTION    , _T("COLOR_INACTIVECAPTION  \t%06lx")},
     { COLOR_MENU               , _T("COLOR_MENU             \t%06lx")},
     { COLOR_WINDOW             , _T("COLOR_WINDOW           \t%06lx")},
     { COLOR_WINDOWFRAME        , _T("COLOR_WINDOWFRAME      \t%06lx")},
     { COLOR_MENUTEXT           , _T("COLOR_MENUTEXT         \t%06lx")},
     { COLOR_WINDOWTEXT         , _T("COLOR_WINDOWTEXT       \t%06lx")},
     { COLOR_CAPTIONTEXT        , _T("COLOR_CAPTIONTEXT      \t%06lx")},
     { COLOR_ACTIVEBORDER       , _T("COLOR_ACTIVEBORDER     \t%06lx")},
     { COLOR_INACTIVEBORDER     , _T("COLOR_INACTIVEBORDER   \t%06lx")},
     { COLOR_APPWORKSPACE       , _T("COLOR_APPWORKSPACE     \t%06lx")},
     { COLOR_HIGHLIGHT          , _T("COLOR_HIGHLIGHT        \t%06lx")},
     { COLOR_HIGHLIGHTTEXT      , _T("COLOR_HIGHLIGHTTEXT    \t%06lx")},
     { COLOR_BTNFACE            , _T("COLOR_BTNFACE          \t%06lx")},
     { COLOR_BTNSHADOW          , _T("COLOR_BTNSHADOW        \t%06lx")},
     { COLOR_GRAYTEXT           , _T("COLOR_GRAYTEXT         \t%06lx")},
     { COLOR_BTNTEXT            , _T("COLOR_BTNTEXT          \t%06lx")},
     { COLOR_INACTIVECAPTIONTEXT, _T("COLOR_INACTIVECAPTEXT  \t%06lx")},
     { COLOR_BTNHIGHLIGHT       , _T("COLOR_BTNHIGHLIGHT     \t%06lx")},
     {  COLOR_3DDKSHADOW        , _T("COLOR_BTNHIGHLIGHT     \t%06lx")},
     {  COLOR_3DLIGHT           , _T("COLOR_3DLIGHT          \t%06lx")},
     {  COLOR_INFOTEXT          , _T("COLOR_INFOTEXT         \t%06lx")},
     {  COLOR_INFOBK            , _T("COLOR_INFOBK           \t%06lx")},
//     {  COLOR_STATIC            , _T("COLOR_STATIC           \t%06lx")},
//     {  COLOR_STATICTEXT        , _T("COLOR_STATICTEXT       \t%06lx")},
     {  COLOR_DESKTOP           , _T("COLOR_DESKTOP          \t%06lx")},
     {  COLOR_3DFACE            , _T("COLOR_3DFACE           \t%06lx")},
     {  COLOR_3DSHADOW          , _T("COLOR_3DSHADOW         \t%06lx")},
     {  COLOR_3DHIGHLIGHT       , _T("COLOR_3DHIGHLIGHT      \t%06lx")},
     {  COLOR_3DHILIGHT         , _T("COLOR_3DHILIGHT        \t%06lx")}
};


/* GetSystemMetrics() codes from WINUSER.H */
LookupEntry  SystemMetrics[] =
    {{ SM_CXSCREEN         , _T("SM_CXSCREEN      \t%d")},
     { SM_CYSCREEN         , _T("SM_CYSCREEN      \t%d")},
     { SM_CXVSCROLL        , _T("SM_CXVSCROLL     \t%d")},
     { SM_CYHSCROLL        , _T("SM_CYHSCROLL     \t%d")},
     { SM_CYCAPTION        , _T("SM_CYCAPTION     \t%d")},
     { SM_CXBORDER         , _T("SM_CXBORDER      \t%d")},
     { SM_CYBORDER         , _T("SM_CYBORDER      \t%d")},
     { SM_CXDLGFRAME       , _T("SM_CXDLGFRAME    \t%d")},
     { SM_CYDLGFRAME       , _T("SM_CYDLGFRAME    \t%d")},
//     { SM_CYVTHUMB         , _T("SM_CYVTHUMB      \t%d")},
//     { SM_CXHTHUMB         , _T("SM_CXHTHUMB      \t%d")},
     { SM_CXICON           , _T("SM_CXICON        \t%d")},
     { SM_CYICON           , _T("SM_CYICON        \t%d")},
//     { SM_CXCURSOR         , _T("SM_CXCURSOR      \t%d")},
//     { SM_CYCURSOR         , _T("SM_CYCURSOR      \t%d")},
     { SM_CYMENU           , _T("SM_CYMENU        \t%d")},
//     { SM_CXFULLSCREEN     , _T("SM_CXFULLSCREEN  \t%d")},
//     { SM_CYFULLSCREEN     , _T("SM_CYFULLSCREEN  \t%d")},
//     { SM_CYKANJIWINDOW    , _T("SM_CYKANJIWINDOW \t%d")},
     { SM_MOUSEPRESENT     , _T("SM_MOUSEPRESENT  \t%d")},
     { SM_CYVSCROLL        , _T("SM_CYVSCROLL     \t%d")},
     { SM_CXHSCROLL        , _T("SM_CXHSCROLL     \t%d")},
     { SM_DEBUG            , _T("SM_DEBUG         \t%d")},
//     { SM_SWAPBUTTON       , _T("SM_SWAPBUTTON    \t%d")},
//     { SM_RESERVED1        , _T("SM_RESERVED1     \t%d")},
//     { SM_RESERVED2        , _T("SM_RESERVED2     \t%d")},
//     { SM_RESERVED3        , _T("SM_RESERVED3     \t%d")},
//     { SM_RESERVED4        , _T("SM_RESERVED4     \t%d")},
//     { SM_CXMIN            , _T("SM_CXMIN         \t%d")},
//     { SM_CYMIN            , _T("SM_CYMIN         \t%d")},
//     { SM_CXSIZE           , _T("SM_CXSIZE        \t%d")},
//     { SM_CYSIZE           , _T("SM_CYSIZE        \t%d")},
//     { SM_CXFRAME          , _T("SM_CXFRAME       \t%d")},
//     { SM_CYFRAME          , _T("SM_CYFRAME       \t%d")},
//     { SM_CXMINTRACK       , _T("SM_CXMINTRACK    \t%d")},
//     { SM_CYMINTRACK       , _T("SM_CYMINTRACK    \t%d")},
     { SM_CXDOUBLECLK      , _T("SM_CXDOUBLECLK   \t%d")},
     { SM_CYDOUBLECLK      , _T("SM_CYDOUBLECLK   \t%d")},
     { SM_CXICONSPACING    , _T("SM_CXICONSPACING \t%d")},
     { SM_CYICONSPACING    , _T("SM_CYICONSPACING \t%d")},
//     { SM_MENUDROPALIGNMENT, _T("SM_MENUDROPALIGNMENT\t%d")},
//     { SM_PENWINDOWS       , _T("SM_PENWINDOWS    \t%d")},
//     { SM_DBCSENABLED      , _T("SM_DBCSENABLED   \t%d")},
//     { SM_CMOUSEBUTTONS    , _T("SM_CMOUSEBUTTONS \t%d")},

#if(WINVER >= 0x0400)
//     {  SM_SECURE            , _T("SM_SECURE          \t%d")},
     {  SM_CXEDGE            , _T("SM_CXEDGE          \t%d")},
     {  SM_CYEDGE            , _T("SM_CYEDGE          \t%d")},
//     {  SM_CXMINSPACING      , _T("SM_CXMINSPACING    \t%d")},
//     {  SM_CYMINSPACING      , _T("SM_CYMINSPACING    \t%d")},
     {  SM_CXSMICON          , _T("SM_CXSMICON        \t%d")},
     {  SM_CYSMICON          , _T("SM_CYSMICON        \t%d")},
//     {  SM_CYSMCAPTION       , _T("SM_CYSMCAPTION     \t%d")},
//     {  SM_CXSMSIZE          , _T("SM_CXSMSIZE        \t%d")},
//     {  SM_CYSMSIZE          , _T("SM_CYSMSIZE        \t%d")},
//     {  SM_CXMENUSIZE        , _T("SM_CXMENUSIZE      \t%d")},
//     {  SM_CYMENUSIZE        , _T("SM_CYMENUSIZE      \t%d")},
//     {  SM_ARRANGE           , _T("SM_ARRANGE         \t%d")},
//     {  SM_CXMINIMIZED       , _T("SM_CXMINIMIZED     \t%d")},
//     {  SM_CYMINIMIZED       , _T("SM_CYMINIMIZED     \t%d")},
//     {  SM_CXMAXTRACK        , _T("SM_CXMAXTRACK      \t%d")},
//     {  SM_CYMAXTRACK        , _T("SM_CYMAXTRACK      \t%d")},
//     {  SM_CXMAXIMIZED       , _T("SM_CXMAXIMIZED     \t%d")},
//     {  SM_CYMAXIMIZED       , _T("SM_CYMAXIMIZED     \t%d")},
//     {  SM_NETWORK           , _T("SM_NETWORK         \t%d")},
//     {  SM_CLEANBOOT         , _T("SM_CLEANBOOT       \t%d")},
//     {  SM_CXDRAG            , _T("SM_CXDRAG          \t%d")},
//     {  SM_CYDRAG            , _T("SM_CYDRAG          \t%d")},
#endif /* WINVER >= 0x0400 */

//     {  SM_SHOWSOUNDS        , _T("SM_SHOWSOUNDS      \t%d")},

#if(WINVER >= 0x0400)
//     {  SM_CXMENUCHECK       , _T("SM_CXMENUCHECK     \t%d")},
//     {  SM_CYMENUCHECK       , _T("SM_CYMENUCHECK     \t%d")},
//     {  SM_SLOWMACHINE       , _T("SM_SLOWMACHINE     \t%d")},
//     {  SM_MIDEASTENABLED    , _T("SM_MIDEASTENABLED  \t%d")},
#endif /* WINVER >= 0x0400 */
     };

static TCHAR szbufW[100];
static TCHAR space[] = _T("   %s");

//function:  ExpandInfo()
//input parameters:
//    hwnd  - parent of the list box with the info.
//    index - the input value which was sent to GetDeviceCaps().
//    value - the return value from calling GetDeviceCaps().
void ExpandInfo (HWND hwnd, int index, int value){
  int i;
  switch (index) {
	case TECHNOLOGY   :
		for (i = 0 ; i< NDevice ; i++) {
			if (value == Device[i].Value) {
				_stprintf(szbufW,space, Device[i].String);
				AddString(hwnd, szbufW);
			}
		}
    break;
	case CURVECAPS    :
		for (i = 0 ; i< NCurveCaps ; i++) {
			if (value & CurveCaps[i].Value) {
				_stprintf (szbufW,space, CurveCaps[i].String);
				AddString(hwnd, szbufW);
			}
		}
    break;
	case LINECAPS     :
		for (i = 0 ; i< NLineCaps ; i++) {
			if (value & LineCaps[i].Value) {
				_stprintf (szbufW,space, LineCaps[i].String);
				AddString(hwnd, szbufW);
			}
		}
    break;
	case POLYGONALCAPS:
		for (i = 0 ; i< NPolygonalCaps ; i++) {
			if (value & PolygonalCaps[i].Value) {
				_stprintf (szbufW,space, PolygonalCaps[i].String);
				AddString(hwnd, szbufW);
			}
		}
    break;
	case TEXTCAPS     :
		for (i = 0 ; i< NTextCaps ; i++) {
			if (value & TextCaps[i].Value) {
				_stprintf (szbufW,space, TextCaps[i].String);
				AddString(hwnd, szbufW);
			}
		}
    break;
	case CLIPCAPS     :
		for (i = 0 ; i< NClipCaps ; i++) {
			if (value & ClipCaps[i].Value) {
				_stprintf (szbufW,space, ClipCaps[i].String);
				AddString(hwnd, szbufW);
			}
		}
    break;
	case RASTERCAPS   :
		for (i = 0 ; i< NRasterCaps ; i++) {
			if (value & RasterCaps[i].Value) {
				_stprintf (szbufW,space, RasterCaps[i].String);
				AddString(hwnd, szbufW);
			}
		}
    break;
	default:
    break;
  }
}

void doSysColors (HWND hwnd)
{
	int i;
	DWORD answer;

  for (i = 0; i < NSYSCOLORS; i++) {
		answer = GetSysColor (SysColors[i].Value);
    _stprintf (szbufW, SysColors[i].String, (int)answer);
    AddString(hwnd, szbufW);
  }
  return;
}


void doInfo(HWND hwnd)
{
	SYSTEM_INFO  si;
  GetSystemInfo (&si);
  _stprintf (szbufW,_T("wProcessorArchitecture      \t%d"),	(int) si.wProcessorArchitecture    );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wReserved                   \t%d"),	(int) si.wReserved                 );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("dwPageSize                  \t%d"),  (int) si.dwPageSize                 );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("lpMinimumApplicationAddress \t%08lx"), (LONG)si.lpMinimumApplicationAddress );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("lpMaximumApplicationAddress \t%08lx"), (LONG)si.lpMaximumApplicationAddress );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("dwActiveProcessorMask       \t%d"),  (int) si.dwActiveProcessorMask      );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("dwNumberOfProcessors        \t%d"),  (int) si.dwNumberOfProcessors       );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("dwProcessorType             \t%d"),  (int) si.dwProcessorType            );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("dwAllocationGranularity     \t%d"),  (int) si.dwAllocationGranularity    );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wProcessorLevel             \t%d"),  (int) si.wProcessorLevel            );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wProcessorRevision          \t%d"),  (int) si.wProcessorRevision         );
  AddString(hwnd, szbufW);
}

void doMetrics   (HWND hwnd){
	int  i;
	int  answer;
	
  for (i = 0; i<sizeof(SystemMetrics)/sizeof(LookupEntry); i++) {
    answer = GetSystemMetrics (SystemMetrics[i].Value);
    _stprintf (szbufW, SystemMetrics[i].String, (int)answer);
    AddString(hwnd, szbufW);
  }
	
  return;
}

void doPalette(HWND hwnd)
{
	int nEntries;
	HDC hdc;
	int i;
	LPPALETTEENTRY  lpp;
	
  hdc = GetDC (hwnd);
  nEntries = GetSystemPaletteEntries (hdc, 0,0, NULL);

  if (nEntries == 0)
    return;

  lpp = (LPPALETTEENTRY)LocalAlloc (LPTR,
		(DWORD) (nEntries* sizeof (PALETTEENTRY)));
	
  if (lpp == NULL) {
    MessageBox (hwnd,_T("Memory allocation failed."),_T("Warning"),  MB_OK);
    return;
  }
  nEntries = GetSystemPaletteEntries (hdc, 0,nEntries, lpp);
  ReleaseDC (hwnd, hdc);
  for (i = 0; i<nEntries; i++) {
    _stprintf (szbufW,_T("%d)%02x \t%02x \t%02x \t%02x"), i,
			lpp[i].peRed, lpp[i].peGreen, lpp[i].peBlue, lpp[i].peFlags);
    AddString(hwnd, szbufW);
  }
  LocalFree (LocalHandle ((LPSTR)lpp));
  return;
}

void doLocalTime (HWND hwnd)
{
	SYSTEMTIME  st;
  GetLocalTime (&st);
  _stprintf (szbufW,_T("wYear         \t%d"),  (int)st.wYear         );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wMonth        \t%d"),  (int)st.wMonth        );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wDayOfWeek    \t%d"),  (int)st.wDayOfWeek    );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wDay          \t%d"),  (int)st.wDay          );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wHour         \t%d"),  (int)st.wHour         );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wMinute       \t%d"),  (int)st.wMinute       );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wSecond       \t%d"),  (int)st.wSecond       );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wMilliseconds \t%d"),  (int)st.wMilliseconds );
  AddString(hwnd, szbufW);
  return;
}

void doTime (HWND hwnd)
{
	SYSTEMTIME  st;
  GetSystemTime (&st);
  _stprintf (szbufW,_T("wYear         \t%d"),  (int)st.wYear         );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wMonth        \t%d"),  (int)st.wMonth        );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wDayOfWeek    \t%d"),  (int)st.wDayOfWeek    );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wDay          \t%d"),  (int)st.wDay          );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wHour         \t%d"),  (int)st.wHour         );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wMinute       \t%d"),  (int)st.wMinute       );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wSecond       \t%d"),  (int)st.wSecond       );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("wMilliseconds \t%d"),  (int)st.wMilliseconds );
  AddString(hwnd, szbufW);
}

void doVersionEx (HWND hwnd)
{
	OSVERSIONINFO osvi;
  osvi.dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
  GetVersionEx (&osvi);
	
  _stprintf (szbufW,_T("dwOSVersionInfoSize \t%d"),  (int)osvi.dwOSVersionInfoSize );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("dwMajorVersion      \t%d"),  (int)osvi.dwMajorVersion );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("dwMinorVersion      \t%d"),  (int)osvi.dwMinorVersion );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("dwBuildNumber       \t%d"),  (int)osvi.dwBuildNumber );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("dwPlatformId        \t%d"),  (int)osvi.dwPlatformId );
  AddString(hwnd, szbufW);
  _stprintf (szbufW,_T("szCSDVersion:       \t%s"),  (LPTSTR) osvi.szCSDVersion );
  AddString(hwnd, szbufW);
	
#ifdef GETVERSION
  _stprintf (szbufW,_T("GetVersion(): \t0x%08x"),  GetVersion () );
  AddString(hwnd, szbufW);
#endif
	
  return;
}

void ExpandSysInfo(HWND hwnd)
{
	doVersionEx(hwnd);
	doLocalTime(hwnd); 
	doTime(hwnd); 
	doInfo(hwnd); 
	doMetrics(hwnd); 
	doSysColors (hwnd); 
	doPalette(hwnd); 
}

void ExpandDeviceCaps(HWND hwnd)
{
	HDC hdc = GetDC(hwnd);
	int i;
	for (i = 0; i < NINDEX ; i++) {
		int value = GetDeviceCaps (hdc, DevCaps[i].Value);
		wsprintf (szbufW,_T("%s \t %8d"), DevCaps[i].String, value);
		AddString(hwnd, szbufW);
		//Expand information returned from GetDeviceCaps if required
		ExpandInfo(hwnd, DevCaps[i].Value, value);
	}
	ReleaseDC(hwnd,hdc);
}

static BOOL devinfo_write_to_file(HWND hWnd, TCHAR *szPath)
{
	HANDLE hFile	= 0;
	int    nLength	= GetWindowTextLength(hWnd);
	UCHAR *pText;
	DWORD  nWritten;

	if (! GetPocketDigiDir(szPath))
		return FALSE;

	_tcscat(szPath, _T("\\devinfo.txt"));
	hFile = CreateFile(szPath, GENERIC_WRITE, FILE_SHARE_READ, 0,
		CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0);
	if (hFile == 0)
		return FALSE;

	pText = (UCHAR*)malloc(sizeof(TCHAR*) * nLength);
	GetWindowText(hWnd, (TCHAR*)pText, nLength);
#ifdef _UNICODE
	{ 
		int i;
		for (i = 1; i < nLength; ++ i)
			pText[i] = pText[i << 1];
	}
#endif
	WriteFile(hFile, (void*)pText, nLength, &nWritten, 0);
	CloseHandle(hFile);
	free(pText);
	return TRUE;
}
