#include "dbgwaterfall.h"
#include "ColorTable.h"
#include <memory.h>
#include <stdio.h>
#include <tchar.h>

#include "../Resource.h"

#define	RULER_HEIGHT	14

/// Forward reference to the waterfall window procedure
LRESULT CALLBACK dbgwaterfall_wndproc(HWND, UINT, WPARAM, LPARAM);

const TBYTE WIN32CLASS_DBGWATERFALL[] = _T("DBGWATERFALL");

#define DBGWATERFALL_FFTLEN_MAX (1024 * 8)
#define DBGWATERFALL_DIRECTION TRUE

/// Register class of the waterfall window
ATOM dbgwaterfall_register_class(HINSTANCE hInstance)
{
	WNDCLASS	wc;

	wc.style			= CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc		= (WNDPROC)dbgwaterfall_wndproc;
	wc.cbClsExtra		= 0;
	wc.cbWndExtra		= 0;
	wc.hInstance		= hInstance;
	wc.hIcon			= 0;
	wc.hCursor			= 0;
	wc.hbrBackground	= (HBRUSH) GetStockObject(WHITE_BRUSH);
	wc.lpszMenuName		= 0;
	wc.lpszClassName	= WIN32CLASS_DBGWATERFALL;

	return RegisterClass(&wc);
}

DbgWaterfall* dbgwaterfall_new()
{
	DbgWaterfall *wf = (DbgWaterfall*)malloc(sizeof(DbgWaterfall));
	if (wf == 0)
		return 0;
	if (! dbgwaterfall_init(wf)) {
		dbgwaterfall_destroy(wf);
		return 0;
	}
	return wf;
}

void dbgwaterfall_free(DbgWaterfall *wf)
{
	if (wf == 0)
		return;
	dbgwaterfall_destroy(wf);
	free(wf);
}

static BOOL alloc_pixbuf(DbgWaterfall *wf, BOOL restart)
{
	RECT	rect;
	int		nWidth, nHeight;
	int		nSize;

	if (! wf->hWnd)
		return FALSE;

	GetClientRect(wf->hWnd, &rect);
	nWidth  = rect.right - rect.left;
	nHeight = rect.bottom - rect.top - RULER_HEIGHT;
	nSize   = nHeight * wf->fftlen;

	if (restart) {
		if (nSize != wf->pixbufsize)
			wf->pixbuf = (COLORREF*)realloc((void*)wf->pixbuf, nSize * sizeof(COLORREF));
		wf->pixbufptr  = 0;
		wf->pixbufsize = nSize;
		if (wf->pixbuf == 0)
			return FALSE;
		memset(wf->pixbuf, 0, nSize * sizeof(COLORREF));
	} else if (nSize != wf->pixbufsize) {
		COLORREF *pBuf = (COLORREF*)malloc(nSize * sizeof(COLORREF));
		int       nPtr = 0;
		if (pBuf == 0)
			return FALSE;
		if (wf->pixbufsize - wf->pixbufptr >= nSize) {
			memcpy(pBuf, wf->pixbuf + wf->pixbufptr, nSize * sizeof(COLORREF));
		} else {
			nPtr = wf->pixbufsize - wf->pixbufptr;
			memcpy(pBuf, wf->pixbuf + wf->pixbufptr, nPtr * sizeof(COLORREF));
			if (nSize - nPtr > wf->pixbufptr) {
				memcpy(pBuf + nPtr, wf->pixbuf, wf->pixbufptr * sizeof(COLORREF));
				nPtr += wf->pixbufptr;
			} else {
				memcpy(pBuf + nPtr, (const void*)wf->pixbuf, (nSize - nPtr) * sizeof(COLORREF));
				nPtr = 0;
			}
		}
		free(wf->pixbuf);
		wf->pixbuf	   = pBuf;
		wf->pixbufptr  = nPtr;
		wf->pixbufsize = nSize;
	}

	{
		BITMAPINFO  bmi;
		DeleteObject(wf->hBitmap);
		memset(&bmi, 0, sizeof(BITMAPINFO));
		nWidth = ((nWidth + 3) >> 2) << 2;
		//FIXME
		nHeight = 1;
		bmi.bmiHeader.biSize		= sizeof(bmi.bmiHeader);
		bmi.bmiHeader.biWidth		= nWidth;
		bmi.bmiHeader.biHeight		= nHeight;
		bmi.bmiHeader.biSizeImage   = nWidth * nHeight * 3; 
		bmi.bmiHeader.biPlanes		= 1;
		bmi.bmiHeader.biBitCount	= 24;
		bmi.bmiHeader.biCompression = BI_RGB;
		bmi.bmiHeader.biXPelsPerMeter = 3780;
		bmi.bmiHeader.biYPelsPerMeter = 3780;
		{
			HDC hDC = GetDC(NULL);
			wf->hBitmap = CreateDIBSection(hDC, &bmi, DIB_RGB_COLORS, (void**)&wf->pBitmapBits, NULL, 0);
			//FIXME
			memset(wf->pBitmapBits, 128, bmi.bmiHeader.biSizeImage);
			ReleaseDC(NULL, hDC);
		}
	}
	return TRUE;
}

BOOL dbgwaterfall_init(DbgWaterfall *wf)
{
	int i;

	memset(wf, 0, sizeof(DbgWaterfall));

	if (FALSE) {
		wf->nColorTableLen = 91;
		wf->aColorTable = malloc(wf->nColorTableLen * sizeof(COLORREF));
		CreatePowerColorGradient16bit(wf->aColorTable, wf->nColorTableLen);
	} else {
		wf->nColorTableLen = 6 * 31 + 1;
		wf->aColorTable = malloc(wf->nColorTableLen * sizeof(COLORREF));
		CreatePowerColorGradientSimple(5, wf->aColorTable, wf->nColorTableLen);
	}

//	wf->fftlen = 1024 * 2;
//	wf->fftlen = 1024 / 2;
	wf->fftlen = 1024 / 4;
//	wf->fftlen = 1024 / 8;
//	wf->fftlen = 1024 * 2 / 16;
	alloc_pixbuf(wf, TRUE);

	wf->inbuf = malloc(DBGWATERFALL_FFTLEN_MAX * sizeof(cmplx));
	memset(wf->inbuf, 0, DBGWATERFALL_FFTLEN_MAX * sizeof(cmplx));
	wf->inptr = 0;

	wf->fftcfg = kiss_fft_alloc(wf->fftlen, 0, 0, 0);

	wf->window = (float*)malloc(sizeof(float) * wf->fftlen);
	for (i = 0; i < (wf->fftlen >> 1); ++ i) {
		wf->window[i] = (float)(0.54 - 0.46 * cos(M_PI * (double)i / (double)((wf->fftlen >> 1) - 1)));
		wf->window[wf->fftlen - i - 1] = wf->window[i];
	}

	return TRUE;
}

BOOL dbgwaterfall_create(DbgWaterfall *wf, DWORD dwStyle,	
	int x, int y, int nWidth, int nHeight, HWND hWndParent, HANDLE hInstance)
{
	wf->hWnd = CreateWindow(WIN32CLASS_DBGWATERFALL, NULL, dwStyle,
		x, y, nWidth, nHeight, hWndParent, NULL, hInstance, (void*)wf);
	if (! wf->hWnd)
		return FALSE;
	SetWindowLong(wf->hWnd, GWL_USERDATA, (long)wf);
	{
		LOGFONT logfont;
		memset(&logfont, 0, sizeof(LOGFONT));
		logfont.lfHeight  = -9;
		logfont.lfWeight  = FW_NORMAL;
		logfont.lfCharSet = ANSI_CHARSET;
		_tcscpy(logfont.lfFaceName, _T("Tahoma"));
		wf->hFont = CreateFontIndirect(&logfont);
	}
	return alloc_pixbuf(wf, FALSE);
}

void dbgwaterfall_destroy(DbgWaterfall *wf)
{
	DeleteObject(wf->hBitmap);
	DeleteObject(wf->hFont);
	DestroyWindow(wf->hWnd);
	wf->hWnd		= NULL;

	free(wf->aColorTable);
	wf->aColorTable		= 0;
	wf->nColorTableLen	= 0;

	free(wf->pixbuf);
	wf->pixbuf		= NULL;
	wf->pixbufptr	= 0;
	wf->pixbufsize	= 0;

	free(wf->inbuf);
	wf->inbuf		= NULL;

	kiss_fft_free(wf->fftcfg);
	free(wf->window);
	wf->window = 0;

//	kiss_fftr_free(wf->kiss_fftr_state);
//	wf->kiss_fftr_state = NULL;
}

static void draw_ruler(DbgWaterfall *wf, HDC hDC)
{
	RECT	rect;
	int		iResolution, iMajorInc, iMinorInc, iFreq, iMin, iMax;
	int		iTickPosMajorTop, iTickPosMinorTop, iMarkerPosTop, iPosTickBottom;
	TCHAR	label[8];
	HGDIOBJ hOldPen, hOldFont;
	POINT	aPoints[2];

	switch (wf->fftlen) {
		case 1024:
			iMajorInc   = 500;
			iMinorInc   = 100;
			iResolution = 8389;
			break;
		case 2 * 1024:
			iMajorInc   = 200;
			iMinorInc   = 20;
			iResolution = 16777;
			break;
		case 4 * 1024:
			iMajorInc   = 100;
			iMinorInc   = 20;
			iResolution = 33554;
			break;
		case 8 * 1024:
			iMajorInc   = 50;
			iMinorInc   = 10;
			iResolution = 67109;
			break;
		default:
			iMajorInc   = 500;
			iMinorInc   = 100;
			iResolution = 8389;
			break;
	}

	GetClientRect(wf->hWnd, &rect);
	iMin = 0;
	iMax = (((rect.right - rect.left) << 16) - (1 << 15)) / iResolution;

	if (DBGWATERFALL_DIRECTION) {
		int  iHeight     = rect.bottom - rect.top - RULER_HEIGHT;
		iPosTickBottom   = iHeight;
		iTickPosMinorTop = iHeight + 2;
		iTickPosMajorTop = iHeight + 5;
		iMarkerPosTop    = iHeight + 7;
		rect.top = rect.bottom - RULER_HEIGHT;
	} else {
		iPosTickBottom   = RULER_HEIGHT - 1;
		iTickPosMinorTop = RULER_HEIGHT - 3;
		iTickPosMajorTop = RULER_HEIGHT - 6;
		iMarkerPosTop    = RULER_HEIGHT - 8;
		rect.bottom      = rect.top + RULER_HEIGHT;
	}
	FillRect(hDC, &rect, GetStockObject(BLACK_BRUSH));

	hOldPen  = SelectObject(hDC, GetStockObject(WHITE_PEN));
	hOldFont = SelectObject(hDC, wf->hFont);
	SetTextColor(hDC, RGB(255, 255, 255));
	SetBkColor(hDC, RGB(0, 0, 0));
	iFreq = 0;
	aPoints[0].y = iPosTickBottom;
	aPoints[1].y = iTickPosMinorTop;
	while (iFreq < iMax) {
		if (iFreq >= iMin) {
			int iPos       = ((iFreq * iResolution + (1 << 15)) >> 16);
			aPoints[0].x   = iPos;
			aPoints[1].x   = iPos;
			Polyline(hDC, aPoints, 2);
		}
		iFreq += iMinorInc;
	}

	iFreq = 0;
	aPoints[1].y = iTickPosMajorTop;
	while (iFreq < iMax) {
		if (iFreq >= iMin) {
			int iPos       = ((iFreq * iResolution + (1 << 15)) >> 16);
			aPoints[0].x   = iPos;
			aPoints[1].x   = iPos;
			Polyline(hDC, aPoints, 2);
			_stprintf(label, _T("%d"), iFreq);
			ExtTextOut(hDC, iPos, iTickPosMajorTop, 0, NULL, label, _tcslen(label), NULL);
		}
		iFreq += iMajorInc;
	}

	SelectObject(hDC, hOldPen);
	SelectObject(hDC, hOldFont);
}

static void draw_waterfall_line(DbgWaterfall *wf, HDC hDC, int iRow, COLORREF *pData, int nLength)
{
	int    i;
	UCHAR *pPixels = (UCHAR*)wf->pBitmapBits;
	for (i = 0; i < nLength; ++ i) {
		COLORREF  clr  = pData[i];
		UCHAR    *pClr = (UCHAR*)(&clr) + 2;
		*pPixels++ = *pClr--;
		*pPixels++ = *pClr--;
		*pPixels++ = *pClr;
	}
	{
		HDC      hBmpDC  = CreateCompatibleDC(hDC);
		HGDIOBJ *pObjOld = SelectObject(hBmpDC, wf->hBitmap);
		BitBlt(hDC, 0, iRow, nLength, 1, hBmpDC, 0, 0, SRCCOPY);
		SelectObject(hBmpDC, pObjOld);
		DeleteDC(hBmpDC);
	}
}

static void draw(DbgWaterfall *wf, HDC hDC, BOOL bRefresh)
{
	RECT rect;
	int  nWidth, nHeight, nWidthBuf, iPos, i;
	GetClientRect(wf->hWnd, &rect);
	nWidth		= rect.right  - rect.left;
	nHeight		= rect.bottom - rect.top - RULER_HEIGHT;
	nWidthBuf	= wf->fftlen;
	iPos		= wf->pixbufptr - (bRefresh ? nHeight : 1) * nWidthBuf;

	if (iPos < 0)
		iPos += wf->pixbufsize;

	if (nWidth > nWidthBuf)
		nWidth = nWidthBuf;

	if (bRefresh) {
		draw_ruler(wf, hDC);
	} else {
		RECT rect;
		rect.left   = 0;
		rect.right  = nWidth;
		rect.top    = 0;
		rect.bottom = nHeight;
		if (DBGWATERFALL_DIRECTION) {
			ScrollWindowEx(wf->hWnd, 0, -1, &rect, 0, 0, 0, 0);
		} else {
			rect.top    += RULER_HEIGHT;
			rect.bottom += RULER_HEIGHT;
			ScrollWindowEx(wf->hWnd, 0,  1, &rect, 0, 0, 0, 0);
		}
	}

	for (i = (bRefresh ? 0 : (nHeight - 1)); i < nHeight; ++ i) {
		draw_waterfall_line(wf, hDC,
			DBGWATERFALL_DIRECTION ? i : (nHeight - i - RULER_HEIGHT),
			wf->pixbuf + iPos, nWidth);
		iPos += nWidthBuf;
		if (iPos == wf->pixbufsize)
			iPos = 0;
	}
}

static BOOL dbgwaterfall_set_data2(DbgWaterfall *wf)
{
	int			  nFFTLen   = wf->fftlen;
	kiss_fft_cpx *aIn		= (kiss_fft_cpx*)malloc(nFFTLen * sizeof(kiss_fft_cpx));
	kiss_fft_cpx *aOut		= (kiss_fft_cpx*)malloc(nFFTLen * sizeof(kiss_fft_cpx));
	COLORREF	 *aLoud		= (COLORREF*)malloc(nFFTLen * sizeof(COLORREF));
	int			  i;

	for (i = 0; i < nFFTLen; ++ i) {
		aIn[i].r = wf->window[i] * ((float)wf->inbuf[i].re) / 32768.0f;
		aIn[i].i = wf->window[i] * ((float)wf->inbuf[i].im) / 32768.0f;
	}

//	window(aReal, nFFTLen);
	kiss_fft(wf->fftcfg, aIn, aOut);
//	fix_loud(aLoud, aReal, aImag, nWidth, 0);

	for (i = 0; i < nFFTLen; ++ i) {
		double loud = log(aOut[i].r * aOut[i].r + aOut[i].i * aOut[i].i);
		int color = loud * wf->nColorTableLen;
		if (color >= wf->nColorTableLen)
			color = wf->nColorTableLen - 1;
		else if (color < 0)
			color = 0;
		aLoud[i] = wf->aColorTable[color];
	}

	if (wf->pixbufptr + nFFTLen <= wf->pixbufsize) {
		COLORREF *ptr = wf->pixbuf + wf->pixbufptr;
		for (i = 0; i < nFFTLen; ++ i)
			*ptr++ = aLoud[i];
		wf->pixbufptr += nFFTLen;
		if (wf->pixbufptr == wf->pixbufsize)
			wf->pixbufptr = 0;
	} else {
		COLORREF *ptr  = wf->pixbuf + wf->pixbufptr;
		int	      nLen = wf->pixbufsize - wf->pixbufptr;
		for (i = 0; i < nLen; ++ i)
			*ptr++ = aLoud[i];
		ptr = 0;
		for (; i < nFFTLen; ++ i)
			*ptr++ = aLoud[i];
		wf->pixbufptr = nFFTLen - nLen;
	}

	free(aIn);
	free(aOut);
	free(aLoud);
	return TRUE;
}

void dbgwaterfall_set_data(DbgWaterfall *wf, cmplx *data, int len)
{
	int i;
	int overlap = wf->fftlen >> 3;
	while (len-- > 0) {
		wf->inbuf[wf->inptr ++] = *data++;
		if (wf->inptr >= wf->fftlen) {
			if (wf->pixbuf != NULL && dbgwaterfall_set_data2(wf)) {
				HDC hDC = GetDC(wf->hWnd);
				draw(wf, hDC, FALSE);
				ReleaseDC(wf->hWnd, hDC);
			}
			for (i = 0; i < wf->inptr - overlap; i++)
				wf->inbuf[i] = wf->inbuf[i + overlap];
			wf->inptr -= overlap;
		}
	}
}

void dbgwaterfall_set_data_raw(DbgWaterfall *wf, cmplx *aIn, int len)
{
	int			  nFFTLen   = wf->fftlen;
	COLORREF	 *aLoud		= (COLORREF*)malloc(nFFTLen * sizeof(COLORREF));
	int			  i;

	for (i = 0; i < nFFTLen; ++ i) {
		double loud;
		int    color;
		if (i >= len) {
			aLoud[i] = 0;
			continue;
		}
		loud = log(aIn[i].re * aIn[i].re + aIn[i].im * aIn[i].im);
		color = loud * wf->nColorTableLen * 0.02;
		if (color >= wf->nColorTableLen)
			color = wf->nColorTableLen - 1;
		else if (color < 0)
			color = 0;
		aLoud[i] = wf->aColorTable[color];
	}

	if (wf->pixbufptr + nFFTLen <= wf->pixbufsize) {
		COLORREF *ptr = wf->pixbuf + wf->pixbufptr;
		for (i = 0; i < nFFTLen; ++ i)
			*ptr++ = aLoud[i];
		wf->pixbufptr += nFFTLen;
		if (wf->pixbufptr == wf->pixbufsize)
			wf->pixbufptr = 0;
	} else {
		COLORREF *ptr  = wf->pixbuf + wf->pixbufptr;
		int	      nLen = wf->pixbufsize - wf->pixbufptr;
		for (i = 0; i < nLen; ++ i)
			*ptr++ = aLoud[i];
		ptr = 0;
		for (; i < nFFTLen; ++ i)
			*ptr++ = aLoud[i];
		wf->pixbufptr = nFFTLen - nLen;
	}

	free(aLoud);

	{
		HDC hDC = GetDC(wf->hWnd);
		draw(wf, hDC, FALSE);
		ReleaseDC(wf->hWnd, hDC);
	}
	return TRUE;
}

LRESULT CALLBACK dbgwaterfall_wndproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	DbgWaterfall *wf = (DbgWaterfall*)GetWindowLong(hWnd, GWL_USERDATA);
	if (message == WM_PAINT) {
		PAINTSTRUCT paint;
		draw(wf, BeginPaint(hWnd, &paint), TRUE);
		EndPaint(hWnd, &paint);
	} else if (message == WM_ERASEBKGND) {
		// ignore background erasing
		return TRUE;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}
