/*
 *    cw.c  --  morse code modem
 *
 *    Copyright (C) 2004
 *      Lawrence Glaister (ve7it@shaw.ca)
 *    This modem borrowed heavily from other gmfsk modems and
 *    also from the unix-cw project. I would like to thank those
 *    authors for enriching my coding experience by providing
 *    and supporting open source.
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "../main/trx.h"
#include "../misc/filter.h"
#include "../main/sndmodem.h"
#include "../main/trxctrl.h"

#include "cw.h"
#include "morse.h"

static void cw_txinit(Trx *trx)
{
	struct cw *c = (struct cw *) trx->modem;
	c->phaseacc   = 0;
	c->nTxLastKey = 0;
	c->nTxBufPos  = 0;
	c->nTxBufLen  = 0;

	if (g_TrxCtrl.nType == TRXCTRL_ATS3_BELL202 || g_TrxCtrl.nType == TRXCTRL_ATS3_MANCHESTER)
		trxctrl_ats3_start_bpsk(&g_TrxCtrl, trx->pModem, trx->nTxFrequency);
}

static void cw_free(struct cw *s)
{
	if (s) {
		filter_free(s->fir1);
		filter_free(s->fir2);
		filter_free(s->fir3);
		free(s);
	}
}

/* 64-tap low pass FIR */
/*
short cw_firc[64] = {
	35,
	45,
	52,
	52,
	44,
	24,
	-10,
	-58,
	-120,
	-195,
	-278,
	-362,
	-440,
	-503,
	-539,
	-538,
	-488,
	-381,
	-208,
	34,
	347,
	727,
	1166,
	1652,
	2170,
	2700,
	3221,
	3711,
	4149,
	4512,
	4786,
	4955,
	5013,
	4955,
	4786,
	4512,
	4149,
	3711,
	3221,
	2700,
	2170,
	1652,
	1166,
	727,
	347,
	34,
	-208,
	-381,
	-488,
	-538,
	-539,
	-503,
	-440,
	-362,
	-278,
	-195,
	-120,
	-58,
	-10,
	24,
	44,
	52,
	52,
	45
};
*/

/*
float cw_firc[64] = {
	5.36E-04f,
	6.88E-04f,
	7.89E-04f,
	7.97E-04f,
	6.69E-04f,
	3.64E-04f,
	-1.49E-04f,
	-8.85E-04f,
	-1.84E-03f,
	-2.98E-03f,
	-4.24E-03f,
	-5.52E-03f,
	-6.72E-03f,
	-7.68E-03f,
	-8.23E-03f,
	-8.21E-03f,
	-7.45E-03f,
	-5.81E-03f,
	-3.18E-03f,
	5.21E-04f,
	5.29E-03f,
	1.11E-02f,
	1.78E-02f,
	2.52E-02f,
	3.31E-02f,
	4.12E-02f,
	4.92E-02f,
	5.66E-02f,
	6.33E-02f,
	6.89E-02f,
	7.30E-02f,
	7.56E-02f,
	7.65E-02f,
	7.56E-02f,
	7.30E-02f,
	6.89E-02f,
	6.33E-02f,
	5.66E-02f,
	4.92E-02f,
	4.12E-02f,
	3.31E-02f,
	2.52E-02f,
	1.78E-02f,
	1.11E-02f,
	5.29E-03f,
	5.21E-04f,
	-3.18E-03f,
	-5.81E-03f,
	-7.45E-03f,
	-8.21E-03f,
	-8.23E-03f,
	-7.68E-03f,
	-6.72E-03f,
	-5.52E-03f,
	-4.24E-03f,
	-2.98E-03f,
	-1.84E-03f,
	-8.85E-04f,
	-1.49E-04f,
	3.64E-04f,
	6.69E-04f,
	7.97E-04f,
	7.89E-04f,
	6.88E-04f
};
*/

/* 300Hz */
/*
float cw_firc[64] = {
	6.877881e-04f,	//H[0]=H[63]
	-1.252935e-04f,	//H[1]=H[62]
	2.814168e-04f,	//H[2]=H[61]
	8.687654e-04f,	//H[3]=H[60]
	1.463702e-03f,	//H[4]=H[59]
	1.803412e-03f,	//H[5]=H[58]
	1.612692e-03f,	//H[6]=H[57]
	7.147544e-04f,	//H[7]=H[56]
	-8.589447e-04f,	//H[8]=H[55]
	-2.803745e-03f,	//H[9]=H[54]
	-4.560505e-03f,	//H[10]=H[53]
	-5.435775e-03f,	//H[11]=H[52]
	-4.808364e-03f,	//H[12]=H[51]
	-2.372128e-03f,	//H[13]=H[50]
	1.660624e-03f,	//H[14]=H[49]
	6.475443e-03f,	//H[15]=H[48]
	1.075044e-02f,	//H[16]=H[47]
	1.294233e-02f,	//H[17]=H[46]
	1.171769e-02f,	//H[18]=H[45]
	6.428127e-03f,	//H[19]=H[44]
	-2.505556e-03f,	//H[20]=H[43]
	-1.343206e-02f,	//H[21]=H[42]
	-2.362028e-02f,	//H[22]=H[41]
	-2.973490e-02f,	//H[23]=H[40]
	-2.855979e-02f,	//H[24]=H[39]
	-1.781238e-02f,	//H[25]=H[38]
	3.151983e-03f,	//H[26]=H[37]
	3.293988e-02f,	//H[27]=H[36]
	6.815725e-02f,	//H[28]=H[35]
	1.039158e-01f,	//H[29]=H[34]
	1.347220e-01f,	//H[30]=H[33]
	1.555605e-01f,	//H[31]=H[32]
	1.629258e-01f,	//H[32]=H[31]
	1.555605e-01f,	//H[33]=H[30]
	1.347220e-01f,	//H[34]=H[29]
	1.039158e-01f,	//H[35]=H[28]
	6.815725e-02f,	//H[36]=H[27]
	3.293988e-02f,	//H[37]=H[26]
	3.151983e-03f,	//H[38]=H[25]
	-1.781238e-02f,	//H[39]=H[24]
	-2.855979e-02f,	//H[40]=H[23]
	-2.973490e-02f,	//H[41]=H[22]
	-2.362028e-02f,	//H[42]=H[21]
	-1.343206e-02f,	//H[43]=H[20]
	-2.505556e-03f,	//H[44]=H[19]
	6.428127e-03f,	//H[45]=H[18]
	1.171769e-02f,	//H[46]=H[17]
	1.294233e-02f,	//H[47]=H[16]
	1.075044e-02f,	//H[48]=H[15]
	6.475443e-03f,	//H[49]=H[14]
	1.660624e-03f,	//H[50]=H[13]
	-2.372128e-03f,	//H[51]=H[12]
	-4.808364e-03f,	//H[52]=H[11]
	-5.435775e-03f,	//H[53]=H[10]
	-4.560505e-03f,	//H[54]=H[9]
	-2.803745e-03f,	//H[55]=H[8]
	-8.589447e-04f,	//H[56]=H[7]
	7.147544e-04f,	//H[57]=H[6]
	1.612692e-03f,	//H[58]=H[5]
	1.803412e-03f,	//H[59]=H[4]
	1.463702e-03f,	//H[60]=H[3]
	8.687654e-04f,	//H[61]=H[2]
	2.814168e-04f,	//H[62]=H[1]
	-1.252935e-04f,	//H[63]=H[0]
};
*/

// 64 taps, 31.25
float cw_fir1c[64] = {
	-0.00044789f,
	-0.00063167f,
	-0.00100256f,
	-0.00156531f,
	-0.00232266f,
	-0.00327063f,
	-0.00439342f,
	-0.00565850f,
	-0.00701253f,
	-0.00837907f,
	-0.00965823f,
	-0.01072878f,
	-0.01145278f,
	-0.01168246f,
	-0.01126911f,
	-0.01007335f,
	-0.00797606f,
	-0.00488924f,
	-0.00076591f,
	0.00439171f,
	0.01052660f,
	0.01752432f,
	0.02521404f,
	0.03337329f,
	0.04173643f,
	0.05000649f,
	0.05786981f,
	0.06501264f,
	0.07113857f,
	0.07598566f,
	0.07934199f,
	0.08105860f,
	0.08105860f,
	0.07934199f,
	0.07598566f,
	0.07113857f,
	0.06501264f,
	0.05786981f,
	0.05000649f,
	0.04173643f,
	0.03337329f,
	0.02521404f,
	0.01752432f,
	0.01052660f,
	0.00439171f,
	-0.00076591f,
	-0.00488924f,
	-0.00797606f,
	-0.01007335f,
	-0.01126911f,
	-0.01168246f,
	-0.01145278f,
	-0.01072878f,
	-0.00965823f,
	-0.00837907f,
	-0.00701253f,
	-0.00565850f,
	-0.00439342f,
	-0.00327063f,
	-0.00232266f,
	-0.00156531f,
	-0.00100256f,
	-0.00063167f,
	-0.00044789f,
};

static void cw_rxinit(Trx *trx)
{
	struct cw *c = (struct cw *) trx->modem;
	c->cw_receive_state = RS_IDLE;	// set us waiting for a tone
	c->s_ctr = 0;			// reset audio sample counter
	c->cw_rr_current = 0;		// reset decoding pointer
	c->agc_peak = 0;		// reset agc
//	c->dec_ctr = 0;			// reset decimation counter
}

static void cw_destructor(Trx *trx)
{
	struct cw *s = (struct cw *) trx->modem;

	cw_free(s);

	trx->modem = NULL;
	trx->txinit = NULL;
	trx->rxinit = NULL;
	trx->txprocess = NULL;
	trx->rxprocess = NULL;
	trx->destructor = NULL;
}

void cw_init(Trx *trx)
{
	struct cw *s;
	int i;

	s = (struct cw*)malloc(sizeof(struct cw));
	memset(s, 0, sizeof(struct cw));

	s->cw_send_speed	= trx->cw_txspeed;
	s->cw_receive_speed = 60; // will be decreased by cw_set_bandwidth
	s->cw_noise_spike_threshold = INITIAL_NOISE_THRESHOLD;
	s->cw_adaptive_receive_threshold = 2 * DOT_MAGIC / trx->cw_rxspeed;
//	s->cw_gap = INITIAL_GAP;	/* Initially no 'Farnsworth' gap */  

	memset(s->cw_receive_representation_buffer, 0,
	       sizeof(s->cw_receive_representation_buffer));

	if (cw_tables_init() == FALSE) {
//		g_warning("cw_tables_init failed\n");
		cw_free(s);
		return;
	}

	// setup function pointers for cw processes
	trx->modem = s;
	trx->txinit = cw_txinit;
	trx->rxinit = cw_rxinit;
	trx->txprocess = cw_txprocess;
	trx->rxprocess = cw_rxprocess;
	trx->destructor = cw_destructor;

//	s->fir1 = filter_init_lowpass(64, DEC_RATIO, 0.5f * 330.0f / (float)SampleRate);
	s->fir1 = filter_init_float(65, DEC_RATIO, cw_fir1c, cw_fir1c);
	cw_set_bandwidth(trx);

	// block of variables that get up dated each time speed changes
	s->cw_in_sync = FALSE;	/* Synchronization flag */
	cw_sync_parameters(s);

	// init the cw rx tracking arrays, sums and indexes to intial rx speed
	s->cw_dt_dot_tracking_sum = AVERAGE_ARRAY_LENGTH * s->cw_send_dot_length;
	s->cw_dt_dash_tracking_sum = AVERAGE_ARRAY_LENGTH * s->cw_send_dash_length;
	for (i = 0; i < AVERAGE_ARRAY_LENGTH; i++) {
		s->cw_dot_tracking_array[i] = s->cw_send_dot_length;
		s->cw_dash_tracking_array[i] = s->cw_send_dash_length;
	}
	s->cw_dt_dot_index = 0;
	s->cw_dt_dash_index = 0;
}

/**
 * cw_sync_parameters()
 *
 * Synchronize the dot, dash, end of element, end of character, and end
 * of word timings and ranges to new values of Morse speed, 'Farnsworth'
 * gap, or receive tolerance.
 */
void cw_sync_parameters(struct cw *c)
{
	// Do nothing if we are already synchronized with speed/gap.
	if (c->cw_in_sync) 
		return;

	// Send parameters:
	// Calculate the length of a Dot as 1200000 / speed in wpm,
	// and the length of a Dash as three Dot lengths.
	c->cw_send_dot_length = DOT_MAGIC / c->cw_send_speed;
	c->cw_send_dash_length = 3 * c->cw_send_dot_length;

	// calculate basic dot timing as number of samples of audio needed
	// this needs to fit in the trx outbuf[OUTBUFSIZE] so tx logic works
	c->symbollen = SampleRate * c->cw_send_dot_length / USECS_PER_SEC;
//	if (c->symbollen > OUTBUFSIZE) {
//FIXME
//		g_error("CW - fatal: dot length too long: %d\n", c->symbollen);
		/* abort() */
//	}

	// An end of element length is one Dot, end of character is
	// three Dots total, and end of word is seven Dots total.

//	c->cw_additional_delay  = c->cw_gap * c->cw_send_dot_length;

	/*
	 * For 'Farnsworth', there also needs to be an adjustment
	 * delay added to the end of words, otherwise the rhythm
	 * is lost on word end.  I don't know if there is an
	 * "official" value for this, but 2.33 or so times the
	 * gap is the correctly scaled value, and seems to sound
	 * okay.
	 *
	 * Thanks to Michael D. Ivey <ivey@gweezlebur.com> for
	 * identifying this in earlier versions of cwlib.
	 */
//	c->cw_adjustment_delay  = (7 * c->cw_additional_delay) / 3;

	// Receive parameters:
	c->cw_receive_speed = DOT_MAGIC / (c->cw_adaptive_receive_threshold / 2);

	// receive routines track speeds, but we put hard limits
	// on the speeds here if necessary.
	// (dot/dash threshold is 2 dots timing)
	if (c->cw_receive_speed < CW_MIN_SPEED) {
		c->cw_receive_speed = CW_MIN_SPEED;
		c->cw_adaptive_receive_threshold = 2 * DOT_MAGIC / CW_MIN_SPEED;
	}

	if (c->cw_receive_speed > CW_MAX_SPEED) {
		c->cw_receive_speed = CW_MAX_SPEED;
		c->cw_adaptive_receive_threshold = 2 * DOT_MAGIC / CW_MAX_SPEED;
	}

	// Calculate the basic receive dot and dash lengths.
	c->cw_receive_dot_length = DOT_MAGIC / c->cw_receive_speed;
	c->cw_receive_dash_length = 3 * c->cw_receive_dot_length;

	// Set the parameters in sync flag.
	c->cw_in_sync = TRUE;
}

void cw_set_bandwidth(Trx *trx)
{
	struct cw *s = (struct cw *)trx->modem;
	int iBW;
	int iTaps;
	int iSpeed;

	switch (trx->cw_bandwidth) {
		case 0:  iBW = 10;  break;
		case 1:  iBW = 20;  break;
		case 2:  iBW = 30;  break;
		case 3:  iBW = 70;  break;
		case 4:  iBW = 150; break;
		case 5:  iBW = 300; break;
		default: iBW = 70;  break;
	}

	switch (trx->cw_firtaps) {
		case 0:  iTaps = 64;  break;
		case 1:  iTaps = 96;  break;
		case 2:  iTaps = 128; break;
		case 3:  iTaps = 256; break;
		default: iTaps = 128; break;
	}

	switch (trx->cw_rxspeed) {
		case 0:  iSpeed =  3; break; //  1 WPM
		case 1:  iSpeed =  5; break; //  2 WPM
		case 3:	 iSpeed = 12; break; //  5 WPM
		case 4:  iSpeed = 24; break; // 10 WPM
		case 5:  iSpeed = 48; break; // 20 WPM
		case 6:  iSpeed = 72; break; // 40 WPM
		case 7:  iSpeed = 96; break; // 60 WPM
		default: iSpeed = 24; break; // 10 WPM
	}

	filter_free(s->fir2);
	s->fir2 = filter_init_lowpass(iTaps, 1, 0.5f * DEC_RATIO * (float)iBW / (float)SampleRate);

	filter_free(s->fir3);
	s->fir3 = filter_init_lowpass(iTaps, 1, 0.5f * DEC_RATIO * (float)iSpeed / (float)SampleRate);

	s->cw_receive_speed = iSpeed * 800 / 1000; // 80 percent of maximal speed
	trx->bandwidth = iBW << FREQ_FRAC_BITS;
}

void cw_set_txspeed(Trx *trx)
{
	struct cw *s = (struct cw *)trx->modem;
	s->cw_send_speed	= trx->cw_txspeed;
	s->symbollen		= (int)((double)SampleRate * (double)DOT_MAGIC / ((double)trx->cw_txspeed * (double)USECS_PER_SEC));
}
