/*
 *    ax25.c  --  morse code modem
 *
 *    Copyright (C) 2007 Vojtech Bubnik OK1IAK <bubnikv@seznam.cz>
 *   
 *    This modem borrowed heavily from other gmfsk modems and
 *    from rtty project by Jesus Arias EB1DIX
 *
 *    This file is part of PocketDigi.
 *
 *    PocketDigi is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    PocketDigi is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with PocketDigi; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "../main/trx.h"
#include "../misc/filter.h"
#include "../main/sndmodem.h"

#include "ax25.h"

static void ax25_txinit(Trx *trx)
{
	struct ax25 *c = (struct ax25*)trx->modem;
	c->phaseacc = 0;
}

static void ax25_free(struct ax25 *s)
{
	if (s) {
		filter_free(s->rxfilter);
		free(s);
	}
}

static void ax25_rxinit(Trx *trx)
{
	struct ax25 *s = (struct ax25*)trx->modem;
	s->phaseacc = 0;
}

static void ax25_destructor(Trx *trx)
{
	struct ax25 *s = (struct ax25*)trx->modem;

	ax25_free(s);

	trx->modem		= 0;
	trx->txinit		= 0;
	trx->rxinit		= 0;
	trx->txprocess	= 0;
	trx->rxprocess	= 0;
	trx->destructor = 0;
}

void ax25_init(Trx *trx)
{
	struct ax25 *s;

	s = (struct ax25*)malloc(sizeof(struct ax25));
	memset(s, 0, sizeof(struct ax25));

	s->bps = 1200;
//	s->bps = 300;
	s->f0  = 1200;
	s->f1  = 2200;

	{
		int    buflen = RXFILTLEN * RXFILTOVER;
		double *f0r	  = (double*)malloc(sizeof(double) * (buflen << 2));
		double *f0i	  = f0r + buflen;
		double *f1r	  = f0i + buflen;
		double *f1i	  = f1r + buflen;
		double ph0, ph1, w, tscale;
		double max, max0, max1, max2, max3;
		unsigned int i, j;

		tscale = (double)s->bps / (double)SampleRate / RXFILTOVER / WINDOWEXPAND;
        ph0 = 2.0 * M_PI * (double)s->f0 / (double)SampleRate / RXFILTOVER;
        ph1 = 2.0 * M_PI * (double)s->f1 / (double)SampleRate / RXFILTOVER;
        for (i = 0; i < RXFILTLEN * RXFILTOVER; i++) {
			w = i * tscale;
            if (w > 1)
				w = 0.0;
            else
				w = hamming(w);
            f0r[i] = w * cos(ph0 * i);
            f0i[i] = w * sin(ph0 * i);
            f1r[i] = w * cos(ph1 * i);
            f1i[i] = w * sin(ph1 * i);
        }
        // determine maximum
        max = 0;
        for (i = 0; i < RXFILTOVER; i++) {
			max0 = max1 = max2 = max3 = 0;
            for (j = i; j < RXFILTLEN * RXFILTOVER; j += RXFILTOVER) {
				max0 += fabs(f0r[j]);
                max1 += fabs(f0i[j]);
                max2 += fabs(f1r[j]);
                max3 += fabs(f1i[j]);
			}
            if (max0 > max)
                max = max0;
            if (max1 > max)
                max = max1;
            if (max2 > max)
                max = max2;
            if (max3 > max)
                max = max3;
        }
        w = 32767.0 / max;
        for (i = 0; i < RXFILTLEN * RXFILTOVER; i++) {
            f0r[i] *= w;
            f0i[i] *= w;
            f1r[i] *= w;
            f1i[i] *= w;
        }
		for (i = 0; i < RXFILTOVER; i++)
		    for (j = 0; j < RXFILTLEN; j++) {
			    s->f0r[RXFILTOVER-1-i][j] = (int)f0r[j*RXFILTOVER+i];
			    s->f0i[RXFILTOVER-1-i][j] = (int)f0i[j*RXFILTOVER+i];
			    s->f1r[RXFILTOVER-1-i][j] = (int)f1r[j*RXFILTOVER+i];
			    s->f1i[RXFILTOVER-1-i][j] = (int)f1i[j*RXFILTOVER+i];
			}
		s->rxphinc = ((SampleRate << 16) + s->bps / 2) / s->bps;
		free(f0r);
	}

	// setup function pointers for ax25 processes
	trx->modem		= s;
	trx->txinit		= ax25_txinit;
	trx->rxinit		= ax25_rxinit;
	trx->txprocess	= ax25_txprocess;
	trx->rxprocess	= ax25_rxprocess;
	trx->destructor = ax25_destructor;
}
