// txrx01.c
// by
// M. Ossmann
// 22. march 2007

// ISO15693 RFID reader
// ATmeag8
// fuse: external clock
// RS232: 19200 Bit/sec
//


#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
                  // on PORTD the following pins are us
#define RXpin   7
#define ADCpin  6
#define TRIGpin 5 // trigger out for tests: high while TX
#define INTpin  4 // interrupt active: high while interrupt-service
#define MODpin  3 // TX modulation
#define LEDpin  2 // low active LED signalizer response was ok

// PORTB
#define SSpin 2

// 13.56 MHz clock  ; divide by 128 yields: 9.439528.. us  means 105.9375 kHz

//----------------------------------------------------------------------------------------  

// RS232 TX routines

int uart_putc(unsigned char c){
   while (!(UCSRA & (1<<UDRE))); 
   UDR = c;                      
   return 0;
   }


void uart_puts (char *s){
   while (*s){   
      uart_putc(*s);
      s++;
      }
   }

void uart_puts_p(const char *progmem_s){
  register char c;
  while ( (c = pgm_read_byte(progmem_s++)) ) { uart_putc(c); }
  }


void uart_blank(){
	 uart_putc(' ') ; 
	 }

void uart_crlf(){
   uart_putc(13) ; 
   uart_putc(10) ;
   }

void decout(int k){
   uart_putc(32) ;
   uart_putc( ((k/100) % 10)+48 ) ;
   uart_putc( ((k/10) % 10)+48 ) ;
   uart_putc( (k % 10)+48 ) ;
   uart_putc(32) ;
   }


void chr_out(uint8_t v){
   uart_putc(v) ; 
   }


void hex_out(uint8_t v){
   v &=0b00001111 ;
   if (v<=9) { chr_out('0'+v) ; } 
        else { chr_out(v+'A'-10) ; } ;
   }

void byte_out(uint8_t v){
   hex_out(v>>4) ; 
   hex_out(v) ; 
   }

void word_out(uint16_t v){
   byte_out(v>>8) ; 
   byte_out(v) ; 
   }

//----------------------------------------------------------------------------------------  

// RS232 RX routines


uint8_t uart_getc(void){
   while (!(UCSRA & (1<<RXC)));  
   return UDR;                   
}

uint8_t uart_getc_echo(void){
char c ;
	c=uart_getc() ;
	uart_putc(c) ;
	return(c) ;
	}

uint8_t uart_chkc(void) {
	return (UCSRA & (1<<RXC)) ;  //  Zeichen verfuegbar ?
	}

uint8_t uart_geth(){
	uint8_t c ;
	uint8_t v ;
	v=0 ;
	uart_putc('>') ;
	while(1){
		c=uart_getc() ;
		if ( ('0'<=c) & (c<='9') ) { v=(v<<4) + (c-'0') ; 	uart_putc(c) ;}
		else if ( ('A'<=c) & (c<='F') ) { v=(v<<4) + (c-'A'+10) ; uart_putc(c) ;}
		else if ( ('a'<=c) & (c<='f') ) { v=(v<<4) + (c-'a'+10) ; uart_putc(c) ;}
		else { uart_blank() ; return(v) ; }
		} ;

}


//----------------------------------------------------------------------------------------  

                            // variables used by interrupt routine
volatile uint8_t SHIFTcnt ; // how many bits to shift, goes from HOLDcnt downto 1
volatile uint8_t HOLDcnt ;  // Bits in TXhold
volatile uint8_t TXshift ;  // pattern shift register
volatile uint8_t TXhold ;   // pattern hold register
volatile uint8_t TXhfull ;  // 1 means is full

volatile uint8_t ADCempty ;
volatile uint8_t ADCval ;


// if TXon:
//   ouput bit-pattern in TXshift in MSB first fashion
//   reload TXshift by value in TXhold
//   number of bits to output is in HOLDcnt resp SHIFTcnt
// else
//   read and retrigger ADC
//


uint8_t TXon ;

ISR(TIMER1_OVF_vect) {
   PORTD |= _BV(INTpin) ;                                    // signalize start of interrupt
   if ( TXon) {                                              // TX: transmit TXshift register
      if (TXshift & 0x80) {  PORTD |=   _BV(MODpin) ; }      // output actual bit of pattern
                   else   {  PORTD &= ~ _BV(MODpin) ; }
      TXshift = TXshift << 1 ;                               // shift to next position
      if ( 0==--SHIFTcnt) 
         { TXshift=TXhold ; TXhfull=0 ; SHIFTcnt=HOLDcnt ; } // reload shift register
      }
    else{                                                    // RX: only sample ADC
      ADCempty=0 ;                                           // ADCval is full
      ADCval=ADCH ;                                          // read value
	  }
   PORTD &= ~ _BV(INTpin) ;                                  // signalize end of interrupt
}

void TXbyte(uint8_t b) {     // routine to transmit one byte serially
  while ( TXhfull ) { } ;    // wait until TXhold is empty
  TXhold=b ;                 // load with new byte
  TXhfull=1 ;                // and signal TXhold is full
  }


uint8_t FIR0,FIR1,FIR2,FIR3 ;
uint8_t FIRsum,FIRsample ;


uint8_t getADC(uint8_t sample){

  PORTB |=   _BV(SSpin) ;           // signal active
  FIRsum=(FIR0+FIR1+FIR2+FIR3)>>2 ; // mean of last samples
  SPDR=FIRsum ;                     // out to SPI for tests

  while ( ADCempty ) { } ;          // wait for new sample
  FIR3=FIR2 ;                       // shift sample trough buffer
  FIR2=FIR1 ;
  FIR1=FIR0 ;
  FIR0=ADCval ;                     // input new sample into buffer
  ADCempty=1 ;                      // signal "fetched"

  if (sample) { 
    PORTD |= _BV(ADCpin) ;          // signalize sampling point
	FIRsample=FIRsum ; }            // sample bit in midth
   else { 
    PORTD &= ~ _BV(ADCpin) ; 
	}

  PORTB &= ~ _BV(SSpin) ; 

  SPDR=FIRsample ;                 // sample to other channel SPI for tests
  
  return FIRsum ;
  }

 
//----------------------------------------------------------------------------------------

uint16_t CRCreg ;
#define CRCpoly 0x8408

void CRCinit(){ CRCreg=0xffff ;}

void CRCadd(uint8_t b) {
	uint8_t k ;
	CRCreg ^= b ;
	for (k=0 ; k<8 ; k++ ) {
		if (CRCreg&1) { CRCreg=(CRCreg>>1) ^ CRCpoly ; }
		else          { CRCreg=(CRCreg>>1)  ; }
		}
}


//------------------------------------------------------------------------------------


uint8_t BYTEbuffer[30] ;
uint8_t BYTEcount ;

void putBYTE(uint8_t b){         // fill in one BYTE into buffer and update CRC
	BYTEbuffer[BYTEcount++]=b ;  // fill in 
	CRCadd(b) ;                  //  and update CRC
	} 

void TXreset(){                  // BYTEbuffer initialisation
	CRCinit() ;                  // CRC computation reset
	BYTEcount=0 ;                // Buffer is empty
}

void TXclose() {                 // close buffer
uint16_t finalCRCreg ;           // save store
    finalCRCreg= CRCreg^0xffff ; // complement CRC
	putBYTE(finalCRCreg) ;       // and append LSB first
	putBYTE(finalCRCreg >> 8) ;  // than MSB
	}

void TXdisplay(){
uint8_t k ;
   uart_crlf() ;
   uart_puts_p(PSTR("TX=[")) ;
   for (k=0 ; k<BYTEcount ; k++ ){
      byte_out(BYTEbuffer[k]) ; 
	  if (k!=BYTEcount-3) { uart_blank() ; } else { uart_putc(';') ; }
	  } ;
   uart_putc(']') ;
}

//---------------------------------------------------------------------------------------------------------
// bits to bytes conversion


uint8_t RXbyteBUFFER[30] ; 


uint8_t RXerrors ;
uint8_t rxSOF ;
uint8_t rxEOF ;

#define CRCerror   0b00000001
#define SOFerror   0b00000010
#define EOFerror   0b00000100
#define STARTerror 0b00001000

int8_t ones ;
uint16_t threshold,val ;

uint8_t searchSTART(){                      // search startbit
uint16_t k ;
   threshold=0 ;
   for (k=0 ; k<8 ; k++ ) {                  // sample noise level
      val=getADC(0) ; 
	  if (val>threshold) { threshold=val ; }
	  }
   ones=0 ;
   for (k=0 ; k<500 ; k++ ) { // search for starting edge for 10ms
      uint16_t val ;
	  val=getADC(0) ;
	  if (val>threshold>>1) { ones++ ; } else { ones=0 ; } 
	  if (val>threshold) { threshold=val ; } 
      if (ones==6) { threshold=threshold>>1 ; return 0 ; } // startbit found, set threshold
	  }
   RXerrors |= STARTerror ;
   return 1 ;
   }

uint8_t RXbyte(){
uint8_t b,theBYTE,k ;
   theBYTE=0 ;
   for ( k=0 ; k<8 ; k++) { 
	  int16_t half1,half2 ;
      getADC(0) ; getADC(0) ; half1=getADC(1) ; getADC(0) ;
      getADC(0) ; getADC(0) ; half2=getADC(1) ; getADC(0) ;
	  if ( half1<half2) { b=1 ; } else { b=0 ;} ;
	  theBYTE += b<<k  ; 
	  }
   return theBYTE ;
}

void CRCcheck(uint8_t Nbytes){
    uint8_t idx ;
    CRCinit() ;
    for (idx=0 ; idx<Nbytes ; idx++ ){ CRCadd(RXbyteBUFFER[idx]) ; } ;
  	if (CRCreg!=0xF0B8) { RXerrors |= CRCerror ; }
	}

uint8_t RXget(uint8_t Nbits){ // get Nbits into buffer
   uint8_t k ;
   uint8_t theBYTE ;
   uint16_t val ;
   theBYTE=0 ;
   for (k=0 ; k<Nbits ; k++){ 
      theBYTE=theBYTE<<1 ;
	  getADC(0) ;                    // sample ADC 4 times and compute sliding mean
	  getADC(0) ; 
	  val=getADC(1) ;                // sample sliding mean here
	  getADC(0) ;
      if (val>threshold) { theBYTE += 1 ; } ; // and get bit value
      }
   return theBYTE ;
   }

void TX(uint16_t delay){
uint8_t k,j,theBYTE ;  
   PORTC=15<<1 ;
   OCR1A=127 ;         // period 128
   HOLDcnt=8 ;         // standard size is BYTE
   TXon=1 ;
   for (k=0 ; k<30 ; k++ ) { TXbyte(0x00) ; }     // switch TX off: card reset
   for (k=0 ; k<190 ; k++ ) { TXbyte(0xFF) ; }     // TX on, charge card 
   PORTD |= _BV(TRIGpin) ;                     // trigger impulse for scope
   TXbyte(0b01111011) ;                        // transmit SOF
   for (k=0 ; k<BYTEcount ; k++ ){             // and emit buffer
      theBYTE=BYTEbuffer[k] ;                  // actual byte 
	  for (j=0 ; j<4 ; j++ ){                  // is four pairs	
	     switch ( theBYTE & 3 ) {              // select pair   
		    case 0: TXbyte( 0b10111111) ; break ; // one of four code
		    case 1: TXbyte( 0b11101111) ; break ; 
		    case 2: TXbyte( 0b11111011) ; break ; 
		    case 3: TXbyte( 0b11111110) ; break ; 
            }                                  // end of switch
         theBYTE=theBYTE>>2 ;                  // select next pair
		 }                                     // end of for j
      }                                        // end of for k
   TXbyte(0b11011111) ;                        // transmt EOF
   PORTD &= ~ _BV(TRIGpin) ;                   // trigger at end of send
   
    if (delay != 0 ) { 
	   for ( k=0 ; k<delay ; k++ ) { TXbyte(0xff) ; }  
       TXbyte(0b11011111) ;                    // transmt EOF
	   }
	
	// send another FF for shift-out 
	for (k=0 ; k<4 ; k++ ) { TXbyte(0xFF) ; } 
   TXon=0 ;            // RX time starts here
   OCR1A=255 ;         // sampling period=1 half-bit
   RXerrors=0 ;
// skip first samples
   PORTD |= _BV(ADCpin) ; 
   PORTD |= _BV(RXpin) ; 
   if ( searchSTART()==0 )  { PORTD &= ~ _BV(ADCpin) ; } 
   PORTD &= ~ _BV(RXpin) ; 
// scan SOF last 4 bits 
   rxSOF=RXget(4) ;
   }


void RXdisplay(uint8_t Nbytes){         // display bytes from RFID and display
uint8_t Byte_idx ;
   CRCcheck(Nbytes) ;
   if (rxSOF!=0b00001101) RXerrors |= SOFerror ;
   if (rxEOF!=0b10111000) RXerrors |= EOFerror ;
   // uart_crlf() ;
   uart_puts_p(PSTR("  RX=[")) ;
   for (Byte_idx=0 ; Byte_idx<Nbytes ; Byte_idx++ ){
      byte_out(RXbyteBUFFER[Byte_idx]) ; 
	  if (Byte_idx!=Nbytes-3) { uart_blank() ; } else { uart_putc(';') ; }
	  } ;
	  uart_putc(']') ;
   
   if (RXerrors){
      uart_puts_p(PSTR(" COMerr: ")) ; 
      if (RXerrors & STARTerror) { uart_puts_p(PSTR("STARTerror")) ;}
	  else if (RXerrors & SOFerror) { uart_puts_p(PSTR("SOFerror")) ;}
	  else if (RXerrors & CRCerror) { uart_puts_p(PSTR("CRCerror")) ;}
	  else if (RXerrors & EOFerror) { uart_puts_p(PSTR("EOFerror")) ;}
	  }

   if (RXerrors==0)  { asm volatile (" cbi 18,2 \n\t" ) ; } 	// 18=PORTD 
	    else { asm volatile (" sbi 18,2 \n\t" ) ; } 	// 18=PORTD 
	uart_crlf() ;
}

void CMDerror(){
uint8_t k ;
  for (k=1 ; k<4 ; k++ ) { RXbyteBUFFER[k]=RXbyte() ; }
    rxEOF=RXget(8) ;
  	RXdisplay(4) ;
	uart_puts("CMDerror !") ; 
}
//----------------------------------------------------------------------------------------
// supported commands

void readINVENTORY(){
uint8_t k ;
   uart_puts_p(PSTR("READinventory")) ; 
   TXreset() ;
   putBYTE(0x24) ; // inventory flags
   putBYTE(0x01) ;
   putBYTE(0x00) ;  // mask length 0 
   TXclose() ;
   TXdisplay() ;
   TX(0) ; // length of answer ( including CRC bytes )
   for (k=0 ; k<12 ; k++ ) { RXbyteBUFFER[k]=RXbyte() ; } // read bytes
   rxEOF=RXget(8) ;                                       // read EOF
   RXdisplay(12) ;                                        // display and check for errors
   if (RXerrors==0){
      uart_puts_p(PSTR("FLAGS=")) ; byte_out(RXbyteBUFFER[0]) ; uart_blank() ; 
	  uart_puts_p(PSTR("DSFID=")) ; byte_out(RXbyteBUFFER[1]) ; uart_blank() ; 
      uart_puts_p(PSTR("UID=")) ;
      for (k=9 ; k>1 ; k--){ byte_out(RXbyteBUFFER[k]) ;} 
      uart_blank() ; 
	  }
    }

void readSYSTEMinfo(){
uint8_t k ;
	uart_puts_p(PSTR("READsystemINFO")) ; 

	TXreset() ;
	putBYTE(0x00) ; // flags
	putBYTE(0x2b) ; // read system info
	TXclose() ;
	TXdisplay() ;
    TX(0) ; 
	RXbyteBUFFER[0]=RXbyte() ;
    if (RXbyteBUFFER[0]!=0) {
      CMDerror() ;
	  }
     else {
	  for (k=1 ; k<10 ; k++ ) { RXbyteBUFFER[k]=RXbyte() ; } // UID
      k=10 ; // next free entry in buffer
	  uint8_t infos=RXbyteBUFFER[1] ;
	  uint8_t DSFIDthere ;
	  DSFIDthere=0 ;
	  uint8_t AFIthere ;
	  AFIthere=0 ;
	  uint8_t MEMINFOthere ;
	  MEMINFOthere=0 ;
	   uint8_t ICINFOthere ;
	  ICINFOthere=0 ;
	  if (infos & 0b00000001) { DSFIDthere=k ; RXbyteBUFFER[k++]=RXbyte() ;  }
	  if (infos & 0b00000010) { AFIthere=k ; RXbyteBUFFER[k++]=RXbyte() ; }
	  if (infos & 0b00000100) { MEMINFOthere=k ; RXbyteBUFFER[k++]=RXbyte() ; RXbyteBUFFER[k++]=RXbyte() ;  }
      if (infos & 0b00001000) { ICINFOthere=k ; RXbyteBUFFER[k++]=RXbyte() ;  }
      RXbyteBUFFER[k++]=RXbyte() ; // CRC bytes
      RXbyteBUFFER[k++]=RXbyte() ;
	  rxEOF=RXget(8) ;
      RXdisplay(17) ;
	  if (RXerrors==0){
	     uart_puts_p(PSTR("UID=")) ;
         for (k=9 ; k>1 ; k--){ byte_out(RXbyteBUFFER[k]) ;} 
         uart_blank() ; 
	     if ( DSFIDthere !=0 ) {  
	         uart_puts_p(PSTR("DSFID=")) ; byte_out(RXbyteBUFFER[DSFIDthere]) ; uart_blank() ; 
		     }
         if ( AFIthere !=0 ) {  
	         uart_puts_p(PSTR("AFI=")) ; byte_out(RXbyteBUFFER[AFIthere]) ; uart_blank() ; 
		     }
         if ( MEMINFOthere !=0 ) {  
	         uart_puts_p(PSTR(" MEM_BLKS=")) ; byte_out(RXbyteBUFFER[MEMINFOthere]) ; uart_blank() ; 
		     uart_puts_p(PSTR(" BYTESinBLK=")) ; byte_out(RXbyteBUFFER[MEMINFOthere+1]+1) ; uart_blank() ; 
		     }
         }

	  }
  }


void readBLK(){
   uint8_t k ;
  
   uart_puts_p(PSTR("READ: enter BLOCK-NR(hex)")) ; 
   uint8_t blk_nr=uart_geth() ;
   uart_puts_p(PSTR(" block=")) ; byte_out(blk_nr) ; 
   TXreset() ;
   putBYTE(0x00+0b00000000) ; // flags
   putBYTE(0x20) ; // read page
   putBYTE(blk_nr) ; // number
   TXclose() ;
   TXdisplay() ;
   TX(0) ; 
   RXbyteBUFFER[0]=RXbyte() ;
   if (RXbyteBUFFER[0]!=0) {
      CMDerror() ;
	  }
     else{
	  for (k=1 ; k<7 ; k++ ) { RXbyteBUFFER[k]=RXbyte() ; }
	  rxEOF=RXget(8) ;
      RXdisplay(7) ;
	  uart_puts_p(PSTR("DATA=")) ;
      for (k=1 ; k<5 ; k++){ byte_out(RXbyteBUFFER[k]) ; uart_blank() ; } 
	  }
}



void write1(){
    uart_puts_p(PSTR("WRITE1: enter BLOCK-NR and CONTENTS(hex)")) ; 
    uint8_t blk_nr=uart_geth() ;
    
	TXreset() ;
	putBYTE(0x00+0b01000000) ; // flags 0b01000000 to ad for TI TAG-IT !
	putBYTE(0x21) ; // write page
	putBYTE(blk_nr) ; // number
	uint8_t k ;
	for (k=0 ; k<4 ; k++) { 
	    uint8_t v=uart_geth() ;
		putBYTE(v) ; 
		}
	TXclose() ;
	TXdisplay() ;
	TX(150) ;  
	RXbyteBUFFER[0]=RXbyte() ;
    if (RXbyteBUFFER[0]!=0) {
      CMDerror() ;
	  }
     else{
	  for (k=1 ; k<3 ; k++ ) { RXbyteBUFFER[k]=RXbyte() ; }
	  rxEOF=RXget(8) ;
      RXdisplay(3) ;
	  }
	}

void write0(){
    uart_puts_p(PSTR("WRITE0: enter BLOCK-NR and CONTENTS(hex)")) ; 
    uint8_t blk_nr=uart_geth() ;
    
	TXreset() ;
	putBYTE(0x00+0b00000000) ;  
	putBYTE(0x21) ; // write page
	putBYTE(blk_nr) ; // number
	uint8_t k ;
	for (k=0 ; k<4 ; k++) { 
	    uint8_t v=uart_geth() ;
		putBYTE(v) ; 
		}
	TXclose() ;
	TXdisplay() ;
	TX(0) ;  
	RXbyteBUFFER[0]=RXbyte() ;
    if (RXbyteBUFFER[0]!=0) {
      CMDerror() ;
	  }
     else{
	  for (k=1 ; k<3 ; k++ ) { RXbyteBUFFER[k]=RXbyte() ; }
	  rxEOF=RXget(8) ;
      RXdisplay(3) ;
	  }
	}

//----------------------------------------------------------------------------------------
// unsupported commands, only for tests

void TXfast(){
uint8_t k,j,theBYTE ;  
   PORTC=15<<1 ;
   OCR1A=127 ;         // period 128
   HOLDcnt=8 ;         // standard size is BYTE
   TXon=1 ;
   //for (k=0 ; k<30 ; k++ ) { TXbyte(0x00) ; }     // switch TX off: card reset
   for (k=0 ; k<10 ; k++ ) { TXbyte(0xFF) ; }     // TX on, charge card 
   PORTD |= _BV(TRIGpin) ;                     // trigger impulse for scope
   TXbyte(0b01111011) ;                        // transmit SOF
   for (k=0 ; k<BYTEcount ; k++ ){             // and emit buffer
      theBYTE=BYTEbuffer[k] ;                  // actual byte 
	  for (j=0 ; j<4 ; j++ ){                  // is four pairs	
	     switch ( theBYTE & 3 ) {              // select pair   
		    case 0: TXbyte( 0b10111111) ; break ; // one of four code
		    case 1: TXbyte( 0b11101111) ; break ; 
		    case 2: TXbyte( 0b11111011) ; break ; 
		    case 3: TXbyte( 0b11111110) ; break ; 
            }                                  // end of switch
         theBYTE=theBYTE>>2 ;                  // select next pair
		 }                                     // end of for j
      }                                        // end of for k
   TXbyte(0b11011111) ;                        // transmt EOF
   PORTD &= ~ _BV(TRIGpin) ;                   // trigger at end of send
   // send another FF for shift-out 
   for (k=0 ; k<150 ; k++ ) { TXbyte(0xFF) ; } 
   //for (k=0 ; k<150 ; k++ ) { TXbyte(0xFF) ; } 
   }


void TXtest(){
uint8_t k ;
	TXreset() ;
	putBYTE(0x00+0b00000000) ; // flags single subcarrier LOWspeed
	putBYTE(0x20) ; // read page
	putBYTE(10) ; // number
	TXclose() ;
	TX(0) ;
	for (k=0 ; k<7 ; k++) { RXbyte() ; }  
	rxEOF=RXget(8) ;
	RXdisplay(7) ;
}

void highSPEED(){
	TXreset() ;
	putBYTE(0x00+0b00000010) ; // flags single subcarrier HIGHspeed
	putBYTE(0x20) ; // read page
	putBYTE(0) ; // number
	TXclose() ;
	TX(0) ;  
}

void twoSUBcarriers(){
	TXreset() ;
	putBYTE(0x00+0b00000001) ; // flags 2 subcarriers, low speed
	putBYTE(0x20) ; // read page
	putBYTE(0) ; // number
	TXclose() ;
	TX(0) ;  
}

void writeDSFID(){
	uart_puts(">> ") ;
	TXreset() ;
	putBYTE(0x00) ; // flags
	putBYTE(0x29) ; // write DSFID
	putBYTE(0x13) ; 
	TXclose() ;
	TX(0) ;  
	}


// generate fast repeated command for spectrum measurement

void spectrum(){
uint8_t k ;
	TXreset() ;
//	putBYTE(0x00) ; // flags
//	putBYTE(0x2b) ; // read system info
//	TXclose() ;

	putBYTE(0x00+0b00000000) ; // flags 2 subcarriers, low speed 01
	putBYTE(0x20) ; // read page
	putBYTE(0) ; // number
	TXclose() ;
   while(1){ TXfast() ; } 
	}


//----------------------------------------------------------------------------------------

void help(){
  uart_puts_p(PSTR("COMMANDs are :")) ; uart_crlf() ; 
  uart_puts_p(PSTR("h: help                                        ")) ; uart_crlf() ; 
  uart_puts_p(PSTR("r: read block                 i: read inventory")) ; uart_crlf() ; 
  uart_puts_p(PSTR("s: read system parameters     x: continous test")) ; uart_crlf() ; 
  uart_puts_p(PSTR("w: write block (option 1)     v: write block (option 0): ")) ;uart_crlf() ; 
}

void dialog(){
  uart_crlf() ;
  help() ;
  while(1){
    uart_crlf() ;
	uart_puts("-->") ;
    char c=uart_getc() ;
	if (c=='h') { uart_crlf() ; help() ; }
	if (c=='r') { readBLK() ; }
	if (c=='w') { write1() ; }
	if (c=='v') { write0() ; }
	if (c=='i') { readINVENTORY() ; }
	if (c=='s') { readSYSTEMinfo() ; }
	if (c=='x') { while(1) { readSYSTEMinfo() ; } }
    }
  }


//----------------------------------------------------------------------------------------



void SPIchk(){
volatile uint16_t k ;
uint8_t j ;
   while(1){
      PORTB &= ~ _BV(SSpin) ; 
	  for (k=0 ; k<2 ; k++ ) { } ;
      PORTB |=   _BV(SSpin) ; 
	  SPDR=j++ ;
      for (k=0 ; k<10 ; k++ ) { } ;
    }
}


//----------------------------------------------------------------------------------------

int main(){

 // 13.56MHz/19200/16=44.14....
 // use 38400 bit/sec UBRRL=21
  UCSRB |= ( 1 << TXEN );					// UART TX einschalten
  UCSRB |= ( 1 << RXEN );					// UART RX einschalten
  UCSRC |= ( 1 << URSEL )|( 3<<UCSZ0 );	    // Asynchron 8N1
  UBRRH  = 0;                               // Highbyte ist 0
  UBRRL  = 43 ;                             // 19200 Bit/sec
 
  DDRD  =0b11111110 ;  // RXD is input, other pins are output
  PORTD =0b11111111 ;  // RXD has pull-up, other pins are HIGH

  DDRB  =0b00101100 ;  // 
  PORTB =0b00101100 ;  // 

                       // TIMER setup
  TCCR1A=0b00000011 ;  // fast PWM mode 15H with TOP=OCR1A
  TCCR1B=0b00011001	;  // clock=XTAL, no prescaler
  OCR1A=127 ;          // period 128
  TIMSK=0b00000100 ;   // Interupt enable for Timer1-Overflow
 
  DDRC  =0b00111110 ; // Bit1,2,3,4,5 R-2R DAC for tests only !
  PORTC =0b00000000 ; // no pullups
  ADMUX =0b01100000 ; // left align channel 0 with AVcc=ref

  ADCSRA=0b11100100 ; // free run mode with 1/16 clock
  SFIOR =0b11000000 ; // Trigger on T1 overflow

  // SPI setup
  SPCR=0b01010000 ;
  SPSR=0b00000000 ;
  // SPIchk() ;
  
  uart_puts_p(PSTR("START ISO15693 RFID-READER V1.0")) ;  

   sei() ;              // start all interrupts
 // spectrum() ; // for tests only

   dialog() ;

// following things only for tests !
 while(1){
// readINVENTORY() ;
   readSYSTEMinfo() ;
// readBLK(64) ;
// readBLK(2) ;
// highSPEED() ;
// twoSUBcarriers() ;
// TXtest() ;
// readBLK(1) ;
// readBLK(2) ;
// readBLK(3) ;
// uart_crlf() ;
   } ;
 return(0) ;
 }
