/*
  Copyright (C) 2004 Pavel Lajsner <ok2ucx@qsl.net>.

  This file is part of the Weather station system.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software Foundation, 
  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

  $File Name        : voicesynth.c$
  $Date             : May-5-2006$
  $Version          : 1.1.15.0$ 
  $Last Modified By : r30323$

*/

#include "VoiceSynth.h"
#include "voice.h"

#pragma CONST_SEG CONST_ROM

const WAVINDEX OnesLookUp[]=
{
  _NULA,
  _JEDNA,
  _DVA,
  _TRI,
  _CTYRI,
  _PET,
  _SEST,
  _SEDM,
  _OSM,
  _DEVET,
  _DESET,
  _JEDENACT,
  _DVANACT,
  _TRINACT,
  _CTRNACT,
  _PATNACT,
  _SESTNACT,
  _SEDMNACT,
  _OSMNACT,
  _DEVATENACT
};

const WAVINDEX TensLookUp[]=
{
  _DVACET,
  _TRICET,
  _CTYRICET,
  _PADESAT,
  _SEDESAT,
  _SEDMDESAT,
  _OSMDESAT,
  _DEVADESAT,
};

const WAVINDEX GenderLookUp[/*TWord*/TWORDCOUNT][/*TGender*/TGENDERCOUNT]=
{
        {_JEDEN,  _JEDNA, _JEDNO}, /* ONE */             /* jeden, jedna, jedno */
        {_DVA,    _DVE,   _DVE}, /* TWO */             /* dva, dve, dve */
        {_CELYCH, _CELA,  _CELE}  /* POINT */           /* celych, cela, cele */
};

const WAVINDEX CaseLookUp[/*TQuan*/TQUANCOUNT][/*TCase*/TCASECOUNT]=
{
        {_METR,     _METRY,    _METRUU,    _METRU},     /* METER */           /* 1 metr, 2 metry, 5 metru, desetin metru*/
        {_HPASCAL,  _HPASCALY, _HPASCALUU, _HPASCALU},  /* HEKTOPASCAL */     /* 1 hPascal, 2 hPascaly, 5 hPascalu, desetin hPascalu */
        {_STUPEN,   _STUPNE,   _STUPNU,    _STUPNE},    /* DEGREE */          /* 1 stupen, 2 stupne, 5 stupnu, desetin stupne */
        {_PROCENTO, _PROCENTA, _PROCENT,   _PROCENTA},  /* PERCENT */         /* 1 procento, 2 procenta, 5 procent, desetin % */
        {_VOLT, 	_VOLTY,    _VOLTUU,    _VOLTU},     /* VOLTAGE */         /* 1 volt, 2 volty, 5 voltu, desetin V % */
        {_AMPER, 	_AMPERY,   _AMPERUU,   _AMPERU},    /* CURRENT */         /* 1 amper, 2 ampery, 5 amperu, desetin A % */
        {_OHM, 		_OHMY,	   _OHMUU,     _OHMU},      /* RESISTANCE */      /* 1 ohm, 2 ohmy, 5 ohmu, desetin ohmu % */
        {_DESETINA, _DESETINY, _DESETIN,   0}           /* DECIMALS */ 
};

const TGender Gender[TQUANCOUNT]=
{
        M,   /* METER */
        M,   /* HEKTOPASCAL */
        M,   /* DEGREE */
        N    /* PERCENT */    
};

#pragma CONST_SEG DEFAULT


void TellWord(TWord Word, TGender Gender)
{
  TellTable(GenderLookUp[Word][Gender]);
}

void TellQuantity(int num, TQuan Quantity)
{
  num %= 100;   /* use only last two digits */

  if (num >= 20) /* all above 20 is only last digit */
    num %= 10;

  if (num == 1)
  { TellTable(CaseLookUp[Quantity][CASEONE]); }
  else if ((num == 2) || (num == 3) || (num == 4))
  { TellTable(CaseLookUp[Quantity][CASETWO]); }
  else
  { TellTable(CaseLookUp[Quantity][CASEMORE]); };
}


void TellNum(int num, TGender Gender)
// num -999.999 ... 0 ... 999.999
// or -32768 .. 0 .. 32767
{

//  if ((num > 999999) || (num < -999999))
//    return; //just discard any numbers above 9999999

  if (num < 0)
  { 
    TellTable(_MINUS);
    num = -num;
  }  
  else 
  {/*Tell(PLUS)*/};

// num here is always positive (or zero)

  if (num == 0)
  {
    Tell1(0, Gender);   // explicitly tell "zero" if it's zero
    return;
  };

  if (num >= 1000)
  {
    if (num < 2000)
    {
      /* DON'T SAY "ONE THOUSAND" BUT JUST "THOUSAND" (CZ)*/
    }
    else
    {
      TellNum(num / 1000, Gender);   // tell how many thousands 2,3 ... 999
    };
    TellThousand(num / 1000);  // tell "thousand(s)" word correctly
    num %= 1000;
  };

// num here is always below 1000

  if ((unsigned int)num >= 100)
  {
    if ((unsigned int)num < 200)
    { 
      /* DON'T SAY "ONE HUNDRED" & JUST "HUNDRED" (CZ)*/
    }
    else if ((unsigned int)num < 300)
    {
      TellTable(_DVE);
    }
    else
    {
      TellNum((unsigned int)(num / 100), Gender);   // tell how many hundreds
    };
    TellHundred((unsigned char)(num / 100));  // tell "hundred(s)" word correctly
    num %= 100;
  };

// num here is always below 100

  if ((unsigned char)num >= 20)
  {
    Tell10((unsigned char)num); // tell 20, 30, 40 ...
    if ((unsigned char)(num % 10) != 0)
    {
      Tell1((unsigned char)(num % 10), Gender);         // tell 1,2,3 ... 
    }
    else
    { 
      /* DON'T SAY ZERO AFTER 20, 30, 40 ... */
    };
  }
  else
  {
    if ((unsigned char)num > 0)
    {
      Tell1((unsigned char)num, Gender);           // tell 1,2 .. 19 
    }
    else
    {
      /* Don't say "ZERO" in xxx0 like numbers */
    }
  }
}

void TellNumDecQua(int num, TQuan Quantity)
{ /* num is int, e.g. 142.4 comes as 1424 */
  if (num < 0)
  { 
    TellTable(_MINUS);
    num = -num;
  }  

  if (!(abs(num) % 10)) // decimals ARE 0, tell ONLY INT
  {
    TellNumQua(num / 10, Quantity);	/* tell it as without decimal point */
  }
  else
  {
    TellNum(num / 10, F);	/* the part before decimal point sounds better if female CZ */
    TellPoint(num / 10);     /* tell decimal point the right way */
    Tell1((unsigned char)(num % 10), F);   /* tell decimal part of num */
    TellDecimal((unsigned char)(num % 10));                /* tell "desetiny" */
    TellTable(CaseLookUp[Quantity][CASEDEG]);  /* tell quantity the right way */
  }
}

void TellNumQua(int num, TQuan Quantity)
{
  TellNum(num, Gender[Quantity]);
  TellQuantity(num, Quantity);
}

void TellDecimal(unsigned char num)
// num 0..9 (19)
{
  if (num == 1)
  { TellTable(_DESETINA); } 
  else if ((num == 0) || (num == 2) || (num == 3) || (num == 4))
  { TellTable(_DESETINY); } 
  else
  { TellTable(_DESETIN); }
}

void TellPoint(unsigned int num)
// num is int 0 ... maxint
{
//  if ((num % 100) >= 20) 
//  {
//    num %= 10;
//    if (num == 0)       /* 20, 30, 40. etc. as _CELYCH! - like 10 */
//      num = 10;
//  }

  if ((num == 0) || (num == 1))
  { TellTable(_CELA); } 
  else if ((num == 2) || (num == 3) || (num == 4))
  { TellTable(_CELE); } 
  else
  { TellTable(_CELYCH); }
}

void TellThousand(unsigned int num)
// num 0...999 (?)
{
  if ((num == 0) || (num == 1))
  { TellTable(_TISIC); }
  else if ((num >= 2) && (num <= 4))
  { TellTable(_TISICE); }
  else if ((num > 4) && (num < 20))
  { TellTable(_TISIC); }
  else /* >= 20 */
  { TellThousand(num % 10); }
}

void TellHundred(unsigned char num)
// num 1..9 (19)
{
  if (num == 1)
  { TellTable(_STO); } 
  else if (num == 2)
  { TellTable(_STE); } 
  else if ((num == 3) || (num == 4))
  { TellTable(_STA); }
  else
  { TellTable(_SET); }
}

void Tell10(unsigned char num)
// num 20, 30, 40 ... (but num could be like 24!)
{
  TellTable(TensLookUp[(unsigned char)(num - 20) / 10]);
}

void Tell1(unsigned char num, TGender Gender)
// num 0 .. 19
{
  if ((num == 0) || (num > 2))
  { TellTable(OnesLookUp[num]); }
  else if (num == 1)
  { TellTable(GenderLookUp[ONE][Gender]); }
  else if (num == 2)
  { TellTable(GenderLookUp[TWO][Gender]); };
}

void TellTemp(signed int value)
{
  if (Valid.Temperature == YES)
  {
    Pause(PAUSE250MS);
    TellTable(_TEPLOTA);
    TellNumDecQua(value, DEGREE);
    TellTable(_CELSIA);
    Pause(PAUSE250MS);
  }
}

void TellPress(unsigned int value)
{
  if (Valid.Pressure == YES)
  {
    Pause(PAUSE250MS);
    TellTable(_TLAK);
    TellNumDecQua(value, HECTOPASCAL);
    Pause(PAUSE250MS);
  }
}

void TellHumid(unsigned char value)
{
  if (Valid.Humidity == YES)
  {
    Pause(PAUSE250MS);
    TellTable(_VLHKOST);
    TellNumQua(value, PERCENT);
    Pause(PAUSE250MS);
  }
}

void TellWindS(unsigned char value)
{
  if (Valid.WindSpeed == YES)
  {
    Pause(PAUSE250MS);
    TellTable(_RYCHLOST);TellTable(_VETRU);
    TellNumQua(value, METER);
    TellTable(_ZA); TellTable(_SEKUNDU);
    Pause(PAUSE250MS);
  }
}

void TellWindD(unsigned int value)
{
  if (Valid.WindDir == YES)
  {
    Pause(PAUSE250MS);
    TellTable(_SMER);TellTable(_VETRU);
    TellNumQua(value, DEGREE);
    Pause(PAUSE250MS);
  }
}


void TellTempU(signed int value)
{
    Pause(PAUSE250MS);
    TellTable(_TEPLOTA);
    TellNumDecQua(value, DEGREE);
    TellTable(_CELSIA);
    Pause(PAUSE250MS);
}


