/*
  Copyright (C) 2004 Pavel Lajsner <ok2ucx@qsl.net>.

  This file is part of the Weather station system.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software Foundation, 
  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

  $File Name        : voicesynth-de.c$
  $Date             : Jan-18-2005$
  $Version          : 1.1.2.0$ 
  $Last Modified By : r30323$

*/

#include "VoiceSynth-de.h"
#include "voice.h"

#pragma CONST_SEG CONST_ROM

const WAVINDEX OnesLookUp[]=
{
  _NULL,
  _EINS,
  _ZWEI,
  _DREI,
  _VIER,
  _FUNF,
  _SECHS,
  _SIEBEN,
  _ACHT,
  _NEUN,
  _ZEHN,
  _ELF,
  _ZWOELF,
  _DREIZEHN,
  _VIERZEHN,
  _FUNFZEHN,
  _SECHSZEHN,
  _SIEBZEHN,
  _ACHTZEHN,
  _NEUNZEHN
};

const WAVINDEX TensLookUp[]=
{
  _ZWANZIG,
  _DREISSIG,
  _VIERZIG,
  _FUNFZIG,
  _SECHZIG,
  _SIEBZIG,
  _ACHTZIG,
  _NEUNZIG
};

const WAVINDEX CaseLookUp[/*TQuan*/TQUANCOUNT]=
{
        {_METER},        /* METER */       /* 1 metr, 2 or more meters */
        {_HEKTOPASCAL},  /* HEKTOPASCAL */ /* 1 hPascal, 2 or more hPascals */
        {_GRAD},         /* DEGREE */      /* 1 degree, 2 or more degrees */
        {_PROZENT},      /* PERCENT */     /* 1, 2 or more percent */
        {_VOLT},         /* VOLTAGE */     /* 1 volt, 2 or more volts */
        {_AMPER},        /* CURRENT */     /* 1 amper, 2 or more ampers */
        {_OHM}           /* RESISTANCE */  /* 1 ohm, 2 or more ohms */
};

#pragma CONST_SEG DEFAULT


void TellQuantity(int num, TQuan Quantity)
{
  TellTable(CaseLookUp[Quantity]);
}

void TellNum(int num)
// num -999.999 ... 0 ... 999.999
// or -32768 .. 0 .. 32767
{

//  if ((num > 999999) || (num < -999999))
//    return; //just discard any numbers above 9999999

  if (num < 0)
  { 
    TellTable(_MINUS);
    num = -num;
  }  
  else 
  {/*Tell(PLUS)*/}; // don't say plus

// num here is always positive (or zero)

  if (num == 0)
  {
    Tell1(0);   // explicitly tell "zero" if it's zero
    return;
  };

  if (num >= 1000)
  {
    TellNum(num / 1000);   // tell how many thousands 2,3 ... 999
    TellThousand(num / 1000);  // tell "thousand(s)" word correctly
    num %= 1000;
  };

// num here is always below 1000

  if ((unsigned int)num >= 100)
  {
    TellNum((unsigned int)(num / 100));   // tell how many hundreds
    TellHundred((unsigned char)(num / 100));  // tell "hundred(s)" word correctly
    num %= 100;
  };

// num here is always below 100

  if ((unsigned char)num >= 20)
  {

    if ((unsigned char)(num % 10) != 0)
    {
      Tell1((unsigned char)(num % 10));         // tell 1,2,3 ... 
	  TellTable(_UND);
    }
    else
    { 
      /* DON'T SAY ZERO AFTER 20, 30, 40 ... */
    };

    Tell10((unsigned char)num); // tell 20, 30, 40 ...
  }
  else
  {
    if ((unsigned char)num > 0)
    {
      Tell1((unsigned char)num);           // tell 1,2 .. 19 
    }
    else
    {
      /* Don't say "ZERO" in xxx0 like numbers */
    }
  }
}

void TellNumDecQua(int num, TQuan Quantity)
{ /* num is int, e.g. 142.4 comes as 1424 */
  if (!(abs(num) % 10)) // decimals ARE 0, tell ONLY INT
  {
    TellNumQua(num / 10, Quantity);	/* tell it as without decimal point */
  }
  else
  {
    if (num < 0)
    { 
      TellTable(_MINUS);
      num = -num;
    }  
    TellNum(num / 10);	/* the part before decimal point */
    TellPoint(num / 10);     /* tell decimal point the right way */
    Tell1((unsigned char)(num % 10));   /* tell decimal part of num */
    TellDecimal((unsigned char)(num % 10));                /* tell "decimals" */
    TellTable(CaseLookUp[Quantity]);  /* tell quantity the right way*/
  }
}

void TellNumQua(int num, TQuan Quantity)
{
  TellNum(num);
  TellQuantity(num, Quantity);
}

void TellDecimal(unsigned char num)
// num 0..9 (19)
{
	/* simply don't say anything in German too ;-) */
}

void TellPoint(unsigned int num)
// num is int 0 ... maxint
{
  TellTable(_KOMMA);
}

void TellThousand(unsigned int num)
// num 0...999 (?)
{
  TellTable(_TAUSEND);
}

void TellHundred(unsigned char num)
// num 1..9 (19)
{
  TellTable(_HUNDERT);
}

void Tell10(unsigned char num)
// num 20, 30, 40 ... (but num could be like 24!)
{
  TellTable(TensLookUp[(unsigned char)(num - 20) / 10]);
}

void Tell1(unsigned char num)
// num 0 .. 19
{
   TellTable(OnesLookUp[num]);
}

void TellTemp(signed int value)
{
  if (Valid.Temperature == YES)
  {
    Pause(PAUSE250MS);
    TellTable(_TEMPERATUR);
    TellNumDecQua(value, DEGREE);
    Pause(PAUSE250MS);
  }
}

void TellPress(unsigned int value)
{
  if (Valid.Pressure == YES)
  {
    Pause(PAUSE250MS);
    TellTable(_LUFTDRUCK);
    TellNumDecQua(value, HECTOPASCAL);
    Pause(PAUSE250MS);
  }
}

void TellHumid(unsigned char value)
{
  if (Valid.Humidity == YES)
  {
    Pause(PAUSE250MS);
    TellTable(_FEUCHTIGKEIT);
    TellNumQua(value, PERCENT);
    Pause(PAUSE250MS);
  }
}

void TellWindS(unsigned char value)
{
  if (Valid.WindSpeed == YES)
  {
    Pause(PAUSE250MS);
    TellTable(_WIND);TellTable(_GESWCHINDIGKEIT);
    TellNumQua(value, METER);
    TellTable(_PRO); TellTable(_SEKUNDE);
    Pause(PAUSE250MS);
  }
}

void TellWindD(unsigned int value)
{
  if (Valid.WindDir == YES)
  {
    Pause(PAUSE250MS);
    TellTable(_WIND);TellTable(_RICHTUNG);
    TellNumQua(value, DEGREE);
    Pause(PAUSE250MS);
  }
}

void TellTempU(signed int value)
{
    Pause(PAUSE250MS);
    TellTable(_TEMPERATUR);
    TellNumDecQua(value, DEGREE);
    Pause(PAUSE250MS);
}


