/*
  Copyright (C) 2004 Pavel Lajsner <ok2ucx@qsl.net>.

  This file is part of the Weather station system.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software Foundation, 
  Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

  $File Name        : main.c$
  $Date             : May-5-2006$
  $Version          : 1.1.20.0$ 
  $Last Modified By : r30323$

*/

#include "map.h"
#include "voice.h"
#include "sci.h"

#ifdef SERIAL
#include "termio08.h"
#endif

#ifdef TEMPERATOR
#include "onewire.h"
#endif

#define FLASHBLANK 0xFF

#pragma DATA_SEG SHORT _DATA_ZEROPAGE

B Sys;

#ifdef SERIAL
V Valid;

unsigned int Pressure;
unsigned char Humidity;
unsigned char WindSpeed;
unsigned char WindDir;
#endif

int Temperature;
unsigned char Sample, ch, OCCnt;
Thiloword Page, Buffer;
unsigned char i, len;

unsigned char databuff[42];

#pragma DATA_SEG DEFAULT

unsigned char FlashByte (void)
{
unsigned char i = 8, retval = 0;

  do {
	SCK = 1;	/* click */
    retval <<= 1;
	if (SO)
		retval++;
    SCK = 0;	/* clock */
  } while (--i);
  return retval;
}

void SendByte(char byte)
{
unsigned char i = 8;

  do {
	SI = ((byte & 0x80) ? 1 : 0);
	byte <<= 1;
	SCK = 1;	/* click */
    SCK = 0;	/* clock */
  } while (--i);
}

//
//	D Ahi Alo len data 	84H xxxxxxxx xxxxxxxA AAAAAAAA len data data 
//	W Phi Plo			83H xxxxxxPP PPPPPPPx xxxxxxxx
//

void PageWrite(void)
{
  CS = 1;
  SCK = 0;
  SISODDR = 1;	/* output */
  CS = 0;

  SendByte(0x83); 	/* buffer write */
  SendByte((unsigned char)(Page.word>>7)); /**/
  SendByte((unsigned char)(Page.word<<1)); /**/
  SendByte(0);  	/* don't care bits */

  CS = 1;
}

void BufferWrite(void)
{
  CS = 1;
  SCK = 0;
  SISODDR = 1;	/* output */
  CS = 0;

  SendByte(0x84); 	/* buffer write */
  SendByte(0);  	/* don't care bits */
  SendByte(Buffer.b.hi); /**/
  SendByte(Buffer.b.lo); /**/

  for (i=0; i<len; i++)
	SendByte(databuff[i]);
  
  CS = 1;
}

interrupt IV_TOVF void TimerOV(void)
{
  /* flags clear */
  TSC_TOF = 0;

  if (!--OCCnt)
  {
    OCCnt = OCCNTVAL;					// reload every 3 PWM periods

    if (Sys.PWMChan == 1)
    {
      Sys.PWMChan = 0;
      TCH0H = 0;
      TCH0L = Sample;      
    }
    else
    {
      Sys.PWMChan = 1;
      TCH1H = 0;
      TCH1L = Sample;      
    };
    Sys.SampleBusy = OFF;
  };
}; 


#ifdef TEMPERATOR
interrupt IV_IRQ1 void IRQISR(void)
{
  /* flags clear */
  ISCR_ACK1 = 1;		// IS IT REALLY NEEDED??

  // just service IRQ so it will lead MCU out of STOP mode
}; 

unsigned char IRQPin(void)
{
  asm{
        BIH IrqHigh
        CLRA  
        BRA IrqLow   
      IrqHigh:
        LDA   #1
      IrqLow:
  };
}

#undef PROG
#define PROG IRQPin()
		
#endif

void VoiceInit(void)
{
  Sample = 128>>PWM_SCALE;
  OCCnt = OCCNTVAL;
  TMOD = BUS_CLK_HZ/PWM_HZ;
  TCH0 = 0;
  TSC0 = 0x2A;		// buffered OC, TOV, clear on compare
  TSC_TOF = 0;		// clear TOF
  TSC = 0x50;		// TOIE + TRST
}

#ifdef SERIAL
void SerialInit(void)
{
  TSC = 0;			// TOIE disabled
  TSC0 = 0;			// PWM off
  SCIAPIInit();
}

void SerialDecode(void)
{
  SerialInit();

  Sys.FIFOFlag = NO;
  Sys.FIFOBusy = NO;
  do {
    FIFOPut(SCIAPIRead());
  } while (Sys.FIFOFlag == NO);
}
#endif

void TellNBytes(unsigned int num)
{
  do {
    while (Sys.SampleBusy == ON)
      COPCTL = 0;
    
    Sample = FlashByte()>>PWM_SCALE;
    Sys.SampleBusy = ON;   
  } while (--num);
}

void TellNBytesMuted(unsigned int num)
{
  do {
    while (Sys.SampleBusy == ON)
      COPCTL = 0;
    
    Sample = 128>>PWM_SCALE;
    Sys.SampleBusy = ON;   
  } while (--num);
}

void FlashAddress(unsigned int page, unsigned char addr)
{
  CS = 1;
  SCK = 0;
  SISODDR = 1;	/* output */
  CS = 0;

  SendByte(0x68); 	/* continuous array read */
  SendByte((unsigned char)(page>>8)); /* first part of wav index */
  SendByte((unsigned char)(page)); /* second part of wav index */
  SendByte(addr); 	/* position of # of samples in wav */

  SendByte(0);  	/* don't care bits */
  SendByte(0);  	/* don't care bits */
  SendByte(0);  	/* don't care bits */
  SendByte(0);  	/* don't care bits */

  SCK = 1;	/* click */
  SISODDR = 0; /* input */
  SCK = 0;	/* clock */
}

void TellWav(unsigned char wav)
{
union {
	struct 
	{
		unsigned char hi;
		unsigned char lo;
	} b;
	unsigned int word;
} lenlo;

  FlashAddress((unsigned int)wav<<4, 40);

  lenlo.b.lo = FlashByte(); 
  lenlo.b.hi = FlashByte();

  (void) FlashByte();	/* skip next two bytes */
  (void) FlashByte();

  TellNBytes(lenlo.word);    

  CS = 1;
}

void TellTable(WAVINDEX index)
{
unsigned char wavv, wavv2;

  FlashAddress(0, 2*index);	/* page 0, index */
  
  wavv = FlashByte();
  wavv2 = FlashByte();

  CS = 1;

  if ((wavv != 0) && (wavv != FLASHBLANK))
    TellWav(wavv);    

  if ((wavv2 != 0) && (wavv2 != FLASHBLANK))
    TellWav(wavv2);    
}

char CheckFlashVersion(void)
{
unsigned char revision, language;

  FlashAddress(0, 0xFF);	/* page 0, index */
  
  (void)FlashByte();
  language = FlashByte();
  revision = FlashByte();

  CS = 1;

  if ((revision == FLASHBLANK) || (revision == 0))
  	return 0x11;	/* no flash revision info available */

  if ((language == FLASHBLANK) || (language == 0))
  	return 0x12;	/* no flash language info available */

  if (language == LANG)
  { /* check for language match */
	  if (revision >= REVISION)
	  	return 1; /* actual Flash is correct one (or better) */
	  else
	  	return -1; /* actual Flash is too old comparing to reqd' one */
  }
  else
  	return -2;	/* language mismatch */
}


#ifdef SERIAL
void TellAll(void)
{
  VoiceInit();

  TellTemp(Temperature);
  TellHumid(Humidity);
  TellPress(Pressure);
  TellWindS(WindSpeed);
  TellWindD(WindDir);
}
#endif

void WaitCycles(unsigned int numOfCycles)
{
  while(--numOfCycles)  // ~20T per cycle, 6.25us per cycle @ 3.2MHz BUS clock, (with COPCTL little longer)
        COPCTL = 0;
}

unsigned char CheckPROG(void) 
{
  unsigned char i = 255, res = 0;

  while (--i) 
  {
    res |= PROG;
    COPCTL = 0;
  }
  
  return res;  
}

void main(void)
{

  PROG_PUE = 1;		/* PROG pin pull-up */

#ifdef SERIAL
  CONFIG2 = 0;
#endif
#ifdef TEMPERATOR
  CONFIG2 = CONFIG2_IRQEN;			// IRQ enable
#endif

  OSCTRIM = OSCTRIM_PRESET; // trim int. oscillator

  WaitCycles(10000);  // ~60ms @ 3.2MHz BUS

  /* FLASH I/F */
  CS = 1;
  CSDDR = 1;
  SCK = 0;
  SCKDDR = 1;

  for(;;)
  {
    if (CheckPROG())
  	{
      signed char flasherrcode;
  	  asm CLI;
    
#ifdef TEMPERATOR
      CONFIG1 = CONFIG1_STOP | CONFIG1_LVIPWRD;		// COP enabled, Stop enabled
#endif /* ifdef TEMPERATOR */

      flasherrcode = CheckFlashVersion();
      
      if (flasherrcode < 0)
      { /* incorrect Flash! */
    		for (;;)
    		{
          VoiceInit();		/* init for voice generation! */
    		  FlashAddress((unsigned int)(-flasherrcode)<<(4+4), 0);
    		  TellNBytes(0xFFFF);    /* start telling Flash from somewhere */
    		  CS = 1;
    		}
      }
      else
      { /* correct Flash! */    
#ifdef SERIAL
        CONFIG1 = CONFIG1_COPD;		// COPdisable

		    PROG_PUE = 0;		/* deselect PROG pin pull-up, will be PTT from now on! */
	    
  		  // #%w,0,%i,0,%h,0,%t,%p,0,0,0,0,0\\n\\r  
        //   #-,0,-,0,-,0,-0.1,-,0,0,0,0,0
  		  for (;;)
  		  {
  		    SerialDecode();
  		    if (Sys.FIFOBusy && Sys.FIFOFlag)
  		    { /* fetch & decode data (if any) in SCI buffer */
  		      unsigned char dummy;
  		      GetPtr = 0; // pointer 0
  		         
  		      WindSpeed = (unsigned char) TERM_GetInt();
  		      Valid.WindSpeed = Sys.SCIValue; // rcvd correct value - or just minus ??
  		      
  		      dummy = (unsigned char) TERM_GetInt();                      // wind gust
  		      WindDir = (unsigned char) TERM_GetInt();
  		      Valid.WindDir = Sys.SCIValue; // rcvd correct value - or just minus ??
  		      
  		      dummy = (unsigned char)TERM_GetInt();                      // humid int
  		      Humidity = (unsigned char) TERM_GetInt();
  		      Valid.Humidity = Sys.SCIValue; // rcvd correct value - or just minus ??
  		      
  		      dummy = (unsigned char)TERM_GetInt();                      // temp int
  		      Temperature = TERM_GetInt();
  		      Valid.Temperature = Sys.SCIValue; // rcvd correct value - or just minus ??
  		      
  		      Pressure = (unsigned int) TERM_GetInt();
  		      Valid.Pressure = Sys.SCIValue; // rcvd correct value - or just minus ??            
  			  }

  			  PTT = 1;	
  	    	VoiceInit();		/* init for voice generation! */
  	    	Pause(PAUSE500MS); 	/* PTT key-up wait */
  			  TellAll();
  	    	Pause(PAUSE500MS); 	/* PTT key-down wait */
  			  PTT = 0;
  		  }
#endif /* ifdef SERIAL */
#ifdef TEMPERATOR

  			for (;;)
  			{
  				Temperature = OneWire();	// acquire the temperature from 1wire 
  				VoiceInit();		/* init for voice generation! */
  				
  				if (Temperature != ERR_ONE_WIRE_ERROR)
  				  TellTempU(Temperature);
  				else
  				  TellTable(_MINUS);		// tell MINUS in case of error

  				TSC = 0;			// TOIE disabled
  				TSC0 = 0;			// PWM off
  				
  				WIREPIN = 0;        // 1wire pin off too
  				WIREDDR = 1;        // 1wire pin out
  		        
      	  asm STOP;		// go to STOP mode, IRQ edge will take out of it!
  			}    
#endif /* ifdef TEMPERATOR */
  		}
  	}
  	else
  	{ /* Flash programming! */
      CONFIG1 = CONFIG1_COPD;	// COP disabled

  	  PROG_PUE = 0;		/* deselect PROG pin pull-up too */
  	
  	  SCIAPIInit();
  	  SCIAPISpeed /= 2;       // tweak communication speed for 19200Bd

  	  SCIAPIWrite(ACK);	// I'm here!
  	  
  	  do 
  	  {
    		switch (ch = SCIAPIRead())
    		{
    			case WRITE:
    				Page.b.hi = SCIAPIRead();
    				Page.b.lo = SCIAPIRead();

    		 		PageWrite();
    			break;
    			case DATA:
    				Buffer.b.hi = SCIAPIRead();
    				Buffer.b.lo = SCIAPIRead();
    				len = SCIAPIRead();
    				for (i=0; i<len; i++)
    					databuff[i] = SCIAPIRead();

    				BufferWrite();
    			break;
    		}
    		SCIAPIWrite(ACK);	// ACK
  	  } while (ch != QUIT);
  	} /* else/if (PROG) */
  } /* for(;;)*/
}
