/*******************************************************************************
 *
 * Motorola Inc. 2003 All rights reserved 
 * 
 ******************************************************************************* 
 * 
 * $File Name        : hc08qtqy.h$
 * $Date             : Dec-1-2003$
 * $Version          : 1.1.1.0$ 
 * $Last Modified By : r30323$
 * 
 * Description       : HC08 device mapping for 68HC908QTQY
 * 
 ******************************************************************************/ 

/*

Register Definitions

    This header file defines all of the registers using the register name exactly
    as listed in the data book. Each register name is entered in ALL CAPS. The
    registers are all unsigned volatile bytes to ensure that the compiler will
    not remove sequential write commands. This still does not guarentee that
    the compiler will execute each assignement exactly as coded, as some compilers
    will may still try to optimize volatile bytes. The compiler should be tested
    with this header file to ensure register assignments are not optimized.
    
Bitfields

    Each bit is defined using the following format:
    
        REGISTERNAME_BITNAME
    
    While this may result in redundant names in some cases, it will
    always prevent duplicate bit names. If a bit has a descriptive name
    such as FLG the descriptive name is used.
    
    If the bit does not have a descriptive name the generic terms BIT0
    through BIT7 are used. 
    
    In some cases the data book may define bits by adding a digit to the
    register name. In these cases the generic BIT0 names are used
    instead. For eample, DDRA_BIT0 is used instead of the redundant
    DDRA_DDRA0. Short Aliases are defined to provide compatibility for
    these particular cases.
    
    
Exceptions
    
Short Aliases

    Short Aliases are defined for particular cases where the customary usage
    is to use short bit names. For example, it is customary to use the short
    term PTA0 instead of PTA_BIT0. This header permits both usages.

*/

#ifndef __hc08qtqy_h
#define __hc08qtqy_h	    /* if this H file not included, include */ 

#define BIT0 0x01
#define BIT1 0x02
#define BIT2 0x04
#define BIT3 0x08
#define BIT4 0x10
#define BIT5 0x20
#define BIT6 0x40
#define BIT7 0x80

/****************************************************************************/
/* Register Mapping Structures and Macros                                   */
/****************************************************************************/

#define DBLREG(a)  (*((volatile unsigned int *)(a)))
#define REGISTER(a)  (*((volatile unsigned char *)(a)))
#define BIT(a,b)  (((vbitfield *)(a))->bit##b)

/* assumes right to left bit order, highware default */

typedef volatile struct{     
  volatile unsigned int bit0   : 1;
  volatile unsigned int bit1   : 1;
  volatile unsigned int bit2   : 1;
  volatile unsigned int bit3   : 1;
  volatile unsigned int bit4   : 1;
  volatile unsigned int bit5   : 1;   
  volatile unsigned int bit6   : 1;
  volatile unsigned int bit7   : 1;
} vbitfield;

/****************************************************************************/
/* Input Output Ports                                                       */
/****************************************************************************/
 
/* Port A Data register */
#define  PTA          REGISTER(0x00)
#define  PTA_BIT0     BIT(0x00,0)
#define  PTA_BIT1     BIT(0x00,1)
#define  PTA_BIT2     BIT(0x00,2)
#define  PTA_BIT3     BIT(0x00,3)
#define  PTA_BIT4     BIT(0x00,4)
#define  PTA_BIT5     BIT(0x00,5)
#define  PTA_AWUL     BIT(0x00,6)



/* Port B Data register */
#define  PTB          REGISTER(0x01)  
#define  PTB_BIT0     BIT(0x01,0)
#define  PTB_BIT1     BIT(0x01,1)
#define  PTB_BIT2     BIT(0x01,2)
#define  PTB_BIT3     BIT(0x01,3)
#define  PTB_BIT4     BIT(0x01,4)
#define  PTB_BIT5     BIT(0x01,5)
#define  PTB_BIT6     BIT(0x01,6)
#define  PTB_BIT7     BIT(0x01,7)


/* Port A Data Direction Register */
#define  DDRA           REGISTER(0x04)
#define  DDRA_BIT0      BIT(0x04,0)
#define  DDRA_BIT1      BIT(0x04,1)
#define  DDRA_BIT2      BIT(0x04,2)
#define  DDRA_BIT3      BIT(0x04,3)
#define  DDRA_BIT4      BIT(0x04,4)
#define  DDRA_BIT5      BIT(0x04,5)


/* Port B Data Direction Register */
#define  DDRB           REGISTER(0x05)
#define  DDRB_BIT0      BIT(0x05,0)
#define  DDRB_BIT1     BIT(0x05,1)
#define  DDRB_BIT2     BIT(0x05,2)
#define  DDRB_BIT3     BIT(0x05,3)
#define  DDRB_BIT4     BIT(0x05,4)
#define  DDRB_BIT5     BIT(0x05,5)
#define  DDRB_BIT6     BIT(0x05,6)
#define  DDRB_BIT7     BIT(0x05,7)
            
/* Port A Pull Up Enable Register */
#define  PTAPUE          REGISTER(0x0b)
#define  PTAPUE_BIT0     BIT(0x0b,0)
#define  PTAPUE_BIT1     BIT(0x0b,1)
#define  PTAPUE_BIT2     BIT(0x0b,2)
#define  PTAPUE_BIT3     BIT(0x0b,3)
#define  PTAPUE_BIT4     BIT(0x0b,4)
#define  PTAPUE_BIT5     BIT(0x0b,5)
#define  PTAPUE_OSC2EN   BIT(0x0b,7)

/* Port B Pull Up Enable Register */
#define  PTBPUE          REGISTER(0x0c)
#define  PTBPUE_BIT0     BIT(0x0c,0)
#define  PTBPUE_BIT1     BIT(0x0c,1)
#define  PTBPUE_BIT2     BIT(0x0c,2)
#define  PTBPUE_BIT3     BIT(0x0c,3)
#define  PTBPUE_BIT4     BIT(0x0c,4)
#define  PTBPUE_BIT5     BIT(0x0c,5)
#define  PTBPUE_BIT6     BIT(0x0c,6)
#define  PTBPUE_BIT7     BIT(0x0c,7)



/****************************************************************************/
/* Keyboard Registers                                                       */
/****************************************************************************/

/* Keyboard Status and Control Register */
#define  KBSCR          REGISTER(0x1a)  
#define  KBSCR_MODEK    BIT(0x1a,0)
#define  KBSCR_IMASKK   BIT(0x1a,1)
#define  KBSCR_ACKK     BIT(0x1a,2)
#define  KBSCR_KEYF     BIT(0x1a,3)
 
/* Keyboard Interrupt Enable Register */
#define  KBIER           REGISTER(0x1b)  
#define  KBIER_KBIE0     BIT(0x1b,0)
#define  KBIER_KBIE1     BIT(0x1b,1)
#define  KBIER_KBIE2     BIT(0x1b,2)
#define  KBIER_KBIE3     BIT(0x1b,3)
#define  KBIER_KBIE4     BIT(0x1b,4)
#define  KBIER_KBIE5     BIT(0x1b,5)
#define  KBIER_AWUIE     BIT(0x1b,6)


/****************************************************************************/
/* IRQ Status and Control Register                                          */
/****************************************************************************/

#define  ISCR           REGISTER(0x1d)  
#define  ISCR_MODE1     BIT(0x1d,0)
#define  ISCR_IMASK1    BIT(0x1d,1)
#define  ISCR_ACK1      BIT(0x1d,2)
#define  ISCR_IRQF1     BIT(0x1d,3)


/****************************************************************************/
/* Configuration Write-Once Registers                                       */
/*                                                                          */
/* note: bit fields or bit manipulation is no permitted on write once reg   */
/*                                                                          */
/****************************************************************************/

#define  CONFIG2    REGISTER(0x1e)  
#define  CONFIG1    REGISTER(0x1f)  

#define CONFIG2_IRQPUD  BIT7
#define CONFIG2_IRQEN   BIT6
#define CONFIG2_OSCOPT1 BIT4
#define CONFIG2_OSCOPT0 BIT3
#define CONFIG2_RSTEN   BIT0

#define CONFIG1_COPRS   BIT7
#define CONFIG1_LVISTOP BIT6
#define CONFIG1_LVIRSTD BIT5
#define CONFIG1_LVIPWRD BIT4
#define CONFIG1_LVI5OR3 BIT3
#define CONFIG1_SSREC   BIT2
#define CONFIG1_STOP    BIT1
#define CONFIG1_COPD    BIT0

/****************************************************************************/
/* Timer Registers                                                        */
/****************************************************************************/

/* Timer Status and Control Register */
#define  TSC            REGISTER(0x20)  
#define  TSC_PS0        BIT(0x20,0)
#define  TSC_PS1        BIT(0x20,1)
#define  TSC_PS2        BIT(0x20,2)
#define  TSC_TRST       BIT(0x20,4)
#define  TSC_TSTOP      BIT(0x20,5)
#define  TSC_TOIE       BIT(0x20,6)
#define  TSC_TOF        BIT(0x20,7)
#define  M_TSC_PS0      BIT0
#define  M_TSC_PS1      BIT1
#define  M_TSC_PS2      BIT2
#define  M_TSC_TRST     BIT4
#define  M_TSC_TSTOP    BIT5
#define  M_TSC_TOIE     BIT6
#define  M_TSC_TOF      BIT7

/* Timer Counter Register */
#define	 TCNT	DBLREG(0x21)

#define  TCNTH  REGISTER(0x21)  
#define  TCNTL  REGISTER(0x22)  

/* Timer Modulo Register  */

#define  TMOD	DBLREG(0x23)  

#define  TMODH  REGISTER(0x23)  
#define  TMODL  REGISTER(0x24)  

/* Timer Status and Control Register Channel 0 */
#define  TSC0   REGISTER(0x25)  
#define  TSC0_CH0MAX  BIT(0x25,0)
#define  TSC0_TOV0    BIT(0x25,1)
#define  TSC0_ELS0A   BIT(0x25,2)
#define  TSC0_ELS0B   BIT(0x25,3)
#define  TSC0_MS0A    BIT(0x25,4)
#define  TSC0_MS0B    BIT(0x25,5)
#define  TSC0_CH0IE   BIT(0x25,6)
#define  TSC0_CH0F    BIT(0x25,7)
#define  M_TSC0_CH0MAX  BIT0
#define  M_TSC0_TOV0    BIT1
#define  M_TSC0_ELS0A   BIT2
#define  M_TSC0_ELS0B   BIT3
#define  M_TSC0_MS0A    BIT4
#define  M_TSC0_MS0B    BIT5
#define  M_TSC0_CH0IE   BIT6
#define  M_TSC0_CH0F    BIT7

/* Timer Channel 0 Register */
#define  TCH0   DBLREG(0x26)  

#define  TCH0H  REGISTER(0x26)  
#define  TCH0L  REGISTER(0x27)  

/* Timer Status and Control Register Channel 1 */
#define  TSC1   REGISTER(0x28)  
#define  TSC1_CH1MAX  BIT(0x28,0)
#define  TSC1_TOV1    BIT(0x28,1)
#define  TSC1_ELS1A   BIT(0x28,2)
#define  TSC1_ELS1B   BIT(0x28,3)
#define  TSC1_MS1A    BIT(0x28,4)
#define  TSC1_CH1IE   BIT(0x28,6)
#define  TSC1_CH1F    BIT(0x28,7)
#define  M_TSC1_CH1MAX  BIT0
#define  M_TSC1_TOV1    BIT1
#define  M_TSC1_ELS1A   BIT2
#define  M_TSC1_ELS1B   BIT3
#define  M_TSC1_MS1A    BIT4
#define  M_TSC1_CH1IE   BIT6
#define  M_TSC1_CH1F    BIT7

/* Timer Channel 1 Register */
#define  TCH1   DBLREG(0x29)  

#define  TCH1H  REGISTER(0x29)  
#define  TCH1L  REGISTER(0x2a)  


/****************************************************************************/
/* Internal Clock Generator Module Registers                                */
/****************************************************************************/

/* ICG Control Register */
#define  OSCSTAT        REGISTER(0x36) 
#define  OSCSTAT_ECGST  BIT(0x36,0)
#define  OSCSTAT_ECGON  BIT(0x36,1)

#define ICGCR       OSCSTAT             /* KX aliases ;) */
#define ICGCR_ECGS  OSCSTAT_ECGST
#define ICGCR_ECGON OSCSTAT_ECGON

/* ICG Trim Register */
#define  OSCTRIM        REGISTER(0x38) 
#define ICGTR OSCTRIM
#define  OSCTRIM_PRESET        REGISTER(0xFFC0) 

/****************************************************************************/
/* Analogue To Digital Converter Registers                                  */
/****************************************************************************/

/* A/D Status and Control Register */
#define  ADSCR          REGISTER(0x3c) 
#define  ADSCR_ADCH0    BIT(0x3c,0)
#define  ADSCR_ADCH1    BIT(0x3c,1)
#define  ADSCR_ADCH2    BIT(0x3c,2)
#define  ADSCR_ADCH3    BIT(0x3c,3)
#define  ADSCR_ADCH4    BIT(0x3c,4)
#define  ADSCR_ADCO     BIT(0x3c,5)
#define  ADSCR_AIEN     BIT(0x3c,6)
#define  ADSCR_COCO     BIT(0x3c,7)
#define  M_ADSCR_ADCO   BIT5
#define  M_ADSCR_AIEN   BIT6
#define  M_ADSCR_COCO   BIT7

/* A/D-Data Register */
#define  ADR            REGISTER(0x3e)  

/* A/D Input Clock Register */
#define  ADCLK          REGISTER(0x3f)  
#define  ADCLK_ADIV0    BIT(0x3f,5)
#define  ADCLK_ADIV1    BIT(0x3f,6)
#define  ADCLK_ADIV2    BIT(0x3f,7)


/****************************************************************************/
/* System Integration Module Registers                                      */
/****************************************************************************/

/* Break Status Register */
#define  BSR        REGISTER(0xFE00) 
#define  BSR_SBSW   BIT(0xFE00,1)

/* SIM Reset Status Register */
#define  SRSR         REGISTER(0xFE01) 
#define  SRSR_LVI     BIT(0xFE01,1)
#define  SRSR_MENRST  BIT(0xFE01,2)
#define  SRSR_ILAD    BIT(0xFE01,3)
#define  SRSR_ILOP    BIT(0xFE01,4)
#define  SRSR_COP     BIT(0xFE01,5)
#define  SRSR_POR     BIT(0xFE01,7)


/****************************************************************************/
/* Low voltage Inhibit Register                                             */
/****************************************************************************/

/* LVI Status Register */
#define  LVISR          REGISTER(0xFE0c)  
#define  LVISR_LVIOUT   BIT(0xFE0c,7)
 
/****************************************************************************/
/* Flash Registers                                                          */
/****************************************************************************/

/* Flash Control Register 1 */
#define  FLCR         REGISTER(0xFE08)  
#define  FLCR_PGM     BIT(0xFE08,0)
#define  FLCR_ERASE   BIT(0xFE08,1)
#define  FLCR_MASS    BIT(0xFE08,2)
#define  FLCR_HVEN    BIT(0xFE08,3)

#define  M_FLCR_PGM     BIT0
#define  M_FLCR_ERASE   BIT1
#define  M_FLCR_MASS    BIT2
#define  M_FLCR_HVEN    BIT3


/* Flash Block Protect Register 1 */
#define  FLBPR        REGISTER(0xFFBE)  
#define  FLBPR_BPR0   BIT(0xFFBE,0)
#define  FLBPR_BPR1   BIT(0xFFBE,1)
#define  FLBPR_BPR2   BIT(0xFFBE,2)
#define  FLBPR_BPR3   BIT(0xFFBE,3)


/****************************************************************************/
/* COP Control Registers                                                    */
/****************************************************************************/

#define  COPCTL REGISTER(0xFFFF) 


/****************************************************************************/
/* Short Aliases                                                    */
/****************************************************************************/

/* Port A short aliases */
#define PTA0    PTA_BIT0
#define PTA1    PTA_BIT1
#define PTA2    PTA_BIT2
#define PTA3    PTA_BIT3
#define PTA4    PTA_BIT4
#define PTA5    PTA_BIT5

/* Port B short aliases */
#define PTB0    PTB_BIT0
#define PTB1    PTB_BIT1
#define PTB2    PTB_BIT2
#define PTB3    PTB_BIT3
#define PTB4    PTB_BIT4
#define PTB5    PTB_BIT5
#define PTB6    PTB_BIT6
#define PTB7    PTB_BIT7 

/****************************************************************************/
/* Interrupt Vectors                                                        */
/****************************************************************************/

#define IV_ADC        16    /* Analogue To Digital Converter   */
#define IV_KBRD       15    /* Keboard                         */
#define IV_TOVF       6     /* Timer A Channel 1               */
#define IV_TCH1       5     /* Timer A Channel 0               */
#define IV_TCHO       4     /* Modulo Timer                    */
#define IV_IRQ1       2     /* Interrupt Request1              */
#define IV_RESET      0     /* Reset                           */

#endif

